"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SponsorBlock = void 0;
const axios_1 = __importDefault(require("axios"));
const crypto_1 = __importDefault(require("crypto"));
const index_1 = require("../../index");
const utils_1 = require("../utils");
/**
 * SponsorBlock API class, to be constructed with a userID.
 *
 * @description Complete API documentation can be found {@link https://wiki.sponsor.ajay.app/index.php/API_Docs here}.
 * Please review the {@link https://gist.github.com/ajayyy/4b27dfc66e33941a45aeaadccb51de71 attriution template}
 * to abide the {@link https://github.com/ajayyy/SponsorBlock/wiki/Database-and-API-License license}.
 */
class SponsorBlock {
    constructor(userID, options = {}) {
        var _a;
        this.userID = userID;
        this.options = options;
        if ((_a = options.baseURL) === null || _a === void 0 ? void 0 : _a.endsWith('/'))
            options.baseURL = options.baseURL.slice(0, -1);
        this.options = Object.assign(Object.assign({}, index_1.defaultOptions), options);
    }
    getSegments(video, categories = ['sponsor'], ...requiredSegments) {
        return __awaiter(this, void 0, void 0, function* () {
            let videoID = utils_1.resolveVideo(video);
            let res = yield axios_1.default.get('/api/skipSegments', { params: Object.assign({ videoID, service: this.options.service, categories: JSON.stringify(categories) }, (requiredSegments.length && { requiredSegments })), baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            let data = res.data;
            let segments = data.map(({ UUID, segment, category, videoDuration }) => {
                return { UUID, startTime: segment[0], endTime: segment[1], category, videoDuration };
            });
            return segments;
        });
    }
    postSegments(video, ...segments) {
        return __awaiter(this, void 0, void 0, function* () {
            let videoID = utils_1.resolveVideo(video);
            let userAgent = this.options.userAgent;
            let dbSegments = segments.map((segment) => {
                // turn segments to objects the api accepts
                let { startTime, endTime, category } = segment;
                return { segment: [startTime, endTime], category };
            });
            let res = yield axios_1.default.post('/api/skipSegments', { videoID, userID: this.userID, segments: dbSegments, userAgent }, { baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            // returns nothing (status code 200)
        });
    }
    getSegmentsPrivately(video, categories = ['sponsor'], ...requiredSegments) {
        return __awaiter(this, void 0, void 0, function* () {
            let videoID = utils_1.resolveVideo(video);
            let hashPrefix = crypto_1.default.createHash('sha256').update(videoID).digest('hex').substr(0, this.options.hashPrefixLength);
            let query = `?service=${this.options.service}&categories=${JSON.stringify(categories)}`;
            if (requiredSegments.length > 0) {
                query += `&requiredSegments=${JSON.stringify(requiredSegments)}`;
            }
            let res = yield axios_1.default.get(`/api/skipSegments/${hashPrefix}`, { baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            let filtered = res.data.find((video) => video.videoID === videoID);
            if (!filtered) {
                throw new Error('[SponsorBlock] Not found within returned videos');
            }
            let segments = filtered.segments.map((val) => {
                return { UUID: val.UUID, startTime: val.segment[0], endTime: val.segment[1], category: val.category, videoDuration: val.videoDuration };
            });
            return segments;
        });
    }
    vote(segment, type) {
        return __awaiter(this, void 0, void 0, function* () {
            let UUID = utils_1.resolveSegment(segment);
            type = type === 'down' ? 0 : type === 'up' ? 1 : type === 'undo' ? 20 : type;
            let res = yield axios_1.default.get('/api/voteOnSponsorTime', { params: { UUID, userID: this.userID, type }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            // returns nothing (status code 200)
        });
    }
    voteCategory(segment, category) {
        return __awaiter(this, void 0, void 0, function* () {
            let UUID = utils_1.resolveSegment(segment);
            let res = yield axios_1.default.get('/api/voteOnSponsorTime', { params: { UUID, userID: this.userID, category }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            // returns nothing (status code 200)
        });
    }
    viewed(segment) {
        return __awaiter(this, void 0, void 0, function* () {
            let UUID = utils_1.resolveSegment(segment);
            let res = yield axios_1.default.get('/api/viewedVideoSponsorTime', { params: { UUID }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            // returns nothing (status code 200)
        });
    }
    getViews() {
        return __awaiter(this, void 0, void 0, function* () {
            let res = yield axios_1.default.get('/api/getViewsForUser', { params: { userID: this.userID }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            let data = res.data;
            return data.viewCount;
        });
    }
    getTimeSaved() {
        return __awaiter(this, void 0, void 0, function* () {
            let res = yield axios_1.default.get('/api/getSavedTimeForUser', { params: { userID: this.userID }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            let data = res.data;
            return data.timeSaved;
        });
    }
    setUsername(username) {
        return __awaiter(this, void 0, void 0, function* () {
            let res = yield axios_1.default.get('/api/setUsername', { params: { userID: this.userID, username }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            // returns nothing (status code 200)
        });
    }
    getUsername() {
        return __awaiter(this, void 0, void 0, function* () {
            let res = yield axios_1.default.get('/api/getUsername', { params: { userID: this.userID }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            let data = res.data;
            return data.userName;
        });
    }
    getTopUsers(sortType) {
        return __awaiter(this, void 0, void 0, function* () {
            sortType = sortType === 'minutesSaved' ? 0 : sortType === 'viewCount' ? 1 : sortType === 'totalSubmissions' ? 2 : sortType;
            let res = yield axios_1.default.get('/api/getTopUsers', { params: { sortType }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            return utils_1.dbuserStatsToUserStats(res.data);
        });
    }
    getOverallStats() {
        return __awaiter(this, void 0, void 0, function* () {
            let res = yield axios_1.default.get('/api/getTotalStats', { baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            return res.data;
        });
    }
    getDaysSaved() {
        return __awaiter(this, void 0, void 0, function* () {
            let res = yield axios_1.default.get('/api/getDaysSavedFormatted', { baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            let data = res.data;
            return parseFloat(data.daysSaved);
        });
    }
    isVIP() {
        return __awaiter(this, void 0, void 0, function* () {
            let res = yield axios_1.default.get('/api/isUserVIP', { params: { userID: this.userID }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            return res.data.vip;
        });
    }
    getHashedUserID() {
        if (this.hashedUserID)
            return this.hashedUserID;
        let value = this.userID;
        for (let i = 0; i < 5000; i++) {
            value = crypto_1.default.createHash('sha256').update(value).digest('hex');
        }
        return (this.hashedUserID = value);
    }
    getSegmentInfo(segments) {
        return __awaiter(this, void 0, void 0, function* () {
            let UUIDs = segments.map((segment) => utils_1.resolveSegment(segment));
            let res = yield axios_1.default.get('/api/segmentInfo', { params: { UUIDs: JSON.stringify(UUIDs) }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            return res.data;
        });
    }
    getUserID(username, exact = false) {
        return __awaiter(this, void 0, void 0, function* () {
            let res = yield axios_1.default.get('/api/userID', { params: { username }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            return res.data;
        });
    }
    getLockCategories(video) {
        return __awaiter(this, void 0, void 0, function* () {
            let videoID = utils_1.resolveVideo(video);
            let res = yield axios_1.default.get('/api/lockCategories', { params: { videoID }, baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            return res.data.categories;
        });
    }
    getLockCategoriesPrivately(video) {
        return __awaiter(this, void 0, void 0, function* () {
            let videoID = utils_1.resolveVideo(video);
            let hashPrefix = crypto_1.default.createHash('sha256').update(videoID).digest('hex').substr(0, this.options.hashPrefixLength);
            let res = yield axios_1.default.get(`/api/skipSegments/${hashPrefix}`, { baseURL: this.options.baseURL, validateStatus: null });
            utils_1.statusCheck(res);
            let filtered = res.data.find((video) => video.videoID === videoID);
            if (!filtered) {
                throw new Error('[SponsorBlock] Not found within returned videos');
            }
            return filtered.categories;
        });
    }
}
exports.SponsorBlock = SponsorBlock;
