/**
 * Type of class definition.
 * It accepts function identified as constructor (FunctionConstructor) or the class object.
 */
export declare type ClassDefinition<T> = (new (...args: any[]) => T) | FunctionConstructor;
/**
 * Type representing the transformed object data.
 * It must be a primitive (`string`, `number`, `boolean`), an `array` of primitive, or an `object` containing primitive.
 *
 * _(It can be an array containing object of primitive. Object can also contain a such array)_
 */
export declare type PlainType = string | number | boolean | object | any[];
/**
 * Serialized a data
 */
export declare function serialize(data: any): string;
/**
 * Deserialize a string
 * @param text The serialized text
 * @param [allowedClasses] List of allowed classes for deserialization. It will be merged with the global allowed class list
 */
export declare function deserialize(text: string, allowedClasses?: Record<string, ClassDefinition<any>> | undefined): any;
/**
 * Get the list of classes found during serialization
 * @param clear reset the list of classes found during serialization after reading it
 */
export declare function getCollectedClasses(clear?: boolean): Array<ClassDefinition<any>>;
/**
 * Clear the list of classes found during serialization
 */
export declare function resetCollectedClasses(): void;
/**
 * Defined the list of allowed classes for deserialization.
 *
 * ---
 *
 * By default, this list contains:
 * - `Error`
 * - `EvalError`
 * - `RangeError`
 * - `AggregateError`
 * - `ReferenceError`
 * - `SyntaxError`
 * - `TypeError`
 * - `URIError`
 * - `Int8Array`
 * - `Uint8Array`
 * - `Uint8ClampedArray`
 * - `Int16Array`
 * - `Uint16Array`
 * - `Int32Array`
 * - `Uint32Array`
 * - `Float32Array`
 * - `Float64Array`
 * - `BigInt64Array`
 * - `BigUint64Array`
 *
 * @param classes List of classes that are allowed for deserialization
 * @param append If `true` the provided classes will be added to the existing list, if `false` (default) the provided list will replace.
 */
export declare function setGlobalAllowedClasses(classes: Record<string, ClassDefinition<any>> | Array<ClassDefinition<any>>, append?: boolean): void;
/**
 * Add a class in the list of allowed classes for deserialization.
 */
export declare function addGlobalAllowedClass(classConstructor: ClassDefinition<any>): void;
/**
 * Add a class handler
 *
 * ---
 *
 * **toPlain** is a function to transform an instance into a plain Javascript object.
 * - Its first parameter is the instant to transform, the second a function to transform any data to a plain Javascript object, a primitive or an array
 * - You can return `undefined` to let the default transformation occurs or an object
 *
 * **fromPlain** is a function to transform a plain Javascript object into a class instance (reverse function of `toPlain`)
 * - Its first parameter is the plain object, the second is a function to transform any data to a primitive, an array, an object or a class instance
 * - You can return `undefined` to let the default transformation occurs or final transformed object
 *
 * @param classname The classname to handle
 * @param toPlain The function to transform the class instance (1rst param) to a JS plain object. The second parameter is the transformation process function. If undefined is return the normal transformation is used
 * @param fromPlain The function to transform plain js object into a class instance. The second parameter is the transformation process function. If undefined is return the normal transformation is used
 */
export declare function addClassHandler<Instance = any, Plain extends object = object>(classname: string, toPlain: (source: Instance, next: (data: any) => PlainType) => Plain | undefined, fromPlain: (source: Plain, next: (data: any) => any) => Instance | undefined): void;
