import { p as patchYoutubeJs } from "./youtubejs.js";
import { Capacitor } from "@capacitor/core";
import { g as get } from "./index3.js";
import { g as instanceStore, h as interfaceRegionStore, b as authStore, j as playerYouTubeJsAlways, k as playerYouTubeJsFallback, r as returnYTDislikesInstanceStore, l as synciousInstanceStore } from "./store.js";
function buildPath(path) {
  return new URL(`${get(instanceStore)}/api/v1/${path}`);
}
function setRegion(url) {
  const region = get(interfaceRegionStore);
  if (region) {
    url.searchParams.set("region", region);
  }
  return url;
}
async function fetchErrorHandle(response) {
  if (!response.ok) {
    let message = "Internal error";
    try {
      const json = await response.json();
      message = "errorBacktrace" in json ? json.errorBacktrace : json.error;
    } catch {
    }
    throw Error(message);
  }
  return response;
}
function buildAuthHeaders() {
  const authToken = get(authStore)?.token ?? "";
  if (authToken.startsWith("SID=")) {
    return { headers: { __sid_auth: authToken } };
  } else {
    return { headers: { Authorization: `Bearer ${authToken}` } };
  }
}
async function getPopular(fetchOptions) {
  const resp = await fetchErrorHandle(await fetch(buildPath("popular"), fetchOptions));
  return await resp.json();
}
async function getResolveUrl(url) {
  const resp = await fetchErrorHandle(await fetch(`${buildPath("resolveurl")}?url=${url}`));
  return await resp.json();
}
async function getVideo(videoId, local = false, fetchOptions) {
  if (get(playerYouTubeJsAlways) && Capacitor.isNativePlatform()) {
    return await patchYoutubeJs(videoId);
  }
  const resp = await fetch(setRegion(buildPath(`videos/${videoId}?local=${local}`)), fetchOptions);
  if (!resp.ok && get(playerYouTubeJsFallback) && Capacitor.isNativePlatform()) {
    return await patchYoutubeJs(videoId);
  } else {
    await fetchErrorHandle(resp);
  }
  return await resp.json();
}
async function getDislikes(videoId, fetchOptions) {
  const resp = await fetchErrorHandle(
    await fetch(`${get(returnYTDislikesInstanceStore)}/votes?videoId=${videoId}`, fetchOptions)
  );
  return await resp.json();
}
async function getComments(videoId, parameters, fetchOptions) {
  const path = buildPath(`comments/${videoId}`);
  path.search = new URLSearchParams(parameters).toString();
  const resp = await fetchErrorHandle(await fetch(path, fetchOptions));
  return await resp.json();
}
async function getChannel(channelId, fetchOptions) {
  const resp = await fetchErrorHandle(
    await fetch(buildPath(`channels/${channelId}`), fetchOptions)
  );
  return await resp.json();
}
async function getChannelContent(channelId, parameters, fetchOptions) {
  if (typeof parameters.type === "undefined") parameters.type = "videos";
  const url = buildPath(`channels/${channelId}/${parameters.type}`);
  if (typeof parameters.continuation !== "undefined")
    url.searchParams.set("continuation", parameters.continuation);
  if (typeof parameters.sortBy !== "undefined") url.searchParams.set("sort_by", parameters.sortBy);
  const resp = await fetchErrorHandle(await fetch(url.toString(), fetchOptions));
  return await resp.json();
}
async function getHashtag(tag, page = 0) {
  const resp = await fetchErrorHandle(await fetch(buildPath(`hashtag/${tag}?page=${page}`)));
  return await resp.json();
}
async function getSearch(search, options, fetchOptions) {
  if (typeof options.sort_by === "undefined") {
    options.sort_by = "relevance";
  }
  if (typeof options.type === "undefined") {
    options.type = "all";
  }
  if (typeof options.page === "undefined") {
    options.page = "1";
  }
  const path = buildPath("search");
  path.search = new URLSearchParams({ ...options, q: search }).toString();
  const resp = await fetchErrorHandle(await fetch(setRegion(path), fetchOptions));
  return await resp.json();
}
async function getFeed(maxResults, page, fetchOptions = {}) {
  const path = buildPath("auth/feed");
  path.search = new URLSearchParams({
    max_results: maxResults.toString(),
    page: page.toString()
  }).toString();
  const resp = await fetchErrorHandle(
    await fetch(path, { ...buildAuthHeaders(), ...fetchOptions })
  );
  return await resp.json();
}
async function getSubscriptions(fetchOptions = {}) {
  const resp = await fetchErrorHandle(
    await fetch(buildPath("auth/subscriptions"), { ...buildAuthHeaders(), ...fetchOptions })
  );
  return await resp.json();
}
async function postHistory(videoId, fetchOptions = {}) {
  await fetchErrorHandle(
    await fetch(buildPath(`auth/history/${videoId}`), {
      method: "POST",
      ...buildAuthHeaders(),
      ...fetchOptions
    })
  );
}
async function getPlaylist(playlistId, page = 1, fetchOptions = {}) {
  let resp;
  if (get(authStore)) {
    resp = await fetch(buildPath(`auth/playlists/${playlistId}?page=${page}`), {
      ...buildAuthHeaders(),
      ...fetchOptions
    });
  } else {
    resp = await fetch(buildPath(`playlists/${playlistId}?page=${page}`), fetchOptions);
  }
  await fetchErrorHandle(resp);
  return await resp.json();
}
async function getPersonalPlaylists(fetchOptions = {}) {
  const resp = await fetchErrorHandle(
    await fetch(buildPath("auth/playlists"), { ...buildAuthHeaders(), ...fetchOptions })
  );
  return await resp.json();
}
function buildApiExtendedAuthHeaders() {
  const authToken = get(authStore)?.token ?? "";
  return { headers: { Authorization: `Bearer ${authToken.replace("SID=", "")}` } };
}
async function saveVideoProgress(videoId, time, fetchOptions = {}) {
  const headers = buildApiExtendedAuthHeaders();
  headers["headers"]["Content-type"] = "application/json";
  await fetchErrorHandle(
    await fetch(`${get(synciousInstanceStore)}/video/${encodeURIComponent(videoId)}`, {
      ...headers,
      method: "POST",
      body: JSON.stringify({
        time
      }),
      ...fetchOptions
    })
  );
}
async function deleteVideoProgress(videoId, fetchOptions = {}) {
  await fetchErrorHandle(
    await fetch(`${get(synciousInstanceStore)}/video/${videoId}`, {
      method: "DELETE",
      ...buildApiExtendedAuthHeaders(),
      ...fetchOptions
    })
  );
}
export {
  getChannelContent as a,
  getChannel as b,
  getHashtag as c,
  getPersonalPlaylists as d,
  getSearch as e,
  getFeed as f,
  getPopular as g,
  getSubscriptions as h,
  getPlaylist as i,
  getVideo as j,
  getComments as k,
  getDislikes as l,
  getResolveUrl as m,
  deleteVideoProgress as n,
  postHistory as p,
  saveVideoProgress as s
};
