/*
 * Copyright (c) 2011 Kurt Aaholst <kaaholst@gmail.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package uk.org.ngo.squeezer.model;

import android.net.Uri;
import android.os.Parcelable;

import androidx.annotation.NonNull;

import java.util.Map;

import uk.org.ngo.squeezer.Util;

/**
 * Base class for slimserver data.
 * <p>
 * Has an id, a getName() method and a few helper methods for parsing records from slimserver.
 *
 * @author Kurt Aaholst
 */
public abstract class Item implements Parcelable {
    private String id;

    public void setId(String id) {
        this.id = id;
    }

    public String getId() {
        return id;
    }

    public abstract String getName();

    @Override
    public int hashCode() {
        return (getId() != null ? getId().hashCode() : 0);
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }

        if (o == null) {
            return false;
        }

        if (o.getClass() != getClass()) {
            // There is no guarantee that slimserver items have globally unique IDs.
            return false;
        }

        // Both might be empty items. For example a Song initialised
        // with an empty token map, because no song is currently playing.
        if (getId() == null && ((Item) o).getId() == null) {
            return true;
        }

        return getId() != null && getId().equals(((Item) o).getId());
    }

    @Override
    public String toString() {
        return "Item{" +
                "id='" + id + '\'' +
                '}';
    }

    @Override
    public int describeContents() {
        return 0;
    }

    Map<String, Object> getRecord(Map<String, Object> record, String recordName) {
        return Util.getRecord(record, recordName);
    }

    protected int getInt(Map<String, Object> record, String fieldName) {
        return Util.getInt(record, fieldName);
    }

    protected int getInt(Map<String, Object> record, String fieldName, int defaultValue) {
        return Util.getInt(record, fieldName, defaultValue);
    }

    protected static String getString(Map<String, Object> record, String fieldName) {
        return Util.getString(record, fieldName);
    }

    protected static String getString(Map<String, Object> record, Map<String, Object> baseRecord, String fieldName) {
        return Util.getString(baseRecord != null && baseRecord.containsKey(fieldName) ? baseRecord : record, fieldName);
    }

    @NonNull
    static String getStringOrEmpty(Map<String, Object> record, String fieldName) {
        return Util.getStringOrEmpty(record, fieldName);
    }

    @NonNull
    static Uri getImageUrl(Map<String, Object> record, String fieldName) {
        return Util.getImageUrl(record, fieldName);
    }
}
