package tribixbite.cleverkeys

import android.content.Context
import android.util.Log
import java.io.IOException
import java.nio.ByteBuffer
import java.nio.ByteOrder
import java.nio.channels.Channels

/**
 * Fast binary dictionary loader with pre-built prefix index.
 *
 * Loads dictionaries from optimized binary format (generated by
 * scripts/generate_binary_dict.py or scripts/build_dictionary.py)
 * instead of parsing JSON at runtime.
 *
 * Performance Benefits:
 * - No JSON parsing overhead (5-10x faster)
 * - Pre-built prefix index (no runtime computation)
 * - Direct byte buffer access (memory-efficient)
 * - Instant loading via memory mapping (future optimization)
 *
 * ## Binary Format V1 (Legacy):
 * Header (32 bytes):
 *   - Magic: b'DICT' (4 bytes)
 *   - Version: 1 (4 bytes)
 *   - Word count: uint32 (4 bytes)
 *   - Dict offset: uint32 (4 bytes)
 *   - Freq offset: uint32 (4 bytes)
 *   - Prefix offset: uint32 (4 bytes)
 *   - Reserved: 8 bytes
 *
 * Dictionary Section (sorted):
 *   - For each word:
 *     - Length: uint16 (2 bytes)
 *     - UTF-8 bytes
 *
 * Frequency Section (parallel):
 *   - For each word:
 *     - Frequency: uint32 (4 bytes)
 *
 * Prefix Index Section:
 *   - Prefix count: uint32 (4 bytes)
 *   - For each prefix:
 *     - Length: uint8 (1 byte)
 *     - UTF-8 bytes
 *     - Match count: uint32 (4 bytes)
 *     - Match indices: uint32[] (4 bytes each)
 *
 * ## Binary Format V2 (Multilanguage with Accent Support):
 * Header (48 bytes):
 *   - Magic: b'CKDT' (4 bytes) - CleverKeys Dict
 *   - Version: 2 (4 bytes)
 *   - Language: 4 bytes (e.g., "es\0\0")
 *   - Word count: uint32 (4 bytes)
 *   - Canonical offset: uint32 (4 bytes)
 *   - Normalized offset: uint32 (4 bytes)
 *   - Accent map offset: uint32 (4 bytes)
 *   - Reserved: 20 bytes
 *
 * Canonical Section:
 *   - For each word (wordCount entries):
 *     - Length: uint16 (2 bytes)
 *     - UTF-8 bytes (display form with accents)
 *     - Frequency rank: uint8 (0=most common, 255=least)
 *
 * Normalized Section:
 *   - Normalized word count: uint32 (4 bytes)
 *   - For each normalized word:
 *     - Length: uint16 (2 bytes)
 *     - UTF-8 bytes (accent-stripped lookup key)
 *
 * Accent Map Section:
 *   - For each normalized word (parallel to normalized section):
 *     - Canonical count: uint8 (how many canonical forms)
 *     - Canonical indices: uint32[] (indices into canonical section)
 *
 * @since v1.0.0 - V1 format
 * @since v1.2.0 - V2 format with accent normalization for multilanguage
 */
object BinaryDictionaryLoader {
    private const val TAG = "BinaryDictionaryLoader"
    private const val MAGIC_V1 = 0x54434944 // "DICT" in little-endian
    private const val MAGIC_V2 = 0x54444B43 // "CKDT" in little-endian (CleverKeys Dict)
    private const val MAGIC = MAGIC_V1 // For backward compatibility
    private const val EXPECTED_VERSION = 1
    private const val EXPECTED_VERSION_V2 = 2
    private const val HEADER_SIZE = 32
    private const val HEADER_SIZE_V2 = 48  // Extended header for v2

    /**
     * Load dictionary from binary format.
     *
     * Supports both V1 and V2 binary formats:
     * - V1: Magic "DICT", word+frequency pairs
     * - V2: Magic "CKDT", canonical words with frequency ranks (converted to frequencies)
     *
     * @param context Android context for asset access
     * @param filename Binary dictionary filename in assets
     * @return Dict mapping words to frequencies, or null if loading fails
     * @since v1.1.91 - Added V2 format support for multilanguage touch typing
     */
    @JvmStatic
    fun loadDictionary(context: Context, filename: String): Map<String, Int>? {
        // OPTIMIZATION v3 (perftodos3.md): Use android.os.Trace for system-level profiling
        android.os.Trace.beginSection("BinaryDictionaryLoader.loadDictionary")
        try {
            val startTime = System.currentTimeMillis()

            try {
                val inputStream = context.assets.open(filename)
                val channel = Channels.newChannel(inputStream)

                // Read entire file into byte buffer for fast access
                val buffer = ByteBuffer.allocate(inputStream.available()).apply {
                    order(ByteOrder.LITTLE_ENDIAN)
                }
                channel.read(buffer)
                buffer.flip()
                channel.close()
                inputStream.close()

                // Parse header - check magic to determine format version
                val magic = buffer.int
                val isV1 = magic == MAGIC_V1
                val isV2 = magic == MAGIC_V2

                if (!isV1 && !isV2) {
                    Log.e(TAG, String.format("Invalid magic number: 0x%08X (expected V1=0x%08X or V2=0x%08X)",
                        magic, MAGIC_V1, MAGIC_V2))
                    return null
                }

                val version = buffer.int
                if (isV1 && version != EXPECTED_VERSION) {
                    Log.e(TAG, String.format("Unsupported V1 version: %d (expected %d)", version, EXPECTED_VERSION))
                    return null
                }
                if (isV2 && version != EXPECTED_VERSION_V2) {
                    Log.e(TAG, String.format("Unsupported V2 version: %d (expected %d)", version, EXPECTED_VERSION_V2))
                    return null
                }

                val dictionary = if (isV1) {
                    loadDictionaryV1(buffer)
                } else {
                    loadDictionaryV2(buffer)
                }

                val loadTime = System.currentTimeMillis() - startTime
                Log.i(TAG, String.format(
                    "Loaded %d words from binary dictionary (V%d) in %dms",
                    dictionary.size, if (isV1) 1 else 2, loadTime
                ))

                return dictionary
            } catch (e: IOException) {
                Log.e(TAG, "Failed to load binary dictionary: $filename", e)
                return null
            }
        } finally {
            android.os.Trace.endSection()
        }
    }

    /**
     * Load V1 format dictionary (word + frequency pairs).
     */
    private fun loadDictionaryV1(buffer: ByteBuffer): Map<String, Int> {
        val wordCount = buffer.int
        val dictOffset = buffer.int
        val freqOffset = buffer.int
        val prefixOffset = buffer.int
        buffer.position(buffer.position() + 8) // Skip reserved bytes

        // Load dictionary words
        buffer.position(dictOffset)
        val words = Array(wordCount) {
            val wordLen = buffer.short.toInt() and 0xFFFF // Unsigned short
            val wordBytes = ByteArray(wordLen)
            buffer.get(wordBytes)
            String(wordBytes, Charsets.UTF_8)
        }

        // Load frequencies
        buffer.position(freqOffset)
        val dictionary = mutableMapOf<String, Int>()
        for (i in 0 until wordCount) {
            val frequency = buffer.int
            dictionary[words[i]] = frequency
        }

        return dictionary
    }

    /**
     * Load V2 format dictionary (canonical words with frequency ranks).
     * Converts ranks (0-255) to frequencies for compatibility with WordPredictor.
     *
     * V2 rank 0 = most common, rank 255 = least common
     * Converted to frequency: freq = 1000000 - (rank * 3900)
     * This gives range ~1M (rank 0) to ~2500 (rank 255)
     *
     * @since v1.1.91
     */
    private fun loadDictionaryV2(buffer: ByteBuffer): Map<String, Int> {
        // Read V2 header (magic+version already consumed)
        val langBytes = ByteArray(4)
        buffer.get(langBytes)
        val language = String(langBytes, Charsets.UTF_8).trim('\u0000')

        val wordCount = buffer.int
        val canonicalOffset = buffer.int
        val normalizedOffset = buffer.int
        val accentMapOffset = buffer.int
        buffer.position(buffer.position() + 20) // Skip reserved

        // Load canonical words with frequency ranks
        buffer.position(canonicalOffset)
        val dictionary = mutableMapOf<String, Int>()

        for (i in 0 until wordCount) {
            val wordLen = buffer.short.toInt() and 0xFFFF
            val wordBytes = ByteArray(wordLen)
            buffer.get(wordBytes)
            val word = String(wordBytes, Charsets.UTF_8)
            val rank = buffer.get().toInt() and 0xFF

            // Convert rank (0-255) to frequency
            // rank 0 -> freq 1000000 (most common)
            // rank 255 -> freq ~5000 (least common)
            val frequency = 1000000 - (rank * 3900)
            dictionary[word] = frequency
        }

        Log.d(TAG, "Loaded V2 dictionary for language '$language': $wordCount words")
        return dictionary
    }

    /**
     * Load dictionary and prefix index from binary format.
     *
     * This method loads both the word-to-frequency mapping and the prefix index.
     * - V1 format: Prefix index is pre-built in the file
     * - V2 format: Prefix index is built at runtime from loaded words
     *
     * @param context Android context for asset access
     * @param filename Binary dictionary filename in assets
     * @param outDictionary Map to populate with word-to-frequency mappings
     * @param outPrefixIndex Map to populate with prefix-to-words mappings
     * @return true if loading succeeded, false otherwise
     * @since v1.1.91 - Added V2 format support for multilanguage touch typing
     */
    @JvmStatic
    fun loadDictionaryWithPrefixIndex(
        context: Context,
        filename: String,
        outDictionary: MutableMap<String, Int>,
        outPrefixIndex: MutableMap<String, MutableSet<String>>
    ): Boolean {
        // OPTIMIZATION v3 (perftodos3.md): Use android.os.Trace for system-level profiling
        android.os.Trace.beginSection("BinaryDictionaryLoader.loadDictionaryWithPrefixIndex")
        try {
            val startTime = System.currentTimeMillis()

            try {
                val inputStream = context.assets.open(filename)
                val channel = Channels.newChannel(inputStream)

                // Read entire file into byte buffer for fast access
                val buffer = ByteBuffer.allocate(inputStream.available()).apply {
                    order(ByteOrder.LITTLE_ENDIAN)
                }
                channel.read(buffer)
                buffer.flip()
                channel.close()
                inputStream.close()

                // Parse header - check magic to determine format version
                val magic = buffer.int
                val isV1 = magic == MAGIC_V1
                val isV2 = magic == MAGIC_V2

                if (!isV1 && !isV2) {
                    Log.e(TAG, String.format("Invalid magic number: 0x%08X (expected V1=0x%08X or V2=0x%08X)",
                        magic, MAGIC_V1, MAGIC_V2))
                    return false
                }

                val version = buffer.int
                if (isV1 && version != EXPECTED_VERSION) {
                    Log.e(TAG, String.format("Unsupported V1 version: %d (expected %d)", version, EXPECTED_VERSION))
                    return false
                }
                if (isV2 && version != EXPECTED_VERSION_V2) {
                    Log.e(TAG, String.format("Unsupported V2 version: %d (expected %d)", version, EXPECTED_VERSION_V2))
                    return false
                }

                outDictionary.clear()
                outPrefixIndex.clear()

                if (isV1) {
                    // V1: Load pre-built prefix index from file
                    loadDictionaryWithPrefixIndexV1(buffer, outDictionary, outPrefixIndex)
                } else {
                    // V2: Load dictionary and build prefix index at runtime
                    loadDictionaryWithPrefixIndexV2(buffer, outDictionary, outPrefixIndex)
                }

                val loadTime = System.currentTimeMillis() - startTime
                Log.i(TAG, String.format(
                    "Loaded %d words + %d prefixes from binary dictionary (V%d) in %dms",
                    outDictionary.size, outPrefixIndex.size, if (isV1) 1 else 2, loadTime
                ))

                return true
            } catch (e: IOException) {
                Log.e(TAG, "Failed to load binary dictionary with prefix index: $filename", e)
                return false
            }
        } finally {
            android.os.Trace.endSection()
        }
    }

    /**
     * Load V1 format dictionary with pre-built prefix index.
     */
    private fun loadDictionaryWithPrefixIndexV1(
        buffer: ByteBuffer,
        outDictionary: MutableMap<String, Int>,
        outPrefixIndex: MutableMap<String, MutableSet<String>>
    ) {
        val wordCount = buffer.int
        val dictOffset = buffer.int
        val freqOffset = buffer.int
        val prefixOffset = buffer.int
        buffer.position(buffer.position() + 8) // Skip reserved bytes

        // Load dictionary words
        buffer.position(dictOffset)
        val words = Array(wordCount) {
            val wordLen = buffer.short.toInt() and 0xFFFF // Unsigned short
            val wordBytes = ByteArray(wordLen)
            buffer.get(wordBytes)
            String(wordBytes, Charsets.UTF_8)
        }

        // Load frequencies
        buffer.position(freqOffset)
        for (i in 0 until wordCount) {
            val frequency = buffer.int
            outDictionary[words[i]] = frequency
        }

        // Load prefix index
        buffer.position(prefixOffset)
        val prefixCount = buffer.int

        for (i in 0 until prefixCount) {
            // Read prefix string
            val prefixLen = buffer.get().toInt() and 0xFF // Unsigned byte
            val prefixBytes = ByteArray(prefixLen)
            buffer.get(prefixBytes)
            val prefix = String(prefixBytes, Charsets.UTF_8)

            // Read matching word indices
            val matchCount = buffer.int
            val matchingWords = mutableSetOf<String>()
            for (j in 0 until matchCount) {
                val wordIdx = buffer.int
                matchingWords.add(words[wordIdx])
            }

            outPrefixIndex[prefix] = matchingWords
        }
    }

    /**
     * Load V2 format dictionary and build prefix index at runtime.
     * V2 format doesn't include a pre-built prefix index, so we generate it.
     *
     * @since v1.1.91
     */
    private fun loadDictionaryWithPrefixIndexV2(
        buffer: ByteBuffer,
        outDictionary: MutableMap<String, Int>,
        outPrefixIndex: MutableMap<String, MutableSet<String>>
    ) {
        // Read V2 header (magic+version already consumed)
        val langBytes = ByteArray(4)
        buffer.get(langBytes)
        val language = String(langBytes, Charsets.UTF_8).trim('\u0000')

        val wordCount = buffer.int
        val canonicalOffset = buffer.int
        buffer.position(buffer.position() + 28) // Skip normalized/accent offsets + reserved

        // Load canonical words with frequency ranks
        buffer.position(canonicalOffset)

        for (i in 0 until wordCount) {
            val wordLen = buffer.short.toInt() and 0xFFFF
            val wordBytes = ByteArray(wordLen)
            buffer.get(wordBytes)
            val word = String(wordBytes, Charsets.UTF_8)
            val rank = buffer.get().toInt() and 0xFF

            // Convert rank (0-255) to frequency
            // rank 0 -> freq 1000000 (most common)
            // rank 255 -> freq ~5000 (least common)
            val frequency = 1000000 - (rank * 3900)
            outDictionary[word] = frequency

            // Build prefix index (1-3 character prefixes)
            val maxPrefixLen = kotlin.math.min(3, word.length)
            for (len in 1..maxPrefixLen) {
                val prefix = word.substring(0, len).lowercase()
                outPrefixIndex.getOrPut(prefix) { mutableSetOf() }.add(word)
            }
        }

        Log.d(TAG, "Loaded V2 dictionary for '$language': $wordCount words, built ${outPrefixIndex.size} prefix entries")
    }

    /**
     * Estimate JSON loading time for performance comparison.
     * Based on observed ~300ms for 50k words.
     */
    private fun estimateJsonLoadTime(wordCount: Int): Long {
        return (wordCount / 50000.0 * 300).toLong()
    }

    /**
     * Estimate JSON + prefix index building time for performance comparison.
     * Based on observed ~500ms total for 50k words.
     */
    private fun estimateJsonWithIndexLoadTime(wordCount: Int): Long {
        return (wordCount / 50000.0 * 500).toLong()
    }

    /**
     * Load dictionary into NormalizedPrefixIndex with accent support.
     *
     * Supports both V1 and V2 binary formats:
     * - V1: Words loaded as-is, no accent mapping
     * - V2: Full accent normalization with canonical/normalized separation
     *
     * For V1 format, words are normalized at load time and stored with
     * frequency converted to rank (0-255).
     *
     * @param context Android context for asset access
     * @param filename Binary dictionary filename in assets
     * @param outIndex NormalizedPrefixIndex to populate
     * @return true if loading succeeded, false otherwise
     */
    @JvmStatic
    fun loadIntoNormalizedIndex(
        context: Context,
        filename: String,
        outIndex: NormalizedPrefixIndex
    ): Boolean {
        android.os.Trace.beginSection("BinaryDictionaryLoader.loadIntoNormalizedIndex")
        try {
            val startTime = System.currentTimeMillis()

            try {
                val inputStream = context.assets.open(filename)
                val channel = Channels.newChannel(inputStream)

                val buffer = ByteBuffer.allocate(inputStream.available()).apply {
                    order(ByteOrder.LITTLE_ENDIAN)
                }
                channel.read(buffer)
                buffer.flip()
                channel.close()
                inputStream.close()

                // Check magic number to determine format version
                val magic = buffer.int
                val isV2 = magic == MAGIC_V2
                val isV1 = magic == MAGIC_V1

                if (!isV1 && !isV2) {
                    Log.e(TAG, String.format("Invalid magic number: 0x%08X", magic))
                    return false
                }

                val version = buffer.int
                if (isV1 && version != EXPECTED_VERSION) {
                    Log.e(TAG, "Unsupported V1 version: $version")
                    return false
                }
                if (isV2 && version != EXPECTED_VERSION_V2) {
                    Log.e(TAG, "Unsupported V2 version: $version")
                    return false
                }

                outIndex.clear()

                if (isV1) {
                    // Load V1 format and convert to normalized index
                    loadV1IntoNormalizedIndex(buffer, outIndex)
                } else {
                    // Load V2 format with native accent support
                    loadV2IntoNormalizedIndex(buffer, outIndex)
                }

                val loadTime = System.currentTimeMillis() - startTime
                Log.i(TAG, "Loaded ${outIndex.size()} words (${outIndex.normalizedCount()} normalized) " +
                        "into NormalizedPrefixIndex in ${loadTime}ms")

                return true
            } catch (e: IOException) {
                Log.e(TAG, "Failed to load dictionary into NormalizedPrefixIndex: $filename", e)
                return false
            }
        } finally {
            android.os.Trace.endSection()
        }
    }

    /**
     * Load V2 binary dictionary from a file path into NormalizedPrefixIndex.
     * Used for loading language packs from internal storage.
     *
     * @param file Binary dictionary file
     * @param outIndex NormalizedPrefixIndex to populate
     * @return true if loading succeeded, false otherwise
     */
    @JvmStatic
    fun loadIntoNormalizedIndexFromFile(
        file: java.io.File,
        outIndex: NormalizedPrefixIndex
    ): Boolean {
        android.os.Trace.beginSection("BinaryDictionaryLoader.loadFromFile")
        try {
            val startTime = System.currentTimeMillis()

            try {
                val inputStream = java.io.FileInputStream(file)
                val channel = Channels.newChannel(inputStream)

                val buffer = ByteBuffer.allocate(file.length().toInt()).apply {
                    order(ByteOrder.LITTLE_ENDIAN)
                }
                channel.read(buffer)
                buffer.flip()
                channel.close()
                inputStream.close()

                // Check magic number - only V2 format supported for language packs
                val magic = buffer.int
                if (magic != MAGIC_V2) {
                    Log.e(TAG, String.format("Invalid magic number for language pack: 0x%08X", magic))
                    return false
                }

                val version = buffer.int
                if (version != EXPECTED_VERSION_V2) {
                    Log.e(TAG, "Unsupported V2 version: $version")
                    return false
                }

                outIndex.clear()
                loadV2IntoNormalizedIndex(buffer, outIndex)

                val elapsed = System.currentTimeMillis() - startTime
                Log.i(TAG, "Loaded ${outIndex.size()} entries from ${file.name} in ${elapsed}ms")
                return true
            } catch (e: IOException) {
                Log.e(TAG, "Failed to load dictionary from file: ${file.absolutePath}", e)
                return false
            }
        } finally {
            android.os.Trace.endSection()
        }
    }

    /**
     * Load V1 binary format into NormalizedPrefixIndex.
     * Converts raw frequencies to ranks (0-255).
     */
    private fun loadV1IntoNormalizedIndex(buffer: ByteBuffer, outIndex: NormalizedPrefixIndex) {
        val wordCount = buffer.int
        val dictOffset = buffer.int
        val freqOffset = buffer.int
        buffer.position(buffer.position() + 12) // Skip prefix offset + reserved

        // Load dictionary words
        buffer.position(dictOffset)
        val words = Array(wordCount) {
            val wordLen = buffer.short.toInt() and 0xFFFF
            val wordBytes = ByteArray(wordLen)
            buffer.get(wordBytes)
            String(wordBytes, Charsets.UTF_8)
        }

        // Load frequencies and find max for rank conversion
        buffer.position(freqOffset)
        val frequencies = IntArray(wordCount)
        var maxFreq = 1
        for (i in 0 until wordCount) {
            frequencies[i] = buffer.int
            if (frequencies[i] > maxFreq) maxFreq = frequencies[i]
        }

        // Convert frequency to rank (0 = most frequent, 255 = least)
        // Use log scale for better distribution
        val logMax = kotlin.math.ln(maxFreq.toDouble() + 1)
        for (i in 0 until wordCount) {
            val logFreq = kotlin.math.ln(frequencies[i].toDouble() + 1)
            val rank = ((1.0 - logFreq / logMax) * 255).toInt().coerceIn(0, 255)
            outIndex.addWord(words[i], rank)
        }
    }

    /**
     * Load V2 binary format into NormalizedPrefixIndex.
     *
     * V2 Format Header (48 bytes):
     *   - Magic: "CKDT" (4 bytes)
     *   - Version: 2 (4 bytes)
     *   - Language: 4 bytes (e.g., "es\0\0")
     *   - Word count: uint32 (4 bytes)
     *   - Canonical offset: uint32 (4 bytes)
     *   - Normalized offset: uint32 (4 bytes)
     *   - Accent map offset: uint32 (4 bytes)
     *   - Reserved: 20 bytes
     *
     * Canonical Section:
     *   - For each word:
     *     - Length: uint16
     *     - UTF-8 bytes
     *     - Frequency rank: uint8 (0-255)
     *
     * Normalized Section:
     *   - Normalized word count: uint32
     *   - For each normalized word:
     *     - Length: uint16
     *     - UTF-8 bytes
     *
     * Accent Map Section:
     *   - For each normalized word (parallel to normalized section):
     *     - Canonical count: uint8
     *     - Canonical indices: uint32[]
     */
    private fun loadV2IntoNormalizedIndex(buffer: ByteBuffer, outIndex: NormalizedPrefixIndex) {
        // Read V2 header (version already read by caller)
        val langBytes = ByteArray(4)
        buffer.get(langBytes)
        val language = String(langBytes, Charsets.UTF_8).trim('\u0000')

        val wordCount = buffer.int
        val canonicalOffset = buffer.int
        val normalizedOffset = buffer.int
        val accentMapOffset = buffer.int
        buffer.position(buffer.position() + 20) // Skip reserved

        // Load canonical words with frequency ranks
        buffer.position(canonicalOffset)
        val canonicals = Array(wordCount) { "" }
        val ranks = IntArray(wordCount)

        for (i in 0 until wordCount) {
            val wordLen = buffer.short.toInt() and 0xFFFF
            val wordBytes = ByteArray(wordLen)
            buffer.get(wordBytes)
            canonicals[i] = String(wordBytes, Charsets.UTF_8)
            ranks[i] = buffer.get().toInt() and 0xFF
        }

        // Load normalized words
        buffer.position(normalizedOffset)
        val normalizedCount = buffer.int
        val normalizeds = Array(normalizedCount) { "" }

        for (i in 0 until normalizedCount) {
            val wordLen = buffer.short.toInt() and 0xFFFF
            val wordBytes = ByteArray(wordLen)
            buffer.get(wordBytes)
            normalizeds[i] = String(wordBytes, Charsets.UTF_8)
        }

        // Load accent map and populate index
        buffer.position(accentMapOffset)
        for (i in 0 until normalizedCount) {
            val canonicalCount = buffer.get().toInt() and 0xFF
            for (j in 0 until canonicalCount) {
                val canonicalIdx = buffer.int
                if (canonicalIdx < wordCount) {
                    outIndex.addWord(canonicals[canonicalIdx], ranks[canonicalIdx])
                }
            }
        }

        Log.d(TAG, "Loaded V2 dictionary: language=$language, " +
                "$wordCount canonical words, $normalizedCount normalized forms")
    }

    /**
     * Check if a binary dictionary file exists in assets.
     *
     * @param context Android context for asset access
     * @param filename Binary dictionary filename in assets
     * @return true if file exists, false otherwise
     */
    @JvmStatic
    fun exists(context: Context, filename: String): Boolean {
        return try {
            val inputStream = context.assets.open(filename)
            inputStream.close()
            true
        } catch (e: IOException) {
            false
        }
    }
}
