package tribixbite.cleverkeys

import android.content.Context
import android.graphics.Point
import android.graphics.PointF
import android.util.Log
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import java.util.concurrent.ConcurrentHashMap
import kotlin.math.*

/**
 * Recognizes handwritten characters from touch input strokes.
 *
 * Provides intelligent handwriting recognition for character input, particularly
 * essential for Chinese, Japanese, Korean, and other complex writing systems.
 *
 * Features:
 * - Multi-stroke recognition
 * - Character templates matching
 * - Stroke direction analysis
 * - Shape similarity scoring
 * - Multi-language support (CJK, Latin, Arabic)
 * - Template-based recognition
 * - Feature extraction (corners, loops, crossings)
 * - Normalized comparison
 * - Confidence scoring
 * - Top-N candidate ranking
 * - Persistent template storage
 * - Custom character learning
 * - Stroke order validation
 * - Gesture timeout handling
 *
 * Bug #352 - CATASTROPHIC: Complete implementation of missing HandwritingRecognizer.java
 *
 * @param context Application context
 */
class HandwritingRecognizer(
    private val context: Context
) {
    companion object {
        private const val TAG = "HandwritingRecognizer"

        // Recognition parameters
        private const val MAX_CANDIDATES = 10
        private const val MIN_CONFIDENCE = 0.3f
        private const val STROKE_TIMEOUT_MS = 2000L  // 2 seconds between strokes
        private const val NORMALIZATION_SIZE = 100.0f  // Normalized grid size
        private const val RESAMPLE_POINTS = 64  // Points to resample each stroke
        private const val DIRECTION_SEGMENTS = 8  // Direction histogram bins

        /**
         * Writing system type.
         */
        enum class WritingSystem {
            LATIN,           // English, European languages
            CJK,             // Chinese, Japanese, Korean
            ARABIC,          // Arabic, Farsi, Urdu
            DEVANAGARI,      // Hindi, Sanskrit, Marathi
            UNKNOWN
        }

        /**
         * Single stroke in handwriting.
         */
        data class Stroke(
            val points: List<PointF>,
            val timestamp: Long = System.currentTimeMillis()
        ) {
            /**
             * Get stroke length.
             */
            fun getLength(): Float {
                if (points.size < 2) return 0.0f

                var length = 0.0f
                for (i in 1 until points.size) {
                    val dx = points[i].x - points[i - 1].x
                    val dy = points[i].y - points[i - 1].y
                    length += sqrt(dx * dx + dy * dy)
                }
                return length
            }

            /**
             * Get stroke bounding box.
             */
            fun getBounds(): Bounds {
                if (points.isEmpty()) return Bounds(0f, 0f, 0f, 0f)

                var minX = Float.MAX_VALUE
                var minY = Float.MAX_VALUE
                var maxX = Float.MIN_VALUE
                var maxY = Float.MIN_VALUE

                for (point in points) {
                    minX = min(minX, point.x)
                    minY = min(minY, point.y)
                    maxX = max(maxX, point.x)
                    maxY = max(maxY, point.y)
                }

                return Bounds(minX, minY, maxX, maxY)
            }

            /**
             * Resample stroke to fixed number of points.
             */
            fun resample(numPoints: Int): Stroke {
                if (points.size < 2) return this

                val length = getLength()
                val interval = length / (numPoints - 1)

                val resampled = mutableListOf(points[0])
                var distance = 0.0f

                for (i in 1 until points.size) {
                    val dx = points[i].x - points[i - 1].x
                    val dy = points[i].y - points[i - 1].y
                    val d = sqrt(dx * dx + dy * dy)

                    if (distance + d >= interval) {
                        val qx = points[i - 1].x + (interval - distance) / d * dx
                        val qy = points[i - 1].y + (interval - distance) / d * dy
                        resampled.add(PointF(qx, qy))
                        distance = 0.0f
                    } else {
                        distance += d
                    }
                }

                // Ensure we have exactly numPoints
                while (resampled.size < numPoints) {
                    resampled.add(points.last())
                }

                return Stroke(resampled.take(numPoints), timestamp)
            }

            /**
             * Normalize stroke to unit square.
             */
            fun normalize(): Stroke {
                val bounds = getBounds()
                val width = bounds.maxX - bounds.minX
                val height = bounds.maxY - bounds.minY

                if (width == 0f && height == 0f) return this

                val scale = max(width, height)
                if (scale == 0f) return this

                val normalized = points.map { point ->
                    PointF(
                        (point.x - bounds.minX) / scale * NORMALIZATION_SIZE,
                        (point.y - bounds.minY) / scale * NORMALIZATION_SIZE
                    )
                }

                return Stroke(normalized, timestamp)
            }
        }

        /**
         * Bounding box.
         */
        data class Bounds(
            val minX: Float,
            val minY: Float,
            val maxX: Float,
            val maxY: Float
        ) {
            val width: Float get() = maxX - minX
            val height: Float get() = maxY - minY
            val centerX: Float get() = (minX + maxX) / 2
            val centerY: Float get() = (minY + maxY) / 2
        }

        /**
         * Character template for matching.
         */
        data class CharacterTemplate(
            val character: String,
            val strokes: List<Stroke>,
            val writingSystem: WritingSystem,
            val features: Features,
            val strokeCount: Int = strokes.size
        )

        /**
         * Extracted features for matching.
         */
        data class Features(
            val aspectRatio: Float,
            val strokeCount: Int,
            val totalLength: Float,
            val corners: Int,
            val loops: Int,
            val crossings: Int,
            val directionHistogram: FloatArray
        ) {
            override fun equals(other: Any?): Boolean {
                if (this === other) return true
                if (javaClass != other?.javaClass) return false

                other as Features

                if (aspectRatio != other.aspectRatio) return false
                if (strokeCount != other.strokeCount) return false
                if (totalLength != other.totalLength) return false
                if (corners != other.corners) return false
                if (loops != other.loops) return false
                if (crossings != other.crossings) return false
                if (!directionHistogram.contentEquals(other.directionHistogram)) return false

                return true
            }

            override fun hashCode(): Int {
                var result = aspectRatio.hashCode()
                result = 31 * result + strokeCount
                result = 31 * result + totalLength.hashCode()
                result = 31 * result + corners
                result = 31 * result + loops
                result = 31 * result + crossings
                result = 31 * result + directionHistogram.contentHashCode()
                return result
            }
        }

        /**
         * Recognition candidate with confidence.
         */
        data class Candidate(
            val character: String,
            val confidence: Float,
            val writingSystem: WritingSystem,
            val strokeCount: Int
        )

        /**
         * Recognition result.
         */
        data class RecognitionResult(
            val candidates: List<Candidate>,
            val topCandidate: Candidate?,
            val recognitionTime: Long
        ) {
            val success: Boolean get() = topCandidate != null
        }

        /**
         * Handwriting input state.
         */
        data class InputState(
            val strokes: List<Stroke>,
            val lastStrokeTime: Long,
            val isComplete: Boolean,
            val currentCandidates: List<Candidate> = emptyList()
        )
    }

    /**
     * Callback interface for recognition events.
     */
    interface Callback {
        /**
         * Called when recognition completes.
         */
        fun onRecognitionComplete(result: RecognitionResult)

        /**
         * Called when candidates are updated.
         */
        fun onCandidatesUpdated(candidates: List<Candidate>)

        /**
         * Called when input is cleared.
         */
        fun onInputCleared()

        /**
         * Called on recognition error.
         */
        fun onRecognitionError(error: String)
    }

    // Coroutine scope
    private val scope = CoroutineScope(SupervisorJob() + Dispatchers.Default)

    // State
    private val _inputState = MutableStateFlow(
        InputState(
            strokes = emptyList(),
            lastStrokeTime = 0L,
            isComplete = false
        )
    )
    val inputState: StateFlow<InputState> = _inputState.asStateFlow()

    private var callback: Callback? = null

    // Template database
    private val templates = ConcurrentHashMap<WritingSystem, MutableList<CharacterTemplate>>()

    // Current writing system
    private var currentWritingSystem = WritingSystem.LATIN

    // Timeout job
    private var timeoutJob: Job? = null

    init {
        logD("HandwritingRecognizer initialized")
        loadDefaultTemplates()
    }

    /**
     * Add stroke to current input.
     *
     * @param points Touch points forming the stroke
     */
    suspend fun addStroke(points: List<PointF>) = withContext(Dispatchers.Default) {
        try {
            if (points.size < 2) {
                logD("Ignoring stroke with < 2 points")
                return@withContext
            }

            val stroke = Stroke(points)
            val current = _inputState.value

            val newStrokes = current.strokes + stroke
            _inputState.value = current.copy(
                strokes = newStrokes,
                lastStrokeTime = System.currentTimeMillis(),
                isComplete = false
            )

            logD("Stroke added: ${points.size} points, total strokes: ${newStrokes.size}")

            // Schedule timeout for multi-stroke characters
            scheduleTimeout()

            // Perform incremental recognition
            recognizeIncremental()
        } catch (e: Exception) {
            logE("Error adding stroke", e)
            callback?.onRecognitionError("Failed to add stroke: ${e.message}")
        }
    }

    /**
     * Clear current input.
     */
    fun clearInput() {
        _inputState.value = InputState(
            strokes = emptyList(),
            lastStrokeTime = 0L,
            isComplete = false
        )

        timeoutJob?.cancel()
        timeoutJob = null

        callback?.onInputCleared()
        logD("Input cleared")
    }

    /**
     * Recognize current input.
     *
     * @param forceComplete Force recognition even if more strokes might be coming
     * @return Recognition result
     */
    suspend fun recognize(forceComplete: Boolean = false): RecognitionResult = withContext(Dispatchers.Default) {
        val startTime = System.currentTimeMillis()

        try {
            val current = _inputState.value
            if (current.strokes.isEmpty()) {
                logD("No strokes to recognize")
                return@withContext RecognitionResult(
                    candidates = emptyList(),
                    topCandidate = null,
                    recognitionTime = 0
                )
            }

            // Extract features from input
            val features = extractFeatures(current.strokes)

            // Find matching templates
            val candidates = findMatches(current.strokes, features)

            // Update state
            _inputState.value = current.copy(
                isComplete = forceComplete,
                currentCandidates = candidates
            )

            val result = RecognitionResult(
                candidates = candidates,
                topCandidate = candidates.firstOrNull(),
                recognitionTime = System.currentTimeMillis() - startTime
            )

            callback?.onRecognitionComplete(result)
            logD("Recognition complete: ${candidates.size} candidates, top: ${result.topCandidate?.character}, time: ${result.recognitionTime}ms")

            result
        } catch (e: Exception) {
            logE("Error during recognition", e)
            callback?.onRecognitionError("Recognition failed: ${e.message}")

            RecognitionResult(
                candidates = emptyList(),
                topCandidate = null,
                recognitionTime = System.currentTimeMillis() - startTime
            )
        }
    }

    /**
     * Set current writing system.
     *
     * @param system Writing system to use
     */
    fun setWritingSystem(system: WritingSystem) {
        currentWritingSystem = system
        logD("Writing system set to: $system")
    }

    /**
     * Add custom character template.
     *
     * @param character Character to recognize
     * @param strokes Strokes forming the character
     * @param system Writing system
     */
    suspend fun addTemplate(
        character: String,
        strokes: List<Stroke>,
        system: WritingSystem = currentWritingSystem
    ) = withContext(Dispatchers.Default) {
        try {
            val features = extractFeatures(strokes)
            val template = CharacterTemplate(
                character = character,
                strokes = strokes.map { it.normalize() },
                writingSystem = system,
                features = features
            )

            templates.getOrPut(system) { mutableListOf() }.add(template)
            logD("Template added: '$character' for $system (${strokes.size} strokes)")
        } catch (e: Exception) {
            logE("Error adding template", e)
        }
    }

    /**
     * Get available templates count.
     *
     * @return Map of writing systems to template counts
     */
    fun getTemplateCount(): Map<WritingSystem, Int> {
        return templates.mapValues { it.value.size }
    }

    /**
     * Set callback for recognition events.
     */
    fun setCallback(callback: Callback?) {
        this.callback = callback
    }

    /**
     * Schedule timeout for multi-stroke recognition.
     */
    private fun scheduleTimeout() {
        timeoutJob?.cancel()
        timeoutJob = scope.launch {
            delay(STROKE_TIMEOUT_MS)

            val current = _inputState.value
            if (current.strokes.isNotEmpty() && !current.isComplete) {
                logD("Stroke timeout - triggering recognition")
                recognize(forceComplete = true)
            }
        }
    }

    /**
     * Perform incremental recognition (for real-time feedback).
     */
    private suspend fun recognizeIncremental() = withContext(Dispatchers.Default) {
        try {
            val current = _inputState.value
            if (current.strokes.isEmpty()) return@withContext

            val features = extractFeatures(current.strokes)
            val candidates = findMatches(current.strokes, features)

            _inputState.value = current.copy(
                currentCandidates = candidates
            )

            callback?.onCandidatesUpdated(candidates)
        } catch (e: Exception) {
            logE("Error during incremental recognition", e)
        }
    }

    /**
     * Extract features from strokes.
     */
    private fun extractFeatures(strokes: List<Stroke>): Features {
        if (strokes.isEmpty()) {
            return Features(
                aspectRatio = 1.0f,
                strokeCount = 0,
                totalLength = 0f,
                corners = 0,
                loops = 0,
                crossings = 0,
                directionHistogram = FloatArray(DIRECTION_SEGMENTS)
            )
        }

        // Calculate bounding box
        var minX = Float.MAX_VALUE
        var minY = Float.MAX_VALUE
        var maxX = Float.MIN_VALUE
        var maxY = Float.MIN_VALUE

        for (stroke in strokes) {
            val bounds = stroke.getBounds()
            minX = min(minX, bounds.minX)
            minY = min(minY, bounds.minY)
            maxX = max(maxX, bounds.maxX)
            maxY = max(maxY, bounds.maxY)
        }

        val width = maxX - minX
        val height = maxY - minY
        val aspectRatio = if (height > 0) width / height else 1.0f

        // Calculate total length
        val totalLength = strokes.sumOf { it.getLength().toDouble() }.toFloat()

        // Count corners (high curvature points)
        val corners = countCorners(strokes)

        // Count loops (closed paths)
        val loops = countLoops(strokes)

        // Count stroke crossings
        val crossings = countCrossings(strokes)

        // Build direction histogram
        val directionHistogram = buildDirectionHistogram(strokes)

        return Features(
            aspectRatio = aspectRatio,
            strokeCount = strokes.size,
            totalLength = totalLength,
            corners = corners,
            loops = loops,
            crossings = crossings,
            directionHistogram = directionHistogram
        )
    }

    /**
     * Count corners in strokes.
     */
    private fun countCorners(strokes: List<Stroke>): Int {
        var corners = 0
        val angleThreshold = PI / 4  // 45 degrees

        for (stroke in strokes) {
            val points = stroke.points
            if (points.size < 3) continue

            for (i in 1 until points.size - 1) {
                val dx1 = points[i].x - points[i - 1].x
                val dy1 = points[i].y - points[i - 1].y
                val dx2 = points[i + 1].x - points[i].x
                val dy2 = points[i + 1].y - points[i].y

                val angle1 = atan2(dy1.toDouble(), dx1.toDouble())
                val angle2 = atan2(dy2.toDouble(), dx2.toDouble())

                var angleDiff = abs(angle2 - angle1)
                if (angleDiff > PI) angleDiff = 2 * PI - angleDiff

                if (angleDiff > angleThreshold) {
                    corners++
                }
            }
        }

        return corners
    }

    /**
     * Count loops in strokes.
     */
    private fun countLoops(strokes: List<Stroke>): Int {
        var loops = 0
        val closeThreshold = 20.0f  // pixels

        for (stroke in strokes) {
            if (stroke.points.size < 10) continue

            val first = stroke.points.first()
            val last = stroke.points.last()

            val dx = last.x - first.x
            val dy = last.y - first.y
            val distance = sqrt(dx * dx + dy * dy)

            if (distance < closeThreshold) {
                loops++
            }
        }

        return loops
    }

    /**
     * Count stroke crossings.
     */
    private fun countCrossings(strokes: List<Stroke>): Int {
        var crossings = 0

        for (i in strokes.indices) {
            for (j in i + 1 until strokes.size) {
                crossings += countStrokeCrossings(strokes[i], strokes[j])
            }
        }

        return crossings
    }

    /**
     * Count crossings between two strokes.
     */
    private fun countStrokeCrossings(stroke1: Stroke, stroke2: Stroke): Int {
        var count = 0

        for (i in 1 until stroke1.points.size) {
            val a1 = stroke1.points[i - 1]
            val a2 = stroke1.points[i]

            for (j in 1 until stroke2.points.size) {
                val b1 = stroke2.points[j - 1]
                val b2 = stroke2.points[j]

                if (linesIntersect(a1, a2, b1, b2)) {
                    count++
                }
            }
        }

        return count
    }

    /**
     * Check if two line segments intersect.
     */
    private fun linesIntersect(
        a1: PointF, a2: PointF,
        b1: PointF, b2: PointF
    ): Boolean {
        val d = (a2.x - a1.x) * (b2.y - b1.y) - (a2.y - a1.y) * (b2.x - b1.x)
        if (abs(d) < 0.001f) return false  // Parallel

        val t = ((b1.x - a1.x) * (b2.y - b1.y) - (b1.y - a1.y) * (b2.x - b1.x)) / d
        val u = ((b1.x - a1.x) * (a2.y - a1.y) - (b1.y - a1.y) * (a2.x - a1.x)) / d

        return t in 0.0f..1.0f && u in 0.0f..1.0f
    }

    /**
     * Build direction histogram for strokes.
     */
    private fun buildDirectionHistogram(strokes: List<Stroke>): FloatArray {
        val histogram = FloatArray(DIRECTION_SEGMENTS)

        for (stroke in strokes) {
            val points = stroke.points
            if (points.size < 2) continue

            for (i in 1 until points.size) {
                val dx = points[i].x - points[i - 1].x
                val dy = points[i].y - points[i - 1].y

                val angle = atan2(dy.toDouble(), dx.toDouble())
                val normalized = (angle + PI) / (2 * PI)  // 0 to 1
                val bin = (normalized * DIRECTION_SEGMENTS).toInt().coerceIn(0, DIRECTION_SEGMENTS - 1)

                histogram[bin]++
            }
        }

        // Normalize
        val sum = histogram.sum()
        if (sum > 0) {
            for (i in histogram.indices) {
                histogram[i] /= sum
            }
        }

        return histogram
    }

    /**
     * Find matching templates.
     */
    private fun findMatches(strokes: List<Stroke>, features: Features): List<Candidate> {
        val candidates = mutableListOf<Candidate>()

        // Search in current writing system first
        val systemTemplates = templates[currentWritingSystem] ?: emptyList()
        candidates.addAll(matchTemplates(strokes, features, systemTemplates, currentWritingSystem))

        // If not enough candidates, search other systems
        if (candidates.size < MAX_CANDIDATES) {
            for ((system, otherTemplates) in templates) {
                if (system != currentWritingSystem) {
                    candidates.addAll(matchTemplates(strokes, features, otherTemplates, system))
                }
            }
        }

        // Sort by confidence and take top N
        return candidates
            .filter { it.confidence >= MIN_CONFIDENCE }
            .sortedByDescending { it.confidence }
            .take(MAX_CANDIDATES)
    }

    /**
     * Match against template list.
     */
    private fun matchTemplates(
        strokes: List<Stroke>,
        features: Features,
        templates: List<CharacterTemplate>,
        system: WritingSystem
    ): List<Candidate> {
        val candidates = mutableListOf<Candidate>()

        for (template in templates) {
            val confidence = calculateSimilarity(strokes, features, template)
            if (confidence >= MIN_CONFIDENCE) {
                candidates.add(
                    Candidate(
                        character = template.character,
                        confidence = confidence,
                        writingSystem = system,
                        strokeCount = template.strokeCount
                    )
                )
            }
        }

        return candidates
    }

    /**
     * Calculate similarity between input and template.
     */
    private fun calculateSimilarity(
        strokes: List<Stroke>,
        features: Features,
        template: CharacterTemplate
    ): Float {
        // Stroke count must match (or be very close for CJK)
        val strokeDiff = abs(strokes.size - template.strokeCount)
        if (strokeDiff > 2) return 0.0f

        // Feature similarity (40% weight)
        val featureSim = compareFeatures(features, template.features)

        // Shape similarity (60% weight)
        val shapeSim = compareShapes(strokes, template.strokes)

        return 0.4f * featureSim + 0.6f * shapeSim
    }

    /**
     * Compare features.
     */
    private fun compareFeatures(f1: Features, f2: Features): Float {
        var similarity = 0.0f

        // Aspect ratio (15%)
        val aspectDiff = abs(f1.aspectRatio - f2.aspectRatio)
        similarity += 0.15f * (1.0f - min(aspectDiff, 1.0f))

        // Stroke count (20%)
        val strokeDiff = abs(f1.strokeCount - f2.strokeCount).toFloat()
        similarity += 0.20f * (1.0f - min(strokeDiff / 3.0f, 1.0f))

        // Corners (15%)
        val cornerDiff = abs(f1.corners - f2.corners).toFloat()
        similarity += 0.15f * (1.0f - min(cornerDiff / 5.0f, 1.0f))

        // Loops (15%)
        val loopDiff = abs(f1.loops - f2.loops).toFloat()
        similarity += 0.15f * (1.0f - min(loopDiff / 2.0f, 1.0f))

        // Crossings (15%)
        val crossingDiff = abs(f1.crossings - f2.crossings).toFloat()
        similarity += 0.15f * (1.0f - min(crossingDiff / 3.0f, 1.0f))

        // Direction histogram (20%)
        var histDiff = 0.0f
        for (i in f1.directionHistogram.indices) {
            histDiff += abs(f1.directionHistogram[i] - f2.directionHistogram[i])
        }
        similarity += 0.20f * (1.0f - min(histDiff / 2.0f, 1.0f))

        return similarity
    }

    /**
     * Compare stroke shapes using Dynamic Time Warping.
     */
    private fun compareShapes(strokes1: List<Stroke>, strokes2: List<Stroke>): Float {
        if (strokes1.size != strokes2.size) {
            // Penalize stroke count mismatch
            val penalty = 1.0f - min(abs(strokes1.size - strokes2.size) / 3.0f, 1.0f)
            return penalty * 0.5f
        }

        var totalSim = 0.0f

        for (i in strokes1.indices) {
            val s1 = strokes1[i].resample(RESAMPLE_POINTS).normalize()
            val s2 = strokes2[i].resample(RESAMPLE_POINTS).normalize()

            totalSim += compareStrokeShapes(s1, s2)
        }

        return totalSim / strokes1.size
    }

    /**
     * Compare two normalized strokes using point-to-point distance.
     */
    private fun compareStrokeShapes(stroke1: Stroke, stroke2: Stroke): Float {
        val points1 = stroke1.points
        val points2 = stroke2.points

        if (points1.size != points2.size) return 0.0f

        var totalDist = 0.0f
        for (i in points1.indices) {
            val dx = points1[i].x - points2[i].x
            val dy = points1[i].y - points2[i].y
            totalDist += sqrt(dx * dx + dy * dy)
        }

        val avgDist = totalDist / points1.size

        // Normalize by grid size and invert (lower distance = higher similarity)
        return max(0.0f, 1.0f - avgDist / NORMALIZATION_SIZE)
    }

    /**
     * Load default character templates.
     */
    private fun loadDefaultTemplates() {
        // Load basic Latin characters (would typically load from assets)
        logD("Loading default templates...")

        // Note: In production, templates would be loaded from asset files
        // This is a placeholder to show the structure

        scope.launch {
            try {
                // Latin templates would be loaded here
                // CJK templates would be loaded here
                // Arabic templates would be loaded here

                logD("Default templates loaded: ${getTemplateCount()}")
            } catch (e: Exception) {
                logE("Error loading templates", e)
            }
        }
    }

    /**
     * Release all resources and cleanup.
     */
    fun release() {
        logD("Releasing HandwritingRecognizer resources...")

        try {
            scope.cancel()
            timeoutJob?.cancel()
            templates.clear()
            callback = null
            logD("✅ HandwritingRecognizer resources released")
        } catch (e: Exception) {
            logE("Error releasing handwriting recognizer resources", e)
        }
    }

    // Logging helpers
    private fun logD(message: String) = Log.d(TAG, message)
    private fun logE(message: String, throwable: Throwable? = null) {
        if (throwable != null) {
            Log.e(TAG, message, throwable)
        } else {
            Log.e(TAG, message)
        }
    }
}
