package tribixbite.cleverkeys

import android.content.Context
import android.util.Log
import java.io.IOException
import java.nio.ByteBuffer
import java.nio.ByteOrder
import java.nio.channels.Channels
import java.nio.charset.StandardCharsets

/**
 * Fast binary contraction loader.
 *
 * Loads contractions from optimized binary format (generated by
 * scripts/generate_binary_contractions.py) instead of parsing JSON at runtime.
 *
 * Performance Benefits:
 * - No JSON parsing overhead (faster startup)
 * - No JSONObject/JSONArray allocations
 * - Direct byte buffer access (memory-efficient)
 * - Single file instead of two JSON files
 *
 * Binary Format V1:
 * -----------------
 * Header (16 bytes):
 *   - Magic: b'CTRB' (4 bytes)
 *   - Version: uint32 (4 bytes)
 *   - Non-paired count: uint32 (4 bytes)
 *   - Paired count: uint32 (4 bytes)
 *
 * Non-Paired Section:
 *   - For each non-paired mapping:
 *     - Key length: uint16 (2 bytes)
 *     - Key UTF-8 bytes
 *     - Value length: uint16 (2 bytes)
 *     - Value UTF-8 bytes
 *
 * Paired Section:
 *   - For each paired contraction:
 *     - Contraction length: uint16 (2 bytes)
 *     - Contraction UTF-8 bytes
 *
 * OPTIMIZATION v1 (perftodos2.md Todo 4): Eliminates slow JSON parsing at startup
 */
object BinaryContractionLoader {
    private const val TAG = "BinaryContractionLoader"
    private const val MAGIC = 0x42525443 // "CTRB" in little-endian
    private const val EXPECTED_VERSION = 1
    private const val HEADER_SIZE = 16

    /**
     * Result container for loaded contraction data.
     */
    data class ContractionData(
        val nonPairedContractions: Map<String, String>,
        val knownContractions: Set<String>
    )

    /**
     * Load contractions from binary format.
     *
     * @param context Android context for asset access
     * @param filename Binary contraction filename in assets
     * @return ContractionData with loaded mappings, or null if loading fails
     */
    @JvmStatic
    fun loadContractions(context: Context, filename: String): ContractionData? {
        val startTime = System.currentTimeMillis()

        try {
            context.assets.open(filename).use { inputStream ->
                val channel = Channels.newChannel(inputStream)

                // Read entire file into byte buffer for fast access
                val buffer = ByteBuffer.allocate(inputStream.available()).apply {
                    order(ByteOrder.LITTLE_ENDIAN)
                }
                channel.read(buffer)
                buffer.flip()
                channel.close()

                // Parse header
                val magic = buffer.int
                if (magic != MAGIC) {
                    Log.e(TAG, String.format("Invalid magic number: 0x%08X (expected 0x%08X)", magic, MAGIC))
                    return null
                }

                val version = buffer.int
                if (version != EXPECTED_VERSION) {
                    Log.e(TAG, String.format("Unsupported version: %d (expected %d)", version, EXPECTED_VERSION))
                    return null
                }

                val nonPairedCount = buffer.int
                val pairedCount = buffer.int

                // Load non-paired contractions
                val nonPaired = mutableMapOf<String, String>()
                val known = mutableSetOf<String>()

                repeat(nonPairedCount) {
                    val key = readString(buffer)
                    val value = readString(buffer)
                    nonPaired[key] = value
                    known.add(value) // Value is the contraction with apostrophe
                }

                // Load paired contractions
                repeat(pairedCount) {
                    val contraction = readString(buffer)
                    known.add(contraction)
                }

                val loadTime = System.currentTimeMillis() - startTime

                if (BuildConfig.ENABLE_VERBOSE_LOGGING) {
                    Log.d(TAG, String.format(
                        "Loaded %d non-paired, %d total known contractions in %dms",
                        nonPaired.size, known.size, loadTime
                    ))
                }

                return ContractionData(nonPaired, known)
            }
        } catch (e: IOException) {
            Log.e(TAG, "Failed to load binary contractions", e)
            return null
        }
    }

    /**
     * Read a length-prefixed UTF-8 string from buffer.
     *
     * @param buffer ByteBuffer to read from
     * @return Decoded string
     */
    private fun readString(buffer: ByteBuffer): String {
        val length = buffer.short.toInt() and 0xFFFF // Unsigned short
        val bytes = ByteArray(length)
        buffer.get(bytes)
        return String(bytes, StandardCharsets.UTF_8)
    }
}
