package tribixbite.cleverkeys

import ai.onnxruntime.*
import android.content.Context
import android.graphics.PointF
import android.util.Log
import kotlinx.coroutines.*
import java.io.IOException
import java.nio.FloatBuffer
import java.util.concurrent.Executors

/**
 * Complete ONNX-based neural swipe predictor with Kotlin coroutines
 * Full implementation of transformer encoder-decoder architecture
 */
class OnnxSwipePredictorImpl private constructor(private val context: Context) {
    
    companion object {
        private const val TAG = "OnnxSwipePredictor"

        // CRITICAL: Sequence length constants matching ONNX model v106
        // These MUST match the ONNX model's expected input shapes
        private const val MAX_SEQUENCE_LENGTH = 250  // Encoder input sequence length (model_config.json: max_seq_length)
        private const val DECODER_SEQ_LENGTH = 20     // Decoder input sequence length (model_config.json: max_word_length)

        private const val TRAJECTORY_FEATURES = 6 // x, y, vx, vy, ax, ay
        private const val NORMALIZED_WIDTH = 1.0f
        private const val NORMALIZED_HEIGHT = 1.0f
        private const val DEFAULT_BEAM_WIDTH = 8
        private const val DEFAULT_MAX_LENGTH = 20  // model_config.json: max_word_length
        private const val DEFAULT_CONFIDENCE_THRESHOLD = 0.1f

        // Special tokens
        private const val PAD_IDX = 0
        private const val UNK_IDX = 1
        private const val SOS_IDX = 2
        private const val EOS_IDX = 3
        
        @Volatile
        private var instance: OnnxSwipePredictorImpl? = null
        
        fun getInstance(context: Context): OnnxSwipePredictorImpl {
            return instance ?: synchronized(this) {
                instance ?: OnnxSwipePredictorImpl(context).also { instance = it }
            }
        }
    }
    
    // ONNX Runtime components
    private val ortEnvironment: OrtEnvironment = OrtEnvironment.getEnvironment()
    private var encoderSession: OrtSession? = null
    private var decoderSession: OrtSession? = null
    
    // Processing components
    private val tokenizer = SwipeTokenizer()
    internal val trajectoryProcessor = SwipeTrajectoryProcessor() // Internal for dimension setting
    private val vocabulary = OptimizedVocabulary(context)
    
    // Configuration
    private var beamWidth = DEFAULT_BEAM_WIDTH
    private var maxLength = DEFAULT_MAX_LENGTH
    private var confidenceThreshold = DEFAULT_CONFIDENCE_THRESHOLD
    
    // State
    var isModelLoaded = false
        private set
    private var isInitialized = false
    private var debugLogger: ((String) -> Unit)? = null
    
    // Pre-allocated tensors for performance
    // Changed from LongArray to IntArray for new model format (int32)
    private var reusableTokensArray = IntArray(DECODER_SEQ_LENGTH)

    // High-performance tensor pooling for 50-70% speedup
    private val tensorPool = OptimizedTensorPool.getInstance(ortEnvironment)
    private val tensorMemoryManager = TensorMemoryManager(ortEnvironment)

    // Executor for async operations
    private val onnxExecutor = Executors.newSingleThreadExecutor { r ->
        Thread(r, "OnnxPredictor").apply { isDaemon = true }
    }
    
    /**
     * Initialize ONNX models and components
     */
    suspend fun initialize(): Boolean = withContext(Dispatchers.IO) {
        try {
            logDebug("🔄 Loading ONNX transformer models...")
            
            // Load encoder model with validation (ONNX model v106)
            val encoderData = loadModelFromAssets("models/swipe_encoder_android.onnx")
            logDebug("📥 Encoder model data loaded: ${encoderData.size} bytes")

            encoderSession = ortEnvironment.createSession(encoderData, createSessionOptions("Encoder"))
            logDebug("✅ Encoder session created successfully")

            // Validate encoder input/output schema
            encoderSession?.let { session ->
                val inputInfo = session.inputInfo
                val outputInfo = session.outputInfo
                logDebug("   Encoder inputs: ${inputInfo.keys}")
                logDebug("   Encoder outputs: ${outputInfo.keys}")

                // Validate expected input names (new model format uses actual_length instead of src_mask)
                val expectedInputs = setOf("trajectory_features", "nearest_keys", "actual_length")
                if (!inputInfo.keys.containsAll(expectedInputs)) {
                    throw RuntimeException("Encoder missing expected inputs: $expectedInputs")
                }
            }

            // Load decoder model with validation (ONNX model v106)
            val decoderData = loadModelFromAssets("models/swipe_decoder_android.onnx")
            logDebug("📥 Decoder model data loaded: ${decoderData.size} bytes")

            decoderSession = ortEnvironment.createSession(decoderData, createSessionOptions("Decoder"))
            logDebug("✅ Decoder session created successfully")

            // Validate decoder input/output schema
            decoderSession?.let { session ->
                val inputInfo = session.inputInfo
                val outputInfo = session.outputInfo
                logDebug("   Decoder inputs: ${inputInfo.keys}")
                logDebug("   Decoder outputs: ${outputInfo.keys}")

                // Validate expected input names (new model format uses actual_src_length instead of masks)
                val expectedInputs = setOf("memory", "target_tokens", "actual_src_length")
                if (!inputInfo.keys.containsAll(expectedInputs)) {
                    throw RuntimeException("Decoder missing expected inputs: $expectedInputs")
                }
            }
            
            // Initialize tokenizer and vocabulary
            tokenizer.initialize()
            val vocabLoaded = vocabulary.loadVocabulary()
            logDebug("📚 Vocabulary loaded: $vocabLoaded (words: ${vocabulary.getStats().totalWords})")
            
            // Perform complete pipeline validation test
            validateCompletePipeline()

            isModelLoaded = true
            isInitialized = true
            logDebug("🧠 ONNX neural prediction system ready and validated!")

            true
        } catch (e: Exception) {
            logE("Failed to initialize ONNX predictor", e)
            false
        }
    }
    
    /**
     * Predict words from swipe input
     */
    suspend fun predict(input: SwipeInput): PredictionResult = withContext(Dispatchers.Default) {
        if (!isInitialized) {
            if (!initialize()) {
                return@withContext PredictionResult.empty
            }
        }
        
        try {
            logDebug("🚀 Starting neural prediction for ${input.coordinates.size} points")

            // Log raw input data (first 10 points)
            logDebug("📍 Raw swipe input (first 10 points):")
            input.coordinates.take(10).forEachIndexed { i, point ->
                val timestamp = if (i < input.timestamps.size) input.timestamps[i] else 0L
                logDebug("   [$i] x=${point.x}, y=${point.y}, t=$timestamp")
            }

            // Extract trajectory features with automatic key detection
            val features = trajectoryProcessor.extractFeatures(input.coordinates, input.timestamps)
            logDebug("📊 Feature extraction complete:")
            logDebug("   Actual length: ${features.actualLength}")
            logDebug("   First 10 nearest keys: ${features.nearestKeys.take(10)}")
            logDebug("   First 3 normalized points: ${features.normalizedCoordinates.take(3).map { "(%.3f, %.3f)".format(it.x, it.y) }}")

            // Run encoder
            val encoderResult = runEncoder(features)
            val memory = encoderResult.get(0) as OnnxTensor

            // Create actual_length tensor (new model format)
            val actualLengthTensor = createActualLengthTensor(features)

            // Run beam search decoder
            logDebug("🔍 Starting beam search decoder...")
            val candidates = runBeamSearch(memory, actualLengthTensor, features)
            logDebug("✅ Beam search returned ${candidates.size} candidates")

            // Create final prediction result
            val result = createPredictionResult(candidates)

            logDebug("🧠 Neural prediction completed: ${result.size} candidates")
            result
            
        } catch (e: Exception) {
            logE("Neural prediction failed", e)
            PredictionResult.empty
        }
    }
    
    /**
     * Run encoder inference
     */
    private suspend fun runEncoder(features: SwipeTrajectoryProcessor.TrajectoryFeatures): OrtSession.Result {
        val encoderSession = this.encoderSession ?: throw IllegalStateException("Encoder not loaded")

        // Create input tensors (new model format with int32 and actual_length)
        val trajectoryTensor = createTrajectoryTensor(features)
        val nearestKeysTensor = createNearestKeysTensor(features)
        val actualLengthTensor = createActualLengthTensor(features)

        // Run encoder (new model format: actual_length instead of src_mask)
        val inputs = mapOf(
            "trajectory_features" to trajectoryTensor,
            "nearest_keys" to nearestKeysTensor,
            "actual_length" to actualLengthTensor
        )

        return encoderSession.run(inputs)
    }
    
    /**
     * Run beam search decoder with batched inference optimization
     */
    private suspend fun runBeamSearch(
        memory: OnnxTensor,
        actualLengthTensor: OnnxTensor,
        features: SwipeTrajectoryProcessor.TrajectoryFeatures
    ): List<BeamSearchCandidate> = withContext(Dispatchers.Default) {

        val decoderSession = this@OnnxSwipePredictorImpl.decoderSession ?: run {
            logE("❌ CRITICAL: Decoder session not initialized - cannot run beam search", RuntimeException("Decoder not initialized"))
            return@withContext emptyList()
        }

        // Initialize beam search
        val beams = mutableListOf<BeamSearchState>()
        beams.add(BeamSearchState(SOS_IDX, 0.0f, false))
        
        // Maintain separate lists for finished and active beams
        val finishedBeams = mutableListOf<BeamSearchState>()

        // Beam search loop with batched processing
        for (step in 0 until maxLength) {
            try {
                // Separate active beams (need further expansion)
                val activeBeams = beams.filter { !it.finished }

                if (activeBeams.isEmpty()) {
                    break  // All beams finished
                }

                // BATCHED PROCESSING: Process all beams in single inference call (30-50% speedup)
                // Returns beamWidth globally-selected beams (already sorted and pruned)
                val newBeams = processBatchedBeams(activeBeams, memory, actualLengthTensor, decoderSession)

                // Separate new finished beams from still-active ones
                val newFinished = newBeams.filter { it.finished }
                val stillActive = newBeams.filter { !it.finished }

                // Add newly finished beams to finished list
                finishedBeams.addAll(newFinished)

                // Update beams list for next iteration
                // No need to sort/take - processBatchedResults already did global top-k selection
                beams.clear()
                beams.addAll(stillActive)

                // Early stopping conditions (matching web demo):
                // 1. All beams finished
                // 2. After step 10, if we have at least 3 finished beams (good enough)
                if (beams.isEmpty() && finishedBeams.isNotEmpty()) {
                    break  // All beams finished
                }
                if (step >= 10 && finishedBeams.size >= 3) {
                    break  // Have enough good predictions
                }

            } catch (e: Exception) {
                logE("❌ Beam search failed at step $step", e)
                break
            }
        }

        // Return best hypotheses from both finished and remaining active beams
        val allFinalBeams = finishedBeams + beams

        // Convert beams to candidates
        val candidates = allFinalBeams.map { beam ->
            val word = tokenizer.tokensToWord(beam.tokens.drop(1)) // Remove SOS token
            BeamSearchCandidate(word, kotlin.math.exp(beam.score).toFloat())
        }

        // Log top 3 results
        val top3 = candidates.sortedByDescending { it.confidence }.take(3)
        logDebug("🏆 Top 3: ${top3.map { "'${it.word}'" }.joinToString(", ")}")

        candidates
    }
    
    /**
     * CRITICAL OPTIMIZATION: Process all beams in single batched inference call
     * Enhanced with tensor pooling for 50-70% additional speedup
     * Updated for new model format with actual_src_length instead of src_mask
     */
    private suspend fun processBatchedBeams(
        activeBeams: List<BeamSearchState>,
        memory: OnnxTensor,
        actualLengthTensor: OnnxTensor,
        decoderSession: OrtSession
    ): List<BeamSearchState> = withContext(Dispatchers.Default) {

        val batchSize = activeBeams.size
        val seqLength = DECODER_SEQ_LENGTH // Standard decoder sequence length (matches web demo)

        // CRITICAL FIX: Expand memory tensor to match batch size
        // Memory shape is [1, seq_len, 256], need [batchSize, seq_len, 256]
        val memoryShape = memory.info.shape
        val expandedMemoryShape = longArrayOf(batchSize.toLong(), memoryShape[1], memoryShape[2])

        val expandedMemory = if (batchSize > 1) {
            expandMemoryTensor(memory, batchSize)
        } else {
            memory  // No expansion needed for single beam
        }

        // OPTIMIZATION: Use tensor pool for batched tensors - eliminates allocation overhead
        // Changed to int32 for new model format
        val batchedTokensShape = longArrayOf(batchSize.toLong(), seqLength.toLong())

        // Create batched target_tokens tensor (int32 for new model format)
        val batchedTokensData = Array(batchSize) { IntArray(seqLength) }
        populateBatchedTokens(activeBeams, batchedTokensData, seqLength)
        val batchedTokensTensor = OnnxTensor.createTensor(ortEnvironment, batchedTokensData)

        try {
            // Prepare decoder inputs (new model format: actual_src_length instead of masks)
            val decoderInputs = mapOf(
                "memory" to expandedMemory,
                "target_tokens" to batchedTokensTensor,
                "actual_src_length" to actualLengthTensor
            )

            // SINGLE BATCHED INFERENCE
            val batchedOutput = decoderSession.run(decoderInputs)

            // Process batched results with enhanced error handling
            val newBeamCandidates = processBatchedResults(batchedOutput, activeBeams)

            // Cleanup
            batchedOutput.close()
            batchedTokensTensor.close()
            if (batchSize > 1 && expandedMemory != memory) {
                expandedMemory.close()
            }

            newBeamCandidates
        } catch (e: Exception) {
            logE("❌ Batched beam processing failed", e)
            emptyList()
        }
    }

    /**
     * TEMPORARY: Process beams ONE AT A TIME (like CLI) to test if batching is the issue
     * This matches the CLI test's beam search logic exactly
     */
    private suspend fun processBeamsNonBatched(
        activeBeams: List<BeamSearchState>,
        memory: OnnxTensor,
        srcMaskTensor: OnnxTensor,
        decoderSession: OrtSession
    ): List<BeamSearchState> = withContext(Dispatchers.Default) {

        // Collect ALL possible next hypotheses with their GLOBAL scores
        val allHypotheses = mutableListOf<Triple<Int, Int, Float>>()

        activeBeams.forEachIndexed { beamIndex, beam ->
            try {
                // Create tensors for THIS BEAM ONLY (batch size = 1, like CLI)
                val seqLength = DECODER_SEQ_LENGTH

                // Target tokens: pad to DECODER_SEQ_LENGTH
                val tokensArray = LongArray(seqLength) { PAD_IDX.toLong() }
                for (i in beam.tokens.indices) {
                    if (i < seqLength) {
                        tokensArray[i] = beam.tokens[i]
                    }
                }
                val tokensTensor = OnnxTensor.createTensor(ortEnvironment, Array(1) { tokensArray })

                // Target mask: false = valid, true = padded
                val maskArray = BooleanArray(seqLength) { true }
                for (i in beam.tokens.indices) {
                    if (i < seqLength) {
                        maskArray[i] = false
                    }
                }
                val maskTensor = OnnxTensor.createTensor(ortEnvironment, Array(1) { maskArray })

                // Src mask: all zeros (all valid) - matches CLI
                val memoryShape = memory.info.shape
                val srcMaskArray = BooleanArray(memoryShape[1].toInt()) { false }
                val decoderSrcMask = OnnxTensor.createTensor(ortEnvironment, Array(1) { srcMaskArray })

                // Run decoder with batch size = 1
                val decoderInputs = mapOf(
                    "memory" to memory,
                    "target_tokens" to tokensTensor,
                    "src_mask" to decoderSrcMask,
                    "target_mask" to maskTensor
                )

                val result = decoderSession.run(decoderInputs)
                val logitsTensor = result.get(0) as OnnxTensor
                val logits = logitsTensor.value as Array<Array<FloatArray>>

                // Get logits for last valid position
                val currentPos = beam.tokens.size - 1
                if (currentPos >= 0 && currentPos < logits[0].size) {
                    val vocabLogits = logits[0][currentPos]
                    val logProbs = applyLogSoftmax(vocabLogits)

                    // DEBUG: Log top 5 tokens for first beam at first step
                    if (beamIndex == 0 && currentPos == 0) {
                        val top5 = logProbs.withIndex().sortedByDescending { it.value }.take(5)
                        val top5Str = top5.joinToString(", ") { (idx, prob) ->
                            val char = tokenizer.tokenToChar(idx)
                            "$char($idx):${String.format("%.3f", prob)}"
                        }
                        logDebug("🔍 Step ${currentPos+1}, Beam 0 top 5 tokens: $top5Str")
                    }

                    // Add ALL tokens to hypothesis pool for global selection
                    logProbs.forEachIndexed { tokenId, logProb ->
                        val newScore = beam.score + logProb
                        allHypotheses.add(Triple(beamIndex, tokenId, newScore))
                    }
                }

                // Cleanup
                result.close()
                tokensTensor.close()
                maskTensor.close()
                decoderSrcMask.close()

            } catch (e: Exception) {
                logE("❌ Non-batched beam processing failed for beam $beamIndex", e)
            }
        }

        // GLOBAL top-k selection: Sort ALL possibilities and take top beamWidth
        // CRITICAL: logProbs are NEGATIVE, beam.score + logProb accumulates negative values
        // HIGHER (less negative) scores are BETTER → sort DESCENDING
        val topHypotheses = allHypotheses.sortedByDescending { it.third }.take(beamWidth)

        // DEBUG: Log selected tokens for first step
        if (activeBeams.firstOrNull()?.tokens?.size == 1) {
            val selectedTokens = topHypotheses.map { (_, tokenId, score) ->
                val char = tokenizer.tokenToChar(tokenId)
                "$char($tokenId):${String.format("%.3f", score)}"
            }
            logDebug("✅ Selected top $beamWidth tokens: ${selectedTokens.joinToString(", ")}")
        }

        // Construct new beams from globally-selected winners
        val newBeams = mutableListOf<BeamSearchState>()
        topHypotheses.forEach { (parentIndex, tokenId, score) ->
            val parentBeam = activeBeams[parentIndex]
            val newTokens = (parentBeam.tokens + tokenId.toLong()).toMutableList()
            val isFinished = (tokenId == EOS_IDX || tokenId == PAD_IDX || newTokens.size >= maxLength)
            // Use primary constructor: BeamSearchState(tokens, score, finished)
            newBeams.add(BeamSearchState(newTokens, score, isFinished))
        }

        newBeams
    }

    /**
     * Expand memory tensor from [1, seq_len, hidden_dim] to [batch_size, seq_len, hidden_dim]
     * by replicating the single batch across multiple batches
     */
    private fun expandMemoryTensor(memory: OnnxTensor, batchSize: Int): OnnxTensor {
        val memoryData = memory.value as Array<Array<FloatArray>>
        val seqLen = memoryData[0].size
        val hiddenDim = memoryData[0][0].size

        // Create expanded array [batch_size, seq_len, hidden_dim]
        val expandedData = Array(batchSize) { Array(seqLen) { FloatArray(hiddenDim) } }

        // Replicate the single batch to all batch positions
        for (b in 0 until batchSize) {
            for (s in 0 until seqLen) {
                System.arraycopy(memoryData[0][s], 0, expandedData[b][s], 0, hiddenDim)
            }
        }

        return OnnxTensor.createTensor(ortEnvironment, expandedData)
    }

    /**
     * Populate batched tensors efficiently from beam states
     */
    private fun populateBatchedTensors(
        activeBeams: List<BeamSearchState>,
        batchedTokensTensor: OnnxTensor,
        batchedMaskTensor: OnnxTensor,
        seqLength: Int
    ) {
        // Get direct access to tensor data for efficient population
        val tokensData = batchedTokensTensor.value as Array<LongArray>
        val maskData = batchedMaskTensor.value as Array<BooleanArray>

        // Populate batch data for all beams efficiently
        activeBeams.forEachIndexed { batchIndex, beam ->
            val tokensArray = tokensData[batchIndex]
            val maskArray = maskData[batchIndex]

            // Fill tokens and mask for this beam
            // CRITICAL: Mask convention is 1 = PADDED, 0 = VALID (matches web demo)
            for (seqIndex in 0 until seqLength) {
                if (seqIndex < beam.tokens.size) {
                    tokensArray[seqIndex] = beam.tokens[seqIndex]
                    maskArray[seqIndex] = false  // Valid token = 0
                } else {
                    tokensArray[seqIndex] = PAD_IDX.toLong()
                    maskArray[seqIndex] = true   // Padded position = 1
                }
            }
        }
    }

    /**
     * Process batched results with GLOBAL top-k selection
     * CRITICAL FIX: Select top-k candidates from ALL beams globally, not locally per beam
     * This prevents beam collapse where all beams originate from single parent
     */
    private fun processBatchedResults(
        batchedOutput: OrtSession.Result,
        activeBeams: List<BeamSearchState>
    ): List<BeamSearchState> {
        val batchedLogitsTensor = batchedOutput.get(0) as OnnxTensor
        val batchedTensorData = batchedLogitsTensor.value

        logDebug("📊 Decoder output shape: ${batchedLogitsTensor.info.shape.contentToString()}, type: ${batchedTensorData::class.simpleName}")

        if (batchedTensorData !is Array<*>) {
            return emptyList()
        }

        val batchedLogits = batchedTensorData as Array<Array<FloatArray>>

        // CRITICAL: Collect ALL possible next hypotheses with their GLOBAL scores
        // Each item is (parentBeamIndex, tokenId, newTotalScore)
        val allHypotheses = mutableListOf<Triple<Int, Int, Float>>()

        activeBeams.forEachIndexed { batchIndex, beam ->
            val currentPos = beam.tokens.size - 1

            if (currentPos >= 0 && currentPos < batchedLogits[batchIndex].size) {
                val vocabLogits = batchedLogits[batchIndex][currentPos]
                val logProbs = applyLogSoftmax(vocabLogits)

                // DEBUG: Log top 5 tokens for first beam at first step
                if (batchIndex == 0 && currentPos == 0) {
                    val top5 = logProbs.withIndex().sortedByDescending { it.value }.take(5)
                    val top5Str = top5.joinToString(", ") { (idx, prob) ->
                        val char = tokenizer.tokenToChar(idx)
                        "$char($idx):${String.format("%.3f", prob)}"
                    }
                    logDebug("🔍 Step ${currentPos+1}, Beam 0 top 5 tokens: $top5Str")
                }

                // CRITICAL FIX: Consider EVERY possible next token for global selection
                // This prevents beam collapse by allowing lower-scoring beams with
                // high-probability tokens to compete with higher-scoring beams
                logProbs.forEachIndexed { tokenId, logProb ->
                    val newScore = beam.score + logProb
                    allHypotheses.add(Triple(batchIndex, tokenId, newScore))
                }
            }
        }

        // GLOBAL top-k selection: Sort ALL possibilities and take top beamWidth
        // CRITICAL: logProbs are NEGATIVE, beam.score + logProb accumulates negative values
        // HIGHER (less negative) scores are BETTER → sort DESCENDING
        val topHypotheses = allHypotheses.sortedByDescending { it.third }.take(beamWidth)

        // DEBUG: Log selected tokens for first step
        if (activeBeams.firstOrNull()?.tokens?.size == 1) {
            val selectedTokens = topHypotheses.map { (_, tokenId, score) ->
                val char = tokenizer.tokenToChar(tokenId)
                "$char($tokenId):${String.format("%.3f", score)}"
            }
            logDebug("✅ Selected top $beamWidth tokens: ${selectedTokens.joinToString(", ")}")
        }

        // Construct new beam generation from globally-selected winners
        val newBeams = mutableListOf<BeamSearchState>()
        topHypotheses.forEach { (parentIndex, tokenId, score) ->
            val parentBeam = activeBeams[parentIndex]
            val newTokens = (parentBeam.tokens + tokenId.toLong()).toMutableList()
            val isFinished = (tokenId == EOS_IDX || tokenId == PAD_IDX || newTokens.size >= maxLength)
            // Use primary constructor: BeamSearchState(tokens, score, finished)
            newBeams.add(BeamSearchState(newTokens, score, isFinished))
        }

        return newBeams
    }

    /**
     * Create trajectory tensor from features - EXACT Java implementation match
     */
    private fun createTrajectoryTensor(features: SwipeTrajectoryProcessor.TrajectoryFeatures): OnnxTensor {
        // EXACT CLI test pattern: create ByteBuffer, fill via one FloatBuffer view, create tensor with fresh view
        val byteBuffer = java.nio.ByteBuffer.allocateDirect(MAX_SEQUENCE_LENGTH * TRAJECTORY_FEATURES * 4) // 4 bytes per float
        byteBuffer.order(java.nio.ByteOrder.nativeOrder()) // Match CLI test (line 118)

        // Fill buffer via first FloatBuffer view
        val fillBuffer = byteBuffer.asFloatBuffer()
        for (i in 0 until MAX_SEQUENCE_LENGTH) {
            val point = features.normalizedCoordinates.getOrNull(i) ?: PointF(0f, 0f)
            val velocity = features.velocities.getOrNull(i) ?: PointF(0f, 0f)
            val acceleration = features.accelerations.getOrNull(i) ?: PointF(0f, 0f)

            // Exact 6-feature layout matching web demo: [x, y, vx, vy, ax, ay]
            fillBuffer.put(point.x)          // Normalized x [0,1]
            fillBuffer.put(point.y)          // Normalized y [0,1]
            fillBuffer.put(velocity.x)       // Velocity x component (delta)
            fillBuffer.put(velocity.y)       // Velocity y component (delta)
            fillBuffer.put(acceleration.x)   // Acceleration x component (delta of delta)
            fillBuffer.put(acceleration.y)   // Acceleration y component (delta of delta)
        }

        // Create tensor with FRESH FloatBuffer view (position=0) - matches CLI test (line 120)
        val tensor = OnnxTensor.createTensor(ortEnvironment, byteBuffer.asFloatBuffer(), longArrayOf(1, MAX_SEQUENCE_LENGTH.toLong(), TRAJECTORY_FEATURES.toLong()))

        // HEX DUMP: First 30 float values (5 points × 6 features)
        val readBuffer = byteBuffer.asFloatBuffer()
        readBuffer.position(0)
        val hexDump = StringBuilder("🔬 Trajectory tensor hex dump (first 30 floats):\n")
        for (i in 0 until 30) {
            val value = readBuffer.get()
            hexDump.append(String.format("   [%2d] %.6f (0x%08x)\n", i, value, java.lang.Float.floatToRawIntBits(value)))
        }
        logD(hexDump.toString())

        // Track tensor with memory manager
        tensorMemoryManager.trackTensor(tensor, "TrajectoryTensor", longArrayOf(1, MAX_SEQUENCE_LENGTH.toLong(), TRAJECTORY_FEATURES.toLong()), MAX_SEQUENCE_LENGTH * TRAJECTORY_FEATURES * 4L)

        return tensor
    }
    
    /**
     * Create nearest keys tensor - int32 format for new model
     */
    private fun createNearestKeysTensor(features: SwipeTrajectoryProcessor.TrajectoryFeatures): OnnxTensor {
        // New model uses int32 for nearest_keys
        val keysData = IntArray(MAX_SEQUENCE_LENGTH)

        // FIX #41: Verify nearest_keys list size matches expected length
        if (features.nearestKeys.size != MAX_SEQUENCE_LENGTH) {
            logD("⚠️ WARNING: nearest_keys size (${features.nearestKeys.size}) != MAX_SEQUENCE_LENGTH ($MAX_SEQUENCE_LENGTH)")
        }

        // Fill array
        for (i in 0 until MAX_SEQUENCE_LENGTH) {
            keysData[i] = if (i < features.nearestKeys.size) {
                features.nearestKeys[i]
            } else {
                // Should never reach here if Fix #36 is working
                logD("⚠️ BUG: Padding nearest_keys with PAD at index $i")
                PAD_IDX
            }
        }

        // Log last 10 keys to verify repeat-last padding
        val last10 = features.nearestKeys.takeLast(10)
        logD("   Last 10 nearest keys: $last10")

        // HEX DUMP: First 15 values
        val hexDump = StringBuilder("🔬 Nearest keys tensor hex dump (first 15 ints):\n")
        for (i in 0 until 15) {
            hexDump.append(String.format("   [%2d] %d\n", i, keysData[i]))
        }
        logD(hexDump.toString())

        // Create tensor with shape [1, MAX_SEQUENCE_LENGTH] as int32
        return OnnxTensor.createTensor(ortEnvironment, Array(1) { keysData })
    }

    /**
     * Create actual_length tensor for new model format
     * This replaces the src_mask boolean tensor
     */
    private fun createActualLengthTensor(features: SwipeTrajectoryProcessor.TrajectoryFeatures): OnnxTensor {
        // New model uses actual_length as int32 scalar [1]
        val lengthData = intArrayOf(features.actualLength)
        logD("📏 Creating actual_length tensor: ${features.actualLength}")
        return OnnxTensor.createTensor(ortEnvironment, lengthData)
    }

    /**
     * Populate batched tokens tensor with int32 format for new model
     */
    private fun populateBatchedTokens(
        activeBeams: List<BeamSearchState>,
        tokensData: Array<IntArray>,
        seqLength: Int
    ) {
        activeBeams.forEachIndexed { batchIndex, beam ->
            val tokensArray = tokensData[batchIndex]

            // Fill tokens for this beam
            for (seqIndex in 0 until seqLength) {
                tokensArray[seqIndex] = if (seqIndex < beam.tokens.size) {
                    beam.tokens[seqIndex].toInt()
                } else {
                    PAD_IDX
                }
            }
        }
    }
    
    /**
     * Update reusable token arrays for beam (int32 format for new model)
     */
    private fun updateReusableTokens(beam: BeamSearchState, seqLength: Int) {
        reusableTokensArray.fill(PAD_IDX)

        for (i in 0 until minOf(beam.tokens.size, seqLength)) {
            reusableTokensArray[i] = beam.tokens[i].toInt()
        }
    }

    /**
     * Apply log-softmax to convert raw logits to log probabilities
     * log_softmax(x) = x - log(sum(exp(x)))
     * Uses numerical stability trick: subtract max before exp to prevent overflow
     */
    private fun applyLogSoftmax(logits: FloatArray): FloatArray {
        // Find max for numerical stability
        val maxLogit = logits.maxOrNull() ?: 0f

        // Compute exp(x - max) and sum
        val expValues = logits.map { kotlin.math.exp((it - maxLogit).toDouble()).toFloat() }
        val sumExp = expValues.sum()

        // Compute log probabilities: log(exp(x - max) / sum) = (x - max) - log(sum)
        val logSumExp = kotlin.math.ln(sumExp.toDouble()).toFloat() + maxLogit
        return logits.map { it - logSumExp }.toFloatArray()
    }

    /**
     * Get top K indices from probability array
     */
    private fun getTopKIndices(array: FloatArray, k: Int): IntArray {
        return array.withIndex()
            .sortedByDescending { it.value }
            .take(k)
            .map { it.index }
            .toIntArray()
    }
    
    /**
     * Load model from assets
     */
    private fun loadModelFromAssets(modelPath: String): ByteArray {
        return context.assets.open(modelPath).use { inputStream ->
            val available = inputStream.available()
            val modelData = ByteArray(available)
            var totalRead = 0
            
            while (totalRead < available) {
                val read = inputStream.read(modelData, totalRead, available - totalRead)
                if (read == -1) break
                totalRead += read
            }
            
            logDebug("Successfully loaded $totalRead bytes from $modelPath")
            modelData
        }
    }
    
    /**
     * Create optimized session options with model caching
     *
     * Model caching provides 50-80% faster loading after first run by creating
     * optimized .ort files in the app's cache directory.
     */
    private fun createSessionOptions(modelName: String): OrtSession.SessionOptions {
        return OrtSession.SessionOptions().apply {
            setOptimizationLevel(OrtSession.SessionOptions.OptLevel.ALL_OPT)
            setIntraOpNumThreads(0) // Auto-detect
            setMemoryPatternOptimization(true)

            // Enable model caching for 50-80% faster loading after first run
            try {
                val cacheDir = context.cacheDir
                val optimizedModelPath = "$cacheDir/onnx_${modelName.lowercase()}_optimized.ort"
                setOptimizedModelFilePath(optimizedModelPath)
                logDebug("📦 Model caching enabled: $optimizedModelPath")
            } catch (e: Exception) {
                logDebug("⚠️  Model caching unavailable: ${e.message}")
            }

            // Use default execution providers (QNN/XNNPACK not available in this ONNX version)
            logDebug("💻 Using default execution providers for $modelName")
        }
    }
    
    /**
     * Create prediction result from beam candidates
     */
    private fun createPredictionResult(candidates: List<BeamSearchCandidate>): PredictionResult {
        if (candidates.isEmpty()) {
            return PredictionResult.empty
        }

        // Filter through vocabulary if available
        val filteredCandidates = if (vocabulary.isLoaded()) {
            val result = vocabulary.filterPredictions(candidates.map { candidate ->
                OptimizedVocabulary.CandidateWord(candidate.word, candidate.confidence)
            }, createSwipeStats())
            logDebug("📋 Vocabulary filter: ${candidates.size} → ${result.size} candidates")

            // FALLBACK: If vocab filter removes ALL candidates, return top 3 raw beam search results
            if (result.isEmpty() && candidates.isNotEmpty()) {
                logDebug("⚠️ Vocabulary filter removed all candidates - returning top 3 raw predictions")
                candidates.take(3).map { OptimizedVocabulary.FilteredPrediction(it.word, it.confidence) }
            } else {
                result
            }
        } else {
            candidates.map { OptimizedVocabulary.FilteredPrediction(it.word, it.confidence) }
        }

        val words = filteredCandidates.map { it.word }
        val scores = filteredCandidates.map { (it.score * 1000).toInt() }

        return PredictionResult(words, scores)
    }
    
    private fun createSwipeStats(): OptimizedVocabulary.SwipeStats {
        // Create basic swipe stats for vocabulary filtering
        return OptimizedVocabulary.SwipeStats(
            pathLength = 0f,
            duration = 0f,
            straightnessRatio = 0f
        )
    }
    
    /**
     * Set configuration from NeuralConfig
     */
    fun setConfig(neuralConfig: NeuralConfig) {
        beamWidth = neuralConfig.beamWidth.coerceIn(neuralConfig.beamWidthRange)
        maxLength = neuralConfig.maxLength.coerceIn(neuralConfig.maxLengthRange)
        confidenceThreshold = neuralConfig.confidenceThreshold.coerceIn(neuralConfig.confidenceRange)

        logDebug("Neural config updated: beam_width=$beamWidth, max_length=$maxLength, threshold=$confidenceThreshold")
    }

    /**
     * Set configuration from Config (fallback for compatibility)
     * Creates NeuralConfig from SharedPreferences
     */
    fun setConfig(config: Config) {
        val prefs = Config.globalPrefs()
        val neuralConfig = NeuralConfig(prefs)
        setConfig(neuralConfig)
    }
    
    /**
     * Set debug logger
     */
    fun setDebugLogger(logger: ((String) -> Unit)?) {
        debugLogger = logger
    }
    
    /**
     * Set keyboard dimensions
     */
    fun setKeyboardDimensions(width: Int, height: Int) {
        trajectoryProcessor.setKeyboardDimensions(width, height)
    }
    
    /**
     * Set real key positions
     */
    fun setRealKeyPositions(keyPositions: Map<Char, PointF>) {
        trajectoryProcessor.setRealKeyPositions(keyPositions)
    }
    
    /**
     * Validate complete pipeline with test input
     */
    private suspend fun validateCompletePipeline() {
        logDebug("🧪 Validating complete ONNX prediction pipeline...")

        try {
            // Create test input that matches Java calibration activity
            val testPoints = listOf(
                PointF(100f, 200f), PointF(200f, 200f), PointF(300f, 200f),
                PointF(400f, 250f), PointF(500f, 200f)
            )
            val testTimestamps = testPoints.indices.map { it * 100L }
            val testInput = SwipeInput(testPoints, testTimestamps, emptyList())

            logDebug("   Test input: ${testInput.coordinates.size} points, ${testInput.pathLength} path length")

            // Test feature extraction (with automatic key detection)
            val features = trajectoryProcessor.extractFeatures(testInput.coordinates, testInput.timestamps)
            logDebug("   Feature extraction: ${features.actualLength} features, ${features.nearestKeys.size} nearest keys")

            // Test encoder
            val trajectoryTensor = createTrajectoryTensor(features)
            val nearestKeysTensor = createNearestKeysTensor(features)
            val actualLengthTensor = createActualLengthTensor(features)

            logDebug("   Tensor creation successful:")
            logDebug("     Trajectory: ${trajectoryTensor.info.shape.contentToString()}")
            logDebug("     Nearest keys: ${nearestKeysTensor.info.shape.contentToString()}")
            logDebug("     Actual length: ${actualLengthTensor.info.shape.contentToString()}")

            // Test encoder inference
            val encoderResult = runEncoder(features)
            val memory = encoderResult.get(0) as OnnxTensor
            logDebug("   Encoder inference: output shape ${memory.info.shape.contentToString()}")

            // Test one step of decoder (simplified validation)
            val beams = listOf(BeamSearchState(SOS_IDX, 0.0f, false))
            val session = decoderSession ?: run {
                logE("Decoder session not initialized for validation")
                return
            }
            val candidates = processBatchedBeams(beams, memory, actualLengthTensor, session)
            logDebug("   Decoder inference: ${candidates.size} beam candidates generated")

            // Cleanup test tensors
            trajectoryTensor.close()
            nearestKeysTensor.close()
            actualLengthTensor.close()
            encoderResult.close()

            logDebug("✅ Complete pipeline validation successful")

        } catch (e: Exception) {
            logE("Pipeline validation failed", e)
            throw RuntimeException("ONNX pipeline validation failed", e)
        }
    }

    /**
     * Debug logging - sends to both callback and global DebugLoggingManager
     */
    private fun logDebug(message: String) {
        Log.d(TAG, message)
        debugLogger?.invoke(message)
        // Also send to SwipeDebugActivity via broadcast if debug mode is enabled
        DebugLoggingManager.sendDebugLog(message)
    }
    
    private fun logE(message: String, throwable: Throwable) {
        Log.e(TAG, message, throwable)
        debugLogger?.invoke("ERROR: $message - ${throwable.message}")
        // Also send errors to SwipeDebugActivity
        DebugLoggingManager.sendDebugLog("ERROR: $message - ${throwable.message}")
    }
    
    /**
     * Cleanup resources with complete memory management and tensor pool.
     * Should be called from a coroutine context during service shutdown.
     */
    suspend fun cleanup() = withContext(Dispatchers.IO) {
        try {
            encoderSession?.close()
            encoderSession = null
        } catch (e: IllegalStateException) {
            logDebug("Encoder session already closed")
        }

        try {
            decoderSession?.close()
            decoderSession = null
        } catch (e: IllegalStateException) {
            logDebug("Decoder session already closed")
        }

        onnxExecutor.shutdown()

        // Cleanup optimized tensor pool (now properly suspended)
        tensorPool.cleanup()

        tensorMemoryManager.cleanup()
        isModelLoaded = false
        isInitialized = false
        logDebug("ONNX predictor cleaned up with optimized tensor pool and memory management")
    }
    
    /**
     * Beam search state
     */
    data class BeamSearchState(
        val tokens: MutableList<Long>,
        var score: Float,
        var finished: Boolean
    ) {
        constructor(startToken: Int, score: Float, finished: Boolean) : this(
            mutableListOf(startToken.toLong()), score, finished
        )
        
        constructor(other: BeamSearchState) : this(
            other.tokens.toMutableList(), other.score, other.finished
        )
    }
    
    /**
     * Beam search candidate result
     */
    data class BeamSearchCandidate(
        val word: String,
        val confidence: Float
    )
}

/**
 * Complete trajectory processor matching Java implementation
 */
class SwipeTrajectoryProcessor {

    companion object {
        private const val TAG = "SwipeTrajectoryProcessor"
        private const val MAX_TRAJECTORY_POINTS = 250  // Match ONNX model v106 max_seq_length
        private const val SMOOTHING_WINDOW = 3
    }
    
    private var keyboardWidth = 1080
    private var keyboardHeight = 400
    private var realKeyPositions = mapOf<Char, PointF>()
    
    data class TrajectoryFeatures(
        val coordinates: List<PointF>,
        val velocities: List<PointF>,  // Now stores (vx, vy) as PointF
        val accelerations: List<PointF>,  // Now stores (ax, ay) as PointF
        val nearestKeys: List<Int>,  // 2D format: single nearest key per point (matches trained model)
        val actualLength: Int,
        val normalizedCoordinates: List<PointF>
    )
    
    fun extractFeatures(coordinates: List<PointF>, timestamps: List<Long>): TrajectoryFeatures {
        // 1. Add validation check for keyboard dimensions
        if (keyboardWidth == 1080 && keyboardHeight == 400) {
            Log.w(TAG, "⚠️ Using default keyboard dimensions (1080x400). Key detection may be inaccurate. Ensure setKeyboardDimensions() is called.")
        }

        // 2. Filter duplicate starting points (FIX #34: prevents EOS prediction on static starts)
        val filteredCoords = filterDuplicateStartingPoints(coordinates)
        if (filteredCoords.size < coordinates.size) {
            Log.d(TAG, "🔧 Filtered ${coordinates.size - filteredCoords.size} duplicate starting points (${coordinates.size} → ${filteredCoords.size})")
        }

        // 3. Normalize coordinates FIRST (0-1 range) - matches web demo
        val normalizedCoords = normalizeCoordinates(filteredCoords)

        // 4. Detect nearest keys from the filtered, un-normalized coordinates
        // This is more accurate as it uses the raw pixel data
        val nearestKeys = detectNearestKeys(filteredCoords)

        // 4. Pad or truncate to MAX_TRAJECTORY_POINTS
        val finalCoords = padOrTruncate(normalizedCoords, MAX_TRAJECTORY_POINTS)

        // FIX #36: Pad nearest_keys by repeating last key (matches CLI test & training data)
        // Model was trained expecting last key to repeat, NOT PAD tokens
        val finalNearestKeys = if (nearestKeys.size >= MAX_TRAJECTORY_POINTS) {
            nearestKeys.take(MAX_TRAJECTORY_POINTS)
        } else {
            val lastKey = nearestKeys.lastOrNull() ?: 0 // Default to PAD_IDX if empty
            val padding = List(MAX_TRAJECTORY_POINTS - nearestKeys.size) { lastKey }
            nearestKeys + padding
        }

        // 5. Calculate velocities and accelerations on normalized coords (simple deltas)
        // This logic correctly matches the working web demo
        val velocities = mutableListOf<PointF>()
        val accelerations = mutableListOf<PointF>()

        for (i in 0 until MAX_TRAJECTORY_POINTS) {
            if (i == 0) {
                velocities.add(PointF(0f, 0f))
                accelerations.add(PointF(0f, 0f))
            } else {
                val vx = finalCoords[i].x - finalCoords[i-1].x
                val vy = finalCoords[i].y - finalCoords[i-1].y
                velocities.add(PointF(vx, vy))

                if (i == 1) {
                    accelerations.add(PointF(0f, 0f))
                } else {
                    val ax = vx - velocities[i-1].x
                    val ay = vy - velocities[i-1].y
                    accelerations.add(PointF(ax, ay))
                }
            }
        }

        // Verification logging
        if (coordinates.isNotEmpty()) {
            Log.d(TAG, "🔬 Feature calculation (first 3 points):")
            for (i in 0..2.coerceAtMost(finalCoords.size - 1)) {
                val nearestKey = finalNearestKeys.getOrNull(i) ?: -1
                Log.d(TAG, "   Point[$i]: x=${String.format("%.4f", finalCoords[i].x)}, y=${String.format("%.4f", finalCoords[i].y)}, " +
                         "vx=${String.format("%.4f", velocities[i].x)}, vy=${String.format("%.4f", velocities[i].y)}, " +
                         "ax=${String.format("%.4f", accelerations[i].x)}, ay=${String.format("%.4f", accelerations[i].y)}, " +
                         "nearest_key=$nearestKey")
            }
        }

        return TrajectoryFeatures(
            coordinates = finalCoords,
            velocities = velocities,
            accelerations = accelerations,
            nearestKeys = finalNearestKeys,
            actualLength = filteredCoords.size.coerceAtMost(MAX_TRAJECTORY_POINTS),
            normalizedCoordinates = finalCoords
        )
    }
    
    /**
     * Smooth trajectory using moving average. This implementation uses an explicit loop
     * to avoid potential issues with the .windowed() extension function on lists of objects.
     */
    private fun smoothTrajectory(coordinates: List<PointF>): List<PointF> {
        if (coordinates.size <= SMOOTHING_WINDOW) {
            return coordinates
        }

        val smoothedResult = mutableListOf<PointF>()
        for (i in coordinates.indices) {
            // Define the window starting at the current index, up to SMOOTHING_WINDOW elements.
            // This replicates the behavior of .windowed(size, step=1, partialWindows=true).
            val windowEnd = (i + SMOOTHING_WINDOW).coerceAtMost(coordinates.size)
            val window = coordinates.subList(i, windowEnd)

            if (window.isEmpty()) continue

            // Manually calculate the average for clarity and safety.
            var sumX = 0.0
            var sumY = 0.0
            for (p in window) {
                sumX += p.x
                sumY += p.y
            }
            smoothedResult.add(PointF((sumX / window.size).toFloat(), (sumY / window.size).toFloat()))
        }
        return smoothedResult
    }
    
    /**
     * Calculate velocity profile
     */
    private fun calculateVelocities(coordinates: List<PointF>, timestamps: List<Long>): List<Float> {
        if (coordinates.size < 2 || timestamps.size < 2) return listOf(0f)
        
        return coordinates.zip(timestamps).zipWithNext { (p1, t1), (p2, t2) ->
            val dx = p2.x - p1.x
            val dy = p2.y - p1.y
            val distance = kotlin.math.sqrt(dx * dx + dy * dy)
            val timeDelta = (t2 - t1) / 1000f // Convert to seconds
            
            if (timeDelta > 0) distance / timeDelta else 0f
        }
    }
    
    /**
     * Calculate acceleration profile
     */
    private fun calculateAccelerations(velocities: List<Float>, timestamps: List<Long>): List<Float> {
        if (velocities.size < 2 || timestamps.size < 2) return listOf(0f)
        
        return velocities.zipWithNext().zip(timestamps.zipWithNext()) { (v1, v2), (t1, t2) ->
            val velocityDelta = v2 - v1
            val timeDelta = (t2 - t1) / 1000f
            
            if (timeDelta > 0) velocityDelta / timeDelta else 0f
        }
    }
    
    /**
     * Filter duplicate starting points to prevent zero velocity (FIX #34)
     *
     * When touch starts, Android may report same coordinates multiple times before
     * finger movement is detected. This causes zero velocity/acceleration features,
     * making the model interpret the gesture as a tap instead of a swipe.
     *
     * Strategy: Remove consecutive duplicates from the start of the trajectory
     */
    private fun filterDuplicateStartingPoints(coordinates: List<PointF>): List<PointF> {
        if (coordinates.isEmpty()) return coordinates

        val threshold = 1f // 1 pixel tolerance for "same" coordinate
        val filtered = mutableListOf(coordinates[0])

        // Skip consecutive duplicates at the start
        var i = 1
        while (i < coordinates.size) {
            val prev = filtered.last()
            val curr = coordinates[i]

            val dx = kotlin.math.abs(curr.x - prev.x)
            val dy = kotlin.math.abs(curr.y - prev.y)

            // If this point is different from the last kept point, keep it and all remaining points
            if (dx > threshold || dy > threshold) {
                // Add this point and all remaining points
                filtered.addAll(coordinates.subList(i, coordinates.size))
                break
            }
            i++
        }

        return filtered
    }

    /**
     * Normalize coordinates to [0, 1] range
     */
    private fun normalizeCoordinates(coordinates: List<PointF>): List<PointF> {
        return coordinates.map { point ->
            PointF(
                (point.x / keyboardWidth).coerceIn(0f, 1f),
                (point.y / keyboardHeight).coerceIn(0f, 1f)
            )
        }
    }
    
    /**
     * Detect nearest key for each coordinate using real keyboard layout
     */
    private fun detectNearestKeys(coordinates: List<PointF>): List<Int> {
        return coordinates.map { point ->
            if (realKeyPositions.isNotEmpty()) {
                // Use actual keyboard layout positions - find nearest key
                val distances = realKeyPositions.map { (char, keyPos) ->
                    val dx = point.x - keyPos.x
                    val dy = point.y - keyPos.y
                    val distance = dx * dx + dy * dy
                    Pair(char, distance)
                }

                // Sort by distance and take closest key
                val closestKey = distances.minByOrNull { it.second }

                // Convert character to token index
                val char = closestKey?.first ?: 'a'
                if (char in 'a'..'z') (char - 'a') + 4 else 0 // a-z mapping to tokens 4-29
            } else {
                // Enhanced grid detection with proper QWERTY mapping
                detectKeysFromQwertyGrid(point)
            }
        }
    }

    /**
     * Detect nearest key using accurate QWERTY grid mapping
     */
    /**
     * Detect nearest key using grid-based approach with row offsets
     * MATCHES CLI test logic (TestOnnxPrediction.kt:52-70) that achieves 50% accuracy
     * FIX #39: Use staggered QWERTY layout (works with any keyboard dimensions)
     */
    private fun detectKeysFromQwertyGrid(point: PointF): Int {
        // QWERTY layout (same as CLI test)
        val row1 = "qwertyuiop"
        val row2 = "asdfghjkl"
        val row3 = "zxcvbnm"

        val keyWidth = keyboardWidth / 10f
        val keyHeight = keyboardHeight / 4f

        // Determine row (0-2)
        val row = (point.y / keyHeight).toInt().coerceIn(0, 2)

        // Determine column with row-specific offsets
        val col = when (row) {
            0 -> (point.x / keyWidth).toInt().coerceIn(0, 9)
            1 -> ((point.x - keyWidth/2) / keyWidth).toInt().coerceIn(0, 8)
            else -> ((point.x - keyWidth) / keyWidth).toInt().coerceIn(0, 6)
        }

        // Get character from layout
        val char = when (row) {
            0 -> row1.getOrNull(col)
            1 -> row2.getOrNull(col)
            else -> row3.getOrNull(col)
        } ?: return 0 // PAD_IDX

        // Convert to token index (a=4, z=29)
        return if (char in 'a'..'z') (char - 'a') + 4 else 0
    }
    
    /**
     * Pad or truncate list to target size.
     * Handles padding of mutable objects like PointF by creating new instances.
     * Returns original list if empty and cannot determine padding.
     */
    private fun <T> padOrTruncate(list: List<T>, targetSize: Int, paddingValue: T? = null): List<T> {
        if (list.size >= targetSize) {
            return list.take(targetSize)
        }

        val lastValue = list.lastOrNull()
        val paddingTemplate = paddingValue ?: lastValue ?: return list // Return original if cannot determine padding

        val paddingList = List(targetSize - list.size) {
            if (paddingTemplate is PointF) {
                // Create new instance to avoid reference sharing
                PointF(paddingTemplate.x, paddingTemplate.y) as T
            } else {
                paddingTemplate
            }
        }
        return list + paddingList
    }
    
    fun setKeyboardDimensions(width: Int, height: Int) {
        keyboardWidth = width
        keyboardHeight = height
        logD("Keyboard dimensions set: ${width}x${height}")
    }
    
    fun setRealKeyPositions(keyPositions: Map<Char, PointF>) {
        realKeyPositions = keyPositions
        logD("Real key positions updated: ${keyPositions.size} keys")
    }

    private fun logD(message: String) {
        Log.d(TAG, message)
        // Also send to SwipeDebugActivity
        DebugLoggingManager.sendDebugLog(message)
    }
}

/**
 * Complete tokenizer matching Java implementation
 */
