import { Injectable } from '@angular/core';
import { HttpService } from '../http/http.service';
import { NetworkService } from '../network/network.service';
import { BehaviorSubject, debounceTime, EMPTY, map, Observable, of, switchMap, tap, timer } from 'rxjs';
import { AuthService } from '../auth/auth.service';
import { environment } from 'src/environments/environment';
import { Console } from 'src/app/utils/console';

const PAGE_SIZE = 50;
const REFRESH_SIZE = 5;

export interface Notification {
  uuid: string;
  date: number;
  read: boolean;
  text: string;
  textElements: string[] | null | undefined;
  email: string;
}

@Injectable({providedIn: 'root'})
export class NotificationsService {

  public readonly notifications$ = new BehaviorSubject<Notification[]>([]);
  private _lastNb = 0;
  private _email = '';
  private _read: string[] = [];
  private _loaded = false;

  constructor(
    private readonly http: HttpService,
    private readonly network: NetworkService,
    private readonly auth: AuthService,
  ) {
    auth.auth$.pipe(
      switchMap(auth => {
        if (!auth || auth.isAnonymous) {
          this._lastNb = 0;
          this._email = '';
          this._loaded = false;
          this.notifications$.next([]);
          return of(false);
        }
        if (this._email !== auth.email) {
          this._email = auth.email;
          this._lastNb = 0;
          this._read = [];
          this._loaded = false;
          this.notifications$.next([]);
        }
        return network.server$;
      }),
      debounceTime(1000),
      switchMap(connected => connected ? timer(0, 5 * 60000) : of(false)),
    ).subscribe(v => {
      if (v === false) return; // not connected
      if (v === 0) this.loadFirstNotifications();
      else this.refreshNotifications(1);
    });
  }

  public get loaded(): boolean { return this._loaded; }

  private loadFirstNotifications(): void {
    if (this._loaded) return;
    const email = this._email;
    this.http.get<Notification[]>(environment.apiBaseUrl + '/notifications/v1?page=0&size=' + PAGE_SIZE).subscribe(list => {
      Console.info('Received ' + list.length + ' notifications from page 0');
      this._lastNb = list.length;
      this._loaded = true;
      this.notifications$.next(list.map(n => this.toDto(n, email)));
    });
  }

  private refreshNotifications(nb: number): void {
    const email = this._email;
    const size = Math.min(200, REFRESH_SIZE * nb);
    this.http.get<Notification[]>(environment.apiBaseUrl + '/notifications/v1?page=0&size=' + size).subscribe(list => {
      const newItems = list.filter(n => !this.notifications$.value.some(i => i.uuid === n.uuid));
      Console.info('Received ' + list.length + ' notifications from page 0 refresh ' + REFRESH_SIZE + ': ' + newItems.length + ' new items');
      if (newItems.length > 0) {
        this.notifications$.next([...newItems.map(n => this.toDto(n, email)), ...this.notifications$.value]);
        this.refreshNotifications(nb + 1);
      }
    });
  }

  public loadMore(): Observable<any> {
    const email = this._email;
    const page = Math.floor(this.notifications$.value.length / PAGE_SIZE);
    return this.http.get<Notification[]>(environment.apiBaseUrl + '/notifications/v1?page=' + page + '&size=' + PAGE_SIZE).pipe(
      tap(list => {
        Console.info('Received ' + list.length + ' notifications from page ' + page);
        this._lastNb = list.length;
        const newItems = list.filter(n => !this.notifications$.value.some(i => i.uuid === n.uuid));
        if (newItems.length > 0) {
          this.notifications$.next([...this.notifications$.value, ...newItems.map(n => this.toDto(n, email))].sort((n1, n2) => n2.date - n1.date));
        }
      }),
    );
  }

  private toDto(n: Notification, email: string): Notification {
    const dto = {...n, email};
    if (this._read.includes(n.uuid)) dto.read = true;
    return dto;
  }

  public get nbUnread$(): Observable<number> {
    return this.notifications$.pipe(map(list => list.reduce((p,v) => p + (v.read ? 0 : 1), 0)));
  }

  public get mayHaveMore() { return this._lastNb === PAGE_SIZE; }

  public markAsRead(notification: Notification): void {
    notification.read = true;
    this._read.push(notification.uuid);
    this.notifications$.next(this.notifications$.value);
    this.auth.auth$.pipe(
      switchMap(auth => {
        if (!auth || auth.isAnonymous || auth.email !== notification.email) return EMPTY;
        return this.network.server$;
      }),
      switchMap(connected => {
        if (!connected) return EMPTY;
        return this.http.put<Notification>(environment.apiBaseUrl + '/notifications/v1/' + notification.uuid, {...notification, email: undefined});
      }),
    ).subscribe(result => {
      let i = this._read.indexOf(notification.uuid);
      if (i >= 0) this._read.splice(i, 1);
      i = this.notifications$.value.findIndex(n => n.uuid === result.uuid);
      if (i >= 0) {
        this.notifications$.value[i] = result;
        this.notifications$.next(this.notifications$.value);
      }
    });
  }

}
