import { Injectable, Injector } from '@angular/core';
import { MenuItem } from 'src/app/components/menus/menu-item';
import { ANONYMOUS_USER, AuthService } from '../auth/auth.service';
import { ModalController, AlertController, Platform } from '@ionic/angular/standalone';
import { TrailCollection } from 'src/app/model/trail-collection';
import { Router } from '@angular/router';
import { Trail } from 'src/app/model/trail';
import { combineLatest, first, firstValueFrom, map, Observable, of, switchMap } from 'rxjs';
import { TrailCollectionService } from './trail-collection.service';
import { TraceRecorderService } from '../trace-recorder/trace-recorder.service';
import { isPublicationCollection, isPublicationLockedCollection, TrailCollectionType } from 'src/app/model/dto/trail-collection';
import { Track } from 'src/app/model/track';
import { TrackService } from './track.service';
import { filterDefined } from 'src/app/utils/rxjs/filter-defined';
import { TrailService } from './trail.service';
import { MyPublicTrailsService } from './my-public-trails.service';
import { MySelectionService } from './my-selection.service';
import { TrackMetadataSnapshot } from 'src/app/model/snapshots';
import { I18nService } from '../i18n/i18n.service';
import { ModerationService } from '../moderation/moderation.service';
import { environment } from 'src/environments/environment';
import Trailence from '../trailence.service';
import { TrailLinkService } from './link.service';

@Injectable({providedIn: 'root'})
export class TrailMenuService {

  constructor(
    private readonly injector: Injector
  ) {}

  public trailToCompare: Trail | undefined;

  public getTrailsMenu(trails: Trail[], fromTrail: boolean = false, fromCollection: TrailCollection | undefined = undefined, onlyGlobal: boolean = false, isAll: boolean = false, isModeration: boolean = false): MenuItem[] { // NOSONAR
    const menu: MenuItem[] = [];
    const email = this.injector.get(AuthService).email;
    if (trails.length === 1 && trails[0].fromModeration) isModeration = true;

    let inTools = false;
    let addTools = () => {
      if (inTools) return;
      menu.push(new MenuItem());
      inTools = true;
    };

    if (!onlyGlobal && !fromTrail && trails.length === 1) {
      menu.push(new MenuItem().setIcon('enter').setI18nLabel('pages.trail.actions.open')
        .setAction(() => this.openTrail(trails[0])))
    }

    if (!onlyGlobal && trails.length > 0 && !isPublicationCollection(fromCollection?.type) && !isModeration) {
      if (menu.length > 0 || trails.length === 1)
        menu.splice(0,0, new MenuItem().setSectionTitle(true).setI18nLabel('pages.trails.actions.prepare').setTextColor('medium'));

      menu.push(new MenuItem().setIcon('download').setI18nLabel('pages.trail.actions.download_map')
        .setAction(() => import('../functions/map-download').then(m => m.openMapDownloadDialog(this.injector, trails))));
      if (trails.length === 1) {
        menu.push(new MenuItem().setIcon('car').setI18nLabel('pages.trail.actions.go_to_departure')
        .setAction(() => import('../functions/go-to-departure').then(m => m.goToDeparture(this.injector, trails[0]))));
        if (email)
          menu.push(new MenuItem().setIcon('play-circle').setI18nLabel('trace_recorder.start_this_trail')
          .setAction(() => {
            this.injector.get(TraceRecorderService).start(trails[0]);
            const url = '/trail/' + trails[0].owner + '/' + trails[0].uuid;
            const router = this.injector.get(Router);
            if (!router.url.includes(url)) router.navigateByUrl(url);
          }));
      }
    }

    const allOwned = email && trails.every(t => t.owner === email);

    let hasPublish = false;
    if (((allOwned && fromCollection && !isPublicationLockedCollection(fromCollection.type)) || isModeration) && !onlyGlobal) {
      const collectionUuid = this.getUniqueCollectionUuid(trails);
      if (collectionUuid) {
        menu.push(new MenuItem().setSectionTitle(true).setI18nLabel('pages.trails.actions.modify').setTextColor('medium'));
        if (trails.length === 1) {
          menu.push(
            new MenuItem().setIcon('edit-text').setI18nLabel('pages.trails.actions.rename')
              .setAction(() => import('../functions/trail-rename').then(m => m.openRenameTrailDialog(this.injector, trails[0]))),
            new MenuItem().setIcon('date').setI18nLabel('pages.trails.actions.edit_date')
              .setAction(() => this.openTrailDatePopup(trails[0], undefined)),
            new MenuItem().setIcon('location').setI18nLabel('pages.trails.actions.edit_location')
              .setAction(() => import('../../components/location-popup/location-popup.component').then(m => m.openLocationDialog(this.injector, trails[0]))),
          );
        }
        menu.push(new MenuItem().setIcon('hiking').setI18nLabel('pages.trails.actions.edit_activity')
          .setAction(() => import('../../components/activity-popup/activity-popup.component').then(m => m.openActivityDialog(this.injector, trails))));
        if (!isPublicationCollection(fromCollection?.type) && !isModeration) {
          menu.push(new MenuItem().setIcon('tags').setI18nLabel('pages.trails.tags.menu_item')
            .setAction(() => import('../../components/tags/tags.component').then(m => m.openTagsDialog(this.injector, trails, collectionUuid))));
          if (trails.length === 1 &&
            !this.injector.get(MyPublicTrailsService).myPublicTrails$.value.some(p => p.privateUuid === trails[0].uuid) &&
            !this.injector.get(TrailService).getAllNow().some(t => t.publishedFromUuid === trails[0].uuid)
          ) {
            menu.push(
              new MenuItem(),
              new MenuItem().setIcon('web').setI18nLabel('publications.publish')
                .setDisabled(() => email === ANONYMOUS_USER)
                .setAction(() => this.startPublication(trails[0])),
            );
            hasPublish = true;
          }
        }
      }
    }

    if (!isPublicationCollection(fromCollection?.type) && email && !onlyGlobal && !isModeration && trails.length > 0) {
      if (!hasPublish) {
        menu.push(new MenuItem());
      }
      const hasAbsent = () => {
        const sel = this.injector.get(MySelectionService).getMySelectionNow();
        for (const trail of trails) {
          const isInSelection = sel.some(s => s.owner === trail.owner && s.uuid === trail.uuid);
          if (!isInSelection) return true;
        }
        return false;
      };
      const hasPresent = () => {
        const sel = this.injector.get(MySelectionService).getMySelectionNow();
        for (const trail of trails) {
          const isInSelection = sel.some(s => s.owner === trail.owner && s.uuid === trail.uuid);
          if (isInSelection) return true;
        }
        return false;
      };
      menu.push(new MenuItem().setIcon('star-filled').setI18nLabel('pages.trails.actions.add_to_my_selection')
        .setTextColor('my-selection')
        .setVisible(() => hasAbsent())
        .setAction(() => this.addToMySelection(trails))
      );
      menu.push(new MenuItem().setIcon('star-empty').setI18nLabel('pages.trails.actions.remove_from_my_selection')
        .setTextColor('my-selection')
        .setVisible(() => hasPresent())
        .setAction(() => this.removeFromMySelection(trails))
      );
    }

    if (trails.length === 2 && email && !onlyGlobal) {
      addTools();
      menu.push(new MenuItem().setIcon('compare').setI18nLabel('pages.trail.actions.compare')
        .setAction(() => {
          this.trailToCompare = undefined;
          const router = this.injector.get(Router);
          router.navigateByUrl('/trail/' + encodeURIComponent(trails[0].owner) + '/' + trails[0].uuid + '/' + encodeURIComponent(trails[1].owner) + '/' + trails[1].uuid + '?from=' + encodeURIComponent(router.url));
        })
      );
    }

    if (trails.length === 1 && !onlyGlobal) {
      if (this.trailToCompare) {
        addTools();
        menu.push(new MenuItem().setIcon('compare').setI18nLabel('pages.trail.actions.compare_with_this_one').setAction(() => {
          const trail1 = this.trailToCompare!;
          this.trailToCompare = undefined;
          const router = this.injector.get(Router);
          router.navigateByUrl('/trail/' + encodeURIComponent(trail1.owner) + '/' + trail1.uuid + '/' + encodeURIComponent(trails[0].owner) + '/' + trails[0].uuid + '?from=' + encodeURIComponent(router.url));
        }).setDisabled(this.trailToCompare === trails[0]));
      } else if (email) {
        addTools();
        menu.push(new MenuItem().setIcon('compare').setI18nLabel('pages.trail.actions.compare_with').setAction(() => {
          this.trailToCompare = trails[0];
          const i18n = this.injector.get(I18nService).texts;
          this.injector.get(AlertController).create({
            header: i18n.pages.trail.actions.compare_with,
            message: i18n.pages.trail.actions.compare_with_explanation,
            buttons: [{
              text: i18n.buttons.ok,
              role: 'cancel'
            }]
          }).then(a => a.present());
        }));
      }
    }
    if (trails.length > 1 && fromCollection && !isPublicationCollection(fromCollection.type) && !onlyGlobal) {
      addTools();
      menu.push(new MenuItem().setIcon('merge').setI18nLabel('pages.trail.actions.merge_trails')
        .setAction(() => import('../functions/merge-trails').then(m => m.mergeTrails(this.injector, trails, fromCollection.uuid))));
    }

    if (onlyGlobal && fromCollection && !isPublicationCollection(fromCollection.type)) {
      addTools();
      menu.push(new MenuItem().setIcon('add-circle').setI18nLabel('tools.import')
        .setAction(() => import('../functions/import').then(m => m.openImportTrailsDialog(this.injector, fromCollection.uuid))));
    }

    if (trails.length > 0) {
      addTools();
      menu.push(new MenuItem().setIcon('export').setI18nLabel('pages.trails.actions.export' + (onlyGlobal ? '_collection' : ''))
        .setAction(() => import('../functions/export').then(m => m.exportTrails(this.injector, trails))));
    }

    if (trails.length > 0 && fromCollection && !isPublicationCollection(fromCollection.type) && trails.some(t => t.owner !== ANONYMOUS_USER) &&
      (onlyGlobal || fromCollection.uuid === this.getUniqueCollectionUuid(trails))) {
      addTools();
      menu.push(new MenuItem().setIcon('share').setI18nLabel('pages.trails.actions.share_' + (onlyGlobal ? 'global' : trails.length === 1 ? 'trail' : 'trails'))
        .setAction(() => import('../../components/share-popup/share-popup.component')
          .then(m => m.openSharePopup(this.injector, fromCollection.uuid, onlyGlobal ? [] : trails))
        ));
    }

    if (trails.length === 1 && !onlyGlobal && trails[0].owner === 'trailence' && trails[0].source?.startsWith(environment.baseUrl)) {
      // public trail
      const link = trails[0].source;
      menu.push(new MenuItem());
      if (this.injector.get(Platform).is('capacitor')) {
        menu.push(new MenuItem().setIcon('share').setI18nLabel('pages.trails.actions.share_link')
          .setAction(() => {
            Trailence.share({link, title: trails[0].name});
          })
        );
      } else {
        menu.push(new MenuItem().setIcon('link').setI18nLabel('pages.trails.actions.copy_link')
          .setAction(() => {
            navigator.clipboard.writeText(link);
          })
        );
      }
    }

    if (trails.length === 1 && !onlyGlobal && trails[0].owner === email && !isPublicationCollection(fromCollection?.type)) {
      menu.push(
        new MenuItem()
          .setIcon('link')
          .setI18nLabel('pages.trails.actions.' + (this.injector.get(TrailLinkService).getLinkForTrail(trails[0].uuid) ? 'edit_link' : 'create_link'))
          .setAction(() => import('../../components/trail-link-popup/trail-link-popup.component').then(m => m.openTrailLink(this.injector, trails[0].uuid))),
      );
    }

    if (trails.length === 1 && !onlyGlobal) {
      menu.push(new MenuItem().setIcon('text').setI18nLabel('pages.pdf_popup.title')
        .setAction(() => import('../../components/pdf-popup/pdf-popup.component').then(m => m.openPdfPopup(this.injector, trails[0]))));
    }

    if (trails.length > 0 && !isAll && !isPublicationCollection(fromCollection?.type) && email && !onlyGlobal) {
      menu.push(
        new MenuItem(),
        new MenuItem().setIcon('collection-copy').setI18nLabel('pages.trails.actions.copy_to_collection')
        .setChildrenProvider(() => this.getCollectionsMenuItems(this.getAllCollectionsUuids(trails, email),
          (col) => import('../functions/copy-trails').then(m => m.copyTrailsTo(this.injector, trails, col, email, fromTrail)))
        )
      );
    }

    if (fromCollection && !isPublicationCollection(fromCollection.type) && !onlyGlobal && trails.length > 0 && allOwned) {
      const collectionUuid = this.getUniqueCollectionUuid(trails);
      if (fromCollection.uuid === collectionUuid) {
        menu.push(
          new MenuItem().setIcon('collection-move').setI18nLabel('pages.trails.actions.move_to_collection')
          .setChildrenProvider(() => this.getCollectionsMenuItems([collectionUuid],
            (col) => import('../functions/copy-trails').then(m => m.moveTrailsTo(this.injector, trails, col, email)))
          ));
      }
    }

    if (fromCollection && !onlyGlobal && trails.length > 0 && allOwned && !isPublicationLockedCollection(fromCollection.type)) {
      menu.push(
        new MenuItem(),
        new MenuItem().setIcon('trash').setI18nLabel('buttons.delete').setTextColor('danger')
          .setAction(() => import('../functions/delete-trails').then(m => m.confirmDeleteTrails(this.injector, trails, fromTrail))),
      );
    }

    if (fromCollection && onlyGlobal && trails.length > 0) {
      menu.push(
        new MenuItem(),
        new MenuItem().setIcon('compare').setI18nLabel('pages.find_duplicates.title')
          .setAction(() => import('../../components/find-duplicates/find-duplicates.component').then(m => m.openFindDuplicates(this.injector, fromCollection.uuid))),
      );
    }

    if (isModeration && this.injector.get(AuthService).auth?.admin) {
      menu.push(
        new MenuItem(),
        new MenuItem()
          .setFixedLabel('[Admin] Decline All')
          .setTextColor('danger')
          .setAction(() => this.declineAll(trails)),
      );
    }
    return menu;
  }

  private getCollectionsMenuItems(excludeUuids: string[], action: (col: TrailCollection) => void): Observable<MenuItem[]> {
    const collectionService = this.injector.get(TrailCollectionService);
    return collectionService.getMyCollectionsReady$().pipe(
      map(cols => {
        const list = cols.filter(col => !excludeUuids.includes(col.uuid));
        collectionService.sort(list);
        return list.map(
          col => {
            const item = new MenuItem();
            if (col.name === '' && col.type === TrailCollectionType.MY_TRAILS)
              item.setI18nLabel('my_trails');
            else
              item.setFixedLabel(col.name);
            item.setAction(() => action(col));
            return item;
          }
        );
      }),
      map(items => {
        items.splice(0, 0, new MenuItem()
          .setI18nLabel('pages.trails.actions.new_collection')
          .setIcon('add')
          .setAction(() => {
            this.injector.get(TrailCollectionService).collectionPopup(undefined, false)
            .then(result => {
              if (result.role !== 'apply' || !result.data) return;
              action(result.data as TrailCollection);
            });
          })
        );
        return items;
      })
    );
  }

  private getUniqueCollectionUuid(trails: Trail[]): string | undefined {
    if (trails.length === 0) return undefined;
    let uuid = trails[0].collectionUuid;
    for (let i = 1; i < trails.length; ++i) {
      if (trails[i].collectionUuid !== uuid) return undefined;
    }
    return uuid;
  }

  private getAllCollectionsUuids(trails: Trail[], owner: string): string[] {
    const result: string[] = [];
    for (const trail of trails) {
      if (trail.owner === owner && !result.includes(trail.collectionUuid))
        result.push(trail.collectionUuid)
    }
    return result;
  }

  public async openTrailDatePopup(trail: Trail, track: Track | TrackMetadataSnapshot | undefined) {
    const getTrackDate$ = track ? of(track.startDate) : this.injector.get(TrackService).getMetadata$(trail.currentTrackUuid, trail.owner).pipe(filterDefined(), map(t => t?.startDate), first());
    const modal = await Promise.all([
      firstValueFrom(getTrackDate$),
      import('../../components/datetime-popup/datetime-popup.component'),
    ]).then(([defaultDate, module]) => this.injector.get(ModalController).create({
      component: module.DateTimePopup,
      componentProps: {
        timestamp: trail.date ?? defaultDate,
        defaultTimestamp: defaultDate,
        maxTimestamp: Date.now(),
      },
      backdropDismiss: true,
      cssClass: 'small-modal',
    }));
    await modal.present();
    const result = await modal.onDidDismiss();
    if (result.role !== 'ok') return false;
    return new Promise<boolean>(resolve => this.injector.get(TrailService).doUpdate(trail, t => t.date = result.data, () => resolve(true)));
  }

  public async startPublication(trail: Trail) {
    const module = await import('../../components/trail/start-publication-modal/start-publication-modal.component');
    const modal = await this.injector.get(ModalController).create({
      component: module.StartPublicationModal,
      componentProps: {
        trail
      },
      cssClass: 'medium-modal'
    });
    await modal.present();
  }

  public addToMySelection(trails: Trail[]): void {
    this.injector.get(MySelectionService).getMySelection().pipe(
      first(),
      switchMap(current => {
        const newSelection = trails.filter(t => !current.some(c => c.owner === t.owner && c.uuid === t.uuid)).map(t => ({owner: t.owner, uuid: t.uuid}));
        if (newSelection.length === 0) return of([]);
        return combineLatest(newSelection.map(s => this.injector.get(MySelectionService).addSelection(s.owner, s.uuid).pipe(first())));
      })
    ).subscribe();
  }

  public removeFromMySelection(trails: Trail[]): void {
    this.injector.get(MySelectionService).getMySelection().pipe(
      first(),
    ).subscribe(current => {
      for (const s of current) {
        if (trails.some(t => t.owner === s.owner && t.uuid === s.uuid))
          this.injector.get(MySelectionService).deleteSelection(s.owner, s.uuid);
      }
    });
  }

  public openTrail(trail: Trail): void {
    const router = this.injector.get(Router);
    router.navigate(['trail', trail.owner, trail.uuid], {queryParams: { from: router.url }});
  }

  private async declineAll(trails: Trail[]) {
    const alert = await this.injector.get(AlertController).create({
      header: 'Decline All',
      message: 'Confirm?',
      buttons: [
        {
          text: 'Yes',
          cssClass: 'danger',
          role: 'ok',
          handler: () => this.injector.get(AlertController).dismiss(null, 'ok')
        }, {
          text: 'Cancel',
          role: 'cancel',
        }
      ]
    });
    alert.onDidDismiss()
    .then(result => {
      if (result.role === 'ok') {
        const service = this.injector.get(ModerationService);
        for (const trail of trails)
          service.reject(trail, 'Administrator declined.', undefined);
      }
    });
    await alert.present();
  }

}
