import { Injectable, Optional } from '@angular/core';
import { NavigationStart } from '@angular/router';
import * as i0 from "@angular/core";
import * as i1 from "./platform";
import * as i2 from "@angular/common";
import * as i3 from "@angular/router";
export class NavController {
    location;
    serializer;
    router;
    topOutlet;
    direction = DEFAULT_DIRECTION;
    animated = DEFAULT_ANIMATED;
    animationBuilder;
    guessDirection = 'forward';
    guessAnimation;
    lastNavId = -1;
    constructor(platform, location, serializer, router) {
        this.location = location;
        this.serializer = serializer;
        this.router = router;
        // Subscribe to router events to detect direction
        if (router) {
            router.events.subscribe((ev) => {
                if (ev instanceof NavigationStart) {
                    // restoredState is set if the browser back/forward button is used
                    const id = ev.restoredState ? ev.restoredState.navigationId : ev.id;
                    this.guessDirection = this.guessAnimation = id < this.lastNavId ? 'back' : 'forward';
                    this.lastNavId = this.guessDirection === 'forward' ? ev.id : id;
                }
            });
        }
        // Subscribe to backButton events
        platform.backButton.subscribeWithPriority(0, (processNextHandler) => {
            this.pop();
            processNextHandler();
        });
    }
    /**
     * This method uses Angular's [Router](https://angular.io/api/router/Router) under the hood,
     * it's equivalent to calling `this.router.navigateByUrl()`, but it's explicit about the **direction** of the transition.
     *
     * Going **forward** means that a new page is going to be pushed to the stack of the outlet (ion-router-outlet),
     * and that it will show a "forward" animation by default.
     *
     * Navigating forward can also be triggered in a declarative manner by using the `[routerDirection]` directive:
     *
     * ```html
     * <a routerLink="/path/to/page" routerDirection="forward">Link</a>
     * ```
     */
    navigateForward(url, options = {}) {
        this.setDirection('forward', options.animated, options.animationDirection, options.animation);
        return this.navigate(url, options);
    }
    /**
     * This method uses Angular's [Router](https://angular.io/api/router/Router) under the hood,
     * it's equivalent to calling:
     *
     * ```ts
     * this.navController.setDirection('back');
     * this.router.navigateByUrl(path);
     * ```
     *
     * Going **back** means that all the pages in the stack until the navigated page is found will be popped,
     * and that it will show a "back" animation by default.
     *
     * Navigating back can also be triggered in a declarative manner by using the `[routerDirection]` directive:
     *
     * ```html
     * <a routerLink="/path/to/page" routerDirection="back">Link</a>
     * ```
     */
    navigateBack(url, options = {}) {
        this.setDirection('back', options.animated, options.animationDirection, options.animation);
        return this.navigate(url, options);
    }
    /**
     * This method uses Angular's [Router](https://angular.io/api/router/Router) under the hood,
     * it's equivalent to calling:
     *
     * ```ts
     * this.navController.setDirection('root');
     * this.router.navigateByUrl(path);
     * ```
     *
     * Going **root** means that all existing pages in the stack will be removed,
     * and the navigated page will become the single page in the stack.
     *
     * Navigating root can also be triggered in a declarative manner by using the `[routerDirection]` directive:
     *
     * ```html
     * <a routerLink="/path/to/page" routerDirection="root">Link</a>
     * ```
     */
    navigateRoot(url, options = {}) {
        this.setDirection('root', options.animated, options.animationDirection, options.animation);
        return this.navigate(url, options);
    }
    /**
     * Same as [Location](https://angular.io/api/common/Location)'s back() method.
     * It will use the standard `window.history.back()` under the hood, but featuring a `back` animation
     * by default.
     */
    back(options = { animated: true, animationDirection: 'back' }) {
        this.setDirection('back', options.animated, options.animationDirection, options.animation);
        return this.location.back();
    }
    /**
     * This methods goes back in the context of Ionic's stack navigation.
     *
     * It recursively finds the top active `ion-router-outlet` and calls `pop()`.
     * This is the recommended way to go back when you are using `ion-router-outlet`.
     *
     * Resolves to `true` if it was able to pop.
     */
    async pop() {
        let outlet = this.topOutlet;
        while (outlet) {
            if (await outlet.pop()) {
                return true;
            }
            else {
                outlet = outlet.parentOutlet;
            }
        }
        return false;
    }
    /**
     * This methods specifies the direction of the next navigation performed by the Angular router.
     *
     * `setDirection()` does not trigger any transition, it just sets some flags to be consumed by `ion-router-outlet`.
     *
     * It's recommended to use `navigateForward()`, `navigateBack()` and `navigateRoot()` instead of `setDirection()`.
     */
    setDirection(direction, animated, animationDirection, animationBuilder) {
        this.direction = direction;
        this.animated = getAnimation(direction, animated, animationDirection);
        this.animationBuilder = animationBuilder;
    }
    /**
     * @internal
     */
    setTopOutlet(outlet) {
        this.topOutlet = outlet;
    }
    /**
     * @internal
     */
    consumeTransition() {
        let direction = 'root';
        let animation;
        const animationBuilder = this.animationBuilder;
        if (this.direction === 'auto') {
            direction = this.guessDirection;
            animation = this.guessAnimation;
        }
        else {
            animation = this.animated;
            direction = this.direction;
        }
        this.direction = DEFAULT_DIRECTION;
        this.animated = DEFAULT_ANIMATED;
        this.animationBuilder = undefined;
        return {
            direction,
            animation,
            animationBuilder,
        };
    }
    navigate(url, options) {
        if (Array.isArray(url)) {
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            return this.router.navigate(url, options);
        }
        else {
            /**
             * navigateByUrl ignores any properties that
             * would change the url, so things like queryParams
             * would be ignored unless we create a url tree
             * More Info: https://github.com/angular/angular/issues/18798
             */
            const urlTree = this.serializer.parse(url.toString());
            if (options.queryParams !== undefined) {
                urlTree.queryParams = { ...options.queryParams };
            }
            if (options.fragment !== undefined) {
                urlTree.fragment = options.fragment;
            }
            /**
             * `navigateByUrl` will still apply `NavigationExtras` properties
             * that do not modify the url, such as `replaceUrl` which is why
             * `options` is passed in here.
             */
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            return this.router.navigateByUrl(urlTree, options);
        }
    }
    /** @nocollapse */ static ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "16.2.12", ngImport: i0, type: NavController, deps: [{ token: i1.Platform }, { token: i2.Location }, { token: i3.UrlSerializer }, { token: i3.Router, optional: true }], target: i0.ɵɵFactoryTarget.Injectable });
    /** @nocollapse */ static ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "16.2.12", ngImport: i0, type: NavController, providedIn: 'root' });
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "16.2.12", ngImport: i0, type: NavController, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: i1.Platform }, { type: i2.Location }, { type: i3.UrlSerializer }, { type: i3.Router, decorators: [{
                    type: Optional
                }] }]; } });
const getAnimation = (direction, animated, animationDirection) => {
    if (animated === false) {
        return undefined;
    }
    if (animationDirection !== undefined) {
        return animationDirection;
    }
    if (direction === 'forward' || direction === 'back') {
        return direction;
    }
    else if (direction === 'root' && animated === true) {
        return 'forward';
    }
    return undefined;
};
const DEFAULT_DIRECTION = 'auto';
const DEFAULT_ANIMATED = undefined;
//# sourceMappingURL=data:application/json;base64,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