"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RULE_DOCS_EXTENSION = exports.RULE_NAME = void 0;
const utils_1 = require("@angular-eslint/utils");
const create_eslint_rule_1 = require("../utils/create-eslint-rule");
exports.RULE_NAME = 'directive-selector';
exports.default = (0, create_eslint_rule_1.createESLintRule)({
    name: exports.RULE_NAME,
    meta: {
        type: 'suggestion',
        docs: {
            description: 'Directive selectors should follow given naming rules. See more at https://angular.dev/style-guide#choosing-directive-selectors.',
        },
        schema: [
            {
                oneOf: [
                    // Single config object
                    {
                        type: 'object',
                        properties: {
                            type: {
                                oneOf: [
                                    { type: 'string' },
                                    {
                                        type: 'array',
                                        items: {
                                            type: 'string',
                                            enum: [
                                                utils_1.SelectorUtils.OPTION_TYPE_ELEMENT,
                                                utils_1.SelectorUtils.OPTION_TYPE_ATTRIBUTE,
                                            ],
                                        },
                                    },
                                ],
                            },
                            prefix: {
                                oneOf: [{ type: 'string' }, { type: 'array' }],
                            },
                            style: {
                                type: 'string',
                                enum: [
                                    utils_1.ASTUtils.OPTION_STYLE_CAMEL_CASE,
                                    utils_1.ASTUtils.OPTION_STYLE_KEBAB_CASE,
                                ],
                            },
                        },
                        additionalProperties: false,
                    },
                    // Array of 1-2 config objects
                    {
                        type: 'array',
                        items: {
                            type: 'object',
                            properties: {
                                type: {
                                    type: 'string',
                                    enum: [
                                        utils_1.SelectorUtils.OPTION_TYPE_ELEMENT,
                                        utils_1.SelectorUtils.OPTION_TYPE_ATTRIBUTE,
                                    ],
                                },
                                prefix: {
                                    oneOf: [{ type: 'string' }, { type: 'array' }],
                                },
                                style: {
                                    type: 'string',
                                    enum: [
                                        utils_1.ASTUtils.OPTION_STYLE_CAMEL_CASE,
                                        utils_1.ASTUtils.OPTION_STYLE_KEBAB_CASE,
                                    ],
                                },
                            },
                            additionalProperties: false,
                            required: ['type'],
                        },
                        minItems: 1,
                        maxItems: 2,
                    },
                ],
            },
        ],
        messages: {
            prefixFailure: 'The selector should start with one of these prefixes: {{prefix}}',
            styleFailure: 'The selector should be {{style}}',
            typeFailure: 'The selector should be used as an {{type}}',
        },
    },
    defaultOptions: [
        {
            type: '',
            prefix: '',
            style: '',
        },
    ],
    create(context, [options]) {
        // Normalize options to a consistent format using shared utility
        const configByType = utils_1.SelectorUtils.normalizeOptionsToConfigs(options);
        return {
            [utils_1.Selectors.DIRECTIVE_CLASS_DECORATOR](node) {
                const rawSelectors = utils_1.ASTUtils.getDecoratorPropertyValue(node, 'selector');
                if (!rawSelectors) {
                    return;
                }
                // Parse selectors once for reuse
                const parsedSelectors = utils_1.SelectorUtils.parseSelectorNode(rawSelectors);
                if (!parsedSelectors || parsedSelectors.length === 0) {
                    return;
                }
                // For multiple configs, determine the actual selector type
                let applicableConfig = null;
                if (configByType.size > 1) {
                    // Multiple configs - need to determine which one applies
                    const actualType = utils_1.SelectorUtils.getActualSelectorType(rawSelectors);
                    if (!actualType) {
                        return;
                    }
                    const config = configByType.get(actualType);
                    if (!config) {
                        // No config defined for this selector type
                        return;
                    }
                    applicableConfig = config;
                }
                else {
                    // Single config or single type extracted from array
                    const firstEntry = configByType.entries().next();
                    if (!firstEntry.done) {
                        applicableConfig = firstEntry.value[1];
                    }
                }
                if (!applicableConfig) {
                    return;
                }
                const { type, prefix, style } = applicableConfig;
                const isValidOptions = utils_1.SelectorUtils.checkValidOptions(type, prefix, style);
                if (!isValidOptions) {
                    return;
                }
                const hasExpectedSelector = utils_1.SelectorUtils.checkSelector(rawSelectors, type, (0, utils_1.arrayify)(prefix), style, parsedSelectors);
                if (hasExpectedSelector === null) {
                    return;
                }
                // Directive-specific validation logic (simpler than component)
                if (!hasExpectedSelector.hasExpectedType) {
                    utils_1.SelectorUtils.reportTypeError(rawSelectors, type, context);
                }
                else if (!hasExpectedSelector.hasExpectedStyle) {
                    utils_1.SelectorUtils.reportStyleError(rawSelectors, style, context);
                }
                else if (!hasExpectedSelector.hasExpectedPrefix) {
                    utils_1.SelectorUtils.reportPrefixError(rawSelectors, prefix, context);
                }
            },
        };
    },
});
exports.RULE_DOCS_EXTENSION = {
    rationale: "Consistent directive selector naming conventions help identify which directives belong to your application versus third-party libraries, prevent naming collisions with native HTML attributes and other directives, and make code reviews and debugging easier. For example, using a camelCase attribute selector with a prefix like 'appHighlight' makes it immediately clear that this is a custom directive from your application.",
};
