import { Injectable } from '@angular/core';
import { Router } from '@angular/router';
import { MenuItem } from 'src/app/components/menus/menu-item';
import { Track } from 'src/app/model/track';
import { Trail } from 'src/app/model/trail';
import { TrackService } from 'src/app/services/database/track.service';
import { TraceRecorderService } from 'src/app/services/trace-recorder/trace-recorder.service';
import { TrackUtils } from 'src/app/utils/track-utils';

@Injectable({
  providedIn: 'root'
})
export class ReplayService {

  public readonly canReplay = true;

  constructor(
    private readonly trackService: TrackService,
    private readonly traceRecorder: TraceRecorderService,
    private readonly router: Router,
  ) {}

  public addTrailMenu(menu: MenuItem[], trail: Trail): void {
    menu.push(
      new MenuItem(),
      new MenuItem().setFixedLabel('[Dev] Replay original').setAction(() => this.replay(trail.originalTrackUuid, trail.owner)),
      new MenuItem().setFixedLabel('[Dev] Replay following original').setAction(() => this.replay(trail.originalTrackUuid, trail.owner, trail)),
      new MenuItem().setFixedLabel('[Dev] Replay following original time').setAction(() => this.replay(trail.originalTrackUuid, trail.owner, trail, 10, false, true)),
      new MenuItem().setFixedLabel('[Dev] Replay current').setAction(() => this.replay(trail.currentTrackUuid, trail.owner)),
      new MenuItem().setFixedLabel('[Dev] Replay current, original time').setAction(() => this.replay(trail.currentTrackUuid, trail.owner, undefined, 25, false, true)),
      new MenuItem().setFixedLabel('[Dev] Replay following current').setAction(() => this.replay(trail.currentTrackUuid, trail.owner, trail)),
      new MenuItem().setFixedLabel('[Dev] Replay following current, slowly').setAction(() => this.replay(trail.currentTrackUuid, trail.owner, trail, 250)),
      new MenuItem().setFixedLabel('[Dev] Replay following current, approximate').setAction(() => this.replay(trail.currentTrackUuid, trail.owner, trail, 50, true)),
      new MenuItem().setFixedLabel('[Dev] Replay following current, original time').setAction(() => this.replay(trail.currentTrackUuid, trail.owner, trail, 25, false, true)),
    );
  }

  public replay(trackUuid: string, owner: string, following?: Trail, speed: number = 50, approximative: boolean = false, originalTime: boolean = false): void {
    this.trackService.getFullTrackReady$(trackUuid, owner).subscribe(track => {
      this.startReplay(track, following, speed, approximative, originalTime);
    });
  }

  private startReplay(track: Track, following: Trail | undefined, speed: number, approximative: boolean, originalTime: boolean): void {
    const originalGetCurrentPosition = window.navigator.geolocation.getCurrentPosition;
    const originalWatchPosition = window.navigator.geolocation.watchPosition;
    const originalClearWatch = window.navigator.geolocation.clearWatch;

    window.navigator.geolocation.getCurrentPosition = function(success, error) {
      if (error) error({code: GeolocationPositionError.POSITION_UNAVAILABLE, message: 'Fake!'} as GeolocationPositionError);
    };

    window.navigator.geolocation.clearWatch = function() {
      window.navigator.geolocation.getCurrentPosition = originalGetCurrentPosition;
      window.navigator.geolocation.watchPosition = originalWatchPosition;
      window.navigator.geolocation.clearWatch = originalClearWatch;
    };

    let segmentIndex = 0;
    let pointIndex = 0;
    let timeDiff: number | undefined = undefined;
    let approximativeDiff = 0;
    let startTime = Date.now();
    let totalIndex = 0;
    const sendNextPoint = (success: PositionCallback, error: PositionErrorCallback | null | undefined) => {
      if ((window as any)['_isDemo'] && pointIndex >= 50) return;
      if (segmentIndex >= track.segments.length) return;
      const segment = track.segments[segmentIndex];
      if (pointIndex >= segment.points.length) {
        segmentIndex++;
        pointIndex = 0;
        setTimeout(() => sendNextPoint(success, error), 10);
        return;
      }
      const point = segment.points[pointIndex];
      totalIndex++;
      let time = originalTime ? point.time : track.metadata.duration !== undefined ? point.time ?? 0 : startTime + totalIndex * 100;
      if (pointIndex > 0 && segment.points[pointIndex - 1].time === time && time) {
        let first = pointIndex - 1;
        while (first > 0 && segment.points[first - 1].time === time) first--;
        let last = pointIndex;
        while (last < segment.points.length - 1 && segment.points[last + 1].time === time) last++;
        const distance = TrackUtils.distanceBetween(segment.points, first, last);
        const pd = TrackUtils.distanceBetween(segment.points, first, pointIndex);
        if (last < segment.points.length - 1) {
          const nextTime = segment.points[last + 1].time;
          if (nextTime)
            time = time + (nextTime - time) / distance * pd;
        } else
          time += (pointIndex - first) * 100;
      }
      timeDiff ??= originalTime || !time ? 0 : Date.now() - time;
      if (approximative && (pointIndex % 5 == 0)) {
        approximativeDiff = approximativeDiff + (Math.random() - 0.5) * 0.0001;
        if (approximativeDiff > 0.0002) approximativeDiff = 0.0002 - Math.random() * 0.00005;
        if (approximativeDiff < -0.0002) approximativeDiff = -0.0002 + Math.random() * 0.00005;
      }
      success({
        coords: {
          latitude: point.pos.lat + approximativeDiff,
          longitude: point.pos.lng + approximativeDiff,
          altitude: point.ele !== undefined ? (point.ele + approximativeDiff * 20000) : null,
          accuracy: point.posAccuracy ?? 1,
          altitudeAccuracy: point.eleAccuracy ?? null,
          heading: point.heading ?? null,
          speed: point.speed ?? null,
          toJSON: function() {},
        },
        timestamp: !time ? Date.now() : time + timeDiff,
        toJSON: function() {},
      });
      pointIndex++;
      setTimeout(() => sendNextPoint(success, error), (pointIndex % 10) == 0 ? Math.max(speed * 3, 150) : speed);
    };

    window.navigator.geolocation.watchPosition = function(success, error) {
      setTimeout(() => sendNextPoint(success, error), 2500);
      return 1;
    };

    this.traceRecorder.start(following);
    if (!following)
      this.router.navigateByUrl('/trail');
  }

}
