import { Injectable, Injector } from "@angular/core";
import { BehaviorSubject, Observable, catchError, combineLatest, defaultIfEmpty, first, map, of, switchMap, takeWhile, tap, throwError, timeout } from "rxjs";
import { TrailCollection } from "src/app/model/trail-collection";
import { OwnedStore, UpdatesResponse } from "./owned-store";
import { isPublicationCollection, TrailCollectionDto, TrailCollectionType } from "src/app/model/dto/trail-collection";
import { DatabaseService, TRAIL_COLLECTION_TABLE_NAME, TRAIL_TABLE_NAME } from "./database.service";
import { environment } from "src/environments/environment";
import { HttpService } from "../http/http.service";
import { VersionedDto } from "src/app/model/dto/versioned";
import { ModalController, AlertController } from '@ionic/angular/standalone';
import { MenuItem } from 'src/app/components/menus/menu-item';
import { I18nService } from '../i18n/i18n.service';
import { TagService } from './tag.service';
import { TrailService } from './trail.service';
import { Progress, ProgressService } from '../progress/progress.service';
import Dexie from 'dexie';
import { Router } from '@angular/router';
import { Trail } from 'src/app/model/trail';
import { DependenciesService } from './dependencies.service';
import { filterDefined } from 'src/app/utils/rxjs/filter-defined';
import { PreferencesService } from '../preferences/preferences.service';
import { QuotaService } from '../auth/quota.service';
import { ShareService } from './share.service';
import { ANONYMOUS_USER, AuthService } from '../auth/auth.service';
import { Console } from 'src/app/utils/console';
import { collection$items } from 'src/app/utils/rxjs/collection$items';

@Injectable({
    providedIn: 'root'
})
export class TrailCollectionService {

  private readonly _store: TrailCollectionStore;

  constructor(
    http: HttpService,
    private readonly injector: Injector,
  ) {
    this._store = new TrailCollectionStore(injector, http);
  }

  public getMyCollectionsReady$(): Observable<TrailCollection[]> {
    return this._store.getAll$().pipe(
      collection$items(),
      map(collections => collections.filter(c => !isPublicationCollection(c.type))),
    );
  }

  public getAllCollectionsReady$(): Observable<TrailCollection[]> {
    return this._store.getAll$().pipe(collection$items());
  }

  public getCollection$(uuid: string, owner: string): Observable<TrailCollection | null> {
    return this._store.getItem$(uuid, owner);
  }

  public getCollection(uuid: string, owner: string): TrailCollection | null {
    return this._store.getItem(uuid, owner);
  }

  public getMyTrails$(): Observable<TrailCollection> {
    return this._store.getAll$().pipe(
      switchMap(collections => collections.length === 0 ? of([]) : combineLatest(collections)),
      map(collections => collections.find(collection => collection?.type === TrailCollectionType.MY_TRAILS)),
      filterDefined(),
      first()
    );
  }

  public getOrCreatePublicationDraft(): Observable<TrailCollection> {
    return this._store.getAllWhenLoaded$().pipe(
      collection$items(),
      switchMap(collections => {
        const col = collections.find(c => c.type === TrailCollectionType.PUB_DRAFT);
        if (col) return of(col);
        return this.create(new TrailCollection({
          owner: this.injector.get(AuthService).email,
          type: TrailCollectionType.PUB_DRAFT,
        }))
      }),
      filterDefined(),
      first(),
    );
  }

  public getOrCreatePublicationSubmit(): Observable<TrailCollection> {
    return this._store.getAllWhenLoaded$().pipe(
      collection$items(),
      switchMap(collections => {
        const col = collections.find(c => c.type === TrailCollectionType.PUB_SUBMIT);
        if (col) return of(col);
        return this.create(new TrailCollection({
          owner: this.injector.get(AuthService).email,
          type: TrailCollectionType.PUB_SUBMIT,
        }))
      }),
      filterDefined(),
      first(),
    );
  }

  public getCollectionName$(uuid: string, owner?: string): Observable<string> {
    return this.getCollection$Name$(this.getCollection$(uuid, owner ?? this.injector.get(AuthService).email ?? ''));
  }

  public getCollection$Name$(col: Observable<TrailCollection | null>): Observable<string> {
    return col.pipe(
      filterDefined(),
      switchMap(col => this.getTrailCollectionName$(col)),
    );
  }

  public getTrailCollectionName$(col: TrailCollection): Observable<string> {
    if (col.name.length === 0 && col.type === TrailCollectionType.MY_TRAILS)
      return this.injector.get(I18nService).texts$.pipe(map(texts => texts.my_trails));
    if (col.type === TrailCollectionType.PUB_DRAFT) return this.injector.get(I18nService).texts$.pipe(map(texts => texts.publications.draft_name));
    if (col.type === TrailCollectionType.PUB_SUBMIT) return this.injector.get(I18nService).texts$.pipe(map(texts => texts.publications.submit_name));
    if (col.type === TrailCollectionType.PUB_REJECT) return this.injector.get(I18nService).texts$.pipe(map(texts => texts.publications.reject_name));
    return of(col.name);
  }

  public getCollectionWithName$(uuid: string, owner?: string): Observable<{collection: TrailCollection, name: string} | null> {
    return this.getCollection$(uuid, owner ?? this.injector.get(AuthService).email ?? '').pipe(
      switchMap(collection => collection ? this.getTrailCollectionName$(collection).pipe(map(name => ({collection, name}))) : of(null)),
    )
  }

  public create(collection: TrailCollection, ondone?: () => void): Observable<TrailCollection | null> {
    if (!this.injector.get(QuotaService).checkQuota(q => q.collectionsUsed + this._store.getNbLocalCreates() >= q.collectionsMax, 'trail_collections'))
      return throwError(() => new Error('quota reached'));
    return this._store.create(collection, ondone);
  }

  public update(collection: TrailCollection, updater: (collection: TrailCollection) => void, ondone?: (collection: TrailCollection) => void): void {
    this._store.updateWithLock(collection, updater, ondone);
  }

  public delete(collection: TrailCollection, progress: Progress): void {
    this.injector.get(DatabaseService).pauseSync();
    progress.workAmount = 100 + 1000 + 1;
    this.injector.get(TagService).deleteAllTagsFromCollections([{uuid: collection.uuid, owner: collection.owner}], progress, 100)
    .pipe(defaultIfEmpty(false), timeout(15000), catchError(e => { Console.error('Error deleting tags', e); return of(false); }))
    .subscribe(() => {
      this.injector.get(TrailService).deleteAllTrailsFromCollections([{uuid: collection.uuid, owner: collection.owner}], progress, 1000)
      .pipe(defaultIfEmpty(false), timeout(30000), catchError(e => { Console.error('Error deleting trails', e); return of(false); }))
      .subscribe(() => {
        this._store.delete(collection);
        progress.addWorkDone(1);
        progress.done();
        this.injector.get(DatabaseService).resumeSync();
      });
    });
  }

  propagateDelete(collections: TrailCollection[]): void {
    const list = collections.map(c => ({owner: c.owner, uuid: c.uuid}));
    this.injector.get(TagService).deleteAllTagsFromCollections(list, undefined, 100).subscribe(() => {
      this.injector.get(TrailService).deleteAllTrailsFromCollections(list, undefined, 1000).subscribe();
    });
  }

  public getCollectionMenu(collection: TrailCollection): MenuItem[] {
    const menu: MenuItem[] = [];
    if (!isPublicationCollection(collection.type)) {
      menu.push(
        new MenuItem().setIcon('edit-text').setI18nLabel('pages.trails.actions.edit_collection').setAction(() => this.collectionPopup(collection)),
        new MenuItem().setIcon('tags').setI18nLabel('pages.trails.tags.collection_menu_item')
          .setAction(() => import('../../components/tags/tags.component').then(m => m.openTagsDialog(this.injector, null, collection.uuid))),
      );
    }
    if (collection.type === TrailCollectionType.CUSTOM) {
      menu.push(new MenuItem().setIcon('trash').setI18nLabel('buttons.delete').setTextColor('danger').setAction(() => this.confirmDelete(collection)));
    }
    return menu;
  }

  public async collectionPopup(collection?: TrailCollection, redirectOnApplied?: boolean) {
    const module = await import('../../components/collection-form-popup/collection-form-popup.component');
    const modal = await this.injector.get(ModalController).create({
      component: module.CollectionFormPopupComponent,
      componentProps: {
        collection,
        redirectOnApplied: redirectOnApplied ?? true,
      },
      backdropDismiss: false,
      cssClass: 'small-modal',
    });
    await modal.present();
    return await modal.onWillDismiss();
  }

  public async confirmDelete(collection: TrailCollection) {
    const i18n = this.injector.get(I18nService);
    const alert = await this.injector.get(AlertController).create({
      header: i18n.texts.collection_menu.delete_confirm.title,
      message: i18n.texts.collection_menu.delete_confirm.message.replace('{{}}', collection.name),
      buttons: [
        {
          text: i18n.texts.collection_menu.delete_confirm.yes,
          role: 'danger',
          handler: () => {
            const progress = this.injector.get(ProgressService).create(i18n.texts.collection_menu.deleting, 1);
            this.delete(collection, progress);
            alert.dismiss();
            this.injector.get(Router).navigateByUrl('/');
          }
        }, {
          text: i18n.texts.collection_menu.delete_confirm.no,
          role: 'cancel'
        }
      ]
    });
    await alert.present();
  }

  public storeLoadedAndServerUpdates$(): Observable<boolean> {
    return combineLatest([this._store.loaded$, this._store.syncStatus$]).pipe(
      map(([loaded, sync]) => loaded && !sync.needsUpdateFromServer)
    );
  }

  public doNotDeleteCollectionWhileTrailNotSync(collectionUuid: string, trail: Trail): Promise<any> {
    const collectionKey = collectionUuid + '#' + trail.owner;
    const trailKey = trail.uuid + '#' + trail.owner;
    return this.injector.get(DependenciesService).addDependencies(
      TRAIL_COLLECTION_TABLE_NAME,
      collectionKey,
      'delete',
      [
        {
          storeName: TRAIL_TABLE_NAME,
          itemKey: trailKey,
          operation: 'update'
        }
      ]
    );
  }

  public doNotDeleteCollectionUntilEvent(collectionUuid: string, collectionOwner: string, eventId: string): void {
    this.injector.get(DependenciesService).addEventDependency(
      TRAIL_COLLECTION_TABLE_NAME,
      collectionUuid + '#' + collectionOwner,
      'delete',
      eventId
    );
  }

  public sort(list: TrailCollection[]): TrailCollection[] {
    const prefs = this.injector.get(PreferencesService).preferences;
    return list.sort((c1, c2) => this.compareCollections(c1, c2, prefs.lang));
  }

  public compareCollections(c1: TrailCollection, c2: TrailCollection, lang: string): number {
    if (c1.type === TrailCollectionType.MY_TRAILS) return -1;
    if (c2.type === TrailCollectionType.MY_TRAILS) return 1;
    return c1.name.localeCompare(c2.name, lang);
  }
}

class TrailCollectionStore extends OwnedStore<TrailCollectionDto, TrailCollection> {

    constructor(
      injector: Injector,
      private readonly http: HttpService,
    ) {
      super(TRAIL_COLLECTION_TABLE_NAME, injector);
      this.quotaService = injector.get(QuotaService);
    }

    private readonly quotaService: QuotaService;

    protected override beforeEmittingStoreLoaded(): void {
      if (this._store.value.length === 0 && this.injector.get(AuthService).email === ANONYMOUS_USER) {
        this.create(new TrailCollection({
          type: TrailCollectionType.MY_TRAILS,
          owner: ANONYMOUS_USER,
        }));
      }
      super.beforeEmittingStoreLoaded();
    }

    protected override isQuotaReached(): boolean {
      const q = this.quotaService.quotas;
      return !q || q.collectionsUsed >= q.collectionsMax;
    }

    protected override fromDTO(dto: TrailCollectionDto): TrailCollection {
      return new TrailCollection(dto);
    }

    protected override toDTO(entity: TrailCollection): TrailCollectionDto {
      return entity.toDto();
    }

    protected override migrate(fromVersion: number, dbService: DatabaseService): Promise<number | undefined> {
      return Promise.resolve(undefined);
    }

    protected override readyToSave(entity: TrailCollection): boolean {
        return true;
    }

    protected override readyToSave$(entity: TrailCollection): Observable<boolean> {
      return of(true);
    }

    protected override createdLocallyCanBeRemoved(entity: TrailCollection): Observable<boolean> {
      return of(false);
    }

    protected override createOnServer(items: TrailCollectionDto[]): Observable<TrailCollectionDto[]> {
      return this.http.post<TrailCollectionDto[]>(environment.apiBaseUrl + '/trail-collection/v1/_bulkCreate', items).pipe(
        tap(created => this.quotaService.updateQuotas(q => q.collectionsUsed += created.length)),
      );
    }

    protected override getUpdatesFromServer(knownItems: VersionedDto[]): Observable<UpdatesResponse<TrailCollectionDto>> {
      return this.http.post<UpdatesResponse<TrailCollectionDto>>(environment.apiBaseUrl + '/trail-collection/v1/_bulkGetUpdates', knownItems);
    }

    protected override sendUpdatesToServer(items: TrailCollectionDto[]): Observable<TrailCollectionDto[]> {
      return this.http.put<TrailCollectionDto[]>(environment.apiBaseUrl + '/trail-collection/v1/_bulkUpdate', items);
    }

    protected override deleteFromServer(uuids: string[]): Observable<void> {
      return this.http.post<void>(environment.apiBaseUrl + '/trail-collection/v1/_bulkDelete', uuids).pipe(
        tap({
          complete: () => this.quotaService.updateQuotas(q => q.collectionsUsed -= uuids.length)
        })
      );
    }

    protected override deleted(deleted: {item$: BehaviorSubject<TrailCollection | null> | undefined, item: TrailCollection}[]): void {
      this.injector.get(TrailCollectionService).propagateDelete(deleted.map(d => d.item));
      super.deleted(deleted);
    }

    protected override signalDeleted(deleted: { uuid: string; owner: string; }[]): void {
      this.injector.get(ShareService).signalCollectionsDeleted(deleted);
    }

    protected override doCleaning(email: string, db: Dexie): Observable<any> {
      return of(false);
    }

    protected override newItemFromServer(dto: TrailCollectionDto, entity: TrailCollection): void {
      // if a publication collection is new on server, and we have it created locally
      // move all trails from the local to the one from the server, and delete the local
      if (!isPublicationCollection(entity.type)) return;
      const local = this._store.value.find(c$ => c$.value && c$.value.type === entity.type && c$.value.isCreatedLocally() && !c$.value.isDeletedLocally())?.value;
      if (!local) return;
      const trailService = this.injector.get(TrailService);
      trailService.getAllWhenLoaded$().pipe(
        collection$items(),
        map(trails => trails.filter(t => t.collectionUuid === local.uuid)),
        takeWhile(trails => trails.length > 0),
      ).subscribe(trails => {
        for (const trail of trails) trailService.doUpdate(trail, t => t.collectionUuid = entity.uuid);
      });
    }

  }
