import { bindLifecycleEvents } from '../../providers/angular-delegate';
import { computeStackId, destroyView, getUrl, insertView, isTabSwitch, toSegments, } from './stack-utils';
// TODO(FW-2827): types
export class StackController {
    containerEl;
    router;
    navCtrl;
    zone;
    location;
    views = [];
    runningTask;
    skipTransition = false;
    tabsPrefix;
    activeView;
    nextId = 0;
    constructor(tabsPrefix, containerEl, router, navCtrl, zone, location) {
        this.containerEl = containerEl;
        this.router = router;
        this.navCtrl = navCtrl;
        this.zone = zone;
        this.location = location;
        this.tabsPrefix = tabsPrefix !== undefined ? toSegments(tabsPrefix) : undefined;
    }
    createView(ref, activatedRoute) {
        const url = getUrl(this.router, activatedRoute);
        const element = ref?.location?.nativeElement;
        const unlistenEvents = bindLifecycleEvents(this.zone, ref.instance, element);
        return {
            id: this.nextId++,
            stackId: computeStackId(this.tabsPrefix, url),
            unlistenEvents,
            element,
            ref,
            url,
        };
    }
    getExistingView(activatedRoute) {
        const activatedUrlKey = getUrl(this.router, activatedRoute);
        const view = this.views.find((vw) => vw.url === activatedUrlKey);
        if (view) {
            view.ref.changeDetectorRef.reattach();
        }
        return view;
    }
    setActive(enteringView) {
        const consumeResult = this.navCtrl.consumeTransition();
        let { direction, animation, animationBuilder } = consumeResult;
        const leavingView = this.activeView;
        const tabSwitch = isTabSwitch(enteringView, leavingView);
        if (tabSwitch) {
            direction = 'back';
            animation = undefined;
        }
        const viewsSnapshot = this.views.slice();
        let currentNavigation;
        const router = this.router;
        // Angular >= 7.2.0
        if (router.getCurrentNavigation) {
            currentNavigation = router.getCurrentNavigation();
            // Angular < 7.2.0
        }
        else if (router.navigations?.value) {
            currentNavigation = router.navigations.value;
        }
        /**
         * If the navigation action
         * sets `replaceUrl: true`
         * then we need to make sure
         * we remove the last item
         * from our views stack
         */
        if (currentNavigation?.extras?.replaceUrl) {
            if (this.views.length > 0) {
                this.views.splice(-1, 1);
            }
        }
        const reused = this.views.includes(enteringView);
        const views = this.insertView(enteringView, direction);
        // Trigger change detection before transition starts
        // This will call ngOnInit() the first time too, just after the view
        // was attached to the dom, but BEFORE the transition starts
        if (!reused) {
            enteringView.ref.changeDetectorRef.detectChanges();
        }
        /**
         * If we are going back from a page that
         * was presented using a custom animation
         * we should default to using that
         * unless the developer explicitly
         * provided another animation.
         */
        const customAnimation = enteringView.animationBuilder;
        if (animationBuilder === undefined && direction === 'back' && !tabSwitch && customAnimation !== undefined) {
            animationBuilder = customAnimation;
        }
        /**
         * Save any custom animation so that navigating
         * back will use this custom animation by default.
         */
        if (leavingView) {
            leavingView.animationBuilder = animationBuilder;
        }
        // Wait until previous transitions finish
        return this.zone.runOutsideAngular(() => {
            return this.wait(() => {
                // disconnect leaving page from change detection to
                // reduce jank during the page transition
                if (leavingView) {
                    leavingView.ref.changeDetectorRef.detach();
                }
                // In case the enteringView is the same as the leavingPage we need to reattach()
                enteringView.ref.changeDetectorRef.reattach();
                return this.transition(enteringView, leavingView, animation, this.canGoBack(1), false, animationBuilder)
                    .then(() => cleanupAsync(enteringView, views, viewsSnapshot, this.location, this.zone))
                    .then(() => ({
                    enteringView,
                    direction,
                    animation,
                    tabSwitch,
                }));
            });
        });
    }
    canGoBack(deep, stackId = this.getActiveStackId()) {
        return this.getStack(stackId).length > deep;
    }
    pop(deep, stackId = this.getActiveStackId()) {
        return this.zone.run(() => {
            const views = this.getStack(stackId);
            if (views.length <= deep) {
                return Promise.resolve(false);
            }
            const view = views[views.length - deep - 1];
            let url = view.url;
            const viewSavedData = view.savedData;
            if (viewSavedData) {
                const primaryOutlet = viewSavedData.get('primary');
                if (primaryOutlet?.route?._routerState?.snapshot.url) {
                    url = primaryOutlet.route._routerState.snapshot.url;
                }
            }
            const { animationBuilder } = this.navCtrl.consumeTransition();
            return this.navCtrl.navigateBack(url, { ...view.savedExtras, animation: animationBuilder }).then(() => true);
        });
    }
    startBackTransition() {
        const leavingView = this.activeView;
        if (leavingView) {
            const views = this.getStack(leavingView.stackId);
            const enteringView = views[views.length - 2];
            const customAnimation = enteringView.animationBuilder;
            return this.wait(() => {
                return this.transition(enteringView, // entering view
                leavingView, // leaving view
                'back', this.canGoBack(2), true, customAnimation);
            });
        }
        return Promise.resolve();
    }
    endBackTransition(shouldComplete) {
        if (shouldComplete) {
            this.skipTransition = true;
            this.pop(1);
        }
        else if (this.activeView) {
            cleanup(this.activeView, this.views, this.views, this.location, this.zone);
        }
    }
    getLastUrl(stackId) {
        const views = this.getStack(stackId);
        return views.length > 0 ? views[views.length - 1] : undefined;
    }
    /**
     * @internal
     */
    getRootUrl(stackId) {
        const views = this.getStack(stackId);
        return views.length > 0 ? views[0] : undefined;
    }
    getActiveStackId() {
        return this.activeView ? this.activeView.stackId : undefined;
    }
    /**
     * @internal
     */
    getActiveView() {
        return this.activeView;
    }
    hasRunningTask() {
        return this.runningTask !== undefined;
    }
    destroy() {
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        this.containerEl = undefined;
        this.views.forEach(destroyView);
        this.activeView = undefined;
        this.views = [];
    }
    getStack(stackId) {
        return this.views.filter((v) => v.stackId === stackId);
    }
    insertView(enteringView, direction) {
        this.activeView = enteringView;
        this.views = insertView(this.views, enteringView, direction);
        return this.views.slice();
    }
    transition(enteringView, leavingView, direction, showGoBack, progressAnimation, animationBuilder) {
        if (this.skipTransition) {
            this.skipTransition = false;
            return Promise.resolve(false);
        }
        if (leavingView === enteringView) {
            return Promise.resolve(false);
        }
        const enteringEl = enteringView ? enteringView.element : undefined;
        const leavingEl = leavingView ? leavingView.element : undefined;
        const containerEl = this.containerEl;
        if (enteringEl && enteringEl !== leavingEl) {
            enteringEl.classList.add('ion-page');
            enteringEl.classList.add('ion-page-invisible');
            if (containerEl.commit) {
                return containerEl.commit(enteringEl, leavingEl, {
                    duration: direction === undefined ? 0 : undefined,
                    direction,
                    showGoBack,
                    progressAnimation,
                    animationBuilder,
                });
            }
        }
        return Promise.resolve(false);
    }
    async wait(task) {
        if (this.runningTask !== undefined) {
            await this.runningTask;
            this.runningTask = undefined;
        }
        const promise = (this.runningTask = task());
        promise.finally(() => (this.runningTask = undefined));
        return promise;
    }
}
const cleanupAsync = (activeRoute, views, viewsSnapshot, location, zone) => {
    if (typeof requestAnimationFrame === 'function') {
        return new Promise((resolve) => {
            requestAnimationFrame(() => {
                cleanup(activeRoute, views, viewsSnapshot, location, zone);
                resolve();
            });
        });
    }
    return Promise.resolve();
};
const cleanup = (activeRoute, views, viewsSnapshot, location, zone) => {
    /**
     * Re-enter the Angular zone when destroying page components. This will allow
     * lifecycle events (`ngOnDestroy`) to be run inside the Angular zone.
     */
    zone.run(() => viewsSnapshot.filter((view) => !views.includes(view)).forEach(destroyView));
    views.forEach((view) => {
        /**
         * In the event that a user navigated multiple
         * times in rapid succession, we want to make sure
         * we don't pre-emptively detach a view while
         * it is in mid-transition.
         *
         * In this instance we also do not care about query
         * params or fragments as it will be the same view regardless
         */
        const locationWithoutParams = location.path().split('?')[0];
        const locationWithoutFragment = locationWithoutParams.split('#')[0];
        if (view !== activeRoute && view.url !== locationWithoutFragment) {
            const element = view.element;
            element.setAttribute('aria-hidden', 'true');
            element.classList.add('ion-page-hidden');
            view.ref.changeDetectorRef.detach();
        }
    });
};
//# sourceMappingURL=data:application/json;base64,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