import { ChangeDetectorRef, Component, EventEmitter, Input, OnDestroy, OnInit, Output, ViewChild } from '@angular/core';
import { IonHeader, IonToolbar, IonTitle, IonIcon, IonLabel, IonButton, IonFooter, IonButtons, IonCheckbox, ModalController, IonTextarea, AlertController, IonSegment, IonSegmentButton } from "@ionic/angular/standalone";
import { BehaviorSubject, combineLatest, firstValueFrom, map, Observable, of, switchMap, tap } from 'rxjs';
import { Photo } from 'src/app/model/photo';
import { PhotoService } from 'src/app/services/database/photo.service';
import { FileService } from 'src/app/services/file/file.service';
import { I18nService } from 'src/app/services/i18n/i18n.service';
import { Progress, ProgressService } from 'src/app/services/progress/progress.service';
import { PhotoComponent } from '../photo/photo.component';
import { Subscriptions } from 'src/app/utils/rxjs/subscription-utils';
import { AuthService } from 'src/app/services/auth/auth.service';
import { BrowserService } from 'src/app/services/browser/browser.service';
import { CompositeOnDone } from 'src/app/utils/callback-utils';
import { ErrorService } from 'src/app/services/progress/error.service';
import { Console } from 'src/app/utils/console';
import { TranslatedString } from 'src/app/services/i18n/i18n-string';
import { TrackService } from 'src/app/services/database/track.service';
import { TrackUtils } from 'src/app/utils/track-utils';
import { Track } from 'src/app/model/track';
import { Trail } from 'src/app/model/trail';
import { TraceRecorderService } from 'src/app/services/trace-recorder/trace-recorder.service';
import { CameraService } from 'src/app/services/camera/camera.service';
import { NgClass, NgStyle } from '@angular/common';

interface PhotoWithInfo {
  photo: Photo;
  selected: boolean;
  editing: string | null;
  blobSize: number | undefined;
  positionOnMap: boolean;
}

@Component({
  selector: 'app-photos-popup',
  templateUrl: './photos-popup.component.html',
  styleUrls: ['./photos-popup.component.scss'],
  imports: [
    IonCheckbox, IonButtons, IonFooter, IonButton, IonLabel, IonIcon, IonTitle, IonToolbar, IonHeader, IonTextarea, IonSegment, IonSegmentButton,
    PhotoComponent,
    NgClass, NgStyle,
  ]
})
export class PhotosPopupComponent  implements OnInit, OnDestroy {

  @Input() trails$!: Observable<Trail | null>[];
  @Input() popup = true;
  @Output() positionOnMapRequested = new EventEmitter<Photo>();

  loaded = false;
  photos: PhotoWithInfo[] = [];

  maxWidth!: number;
  maxHeight!: number;
  width!: number;
  height!: number;
  canEdit = false;
  canAdd = false;
  canPositionOnMap = false;
  nbSelected = 0;
  sliderIndex = 0;
  metaColumns!: string;

  tabs: string[] = [];
  trails: Trail[] = [];
  activeTrail$ = new BehaviorSubject<Trail | null>(null);
  tabIndex = 0;

  canTakePhoto = false;
  deviceCanTakePhoto?: boolean;

  private readonly subscriptions: Subscriptions = new Subscriptions();

  @ViewChild('descriptionEditor') descriptionEditor?: IonTextarea;

  constructor(
    public i18n: I18nService,
    private readonly photoService: PhotoService,
    private readonly fileService: FileService,
    private readonly progressService: ProgressService,
    browser: BrowserService,
    private readonly auth: AuthService,
    private readonly modalController: ModalController,
    private readonly changesDetector: ChangeDetectorRef,
    private readonly errorService: ErrorService,
    private readonly trackService: TrackService,
    private readonly alertController: AlertController,
    private readonly traceRecorder: TraceRecorderService,
    private readonly cameraService: CameraService,
  ) {
    this.updateSize(browser);
    this.subscriptions.add(browser.resize$.subscribe(() => this.updateSize(browser)));
  }

  private updateSize(browser: BrowserService): void {
    this.width = browser.width;
    this.height = browser.height;
    this.maxWidth = Math.min(Math.floor(this.width * 0.9) - 20, 300);
    this.maxHeight = Math.min(Math.floor(this.height * 0.4) - 50, 300);
    this.metaColumns = this.maxWidth === 300 ? 'two-columns' : 'one-column';
  }

  ngOnInit() {
    this.subscriptions.add(
      this.auth.auth$.pipe(
        switchMap(auth => combineLatest(this.trails$).pipe(
          tap(() => this.loaded = false),
          switchMap(trails => {
            this.trails = trails.filter(t => !!t);
            if (this.activeTrail$.value !== null)
              this.activeTrail$.next(this.trails.find(t => t.owner === this.activeTrail$.value?.owner && t.uuid === this.activeTrail$.value?.uuid) ?? null);
            if (this.activeTrail$.value === null && this.trails.length > 0) this.activeTrail$.next(this.trails[0]);
            this.tabs = this.trails.map(t => {
              if (t === this.traceRecorder.current?.trail) return this.i18n.texts.trace_recorder.notif_message;
              return t.name;
            });
            return this.activeTrail$;
          }),
          switchMap(t => {
            this.tabIndex = t ? this.trails.indexOf(t) : 0;
            if (!t) return of([[] as Photo[], {track: null, canEdit: false, canAdd: false}] as [Photo[], {track: Track | null, canEdit: boolean, canAdd: boolean}]);
            const photos = t === this.traceRecorder.current?.trail ? this.traceRecorder.current.photos$ : this.photoService.getTrailPhotos$(t);
            const track = auth?.email === t.owner ?
              this.trackService.getFullTrack$(t.currentTrackUuid, t.owner).pipe(map(track => ({track, canEdit: true, canAdd: true}))) :
              of({track: null, canEdit: t.fromModeration, canAdd: false});
            return combineLatest([photos, track]);
          }),
        )),
      ).subscribe(result => {
        if (this.activeTrail$.value === this.traceRecorder.current?.trail) {
          if (this.deviceCanTakePhoto === undefined) {
            this.cameraService.canTakePhoto().then(result => {
              this.deviceCanTakePhoto = result;
              if (this.activeTrail$.value === this.traceRecorder.current?.trail && this.deviceCanTakePhoto) {
                this.canTakePhoto = true;
                this.changesDetector.detectChanges();
              }
            });
          } else {
            this.canTakePhoto = this.deviceCanTakePhoto;
          }
        }
        this.canEdit = result[1].canEdit;
        this.canAdd = result[1].canAdd;
        this.canPositionOnMap = this.canEdit && !this.traceRecorder.current;
        result[0].sort((p1, p2) => p1.index - p2.index);
        this.photos = result[0].map(p => {
          return {
            photo: p,
            selected: this.photos?.find(prev => prev.photo.owner === p.owner && prev.photo.uuid === p.uuid)?.selected ?? false,
            editing: this.photos?.find(prev => prev.photo.owner === p.owner && prev.photo.uuid === p.uuid)?.editing ?? null,
            blobSize: this.photos?.find(prev => prev.photo.owner === p.owner && prev.photo.uuid === p.uuid)?.blobSize,
            positionOnMap: !!this.getPhotoPosition(p, result[1].track),
          } as PhotoWithInfo;
        });
        this.nbSelected = this.photos.reduce((p, pi) => p + (pi.selected ? 1 : 0), 0);
        this.loaded = true;
        this.changesDetector.detectChanges();
      })
    );
  }

  private readonly _dateToPosCache = new Map<number, L.LatLngLiteral | null>();
  private getPhotoPosition(photo: Photo, track: Track | null): L.LatLngLiteral | undefined {
    if (photo.latitude !== undefined && photo.longitude !== undefined) {
      const pos = {lat: photo.latitude, lng: photo.longitude};
      if (!track) return pos;
      const ref = TrackUtils.findClosestPointInTrack(pos, track, 100);
      if (ref) return track.segments[ref.segmentIndex].points[ref.pointIndex].pos;
    }
    if (photo.dateTaken !== undefined) {
      let point: L.LatLngLiteral | null | undefined = this._dateToPosCache.get(photo.dateTaken);
      if (point === undefined && track) {
        const closest = TrackUtils.findClosestPointForTime(track, photo.dateTaken);
        point = closest ? {lat: closest.pos.lat, lng: closest.pos.lng} : null;
        this._dateToPosCache.set(photo.dateTaken, point);
      }
      return point ?? undefined;
    }
    return undefined;
  }

  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
  }

  close(data: Photo | null = null): void {
    this.modalController.dismiss(data, 'close');
  }

  setTab(value: any) {
    const index = value as number;
    if (this.tabIndex === index) return;
    this.tabIndex = index;
    this.activeTrail$.next(this.trails[index]);
  }

  setBlobSize(p: PhotoWithInfo, size: number): void {
    p.blobSize = size;
    this.changesDetector.detectChanges();
  }

  setSelected(p: PhotoWithInfo, selected: boolean): void {
    if (p.selected === selected) return;
    p.selected = selected;
    if (selected) this.nbSelected++; else this.nbSelected--;
    this.changesDetector.detectChanges();
  }

  setAllSelected(selected: boolean): void {
    for (const p of this.photos) p.selected = selected;
    if (selected) this.nbSelected = this.photos.length; else this.nbSelected = 0;
    this.changesDetector.detectChanges();
  }

  positionOnMap(photo: Photo): void {
    if (this.popup) this.close(photo);
    else this.positionOnMapRequested.emit(photo);
  }

  clearPosition(photo: PhotoWithInfo): void {
    if (!photo.positionOnMap) {
      this.photoService.update(photo.photo, p => { p.latitude = undefined; p.longitude = undefined; });
      return;
    }
    this.alertController.create({
      header: this.i18n.texts.pages.photos_popup.position.clear_title,
      message: this.i18n.texts.pages.photos_popup.position.clear_message,
      buttons: [
        {
          text: this.i18n.texts.buttons.confirm,
          role: 'danger',
          handler: () => {
            this.alertController.dismiss();
            this.photoService.update(photo.photo, p => {
              if (p.latitude !== undefined && p.longitude !== undefined) {
                p.latitude = undefined;
                p.longitude = undefined;
              } else {
                p.dateTaken = undefined;
              }
            });
          }
        }, {
          text: this.i18n.texts.buttons.cancel,
          role: 'cancel'
        }
      ]
    }).then(a => a.present());
  }

  addPhotos(withCamera: boolean): void {
    const trail = this.activeTrail$.value;
    if (!trail) return;
    if (withCamera && trail === this.traceRecorder.current?.trail) {
      this.traceRecorder.takePhoto();
      return;
    }
    let photoIndex = this.photos.length + 1;
    this.fileService.openFileDialog({
      types: [
        {
          mime: 'image/jpeg',
          extensions: ['jpg', 'jpeg']
        },
        {
          mime: 'image/png',
          extensions: ['png']
        }
      ],
      multiple: true,
      description: this.i18n.texts.pages.photos_popup.importing,
      onstartreading: (nbFiles: number) => {
        if (this.photos.length + nbFiles > 25) {
          return Promise.reject(new Error(new TranslatedString('quota_reached.photos_max_by_trail', [this.photos.length, 25, nbFiles]).translate(this.i18n)));
        }
        const quota = this.photoService.getQuota();
        if (quota.current + nbFiles > quota.max)
          return Promise.reject(new Error(new TranslatedString('quota_reached.photos_max', [quota.max, quota.current, nbFiles]).translate(this.i18n)));
        const progress = this.progressService.create(this.i18n.texts.pages.photos_popup.importing, nbFiles);
        progress.subTitle = '0/' + nbFiles;
        return Promise.resolve(progress);
      },
      onfileread: (index: number, nbFiles: number, progress: Progress, filename: string, file: ArrayBuffer) => {
        return firstValueFrom(this.photoService.addPhoto(trail.owner, trail.uuid, filename, photoIndex++, file))
        .then(p => {
          progress.subTitle = '' + (index + 1) + '/' + nbFiles;
          progress.addWorkDone(1);
          return true;
        });
      },
      ondone: (progress: Progress | undefined, result: boolean[], errors: any[]) => {
        progress?.done();
        if (errors.length > 0) {
          Console.error('Errors reading photos', errors);
          this.errorService.addErrors(errors);
        };
      }
    })
  }

  deleteSelected(): void {
    const photos = this.getSelection();
    const trail = this.activeTrail$.value;
    if (trail === this.traceRecorder.current?.trail) {
      for (const photo of photos) this.traceRecorder.deletePhoto(photo.uuid);
      return;
    }
    const progress = this.progressService.create(this.i18n.texts.pages.photos_popup.deleting, photos.length);
    const done = new CompositeOnDone(() => progress.done());
    for (const p of photos) this.photoService.delete(p, done.add(() => progress.addWorkDone(1)));
    done.start();
  }

  moveBack(index: number): void {
    const photo = this.photos.splice(index, 1)[0];
    const previous = this.photos[index - 1];
    this.photos.splice(index - 1, 0, photo);
    const newIndex = --photo.photo.index;
    previous.photo.index = newIndex + 1;
    const trail = this.activeTrail$.value;
    if (trail === this.traceRecorder.current?.trail) {
      this.traceRecorder.current?.photos$.next(this.traceRecorder.current?.photos$.value);
      return;
    }
    this.photoService.update(photo.photo, p => p.index = newIndex);
    this.photoService.update(previous.photo, p => p.index = newIndex + 1);
  }

  moveForward(index: number): void {
    this.moveBack(index + 1);
  }

  private getSelection(): Photo[] {
    return this.photos.filter(p => p.selected).map(p => p.photo);
  }

  openSlider(index: number): void {
    this.photoService.openSliderPopup(this.photos.map(p => p.photo), index);
  }

  editDescription(photo: PhotoWithInfo): void {
    if (!this.canEdit) return;
    photo.editing = photo.photo.description;
    this.changesDetector.detectChanges();
    setTimeout(() => {
      if (this.descriptionEditor) this.descriptionEditor.setFocus();
    }, 0);
  }

  descriptionChanging(photo: PhotoWithInfo, text: string | null | undefined): void {
    photo.editing = text ?? null;
  }

  descriptionChanged(photo: PhotoWithInfo, text: string | null | undefined): void {
    if (!photo.editing || !text) return;
    if (photo.photo.description !== text) {
      photo.photo.description = text;
      this.photoService.update(photo.photo, p => p.description = text);
    }
    this.exitEditDescription(photo);
  }

  exitEditDescription(photo: PhotoWithInfo): void {
    if (photo.editing === null) return;
    photo.editing = null;
    this.changesDetector.detectChanges();
  }

}
