// ProseReader Copyright 2025 timur.mobi. All rights reserved.
//go:build wasm
// +build wasm
package main

import (
	"fmt"
	"io"
	"strings"
	"strconv"
	"time"
	"net/url" // url.QueryUnescape()
	"unicode"
	"sort"
	"archive/zip"

	"syscall/js"
	b64 "encoding/base64"

	"gopkg.in/xmlpath.v2"

	md "github.com/JohannesKaufmann/html-to-markdown/v2"
	"github.com/JohannesKaufmann/dom"
	"github.com/JohannesKaufmann/html-to-markdown/v2/converter"
	"github.com/JohannesKaufmann/html-to-markdown/v2/plugin/base"
	"github.com/JohannesKaufmann/html-to-markdown/v2/plugin/commonmark"
	"github.com/JohannesKaufmann/html-to-markdown/v2/plugin/table"
	"golang.org/x/net/html"

	"codeberg.org/timurmobi/goreader/epub"
	"bytes"
)

const maxBlocks int = 10000
const runeTextMax int = 3500000

const LFCHAR byte = 10
const LFRUNE rune = rune(LFCHAR)
const LINEFEEDCHAR = 64

type BlockData struct {
	Idx int
	CharLength int
	LinefeedCount int
	BlockEndCriteria int
	italicOnEnd bool
	boldOnEnd bool
	blockquoteOnEnd bool
	dbgInfo1 int
	dbgInfo2 int
	dbgInfoStr string
}
var blockArray []BlockData // * maxBlocks

type ChapterType struct {
	Title string
	Idx int
	HLevel int
}
var ChapterSlice []ChapterType

var runeText []rune
var runeTextSizeMax = 0
var textLen int = 0
var conv *converter.Converter = nil
var pidx int = 0
var totalBlocks = 0
var quote1open = -1 // '»'
var quote2open = -1 // '"' or '“' (8220 &ldquo; LEFT DOUBLE QUOTATION MARK)
var quote3open = -1 // '›'
var guillemetOpen = '»'
var guillemetClose = '«'

var blockSizeHighMark280 = 700                                                 // = semicolon, long-dash
var blockSizeHighMark170 = int(float64(blockSizeHighMark280) * float64(0.90))  // = dot !?
var blockSizeHighMark200 = int(float64(blockSizeHighMark280) * float64(0.93))  // = coma
var blockSizeHighMark240 = int(float64(blockSizeHighMark280) * float64(0.96))  // = blank
var blockSizeHighMark260 = int(float64(blockSizeHighMark280) * float64(0.98))  // = full-end

var	codetag string

var subBookTitle = ""
var subBookFolder = ""

var book *epub.Rootfile = nil
var xpathSelector *xmlpath.Path
var tocNcxRoot *xmlpath.Node
var coverImage string = ""

var xhtmlFileMap map[string]int

var linkIdToIdxMap map[string]int
var linkIdToTextMap map[string]string

var booksTitle []string

var booksStartUrl []string

var dataLen int = 0
var dbg = false
var dbgLevel = 1
var rtlFlag = false
var loadedMd = 0

func myPrint(format string, args ...interface{}) {
	if dbg {
		timeNow := time.Now()
		millisecs := (timeNow.UnixNano() / 1e6)
		millisecsStr := fmt.Sprintf("%d",millisecs)
		millisecsStr = millisecsStr[len(millisecsStr)-3:]
		clockStr := fmt.Sprintf("%d;%s",timeNow.Second(), millisecsStr)
		fmt.Printf(clockStr+" "+format, args...)
	}
}

const maxBlockLen = 6000
var stateItalic bool
var stateBold bool
func nextTextBlock() (int,int,int,int,int,int) {
	endOfBlockMax := pidx + maxBlockLen
	if endOfBlockMax>textLen {
		endOfBlockMax = textLen
	}
	if endOfBlockMax>runeTextMax {
		endOfBlockMax = runeTextMax
	}

	blockLen := 0
	skipEolChars := 0
	linefeedCount := 0
	imagePage := false

	blockLenSave := 0
	skipEolCharsSave := 0

	var pos int
	var runeVal rune

	blockEndCriteria := 0
	blockEndInfo1 := 0
	blockEndInfo2 := 0
	squareBracketOpen := 0
	roundBracketOpen := 0
	for pos = pidx; pos < endOfBlockMax; pos++ {
		runeVal = runeText[pos]
		if runeVal==0 {
			if runeText[pos+1]==0 {
				blockLen = pos - pidx
				break
			}
			continue
		}

		startPos := pos
		if runeVal=='[' && (pos==pidx || runeText[pos-1]!='\\') {
			posSqOpen := pos
			squareBracketOpen++

			mysquareBracketOpen := squareBracketOpen
			myroundBracketOpen := 0
			pos2 := 0
			for pos2 = pos+1; pos2 < endOfBlockMax; pos2++ {
				var runeVal2 = runeText[pos2]
				if runeVal2 == '[' {
					mysquareBracketOpen++
				} else if runeVal2 == ']' {
					mysquareBracketOpen--
				}

				if runeText[pos2]=='/' && runeText[pos2+1]=='/' && (runeText[pos2+2]=='=' || runeText[pos2+2]=='!') {
					if posSqOpen > pidx {
						blockLen = posSqOpen - pidx
						skipEolChars = 0
						quote1open = -1
						quote2open = -1
						quote3open = -1
						blockEndCriteria=71
						break
					}

					if posSqOpen == pos {
						imagePage = true
					}
				}

				if runeVal2 == ']' && mysquareBracketOpen==0 {
					if runeText[pos2+1] != '(' {
						break
					}
					myroundBracketOpen++
				} else if myroundBracketOpen>0 && runeVal2==')' {
					myroundBracketOpen--
					if myroundBracketOpen==0 && imagePage {
						blockLen = pos2 - pidx +1
						skipEolChars = 0
						quote1open = -1
						quote2open = -1
						quote3open = -1
						blockEndCriteria=72
						break
					}

					squareBracketOpen=0
					pos = pos2+1
					break
				}
			}

			if blockLen>0 {
				break
			}

			if pos - pidx < blockSizeHighMark280 -100 {
				if pos > startPos+1 {
					pos--
				}
				continue
			}

			if pos > startPos+1 {
				pos--
			}

		} else if runeVal==']' /*&& (pos==pidx || runeText[pos-1]!='\\')*/ {
			if squareBracketOpen>0 {
				squareBracketOpen--
			}

		} else if roundBracketOpen>0 && runeVal==')' && (pos==pidx || runeText[pos-1]!='\\') {
			roundBracketOpen--

		} else if roundBracketOpen==0 && runeVal=='/' && runeText[pos+1]=='/' && (runeText[pos+2]=='=' || runeText[pos+2]=='!') {
			if pos > pidx {
				preImgStr := string(runeText[pidx:pos])
				preImgStr = strings.Replace(preImgStr,"[]()","",-1)
				preImgStr = strings.Replace(preImgStr,"\n","",-1)
				preImgStr = strings.Replace(preImgStr,"<div></div>","",-1)
				preImgStr = strings.TrimSpace(string(preImgStr))

				myIdx1 := strings.Index(preImgStr,"]()");
				for myIdx1>0 {
					countSqOpen := 1
					myIdx2 := myIdx1-1
					for myIdx2>=0 {
						if preImgStr[myIdx2]==']' {
							countSqOpen++
						} else if preImgStr[myIdx2]=='[' {
							countSqOpen--
						}
						if countSqOpen==0 {
							myLabel := preImgStr[myIdx2+1:myIdx1]
							if len(myLabel)>0 {
								preImgStr = preImgStr[0:myIdx2] + preImgStr[myIdx1+3:]
							}
							break
						}
						myIdx2--
					}
					myIdx1 = strings.Index(preImgStr,"]()");
				}

				myIdx := strings.Index(preImgStr,"[](#")
				for myIdx>=0 {
					myIdx2 := strings.Index(preImgStr[myIdx:],")")
					if myIdx2>0 {
						preImgStr = preImgStr[:myIdx] + preImgStr[myIdx+myIdx2+1:]
						myIdx = strings.Index(preImgStr,"[](#")
					} else {
						myIdx = -1
					}
				}

				if preImgStr!="" {
					saveCurPos := pos
					curPos := pos
					expectNext := 4
					sqBrClose := 0
					sqBrPos := 0
					curPos--
					for curPos >= pidx {
						if expectNext==4 && pos-curPos > 20 {
							// there seem to be no md-links in front of the image
							break
						}
						ru := runeText[curPos]
						if expectNext==4 && ru==')' {
							// now wait for '('
							expectNext = 3
						} else if expectNext==3 && ru=='(' {
							// now wait for ']'
							expectNext = 2
							sqBrClose = 0
						} else if expectNext==2 && ru==']' {
							expectNext = 1
							// now wait for '['
							sqBrClose++
							sqBrPos=curPos
						} else if expectNext==1 && ru==']' {
							sqBrClose++
						} else if expectNext==1 && ru=='[' {
							if sqBrClose>0 {
								sqBrClose--
							}
							if sqBrClose==0 {
								if sqBrPos-curPos>1 {
									break
								}
								expectNext = 4
								saveCurPos = curPos
							}
						}
						curPos--
					}

					blockLen = saveCurPos - pidx
					if blockLen>0 {
						skipEolChars = 0
						quote1open = -1
						quote2open = -1
						quote3open = -1
						blockEndCriteria=731
						blockEndInfo1 = pos - pidx
						blockEndInfo2 = blockLen
						break
					}
				}
			}

			for pos2 := pos+1; pos2 < endOfBlockMax; pos2++ {
				if runeText[pos2]==LFRUNE || runeText[pos2]=='#' || runeText[pos2]==' ' {
					blockLen = pos2 - pidx +1
					skipEolChars = 0
					quote1open = -1
					quote2open = -1
					quote3open = -1
					blockEndCriteria=732
					break
				}
			}
			break

		} else if squareBracketOpen==0 && roundBracketOpen==0 && runeVal==LFRUNE {
			linefeedCount++

			//!!¡ end paŋe if ¢hap following
			if runeText[pos+1]=='#' {
				blockLen = pos - pidx
				skipEolChars = 1
				quote1open = -1
				quote2open = -1
				quote3open = -1
				blockEndCriteria=75
				blockEndInfo2 = linefeedCount
				break
			}

			if runeText[pos+1]==LFRUNE {
				if(pos==pidx) {
					pos++
					pidx = pos
					continue
				}
				blockLen = pos - pidx +1
				linefeedCount++
				blockEndCriteria=8
				blockEndInfo1 = linefeedCount
				blockEndInfo2 = linefeedCount
				skipEolChars = 1
				quote1open = -1
				quote2open = -1
				quote3open = -1
				break
			}
			if (pos - pidx) + linefeedCount*LINEFEEDCHAR > blockSizeHighMark280 {
				blockLen = pos - pidx
				skipEolChars = 0
				quote1open = -1
				quote2open = -1
				quote3open = -1
				blockEndCriteria=74
				blockEndInfo1 = blockSizeHighMark280
				blockEndInfo2 = linefeedCount
				break
			}

		} else if runeVal=='#' {

			if pos+5<endOfBlockMax &&
				runeText[pos+1]=='#' && runeText[pos+2]=='#' && runeText[pos+3]=='#' && runeText[pos+4]=='#' &&
					(runeText[pos+5]==' ' || runeText[pos+5]==160 || runeText[pos+5]==8239) {
				quote1open = -1
				quote2open = -1
				quote3open = -1

				if pos-1 > pidx {
					tmpStr := string(runeText[pidx:pos-1])
					tmpStr = strings.Replace(tmpStr,"\n","",-1)
					tmpStr = strings.Replace(tmpStr,"[]()","",-1)
					myIdx := strings.Index(tmpStr,"[](#")
					for myIdx>=0 {
						myIdx2 := strings.Index(tmpStr[myIdx:],")")
						if myIdx2>0 {
							tmpStr = tmpStr[:myIdx] + tmpStr[myIdx+myIdx2+1:]
							myIdx = strings.Index(tmpStr,"[](#")
						} else {
							myIdx = -1
						}
					}
					tmpStr = strings.TrimSpace(tmpStr)
					if tmpStr!="" {
						skipEolChars = 0
						if runeText[pos-1]==LFRUNE {
							skipEolChars = 1
						}
						blockEndCriteria=11
						blockLen = pos-1 - pidx
						break
					}
				}

				myroundBracketOpen := 0
				for pos1 := pos+6; pos1 < endOfBlockMax; pos1++ {
					runeVal1 := runeText[pos1]
					if runeVal1=='(' {
						myroundBracketOpen++
					} else if runeVal1==')' {
						myroundBracketOpen--
					}
					if runeVal1==LFRUNE || runeVal1=='>' || (runeVal1=='#' && myroundBracketOpen==0) {
						chapNameRun := runeText[pos+6:pos1]
						chapName := string(chapNameRun)
						blockLen = pos1 - pidx
						skipEolChars = 0
						if chapName!="" {
							addChapName(chapNameRun,pos,5,pos1-pos)
						}
						break
					}
				}
				blockEndCriteria=1
				stateItalic=false
				stateBold=false
				break
			}

			if pos+4 < endOfBlockMax &&
				runeText[pos+1]=='#' && runeText[pos+2]=='#' && runeText[pos+3]=='#' && 
					(runeText[pos+4]==' ' || runeText[pos+4]==160 || runeText[pos+4]==8239) {
				quote1open = -1
				quote2open = -1
				quote3open = -1

				if pos-1 > pidx {
					tmpStr := string(runeText[pidx:pos-1])
					tmpStr = strings.Replace(tmpStr,"\n","",-1)
					tmpStr = strings.Replace(tmpStr,"[]()","",-1)
					myIdx := strings.Index(tmpStr,"[](#")
					for myIdx>=0 {
						myIdx2 := strings.Index(tmpStr[myIdx:],")")
						if myIdx2>0 {
							tmpStr = tmpStr[:myIdx] + tmpStr[myIdx+myIdx2+1:]
							myIdx = strings.Index(tmpStr,"[](#")
						} else {
							myIdx = -1
						}
					}
					tmpStr = strings.TrimSpace(tmpStr)
					if tmpStr!="" {
						skipEolChars = 0
						if runeText[pos-1]==LFRUNE {
							skipEolChars = 1
						}
						blockEndCriteria=12
						blockLen = pos-1 - pidx
						break
					}
				}

				myroundBracketOpen := 0
				for pos1 := pos+5; pos1 < endOfBlockMax; pos1++ {
					runeVal1 := runeText[pos1]
					if runeVal1=='(' {
						myroundBracketOpen++
					} else if runeVal1==')' {
						myroundBracketOpen--
					}

					if runeVal1==LFRUNE || runeVal1=='>' || (runeVal1=='#' && myroundBracketOpen==0) {
						chapNameRun := runeText[pos+5:pos1]
						chapName := string(chapNameRun)
						blockLen = pos1 - pidx
						skipEolChars = 0
						if chapName!="" {
							addChapName(chapNameRun,pos,4,pos1-pos)
						}
						break
					}
				}
				blockEndCriteria=2
				stateItalic=false
				stateBold=false
				break
			}

			if pos+3<endOfBlockMax && runeText[pos+1]=='#' && runeText[pos+2]=='#' && 
					(runeText[pos+3]==' ' || runeText[pos+3]==160 || runeText[pos+3]==8239) {
				quote1open = -1
				quote2open = -1
				quote3open = -1

				if pos-1>pidx {
					tmpStr := string(runeText[pidx:pos-1])
					tmpStr = strings.Replace(tmpStr,"\n","",-1)
					tmpStr = strings.Replace(tmpStr,"[]()","",-1)
					myIdx := strings.Index(tmpStr,"[](#")
					for myIdx>=0 {
						myIdx2 := strings.Index(tmpStr[myIdx:],")")
						if myIdx2>0 {
							tmpStr = tmpStr[:myIdx] + tmpStr[myIdx+myIdx2+1:]
							myIdx = strings.Index(tmpStr,"[](#")
						} else {
							myIdx = -1
						}
					}
					tmpStr = strings.TrimSpace(tmpStr)
					if tmpStr!="" {
						skipEolChars = 0
						if runeText[pos-1]==LFRUNE {
							skipEolChars = 1
						}
						blockEndCriteria=13
						blockLen = pos-1 - pidx
						break
					}
				}

				myroundBracketOpen := 0
				for pos1 := pos+4; pos1 < endOfBlockMax; pos1++ {
					runeVal1 := runeText[pos1]
					if runeVal1=='(' {
						myroundBracketOpen++
					} else if runeVal1==')' {
						myroundBracketOpen--
					}
					if runeVal1==LFRUNE || runeVal1=='>' || (runeVal1=='#' && myroundBracketOpen==0) {
						chapNameRun := runeText[pos+4:pos1]
						chapName := string(chapNameRun)
						blockLen = pos1 - pidx
						skipEolChars = 0
						if chapName!="" {
							addChapName(chapNameRun,pos,3,pos1-pos)
						}
						break
					}
				}
				blockEndCriteria=3
				stateItalic=false
				stateBold=false
				break
			}

			if pos+2<endOfBlockMax && runeText[pos+1]=='#' && 
				(runeText[pos+2]==' ' || runeText[pos+2]==160 || runeText[pos+2]==8239) {
				quote1open = -1
				quote2open = -1
				quote3open = -1

				if pos-1>pidx {
					tmpStr := string(runeText[pidx:pos-1])
					tmpStr = strings.Replace(tmpStr,"\n","",-1)
					tmpStr = strings.Replace(tmpStr,"[]()","",-1)
					myIdx := strings.Index(tmpStr,"[](#")
					for myIdx>=0 {
						myIdx2 := strings.Index(tmpStr[myIdx:],")")
						if myIdx2>0 {
							tmpStr = tmpStr[:myIdx] + tmpStr[myIdx+myIdx2+1:]
							myIdx = strings.Index(tmpStr,"[](#")
						} else {
							myIdx = -1
						}
					}
					tmpStr = strings.TrimSpace(tmpStr)
					if tmpStr!="" {
						skipEolChars = 0
						if runeText[pos-1]==LFRUNE {
							skipEolChars = 1
						}
						blockEndCriteria=14
						blockLen = pos-1 - pidx
						break
					}
				}

				myroundBracketOpen := 0
				for pos1 := pos+3; pos1 < endOfBlockMax; pos1++ {
					runeVal1 := runeText[pos1]
					if runeVal1=='(' {
						myroundBracketOpen++
					} else if runeVal1==')' {
						myroundBracketOpen--
					}
					if runeVal1==LFRUNE || runeVal1=='>' || (runeVal1=='#' && myroundBracketOpen==0) {
						chapNameRun := runeText[pos+3:pos1]
						blockLen = pos1 - pidx
						skipEolChars = 0
						chapName := string(chapNameRun)
						if chapName!="" {
							addChapName(chapNameRun,pos,2,pos1-pos)
						}
						break
					}
				}
				blockEndCriteria=4
				stateItalic=false
				stateBold=false
				break
			}

			if pos+1<endOfBlockMax && (runeText[pos+1]==' ' || runeText[pos+1]==160 || runeText[pos+1]==8239) {
				quote1open = -1
				quote2open = -1
				quote3open = -1

				if pos-1>pidx {
					tmpStr := string(runeText[pidx:pos-1])
					tmpStr = strings.Replace(tmpStr,"\n","",-1)
					tmpStr = strings.Replace(tmpStr,"[]()","",-1)
					myIdx := strings.Index(tmpStr,"[](#")
					for myIdx>=0 {
						myIdx2 := strings.Index(tmpStr[myIdx:],")")
						if myIdx2>0 {
							tmpStr = tmpStr[:myIdx] + tmpStr[myIdx+myIdx2+1:]
							myIdx = strings.Index(tmpStr,"[](#")
						} else {
							myIdx = -1
						}
					}
					tmpStr = strings.TrimSpace(tmpStr)
					if tmpStr!="" {
						skipEolChars = 0
						if runeText[pos-1]==LFRUNE {
							skipEolChars = 1
						}
						blockEndCriteria=15
						blockLen = pos-1 - pidx
						break
					}
				}

				myroundBracketOpen := 0
				for pos1 := pos+2; pos1 < endOfBlockMax; pos1++ {
					runeVal1 := runeText[pos1]
					if runeVal1=='(' {
						myroundBracketOpen++
					} else if runeVal1==')' {
						myroundBracketOpen--
					}
					if runeVal1==LFRUNE || runeVal1=='>' || (runeVal1=='#' && myroundBracketOpen==0) {
						chapNameRun := runeText[pos+2:pos1]
						chapName := string(chapNameRun)
						blockLen = pos1 - pidx
						skipEolChars = 0
						if chapName!="" {
							addChapName(chapNameRun,pos,1,pos1-pos)
						}
						break
					}
				}
				blockEndCriteria=5
				stateItalic=false
				stateBold=false
				break
			}
		} else if squareBracketOpen==0 && roundBracketOpen==0 && runeVal=='.'{
			if pos-pidx >= 2 {
				if runeText[pos+1]>=32 &&
				    (runeText[pos-2]==' ' || runeText[pos-2]==160 || runeText[pos-2]==8239 || runeText[pos-2]=='(' ||
					runeText[pos-2]==LFRUNE || runeText[pos-2]=='>' || runeText[pos-2]=='*' || runeText[pos-2]=='“' ||
					runeText[pos-2]=='‘') {
					continue
				}
				if runeText[pos-2]=='.' || runeText[pos-2]==' ' || runeText[pos-2]=='‑' || runeText[pos-2]=='-' ||
						runeText[pos-2]==')' {
					if (runeText[pos-1]>='a' && runeText[pos-1]<='z') || (runeText[pos-1]>='A' && runeText[pos-1]<='Z') {
						continue
					}
				}
			}

			if pos-pidx >= 3 && 
				(runeText[pos-3]==' ' || runeText[pos-3]==160 || runeText[pos-3]==8239 ||
				runeText[pos-3]==guillemetOpen || runeText[pos-3]=='"' || runeText[pos-3]==8220 ||
				runeText[pos-3]=='„' || runeText[pos-3]=='“' || runeText[pos-3]==')' ||
				runeText[pos-3]=='(' || runeText[pos-3]=='[' ||
				runeText[pos-3]==LFRUNE || runeText[pos-3]==10 || runeText[pos-3]==13 ||
				runeText[pos-3]=='>' || runeText[pos-3]=='-' || runeText[pos-3]=='*' || runeText[pos-3]=='—') &&
				runeText[pos+1]>=32 {

				if (runeText[pos-2]>='0' && runeText[pos-2]<='9') ||
				   runeText[pos-2]=='I' || runeText[pos-2]=='V' || runeText[pos-2]=='X' || runeText[pos-2]=='M' {
					continue
				}
				twoLetterAbkuerzung := string(runeText[pos-2:pos+1])
				if twoLetterAbkuerzung=="Nr." ||
				   twoLetterAbkuerzung=="Mr." ||
				   twoLetterAbkuerzung=="jr." ||
				   twoLetterAbkuerzung=="Jr." ||
				   twoLetterAbkuerzung=="Tl." ||
				   twoLetterAbkuerzung=="Ew." ||
				   twoLetterAbkuerzung=="Bd." ||
				   twoLetterAbkuerzung=="Dr." ||
				   twoLetterAbkuerzung=="Jh." ||
				   twoLetterAbkuerzung=="Ms." ||
				   twoLetterAbkuerzung=="dt." ||
				   twoLetterAbkuerzung=="ff." ||
				   twoLetterAbkuerzung=="St." ||
				   twoLetterAbkuerzung=="II." ||
				   twoLetterAbkuerzung=="Sr." ||
				   twoLetterAbkuerzung=="Se." ||
				   twoLetterAbkuerzung=="op." ||
				   twoLetterAbkuerzung=="cf." ||
				   twoLetterAbkuerzung=="ii." ||
				   twoLetterAbkuerzung=="ib." ||
				   twoLetterAbkuerzung=="iv." ||
				   twoLetterAbkuerzung=="Th." ||
				   twoLetterAbkuerzung=="Fr." ||
				   twoLetterAbkuerzung=="fr." ||
				   twoLetterAbkuerzung=="Gr." ||
				   twoLetterAbkuerzung=="gr." ||
				   twoLetterAbkuerzung=="Pl." ||
				   twoLetterAbkuerzung=="Op." ||
				   twoLetterAbkuerzung=="vs." {
					continue
				}

				if strings.HasSuffix(twoLetterAbkuerzung,"M.") {
					myPrint("! nextTextBlock twoLetterAbkuerzung (%s) %d\n",twoLetterAbkuerzung, totalBlocks)
					continue
				}
			}

			if pos-pidx >= 4 && 
				(runeText[pos-4]==' ' || runeText[pos-4]==160 || runeText[pos-4]==8239 || runeText[pos-4]=='(' ||
				runeText[pos-4]==guillemetOpen || runeText[pos-4]=='"' || runeText[pos-4]==8220 ||
				runeText[pos-4]=='„' || runeText[pos-4]=='“' || runeText[pos-4]==')' ||
				runeText[pos-4]=='(' || runeText[pos-4]=='[' ||
				runeText[pos-4]==LFRUNE || runeText[pos-4]==10 || runeText[pos-4]==13 ||
				runeText[pos-4]=='>' || runeText[pos-4]=='-' || runeText[pos-4]=='*' || runeText[pos-4]=='—') &&
				runeText[pos+1]>=32 {
				if runeText[pos-3]=='I' || runeText[pos-3]=='V' || runeText[pos-3]=='X' || runeText[pos-3]=='M' ||
				   (runeText[pos-3]>='0' && runeText[pos-3]<='9') {
					if runeText[pos-2]=='I' || runeText[pos-2]=='V' || runeText[pos-2]=='X' || runeText[pos-2]=='M' ||
					   (runeText[pos-2]>='0' && runeText[pos-2]<='9') {
						if runeText[pos-1]=='I' || runeText[pos-1]=='V' || runeText[pos-1]=='X' || runeText[pos-1]=='M' ||
						   (runeText[pos-1]>='0' && runeText[pos-1]<='9') {
							continue
						}
					}
				}

				threeLetterAbkuerzung := string(runeText[pos-3:pos+1])

				if threeLetterAbkuerzung=="Rev." ||
				   threeLetterAbkuerzung=="Mrs." ||
				   threeLetterAbkuerzung=="Mme." ||
				   threeLetterAbkuerzung=="geb." ||
				   threeLetterAbkuerzung=="www." ||
				   threeLetterAbkuerzung=="frz." ||
				   threeLetterAbkuerzung=="bzw." ||
				   threeLetterAbkuerzung=="usw." ||
				   threeLetterAbkuerzung=="etc." ||
				   threeLetterAbkuerzung=="vgl." ||
				   threeLetterAbkuerzung=="sog." ||
				   threeLetterAbkuerzung=="Kap." ||
				   threeLetterAbkuerzung=="Jan." ||
				   threeLetterAbkuerzung=="Feb." ||
				   threeLetterAbkuerzung=="Apr." ||
				   threeLetterAbkuerzung=="Jun." ||
				   threeLetterAbkuerzung=="Jul." ||
				   threeLetterAbkuerzung=="Aug." ||
				   threeLetterAbkuerzung=="Sep." ||
				   threeLetterAbkuerzung=="Okt." ||
				   threeLetterAbkuerzung=="Oct." ||
				   threeLetterAbkuerzung=="Nov." ||
				   threeLetterAbkuerzung=="Dez." ||
				   threeLetterAbkuerzung=="Dec." ||
				   threeLetterAbkuerzung=="ebd." ||
				   threeLetterAbkuerzung=="rev." ||
				   threeLetterAbkuerzung=="Ltg." ||
				   threeLetterAbkuerzung=="s.w." ||
				   threeLetterAbkuerzung=="u.s." ||
				   threeLetterAbkuerzung=="U.S." ||
				   threeLetterAbkuerzung=="Geh." ||
				   threeLetterAbkuerzung=="Esq." ||
				   threeLetterAbkuerzung=="cit." ||
				   threeLetterAbkuerzung=="iii." ||
				   threeLetterAbkuerzung=="Ing." ||
				   threeLetterAbkuerzung=="Vol." ||
				   threeLetterAbkuerzung=="vol." ||
				   threeLetterAbkuerzung=="Abk." ||
				   threeLetterAbkuerzung=="Zus." ||
				   threeLetterAbkuerzung=="Gov." ||
				   threeLetterAbkuerzung=="Exz." ||
				   threeLetterAbkuerzung=="dgl." ||
				   threeLetterAbkuerzung=="Rbl." ||
				   threeLetterAbkuerzung=="Sbr." ||
				   threeLetterAbkuerzung=="z.B." {
					continue
				}
			}
			if pos-pidx >= 5 {
				fourLetterAbkuerzung := string(runeText[pos-4:pos+1])
				if strings.HasSuffix(fourLetterAbkuerzung,"Mlle.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"Corr.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"inkl.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"Aufl.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"Ausg.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"komm.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"Hrsg.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"Verz.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"Komm.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"Verf.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"stud.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"math.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"Prof.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"resp.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"pers.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"incl.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"inkl.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"Mlle.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"verw.") ||
				   strings.HasSuffix(fourLetterAbkuerzung,"gest.") {
					continue
				}

				if (runeText[pos-5]==' ' || runeText[pos-5]==160 || runeText[pos-5]==8239 || runeText[pos-5]=='(' ||
					runeText[pos-5]==LFRUNE || runeText[pos-5]=='>') &&
					runeText[pos+1]>=32 {
					if runeText[pos-4]=='I' || runeText[pos-4]=='V' || runeText[pos-4]=='X' ||
					   runeText[pos-4]=='M' || (runeText[pos-4]>='0' && runeText[pos-4]<='9') {
						if runeText[pos-3]=='I' || runeText[pos-3]=='V' || runeText[pos-3]=='X' ||
						   runeText[pos-3]=='M' || (runeText[pos-3]>='0' && runeText[pos-3]<='9') {
							if runeText[pos-2]=='I' || runeText[pos-2]=='V' || runeText[pos-2]=='X' ||
							   runeText[pos-2]=='M' || (runeText[pos-2]>='0' && runeText[pos-2]<='9') {
								if runeText[pos-1]=='I' || runeText[pos-1]=='V' || runeText[pos-1]=='X' ||
								   runeText[pos-1]=='M' || (runeText[pos-1]>='0' && runeText[pos-1]<='9') {
									continue
								}
							}
						}
					}
				}

				if pos-pidx >= 6 {
					fiveLetterAbkuerzung := string(runeText[pos-5:pos+1])
					if strings.HasSuffix(fiveLetterAbkuerzung,"Nachw.") ||
					   strings.HasSuffix(fiveLetterAbkuerzung,"Übers.") ||
					   strings.HasSuffix(fiveLetterAbkuerzung,"übers.") ||
					   strings.HasSuffix(fiveLetterAbkuerzung,"Preuß.") {
						continue
					}

					if pos-pidx >= 7 {
						sixLetterAbkuerzung := string(runeText[pos-6:pos+1])
						if strings.HasSuffix(sixLetterAbkuerzung,"Königl.") ||
						   strings.HasSuffix(sixLetterAbkuerzung,"königl.") ||
						   strings.HasSuffix(sixLetterAbkuerzung,"insbes.") ||
						   strings.HasSuffix(sixLetterAbkuerzung,"übertr.") ||
						   strings.HasSuffix(sixLetterAbkuerzung,"Grundl.") ||
						   strings.HasSuffix(sixLetterAbkuerzung,"Geweih.") {
							continue
						}

						if pos-pidx >= 8 {
							sevenLetterAbkuerzung := string(runeText[pos-7:pos+1])
							if strings.HasSuffix(sevenLetterAbkuerzung,"vermutl.") ||
							   strings.HasSuffix(sevenLetterAbkuerzung,"Überarb.") ||
							   strings.HasSuffix(sevenLetterAbkuerzung,"überarb.") ||
							   strings.HasSuffix(sevenLetterAbkuerzung,"Neuausg.") {
								continue
							}
						}
					}
				}
			}

		} else if runeVal==guillemetOpen || runeVal=='„' || runeVal=='“' {
			quote1open = pos - pidx
		} else if runeVal==guillemetClose || runeVal==8221 {
			quote1open = -1

		} else if runeVal=='"' {
			if quote2open>=0 {
				quote2open = -1
			} else {
				quote2open = pos - pidx
			}

		} else if runeVal=='›' {
			quote3open = pos - pidx
		} else if runeVal=='‹' {
			quote3open = -1
		}

		if squareBracketOpen==0 && roundBracketOpen==0 && (runeVal=='.' || runeVal=='?' || runeVal=='!') {
			if pos-pidx-quote1open<12 || pos-pidx-quote2open<12 || pos-pidx-quote3open<12 {
				continue
			}

			dotPos := pos
			pos++
			if pos < endOfBlockMax {
				if runeVal=='!' || runeVal=='?' {
					if pos+1 < endOfBlockMax {
						if runeText[pos]==' ' && runeText[pos+1]==unicode.ToLower(runeText[pos+1]) {
							continue
						}
						if runeText[pos]==')' && runeText[pos+1]==' ' {
							continue
						}
						if pos+2 < endOfBlockMax {
							if runeText[pos]=='"' || runeText[pos]=='“' || runeText[pos]=='”' ||
									runeText[pos]==guillemetClose {
								if runeText[pos+1]==' ' && unicode.ToLower(runeText[pos+2])==runeText[pos+2] {
									continue
								}
							}
						}
					}
				}

				runeVal = runeText[pos]
				if runeVal == LFRUNE {
					linefeedCount++
				}
				if runeVal==',' || (runeVal>='a' && runeVal<='z') || (runeVal>='A' && runeVal<='Z') {
					continue
				} 

				if runeVal=='.' || runeVal==';' || runeVal==':' || runeVal=='?' || runeVal=='!' || runeVal=='*' || runeVal==' ' || runeVal==160 || runeVal==8239 || runeVal=='‹' || runeVal=='"' || runeVal=='“' || runeVal==guillemetClose || runeVal==8221 || runeVal==')' || runeVal==']' || runeVal=='\'' || runeVal=='’' || runeVal=='|' {
					pos++
					if pos < endOfBlockMax {
						runeVal = runeText[pos]
						if runeVal == LFRUNE {
							linefeedCount++
						}
						if runeVal==',' {
							continue
						}
						if (runeVal=='\'' || runeVal=='"' || runeVal=='“' || runeVal=='„' || runeVal==8221) && (runeText[pos-1]==' ' || runeText[pos-1]==160 || runeText[pos-1]==8239 || runeVal=='*') {
							continue
						}
						if runeVal=='.' || runeVal==';' || runeVal=='?' || runeVal=='!' || runeVal=='*' || runeVal==' ' || runeVal==160 || runeVal==8239 || runeVal=='_' || runeVal=='\'' || runeVal=='"' || runeVal=='“' || runeVal=='‹' || runeVal==guillemetClose || runeVal==8221 || runeVal==')' || runeVal==']' || runeVal=='\'' || runeVal=='|' {
							pos++
							if pos < endOfBlockMax {
								runeVal = runeText[pos]
								if runeVal == LFRUNE {
									linefeedCount++
								}
								if runeVal==',' {
									continue
								}
								if (runeVal=='\'' || runeVal=='"' || runeVal=='“' || runeVal=='„' || runeVal==8221) && (runeText[pos-1]==' ' || runeText[pos-1]==160 || runeText[pos-1]==8239 || runeVal=='*') {
									continue
								}
								if runeVal=='.' || runeVal==';' || runeVal=='?' || runeVal=='!' || runeVal==' ' || runeVal=='*' || runeVal==160 || runeVal==8239 || runeVal=='‹' || runeVal=='"' || runeVal=='“' || runeVal==guillemetClose || runeVal==8221 || runeVal==')' || runeVal==']' || (runeVal=='_' && runeText[pos-1]!=' ' && runeText[pos-1]!=160) || runeVal=='|' || (runeVal=='\'' && runeText[pos-1]!=' ' && runeText[pos-1]!=160 && runeText[pos-1]!=8239) {
									pos++
									if pos < endOfBlockMax {
										runeVal = runeText[pos]
										if runeVal == LFRUNE {
											linefeedCount++
										}
										if runeVal==',' {
											continue
										} 
										if runeVal=='.' || runeVal==';' || runeVal=='?' || runeVal=='!' || runeVal==' ' || runeVal=='*' || runeVal==160 || runeVal==8239 || runeVal=='‹' || runeVal=='"'  || runeVal=='“' || runeVal=='”' || runeVal==guillemetClose || runeVal==8221 || runeVal==')' || runeVal==']' || runeVal=='|' || (runeVal=='\'' && runeText[pos-1]!=' ' && runeText[pos-1]!=160 && runeText[pos-1]!=8239) {
											pos++

											if pos < endOfBlockMax {
												runeVal = runeText[pos]
												if runeVal == LFRUNE {
													linefeedCount++
												}
												if runeVal==',' {
													continue
												} 
												if runeVal=='.' || runeVal==';' || runeVal=='?' || runeVal=='!' || runeVal==' ' || runeVal==160 || runeVal==8239 || runeVal==guillemetClose || runeVal=='‹' || runeVal=='"' || runeVal=='“' || runeVal=='”' || runeVal==8221 || runeVal==')' || runeVal==']' || runeVal=='\'' || runeVal=='|' {
													pos++
													if pos < endOfBlockMax {
														runeVal = runeText[pos]
														if runeVal == LFRUNE {
															linefeedCount++
														}
														if runeVal==',' {
															continue
														} 
														if runeVal=='.' || runeVal==';' || runeVal=='?' || runeVal=='!' || runeVal==' ' || runeVal==160 || runeVal==8239 || runeVal==guillemetClose || runeVal=='‹' || runeVal=='"'  || runeVal=='“' || runeVal=='”' || runeVal==8221 || runeVal==')' || runeVal==']' || runeVal=='\'' {
															pos++

															if pos < endOfBlockMax {
																runeVal = runeText[pos]
																if runeVal == LFRUNE {
																	linefeedCount++
																}
																if runeVal==',' {
																	continue
																} 
																if runeVal=='.' || runeVal==';' || runeVal=='?' || runeVal=='!' || runeVal==' ' || runeVal==160 || runeVal==8239 || runeVal==guillemetClose || runeVal=='‹' || runeVal=='"'  || runeVal=='“' || runeVal=='”' ||runeVal==8221 || runeVal==')' || runeVal==']' || runeVal=='\'' {
																	pos++
																	if pos < endOfBlockMax {
																		runeVal = runeText[pos]
																		if runeVal=='.' || runeVal==';' || runeVal=='?' || runeVal=='!' || runeVal==' ' || runeVal==160 || runeVal==8239 || runeVal==guillemetClose || runeVal=='‹' || runeVal=='"'  || runeVal=='“' || runeVal=='”' ||runeVal==8221 || runeVal==')' || runeVal==']' || runeVal=='\'' {
																			pos++
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}

			blockLenSave = pos - pidx
			blockEndCriteria=9 // dot or similar after long text
			blockEndInfo1 = pos
			blockEndInfo2 = dotPos
			skipEolChars = 0
			skipEolCharsSave = skipEolChars // in case we do NOT break out for page end

			for runeText[pos]=='[' {
				foundMdLink := false
				if (runeText[pos+1]==']' && runeText[pos+2]=='(') ||
				   (runeText[pos+2]==']' && runeText[pos+3]=='(') ||
				   (runeText[pos+3]==']' && runeText[pos+4]=='(') ||
				   (runeText[pos+4]==']' && runeText[pos+5]=='(') {

					for i:=0; i<160 && pos-pidx+3+i + linefeedCount*LINEFEEDCHAR < endOfBlockMax; i++ {
						if runeText[pos+3+i]==')' {
							foundMdLink = true
							pos += 3+i+1
							blockEndInfo1 += 3+i+1
							blockEndInfo2 = linefeedCount
							blockEndCriteria=31 // md-link (footNote) at the very end of the page

							if runeText[pos]==guillemetClose || runeText[pos]=='"' || runeText[pos]=='“' || runeText[pos]==',' {
								pos += 1
								blockEndCriteria=231
							} else if runeText[pos+1]==guillemetClose || runeText[pos+1]=='"' || runeText[pos+1]=='“' {
								pos += 2
								blockEndCriteria=232
							} else {
							}
							blockLenSave = pos - pidx
							break
						}
					}
				}
				if !foundMdLink {
					break
				}
			}

			if runeText[pos]=='.' {
				pos++
				blockLenSave = pos - pidx
			}

			for i:=0; i<50 && pos+i < endOfBlockMax; i++ {
				runeVal1 := runeText[pos+i]
				if runeVal1==LFRUNE && pos+i-pidx+linefeedCount*LINEFEEDCHAR > blockSizeHighMark240 {
					blockLenSave = pos+i - pidx
					quote1open = -1
					quote2open = -1
					quote3open = -1
					blockEndCriteria=32 // newline is less than a few chars after long text
					blockEndInfo2 = linefeedCount
					break
				}
				if runeVal1=='#' && (runeText[pos+i+1]=='#' || runeText[pos+i+1]==' ') {
					blockLenSave = pos+i-1 - pidx
					quote1open = -1
					quote2open = -1
					quote3open = -1
					blockEndCriteria=33 // head is less than a few chars after long text
					blockEndInfo2 = -1
					break
				}
			}
		}

		if squareBracketOpen==0 && roundBracketOpen==0 && pos-pidx+linefeedCount*LINEFEEDCHAR > blockSizeHighMark240 {

			if blockEndCriteria>0 && pos - pidx > 30 {
				blockLen = blockLenSave
				skipEolChars = skipEolCharsSave
				quote1open = -1
				quote2open = -1
				quote3open = -1
				blockEndCriteria += 100
				break
			}

			if runeVal==';'  || runeVal=='…' {
				pos++
				if pos+1 < endOfBlockMax {
					if runeText[pos]=='"' || runeText[pos]==8221 || runeText[pos]==guillemetClose {
						pos++
					}
				}
				blockLen = pos - pidx
				blockEndCriteria=34
				quote1open = -1
				quote2open = -1
				quote3open = -1
				break
			}

			if pos - pidx > blockSizeHighMark200 {
				if runeVal==':' {
					pos++
					blockLen = pos - pidx
					blockEndCriteria=36
					quote1open = -1
					quote2open = -1
					quote3open = -1
					break
				}

				if pos - pidx > blockSizeHighMark260 {
					endStartPoint := pidx + blockSizeHighMark260

					roundColonOpen := 0
					squareColonOpen := 0
					squareColonIdxOpen := -1
					squareColonIdxClose := -1
					lastPosOutsideBrackets := 0
					lastRoundBracketClose := -1
					for idx := pidx; idx < endStartPoint; idx++ {
						if runeText[idx]=='[' {
							squareColonOpen++
							squareColonIdxOpen = idx
						} else if runeText[idx]==']' {
							if squareColonOpen>0 {
								squareColonOpen--
								if squareColonOpen==0 && runeText[idx+1]== '(' {
									squareColonIdxClose = idx
									roundColonOpen++
								}
							}
						} else if roundColonOpen>0 && runeText[idx]==')' {
							roundColonOpen--
							if roundColonOpen==0 && squareColonOpen==0 && idx-pidx > 100 {
								if squareColonIdxClose-squareColonIdxOpen>1 /*&& lastRoundBracketClose<0*/ {
									lastRoundBracketClose = idx+1
								}
							}
						} else if roundColonOpen==0 && squareColonOpen==0 {
							lastPosOutsideBrackets = idx
						}
					}

					if lastRoundBracketClose > -1 {
						pos = lastRoundBracketClose
						if runeText[pos]==',' || runeText[pos]=='.' {
							pos++
						}
						blockLen = pos - pidx
						blockEndCriteria=244
						blockEndInfo1 = pos
						blockEndInfo2 = blockLen
						quote1open = -1
						quote2open = -1
						quote3open = -1
						break
					}

					if lastPosOutsideBrackets>0 {
						backwardsBlockLen := -1
						roundColonOpen = 0
						squareColonOpen = 0
						for idx := lastPosOutsideBrackets; idx > pidx + 30; idx-- {
							if runeText[idx]==')' {
								roundColonOpen++
							} else if runeText[idx]=='(' {
								if(roundColonOpen>0) {
									roundColonOpen--
								}
							} else if runeText[idx]==']' {
								squareColonOpen++
							} else if runeText[idx]=='[' {
								if(squareColonOpen>0) {
									squareColonOpen--
								}
							}

							if roundColonOpen==0 && squareColonOpen==0 {
								if runeText[idx]=='[' {
									backwardsBlockLen = idx -pidx
									blockEndCriteria=43
									blockEndInfo1 = idx
									blockEndInfo2 = lastPosOutsideBrackets
									quote1open = -1
									quote2open = -1
									quote3open = -1
									break
								}
								if runeText[idx]==';' || runeText[idx]==':' || runeText[idx]==',' {
									backwardsBlockLen = idx -pidx +1

									blockEndInfo1 = blockLen
									blockEndInfo2 = blockEndCriteria
									blockEndCriteria=41
									quote1open = -1
									quote2open = -1
									quote3open = -1
									break
								}
							}
						} // for

						if backwardsBlockLen>0 {
							blockLen = backwardsBlockLen
							break
						}
					}

					if runeVal==' ' {
						pos++
						blockLen = pos - pidx
						blockEndCriteria=39
						quote1open = -1
						quote2open = -1
						quote3open = -1
						break
					}

					if pos - pidx > blockSizeHighMark280 {
						pos++
						blockLen = pos - pidx
						blockEndCriteria=40
						quote1open = -1
						quote2open = -1
						quote3open = -1
						break
					}
				}
			}
		}
	}
	if blockEndCriteria==0 && blockLen==0 && pos>pidx {
		blockLen = pos-pidx
	} else if blockEndCriteria==9 && blockLen==0 && blockLenSave>0 {
		blockLen = blockLenSave;
	}

	italicIsOpenIdx := -1
	if stateItalic {
		italicIsOpenIdx = 0
	}
	boldIsOpenIdx := -1
	if stateBold {
		boldIsOpenIdx = 0
	}

	squareBracketOpenCount := 0
	roundBracketOpenCount := 0
	parsingImage := false
	for pos2 := pidx; pos2 < pidx+blockLen; pos2++ {
		runeVal = runeText[pos2]
		if runeVal==0 {
			break
		}

		if runeVal=='[' {
			squareBracketOpenCount++
		} else if runeVal==']' {
			squareBracketOpenCount--
			if squareBracketOpenCount==0 && runeText[pos2+1]=='(' {
				roundBracketOpenCount++
			}
		}
		if roundBracketOpenCount>0 {
			if runeVal==')' {
				roundBracketOpenCount--
			}
		}

		if runeVal=='/' && runeText[pos2+1]=='/' && runeText[pos2+2]=='!' {
			parsingImage = true
		} else if runeVal==LFRUNE {
			parsingImage = false
		}

		if squareBracketOpenCount==0 && roundBracketOpenCount==0 {
			if runeVal=='*' {
				if (runeText[pos2+1]==' ' && runeText[pos2+2]=='*' && runeText[pos2+3]==' ' && runeText[pos2+4]=='*') {
					blockEndCriteria=45
					blockLen = pos2+4+1 - pidx
					blockEndInfo2 = linefeedCount
					skipEolChars = 1
					quote1open = -1
					quote2open = -1
					quote3open = -1
					break
				}

				if runeText[pos2+1]=='*' {
					if boldIsOpenIdx<0 {
						if pos2+1 >= pidx+blockLen {
						} else if pos2>pidx && (runeText[pos2-1]=='*' || runeText[pos2-1]=='\\') {
						} else if runeText[pos2+2]==' ' || runeText[pos2+2]=='<' || runeText[pos2+2]=='*' {
						} else {
							boldIsOpenIdx = pos2-pidx
						}
					} else {
						if pos2==pidx {
							boldIsOpenIdx = -1
							stateBold = false
							blockArray[totalBlocks-1].boldOnEnd = false
							pidx += 2
							blockLen -= 2
							pos2 = pidx-1
						} else if pos2>pidx && (runeText[pos2-1]==' ' ||runeText[pos2-1]=='*' || runeText[pos2-1]=='\\') {
						} else {
							boldIsOpenIdx = -1
						}
					}
					continue
				}
			}

			if !parsingImage && (runeVal=='_' || runeVal=='*') {
				if runeText[pidx]=='#' {
					continue
				}

				var preChar rune = 0
				if pos2>0 {
					preChar = runeText[pos2-1]
				}
				postChar := runeText[pos2+1]

				if italicIsOpenIdx<0 {
					if pos2 >= pidx+blockLen {
					} else if postChar==' ' {
					} else if postChar=='<' {
					} else if postChar==guillemetClose {
					} else if postChar==']' {
					} else if postChar=='*' {
					} else if postChar==LFRUNE {
					} else if postChar==runeVal {
					} else if preChar==runeVal {
					} else if preChar=='\\' {
					} else {
						italicIsOpenIdx = pos2-pidx
					}
				} else {
					if preChar==' ' || preChar==runeVal || preChar=='\\' || postChar==runeVal {
					} else {
						italicIsOpenIdx = -1
					}
				}
			}
		}
	}

	endsWithItalicOpen := false
	endsWithBoldOpen := false
	if blockLen + skipEolChars <= 0 {
		myPrint("! nextTextBlock eol blockLen+skipEolChars <= 0 (%d %d %d)\n",blockLen,skipEolChars,blockEndCriteria)
	} else {
		if !endsWithItalicOpen && italicIsOpenIdx>=0 && italicIsOpenIdx < blockLen {
			endsWithItalicOpen = true
		}
		if !endsWithBoldOpen && boldIsOpenIdx>=0 && boldIsOpenIdx < blockLen {
			endsWithBoldOpen = true
		}
	}

	stateItalic = endsWithItalicOpen
	stateBold = endsWithBoldOpen
	return blockLen, skipEolChars, blockEndCriteria, blockEndInfo1, blockEndInfo2, linefeedCount
}

func addChapName(chapNameStartRune []rune, idx int, hlevel int, blockLen int) {
	str := string(chapNameStartRune)
	str = strings.Replace(str,"[]()","",-1)
	if len(str) <= 0 {
		myPrint("! addChapName empty str=(%s) (%s) %d %d\n",str,string(runeText[idx:idx+blockLen]),idx,hlevel)
		return
	}
	if len(str) > 280 {
		str = str[:280]
	}
	chapNameRune := []rune(str)
	max := len(chapNameRune)
	chapName := string(chapNameRune[:max])
	chapName = strings.TrimSpace(chapName)
	if chapName=="" {
		myPrint("! addChapName no chapname=(%s) (%s) (%s) idx=%d max=%d hlevel=%d %d %d\n", chapName, string(chapNameRune), string(runeText[idx:idx+blockLen]), idx, max, hlevel,len(ChapterSlice),blockLen)
		return
	}

	chapNameStore := strings.Replace(chapName,"[]()","",-1)

	myIdx := strings.Index(chapNameStore,"[](")
	for myIdx>=0 {
		myIdx2 := strings.Index(chapNameStore[myIdx:],")")
		if myIdx2>=0 {
			chapNameStore = chapNameStore[:myIdx] + chapNameStore[myIdx+myIdx2+1:]
			myIdx = strings.Index(chapNameStore,"[](")
		} else {
			myIdx = -1
		}
	}

	for {
		idxSqBr := strings.Index(chapNameStore,"[")
		if idxSqBr>=0 {
			idxSqBrClose := strings.Index(chapNameStore[idxSqBr:],"]")
			if idxSqBrClose>0 {
				if len(chapNameStore)>idxSqBr+idxSqBrClose+1 && chapNameStore[idxSqBr+idxSqBrClose+1]=='(' {
					idxRdBrClose := strings.Index(chapNameStore[idxSqBr+idxSqBrClose+2:],")")
					if idxRdBrClose==0 {
						chapNameStore = strings.TrimSpace(chapNameStore[0:idxSqBr] +
							chapNameStore[idxSqBr+1:idxSqBr+idxSqBrClose] +
							chapNameStore[idxSqBr+idxSqBrClose+1 + idxRdBrClose+2:])
						continue
					}
				}
			}
		}
		break
	}

	chapNameStore = strings.Replace(chapNameStore,"*","",-1)
	chapNameStore = strings.Replace(chapNameStore,"_","",-1)
	chapNameStore = mdLinkToLabel(chapNameStore,2,1)

	if chapNameStore!="" {
		ChapterSlice = append(ChapterSlice,ChapterType{chapNameStore,idx,hlevel})

		chapNameRune = []rune(chapNameStore)
		if max < len(chapNameRune) {
			myPrint("! addChapName max=%d <= len(chapNameRune)=%d\n", max, len(chapNameRune))
		} else if max == len(chapNameRune) {
			// ignore: chapNameRune did not change
		} else {
			i := 0
			for i < blockLen - (hlevel+1) {
				runeText[idx+hlevel+1+i] = ' '
				i++
			}
			copy(runeText[idx+hlevel+1:],[]rune(chapName)) // dst,src
		}
	}
}

func searchRuneArray(text []rune, search string, direc int) int {
	searchRunes := []rune(search)
	myTextLen := len(text)

	for i := range text {
		found := true
		ii := i
		if direc<0 {
			ii = myTextLen - i
		}
		for j := range searchRunes {
		    if ii+j>=myTextLen || unicode.ToLower(text[ii+j]) != unicode.ToLower(searchRunes[j]) {
		        found = false
		        break
		    }
		}
		if found {
		    return ii
		}
	}
	return -1
}

func idxText(runeText []rune, searchStr string, maxlen int) int {
	if searchStr=="" {
		myPrint("! idxText searchStr is empty\n")
		return 0
	}
	if len(searchStr)>maxlen {
		myPrint("idxText searchStr=(%s) len=%d cut to maxlen=%d\n",searchStr,len(searchStr),maxlen)
		searchStr = searchStr[:maxlen]
	}
	searchRunes := []rune(searchStr)
	if len(searchRunes)>3 {
		myPrint("idxText searchStr=(%s) len searchRunes=%d in runeText len=%d\n",searchStr,len(searchRunes),len(runeText))
	}
	biggestFound:=0
	startSearchRune := searchRunes[0]
	for i,r1 := range runeText[:len(runeText)-len(searchRunes)] {
		if r1 != startSearchRune {
			continue
		}

		found := 0
		for j,r := range searchRunes {
			if runeText[i+j] != r {
				break
			}
			found++
		}
		if found > biggestFound {
			biggestFound = found
			if found == len(searchRunes) {
				if len(searchRunes)>3 {
					myPrint("idxText found: i=%d (%s) biggestFound=%d\n",i,searchStr,biggestFound)
				}
				return i
			}
		}
	}

	return -1
}

type PageLenArray struct {
	Pagenum int
	CharLength int
}
func longestBlocks() []PageLenArray {
	longestArray := make([]PageLenArray, totalBlocks)
	lfMultiply := 50
	for i := range totalBlocks {
		longestArray[i].Pagenum = i
		longestArray[i].CharLength = blockArray[i].CharLength + blockArray[i].LinefeedCount*lfMultiply
	}
	sort.Slice(longestArray, func(i, j int) bool {
		return longestArray[i].CharLength > longestArray[j].CharLength
	})
	return longestArray
}

func blockForIdx(idx int) int {
	var page=0
	for page < totalBlocks && blockArray[page].Idx < idx {
		page++
	}
	if page>0 && (page>=totalBlocks || blockArray[page].Idx>idx) {
		page--
	}
	return page
}

func textForIdx(idx int, leng int, blockNumber int, fastmode bool) string {
	runes := runeText[idx:idx + leng]
	runesLen := len(runes)
	str := strings.TrimSpace(string(runes))
	str = strings.Replace(str,"[]()","",-1)

	myIdx1 := strings.Index(str,"[](#")
	for myIdx1>=0 {
		myIdx2 := strings.Index(str[myIdx1:],")")
		if myIdx2>0 {
			str = str[:myIdx1] + str[myIdx1+myIdx2+1:]
			myIdx1 = strings.Index(str,"[](#")
		} else {
			myIdx1 = -1
		}
	}
	str = strings.TrimSpace(str)
	for str!="" && str[0]==LFCHAR {
		str = str[1:]
	}

	if strings.HasPrefix(str,"#") {
		if blockNumber >= 1 {
			blockArray[blockNumber-1].blockquoteOnEnd = false
			blockArray[blockNumber-1].italicOnEnd = false
			blockArray[blockNumber-1].boldOnEnd = false
		}
	}
	runes = []rune(str)
	runesLen = len(runes)

	s := []rune("")
	lineStart := true
	blockquoteOpenCount := 0
	blockquoteOpenIdx := -1
	if blockNumber>0 && blockArray[blockNumber-1].blockquoteOnEnd {
		blockquoteOpenIdx = 0
		blockquoteOpenCount++
	}

	nornalIdx := -1
	i := 0
	for ; i < runesLen; i++ {
		chr := runes[i]
		if chr=='>' || chr==LFRUNE {
			i2 := i
			if blockquoteOpenIdx >= 0 {
				s = append(s,[]rune("<blockquote>")...)
				s = append(s,runes[blockquoteOpenIdx:i2]...)
				s = append(s,[]rune("</blockquote>")...)
				blockquoteOpenIdx = -1
			}
			lineStart = true
		}

		if chr=='>' && lineStart {
			if nornalIdx>=0 {
				s = append(s,runes[nornalIdx:i]...)
				nornalIdx = -1
			}
			lineStart = false
			if i+1<len(runes) && runes[i+1] == ' ' {
				i++
			}
			if i+1<len(runes) && runes[i+1] == '>' {
				i++
			}
			if i+1<len(runes) && runes[i+1] == ' ' {
				i++
			}

			blockquoteOpenIdx = i
			blockquoteOpenCount++
		} else {

			if chr!=' ' {
				lineStart = false
			}
			if chr==LFRUNE {
				lineStart = true
			}
			if blockquoteOpenIdx<0 && nornalIdx<0 {
				nornalIdx = i
			}
		}
	}
	if blockquoteOpenCount>0 {
		if nornalIdx > -1 {
			s = append(s,runes[nornalIdx:i]...)
			nornalIdx = -1
		}
		if blockquoteOpenIdx > -1 {
			s = append(s,[]rune("<blockquote>")...)
			s = append(s,runes[blockquoteOpenIdx:i]...)
			s = append(s,[]rune("</blockquote>")...)

			blockArray[blockNumber].blockquoteOnEnd = true
		}
		runes = s
	}

	str = string(runes)
	str = strings.Replace(str,"</blockquote><blockquote>","",-1)
	str = strings.Replace(str,"<blockquote>></blockquote>","",-1)
	str = strings.TrimSpace(str)
	if str=="" {
		return str
	}

	italicOnStart := false
	if blockNumber>0 && blockArray[blockNumber-1].italicOnEnd {
		italicOnStart = true
	}
	if(!fastmode) {
		str = replaceItalic(str,italicOnStart,false)
		str = strings.Replace(str,"</i> <i>"," ",-1)

		boldOnStart := false
		if blockNumber>0 && blockArray[blockNumber-1].boldOnEnd {
			boldOnStart = true
		}
		str = replaceBold(str,boldOnStart)
		str = replaceMdTable(str)
	}

	str = strings.Replace(str,"```","",-1)

	if(!fastmode) {
		str = parseHyperlink(str,idx)
		str = strings.TrimSpace(str)
	}

	runes = []rune(str)
	runesLen = len(runes)
	var runes2 = make([]rune, len(runes)+4000)
	outIdx := 0
	for lidx,chr := range runes {
		if chr==LFRUNE {
			if outIdx<=0 {
				continue
			}
			idx2 := lidx+1
			for idx2 < len(runes) {
				if runes[idx2]!=LFRUNE {
					break
				}
				idx2++
			}
			if idx2 >= len(runes) {
				break
			}

			insertRunes := []rune("<div></div>")
			copy(runes2[outIdx:],insertRunes)
			outIdx += len(insertRunes)
		} else {
			runes2[outIdx] = chr
			outIdx++
		}
	}

	str = string(runes2[0:outIdx])
	myIdx := runeSearchString(runes2[0:outIdx], "]()")
	if myIdx>0 {
		closeBrCount := 1
		i := 1
		for myIdx-i>=0 {
			ru := runes2[myIdx-i]
			if ru == ']' {
				closeBrCount++
			} else if ru == '[' {
				closeBrCount--
				if closeBrCount==0 {
					copy(runes2[myIdx-i:],runes2[myIdx+3:])
					outIdx -= i+3
					break
				}
			}
			i++
		}
		if outIdx>0 {
			str = string(runes2[0:outIdx])
		}
	}
	return str
}

func textForBlocknumber(blockNumber int, fastmode bool) string {
	if blockNumber >= len(blockArray) {
		myPrint("textForBlocknumber blockNumber=(%d) >= len(blockArray)=%d\n",blockNumber,len(blockArray))
		return ""
	}
	if blockNumber < 0 {
		myPrint("textForBlocknumber blockNumber=(%d) < 0\n",blockNumber)
		return ""
	}
	return textForIdx(blockArray[blockNumber].Idx, blockArray[blockNumber].CharLength, blockNumber, fastmode)
}

func parseHyperlink(str string, idx int) string {
	if str=="" {
		return str
	}
	if strings.HasPrefix(str,"](#") {
		str = "["+str
	}

	startIdx:=0
	strRunes := []rune(str)

	squareBracketOpenCount := 0
	roundBracketOpenCount := 0
	hyper1 := make([]int,100)
	hyper2 := -1
	hyper3 := -1

	i := 0
	for i < len(strRunes) {
		runeVal := strRunes[i]
		if runeVal==0 {
			i++
			continue
		}

		if runeVal=='[' {
			if squareBracketOpenCount>=0 {
				hyper1[squareBracketOpenCount] = i
			}
			squareBracketOpenCount++
		} else if runeVal==']' {
			squareBracketOpenCount--
			if squareBracketOpenCount>=0 && len(strRunes)>i+1 && strRunes[i+1]=='(' {
				hyper2 = i - hyper1[squareBracketOpenCount]
				roundBracketOpenCount++
			}
		}

		if roundBracketOpenCount>0 {
			if runeVal==')' {
				roundBracketOpenCount--
				if roundBracketOpenCount==0 {
					hyper3 = i - (hyper1[squareBracketOpenCount] + hyper2 + 2)
					strRunes,startIdx = foundHyperlink(strRunes, hyper1[squareBracketOpenCount],hyper2,hyper3, idx, str)
					if startIdx<0 || strings.Index(string(strRunes),"[")<0 {
						break
					}
					i = startIdx -1
				}
			}
		}
		i++
	}
	str = string(strRunes)
	return strings.TrimSpace(str)
}

func foundHyperlink(runes []rune, hyper int, hyper2 int, hyper3 int, idx int, str string) ([]rune,int) {
	label := string(runes[hyper+1:hyper+hyper2])

	linkId := string(runes[hyper+hyper2+2:hyper+hyper2+2+hyper3])
	linkId = strings.TrimSpace(linkId)
	linkIdIdx := strings.Index(linkId,"\"")
	if linkIdIdx>=0 {
		linkId = linkId[:linkIdIdx]
		linkId = strings.TrimSpace(linkId)
	}
	if linkId=="" || label=="" {
		startIdx := hyper
		preLabel := string(runes[:hyper])
		postLabel := string(runes[hyper+hyper2+2+hyper3+1:])
			myLabel := ""
			if runes[0]=='#' {
				myLabel = label
			}
			str = preLabel + myLabel + postLabel
		runes = []rune(str)
		return runes,startIdx
	}

	linkIdShort := linkId
	linkIdIdx = strings.LastIndex(linkId,"#")
	if linkIdIdx>=0 {
		linkIdShort = strings.TrimSpace(linkId[linkIdIdx+1:])
	}

	var followupChar byte = 0
	nextCharIdx := hyper+hyper2+2+hyper3+1
	for(true) {
		if nextCharIdx >= len(str) {
			break
		}
		followupChar = str[nextCharIdx]
		if followupChar == ' ' {
			nextCharIdx++
			continue
		}
		break
	}

	footnoteText := ""
	isFootnote := false

	labelPlaintext, err := md.ConvertString(label)
	if err!=nil {
		labelPlaintext = label
	}
	labelLen := len(label)
	if len(labelPlaintext) < labelLen {
		label = labelPlaintext
		labelLen = len(label)
	}
	if(labelLen>300) {
		trippleDots := "&mldr;"
		label = label[:300]+trippleDots
		if label[299]==' ' {
			label = label[:299]+trippleDots
		} else if label[298]==' ' {
			label = label[:298]+trippleDots
		} else if label[297]==' ' {
			label = label[:297]+trippleDots
		} else if label[296]==' ' {
			label = label[:296]+trippleDots
		} else if label[295]==' ' {
			label = label[:295]+trippleDots
		} else if label[294]==' ' {
			label = label[:294]+trippleDots
		} else if label[293]==' ' {
			label = label[:293]+trippleDots
		} else if label[292]==' ' {
			label = label[:292]+trippleDots
		} else if label[291]==' ' {
			label = label[:291]+trippleDots
		} else if label[290]==' ' {
			label = label[:290]+trippleDots
		}
		labelLen = len(label)
	}
	if label!="" {
		if labelLen>=1 && labelLen<=7 {
			isFootnote = true
			myPrint("foundHyperlink assume short label '%s' is footnote-link\n",label)
		} else {
			myPrint("foundHyperlink assume label '%s' is NOT footnote-link\n",label)
		}
	}

	idxMdID,ok := linkIdToIdxMap[linkIdShort]
	if !ok {
		idxMdID = -1
	}
	myPrint("foundHyperlink idxMdID=(%d) ok=%v from linkIdShort=%s\n", idxMdID, ok, linkIdShort)
	if idxMdID<0 {
		idxMdID = offsetChapName(label)
		if idxMdID>=0 {
			isFootnote = false
			linkId = fmt.Sprintf("!%d",idxMdID)
			myPrint("foundHyperlink linkId=(%s) from label=chapName\n", linkId)
		}
	}
	if isFootnote && idxMdID>=0 && idxMdID<idx {
		myPrint("! foundHyperlink no footnote: idxMdID=%d < idx=%d linkId=(%s) from label=chapName\n", idxMdID,idx,linkId)
		isFootnote = false
	}

	if isFootnote {
		footnoteText = getTextForLinkId2(linkIdShort, 0)

	}

	if footnoteText!="" {
		footnoteText = strings.Replace(footnoteText,"[]()","",-1)
		footnoteText = strings.Replace(footnoteText,"[](#"+linkIdShort+")","",-1)
	}

	if footnoteText!="" {
		myIdx := strings.Index(footnoteText,"[](#")
		for myIdx>=0 {
			myIdx2 := strings.Index(footnoteText[myIdx:],")")
			if myIdx2>0 {
				footnoteText = footnoteText[:myIdx] + footnoteText[myIdx+myIdx2+1:]
				myIdx = strings.Index(footnoteText,"[](#")
			} else {
				myIdx = -1
			}
		}
		dispLen := len(footnoteText)
		if dispLen > 300 {
			dispLen = 300
		}
	}

	if footnoteText!="" {
		myIdx1 := strings.Index(footnoteText,"]()");
		for myIdx1>0 {
			countSqOpen := 1
			myIdx2 := myIdx1-1
			for myIdx2>=0 {
				if footnoteText[myIdx2]==']' {
					countSqOpen++
				} else if footnoteText[myIdx2]=='[' {
					countSqOpen--
				}
				if countSqOpen==0 {
					myLabel := footnoteText[myIdx2+1:myIdx1]
					if len(myLabel)>0 {
						footnoteText = footnoteText[0:myIdx2] + myLabel + footnoteText[myIdx1+3:]
					}
					break
				}
				myIdx2--
			}
			myIdx1 = strings.Index(footnoteText,"]()");
		}

		footnoteText = strings.TrimSpace(footnoteText)
	}

	if isFootnote && footnoteText!="" {
		myIdx := strings.Index(footnoteText,"[](");
		for myIdx>=0 {
			myIdx2 := strings.Index(footnoteText[myIdx+3:],")");
			if myIdx2>=0 {
				if myIdx>0 {
					footnoteText = footnoteText[0:myIdx] + footnoteText[myIdx+3+myIdx2+1:]
				} else {
					footnoteText = footnoteText[3+myIdx2+1:]
				}
				myIdx = strings.Index(footnoteText,"[](");
			} else {
				break
			}
		}
	}

	if footnoteText!="" {
		for strings.Index(footnoteText,"\n")==0 {
			footnoteText = footnoteText[1:]
		}
	}

	if isFootnote && footnoteText!="" && len(footnoteText)>300 {
		myIdx := strings.Index(footnoteText,"\n")
		if myIdx<60 {
			myIdx2 := strings.Index(footnoteText[myIdx+1:],"\n")
			if myIdx2>=0 {
				myIdx = myIdx +1 + myIdx2
			}
		}
		if myIdx>=0 {
			footnoteText = strings.TrimSpace(footnoteText[:myIdx])
		}
	}

	if isFootnote && footnoteText!="" {
		myIdx3 := strings.Index(footnoteText,"# ")
		if myIdx3>=0 && myIdx3<10 {
			isFootnote = false
			myPrint("foundHyperlink text contains early '# ' at %d isFootnote=false '%s'\n",myIdx3,footnoteText)
		}
	}

	if isFootnote && strings.Index(footnoteText,"//!")>=0 {
		myPrint("foundHyperlink image found: isFootnote=false\n")
		isFootnote = false
	}

	if(isFootnote) {
		if len(footnoteText)>9 {
			linkIdToTextMap[linkIdShort] = footnoteText
		} else {
			isFootnote = false
		}
	}

	oldStartIdx := hyper
	startIdx := hyper+hyper2+2+hyper3 +1

	if !isFootnote {
		titleString := ""
		if strings.HasPrefix(linkId,"http://") || strings.HasPrefix(linkId,"https://") {
			if label!="" {
				label = "&nearr;"+label
				titleString = "external hyperlink "+linkId
			} else {
				myPrint("! foundHyperlink ext link, but no label linkId=(%s)\n", linkId)
			}

		} else if strings.HasPrefix(linkId,"!") {
			titleString = "internal hyperlink "+linkId

		} else if linkId!="" {
			filename := linkId
			idxPound := strings.Index(filename,"#")
			if idxPound>=0 {
				filename = filename[:idxPound]
			}
			if filename!="" {
				origFilename := filename
				_,ok := xhtmlFileMap[filename]
				if !ok && subBookFolder!="" {
					filename = subBookFolder+"/"+filename
					_,ok = xhtmlFileMap[filename]
				}

				if ok {
					titleString = "internal hyperlink "+linkId
					myPrint("foundHyperlink int-link2 OK filename=(%s)\n", filename)
				} else {
					filename = origFilename
					idxSlash := strings.Index(filename,"/")
					if idxSlash>=0 {
						myLinkId := filename[idxSlash+1:]
						myIdx,ok := linkIdToIdxMap[myLinkId]
						if ok && myIdx>=0 {
							linkId = fmt.Sprintf("!%d",myIdx)
							titleString = "internal hyperlink "+linkId
						}
					} else {
						idxPound := strings.Index(linkId,"#")
						if idxPound>=0 {
							myLinkId := linkId[idxPound+1:]
							myIdx,ok := linkIdToIdxMap[myLinkId]
							if ok && myIdx>=0 {
								linkId = fmt.Sprintf("!%d",myIdx)
								titleString = "internal hyperlink "+linkId
							}
						}
					}
					if titleString=="" {
						myPrint("! foundHyperlink NOT FOUND int-link (%s) (%s) (%s) %d|%v\n", filename, linkId, subBookFolder, len(xhtmlFileMap), xhtmlFileMap)
					}
				}
			} else {
				linkId = fmt.Sprintf("!%d",idxMdID)
				titleString = "internal hyperlink "+linkId
			}
		}

		linkMarkup := ""
		if label!="" {
			idxImg := strings.Index(label,"//!")
			if titleString!="" {
				if(idxImg>=0 && idxImg<20) {
					linkMarkup =
						"<div class='jumplinkimg'>" +
							"<a onClick=\"jumpHyper('"+linkId+"'); event.stopPropagation();\"" +
								" title=\""+titleString+"\">"+label+"</a>"+
						"</div>";
				} else {
					linkMarkup = "<a class='jumplink' onClick=\"jumpHyper('"+linkId+"'); event.stopPropagation();\""+
						" title=\""+titleString+"\">"+label+"</a>";
				}

				if followupChar=='[' {
					linkMarkup = linkMarkup+"<br>"
				} else if followupChar!=' ' {
				}
			} else {
				if linkId=="" || idxMdID<0 {
					linkMarkup = " "+label
				} else {
					titleString = "internal hyperlink "+linkId;

					if(idxImg>=0 && idxImg<20) {
						linkMarkup =
							"<div class='jumplinkimg'>" +
								"<a onClick=\"jumpHyper('"+linkId+"'); event.stopPropagation();\">"+label+"</a>"+
							"</div>";
					} else {
						linkMarkup = "<a class='jumplink' onClick=\"jumpHyper('"+linkId+"'); event.stopPropagation();\""+
							" >"+label+"</a>";
					}
				}
			}
		}
		str = string(runes[:oldStartIdx]) + string([]rune(linkMarkup)) + string(runes[startIdx:])
		runes = []rune(str)
		startIdx = oldStartIdx + len([]rune(linkMarkup))

	} else {
		pretext := string(runes[:oldStartIdx])
		if len(pretext) > 0 {
			retryLoop := true
			for retryLoop {
				retryLoop = false
				squareBracketOpen := 0
				pointyBracketOpen := 0
				roundColonOpenIdx := 0
				pointyBracketOpenIdx := -1
				for idx := 0; idx<len(pretext); idx++ {
					if pretext[idx]=='<' {
						if pointyBracketOpen==0 {
							pointyBracketOpenIdx = idx
						}
						pointyBracketOpen++
					} else if pretext[idx]=='>' {
						pointyBracketOpen--
						if pointyBracketOpen==0 && pointyBracketOpenIdx>=0 {
							pretext = pretext[:pointyBracketOpenIdx] + pretext[idx+1:]
							retryLoop = true
							break
						}
					}
					if pointyBracketOpen==0 {
						if pretext[idx]=='[' {
							squareBracketOpen++
						} else if pretext[idx]==']' {
							squareBracketOpen--
							if idx+1<len(pretext) && pretext[idx+1]=='(' {
								roundColonOpenIdx = idx+1
							}
						} else if pretext[idx]==')' && roundColonOpenIdx>0 {
							pretext = pretext[:roundColonOpenIdx] + pretext[idx+1:]
							retryLoop = true
							break
						}
					}
				}
			}
		}

		if strings.HasPrefix(pretext,"#") {
			idxSpace := strings.Index(pretext," ")
			if idxSpace>0 && idxSpace<10 {
				pretext = pretext[idxSpace+1:]
			}
		}

		if len(pretext) > 100 {
			pretext = pretext[len(pretext) - 100:]
		}

		if len(pretext) > 50 {
			for idx := 0; idx<len(pretext); idx++ {
				if pretext[idx]==' ' {
					pretext = pretext[idx+1:]
					break
				}
			}
		}

		pretext = url.QueryEscape(pretext)
		pretext = strings.Replace(pretext,"/'/g","%27",-1)

		if pretext!="" {
			pretext = "..."+pretext
		}

		footnotePossible := false
		if footnoteText!="" {
			footnotePossible = true
		}

		jumptoPossible := false
		idxMdID,ok := linkIdToIdxMap[linkIdShort]
		if !ok || idxMdID<0 {
			myPrint("# foundHyperlink BAD BAD '%s'\n",linkIdShort)
		}
		if idxMdID<idx {
			footnotePossible = false
		}
		linkIdShortJump := linkIdShort;
		if idxMdID>=0 {
			jumptoPossible = true
			linkIdShortJump = fmt.Sprintf("!%d",idxMdID)
		}
		linkMarkup := ""
		if footnotePossible || jumptoPossible {
			if footnotePossible && jumptoPossible {
				linkMarkup += "<a class=\"footnotelink\""+
					" onClick=\"showFootnote('"+linkIdShort+"','"+pretext+"'); event.stopPropagation();\""+
					" oncontextmenu=\"event.stopPropagation(); jumpHyper('"+linkIdShortJump+"',0);\""+
					" title=\"footnote (long-press hyperlink) "+linkIdShort+"\">"+label+"</a>"
			} else if footnotePossible {
				linkMarkup += "<a class=\"footnotelink\""+
					" onClick=\"showFootnote('"+linkIdShort+"','"+pretext+"'); event.stopPropagation();\""+
					" title=\"footnote\">"+label+"</a>"
			} else {
				linkMarkup += "<a class=\"jumplink\""+
					" onClick=\"event.stopPropagation(); jumpHyper('"+linkIdShortJump+"',0);\""+
					" oncontextmenu=\"event.stopPropagation(); jumpHyper('"+linkIdShortJump+"',0);\""+
					" title=\"internal hyperlink: "+linkIdShortJump+"\">"+label+"</a>"
			}
		} else {
			if(label!="") {
				linkMarkup = " "+label+" "
			} else {
				linkMarkup = " - "
			}
		}

		lenOfRest := len(string(runes[startIdx:]))
		if lenOfRest <= 0 {
			str = string(runes[:oldStartIdx]) + linkMarkup
			runes = []rune(str)
			startIdx = oldStartIdx + len(linkMarkup)
			return runes,startIdx
		}

		nextChar := runes[startIdx]
		if (nextChar>='a' && nextChar<='z') || (nextChar>='A' && nextChar<='Z') {
			linkMarkup = linkMarkup+" "
		}

		lenOfMarkup := len([]rune(linkMarkup))
		myPrint("foundHyperlink new footnote %d %d %d\n", oldStartIdx,hyper,startIdx)
		str = string(runes[:oldStartIdx]) + linkMarkup + strings.TrimSpace(string(runes[startIdx:]))
		runes = []rune(str)

		startIdx = oldStartIdx + lenOfMarkup
		myPrint("foundHyperlink new footnote done startIdx=(%d)\n", startIdx)
	}
	return runes,startIdx
}

func offsetChapName(chapName string) int {
	if chapName!="" {
		for _,chap := range ChapterSlice {
			if chap.Title == chapName {
				myPrint("offsetChapName chapName=(%s) ret=%d\n", chapName,chap.Idx)
				return chap.Idx
			}
		}
	}
	myPrint("offsetChapName chapName=(%s) ret=-1\n", chapName)
	return -1
}

func cleanupTextFromHtmlConverter(textStr string) string {
	replacer := strings.NewReplacer(
			"\n\n", "\a",
			"\n", " ",
			"\\.", ".",
			"\\#", "#",
			"\\+", "+",
			"\\-", "-",
			"<!--THE END-->", "",
	)
	textStr = replacer.Replace(textStr)
	textStr = strings.Join(strings.Fields(textStr), " ")
	textStr = strings.Replace(textStr,"\a","\n",-1)
	return textStr
}

func replaceItalic(str string, italicOnStart bool, removeItalic bool) string {
	if str=="" {
		return str
	}
	if strings.HasPrefix(str,"#") {
		return str
	}

	italicReplace := 0
	strRunesLen := len([]rune(str))
	strRunes := []rune(str+"                                                                                          ")
	idxUnderline := -1
	idxUnderlen := 0
	parsingImage := false
	closeNeeded := 0

	if italicOnStart {
		idxUnderline = 0
		idxUnderlen = 0
	}

	squareBracketOpenCount := 0
	roundBracketOpenCount := 0
	idx := 0

	var rpItClose = func() {
		s := []rune("")
		if idxUnderline>0 {
			s = append(s,strRunes[:idxUnderline]...)
		}

		if !removeItalic {
			if idxUnderline>0 && strRunes[idxUnderline-1]!='\'' && strRunes[idxUnderline-1]!='"' &&
					strRunes[idxUnderline-1]!='„' && strRunes[idxUnderline-1]!=8221 {
				s = append(s,[]rune("<i>")...)
			} else {
				s = append(s,[]rune("<i>")...)
			}
		}

		s = append(s,strRunes[idxUnderline+idxUnderlen: idx]...)
		if !removeItalic {
			s = append(s,[]rune("</i>")...)
		}
		if idx +1 <= len(strRunes) {
			s = append(s,strRunes[idx+1:]...)
		}

		copy(strRunes,s)
		strRunesLen = len(strRunes)
		italicReplace++
		idxUnderline = -1
	}

	for idx < strRunesLen {
		runeVal := strRunes[idx]
		if runeVal==0 {
			break
		}

		if runeVal=='[' {
			squareBracketOpenCount++
		} else if runeVal==']' {
			squareBracketOpenCount--
			if strRunes[idx+1]=='(' {
				roundBracketOpenCount++
			}
		}
		if roundBracketOpenCount>0 {
			if runeVal==')' {
				roundBracketOpenCount--
			}
		}
		if squareBracketOpenCount==0 && roundBracketOpenCount==0 {
			if runeVal=='/' && strRunes[idx+1]=='/' && strRunes[idx+2]=='!' {
				parsingImage = true
			} else if runeVal==LFRUNE || (runeVal=='<' && strRunes[idx+1]=='b' && strRunes[idx+2]=='r') {
				parsingImage = false
			}
			if(parsingImage) {
				idx++
				continue
			}

			if idx+1 < len(strRunes) {
				var preChar rune = 0
				if idx>0 {
					preChar = strRunes[idx-1]
				}
				postChar := strRunes[idx+1]
				if idxUnderline<0 {
					if (runeVal=='_' && preChar!='\\' && postChar!=' ' && postChar!=LFRUNE && postChar!='<' && postChar!=']') ||
					   (runeVal=='*' && postChar!=' ' && postChar!=LFRUNE && postChar!='<' && postChar!=']' &&
						postChar!=runeVal && preChar!=runeVal && preChar!='\\') {
						idxUnderline = idx
						idxUnderlen = 1
					}
				} else {
					if (runeVal=='_' && preChar!='\\') ||
					   (runeVal=='*' && preChar!='\\') {
						closeNeeded = 1
					}
				}
			}
		}

		if runeVal==' ' {
			countNonBlank := 0
			for i:=idx; i<len(strRunes); i++ {
				if strRunes[i] != ' ' {
					countNonBlank++
				}
			}
			if countNonBlank==0 {
				closeNeeded=2
			}
		}
		
		if closeNeeded>0 {
			if idxUnderline>=0 {
				rpItClose()
			}

			if closeNeeded==2 {
				break
			}

			closeNeeded = 0
		}
		idx++
	}

	if idxUnderline>=0 {
		rpItClose()
	}

	str = strings.TrimSpace(string(strRunes))
	return str
}

func replaceBold(str string, boldOnStart bool) string {
	boldReplace := 0
	idxSearch := 0
	idxBold := 0
	if !boldOnStart {
retry:
		mystr := str[idxSearch:]
		idxBold = strings.Index(mystr, "**")
		if idxBold>=0 {
			if idxSearch+idxBold>0 && str[idxSearch+idxBold-1]=='\\' {
				idxSearch = idxSearch + idxBold+2
				goto retry
			}
			if len(mystr)>idxSearch+idxBold+2 && mystr[idxSearch+idxBold+2]=='<' {
				myPrint("replaceBold false idxBold=%d '%s'\n",idxBold,mystr[idxSearch+idxBold:idxSearch+idxBold+6])
				idxBold = -1
			}
		}
	}

	for idxBold>=0 && len(str)>idxSearch+idxBold+2 {
		if str[idxSearch+idxBold+2]=='*' {
			myPrint("replaceBold skip over *** in (%s) %d %d\n",str,idxSearch,idxBold)
			idxSearch = idxSearch + idxBold + 2
			idxBold = strings.Index(str[idxSearch:], "**")
			continue			
		}

		if idxSearch+idxBold>0 && str[idxSearch+idxBold-1]=='\\' {
			idxSearch = idxSearch + idxBold + 2
			idxBold = strings.Index(str[idxSearch:], "**")
			continue			
		}

		if idxSearch+idxBold < len(str) {
			startSearch := idxSearch + idxBold
			if !boldOnStart {
				startSearch++
			}
			if boldOnStart {
				str = "<b>" + str
				boldOnStart = false
			} else {
				str = str[:idxSearch] + strings.Replace(str[idxSearch:],"**","<b>",1)
			}

			mystr := str[startSearch:]
			idxBold2 := strings.Index(mystr, "**")
			if idxBold2>=0 {
				if idxSearch+idxBold2>0 && str[idxSearch+idxBold2-1]=='\\' {
					idxBold2 = -1
				} else if len(mystr) > idxBold2+2 {

				}
			}
			if idxBold2<0 {
				str = str + "</b>"
				boldReplace++
				break
			}

			str = str[:idxSearch] + strings.Replace(str[idxSearch:],"**","</b>",1)

			boldReplace++
			idxSearch = idxSearch + idxBold + 2
			idxBold = strings.Index(str[idxSearch:], "**")
		} else {
			break
		}
	}
	if boldReplace>0 {
		myPrint("replaceBold count=%d (%s)\n",boldReplace, str)
	} else if idxBold>=0 {
	} else {
	}
	return str
}

func replaceMdTable(str string) string {
	if str=="" {
		return str
	}
	if strings.HasPrefix(str,"#") {
		return str
	}
	if strings.Index(str,"|")<0 {
		return str
	}

	strRunes := []rune(str+"                                                          ")
	idxBlockStart := -1
	doneHeader := false
	replaceStr := ""
	strRunesLen := len(strRunes)
	idx := 0
	idxPipeStart := -1

	myPrint("replaceMdTable start...\n")
	idxHeadEnd := -1
	columns := 0
	row := 0
	col := 0
	var columnAligned [10]int // 0=left, 1=centered, 2=right
	for idx < strRunesLen {
		runeVal := strRunes[idx]
		if runeVal==0 {
			break
		}
		if runeVal=='|' {
			if !doneHeader {
				idxPipeStart = idx
				structErr := 0
				for strRunes[idx+1]==' ' && strRunes[idx+2]=='|' {
					columns++
					idx += 2
				}
				if columns<2 {
					structErr = 5000
					break
				}
				columns--
				if strRunes[idx+1]=='-' || (strRunes[idx+1]==':' && strRunes[idx+2]=='-') {
					// continue below
				} else {
					structErr = 1000
					break
				}

				for col:=0; col<columns; col++ {
					leftAligned := false
					rightAligned := false
					if strRunes[idx+1]=='-' {
						idx++
					} else if strRunes[idx+1]==':' && strRunes[idx+2]=='-' {
						leftAligned = true
						idx+=2
					} else {
						if col < columns-1 {
							structErr = 2000 + col
							break
						}
					}
					if col < columns {
						for strRunes[idx+1]=='-' {
							idx++
						}
						if strRunes[idx+1]==':' {
							rightAligned = true
							idx++
						}
						if strRunes[idx+1]!='|' {
							structErr = 3000 + int(strRunes[idx+1])
							break
						}
					} else {
						if strRunes[idx+1]!=' ' {
							structErr = 4000 + int(strRunes[idx+1])
							break
						}
					}

					if leftAligned && rightAligned {
						columnAligned[col] = 1
					} else if rightAligned {
						columnAligned[col] = 2
					} else {
						columnAligned[col] = 0
					}
					idx++
					// strRunes[idx] = '|'
				}
				if structErr>0 {
					myPrint("replaceMdTable structErr=%d columns=%d\n",structErr,columns)
					break
				}
				myPrint("replaceMdTable found columns=%d\n",columns)
				for col:=0; col<columns; col++ {
					myPrint("replaceMdTable col=%d align=%d\n",col,columnAligned[col])
				}
				replaceStr = "<table style='width:100%;margin-top:14px;margin-bottom:16px;font-size:0.9em;border-spacing: 0 0.4em;'>"
				doneHeader = true
				idxBlockStart = idxPipeStart
				idxHeadEnd = idx
				idxPipeStart = -1
			} else {
				if idxPipeStart<0 {
					idxPipeStart = idx
					replaceStr += "<tr>"
				} else {
					cellContent := strings.TrimSpace(string(strRunes[idxPipeStart+1:idx]))
					if row==0 {
						if columnAligned[col]==1 {
							replaceStr += "<td style='min-width:12vw;text-align:center;'>"+ cellContent + "</td>"
						} else if columnAligned[col]==2 {
							replaceStr += "<td style='min-width:12vw;text-align:right;'>"+ cellContent + "</td>"
						} else {
							replaceStr += "<td style='min-width:12vw'>"+ cellContent + "</td>"
						}
					} else {
						if columnAligned[col]==1 {
							replaceStr += "<td style='text-align:center;'>"+ cellContent + "</td>"
						} else if columnAligned[col]==2 {
							replaceStr += "<td style='text-align:right;'>"+ cellContent + "</td>"
						} else {
							replaceStr += "<td>" + cellContent + "</td>"
						}
					}
					idxPipeStart = idx
					if col==columns-1 {
						replaceStr += "</tr>"
						col = 0
						row++
						if row >= 10 {
							break
						}
						idxPipeStart = -1
					} else {
						col++
					}
				}
			}
		}
		idx++
	}

	if col!=0 || row>=10 {
		myPrint("replaceMdTable interrupt doneHeader=%v col=%d row=%d\n",doneHeader,col,row)
		if doneHeader {
			str = string(strRunes[:idxBlockStart]) + string(strRunes[idxHeadEnd+1:])
		}
		str = strings.Replace(str,"|","",-1)
	} else if idxBlockStart>-1 && replaceStr!="" {
		replaceStr += "</tr></table>"
		myPrint("replaceMdTable done idxBlockStart=%d row=%d replaceStr=(%s)\n",idxBlockStart,row,replaceStr)
		str = string(strRunes[:idxBlockStart]) + "<div style='overflow-x:auto;'>" + replaceStr + "</div>"
		if idx+1 < strRunesLen {
			str += string(strRunes[idx+1:])
		}
		str = strings.TrimSpace(str)
		myPrint("replaceMdTable done str=(%s)\n",str)
	} else {
		myPrint("replaceMdTable done\n")
		str = strings.Replace(str,"|","",-1)
	}
	return str
}

var epubReader *epub.ReadCloser = nil
var epubReaderFiles map[string]*zip.File = nil
var	startTimeLoad time.Time
var	startTimeReadLoop time.Time
func loadBook(filename string, loop int, byteArray []uint8, setDataLen int, copyCount int, multiPageMode int, setSubbookTitle string, loadRemote bool) int {
	subBookTitle = setSubbookTitle
	if loop==0 {
		dataLen = setDataLen
		myPrint("LoadBook filename=%s dataLen=%d multiPageMode=%d subBookTitle=%s copyCount=%d loadRemote=%v\n", filename, dataLen, multiPageMode, subBookTitle, copyCount, loadRemote)
		startTimeLoad = time.Now()
		startTimeReadLoop = time.Now()
		totalBlocks = 0
		blockArray = make([]BlockData, maxBlocks)
		linkIdToIdxMap = make(map[string]int)
		linkIdToTextMap = make(map[string]string)
		subBookFolder = ""
		loadedMd = 0

		if epubReader!=nil {
			epubReader.Close()
			epubReader = nil
			epubReaderFiles = nil
		}

		myPrint("LoadBook create big %d bytes in %d buffers\n",bufSize,bufArrayCount)
		if(byteArray==nil) {
			bufArrayPos := 0
			bufArrayCounter := 0
			byteArray = make([]uint8, bufSize)
			if(byteArray==nil) {
				myPrint("# LoadBook: out of memory %d\n",bufSize)
				for bufArrayCounter < bufArrayCount {
					bufArray[bufArrayCounter] = nil
				}
				bufArrayCount = 0
				return -4
			}
			myPrint("LoadBook: not out of memory %d\n",bufSize)
			for bufArrayCounter < bufArrayCount {
				myPrint("LoadBook loop=%d len=%d bufArrayPos=%d\n", bufArrayCounter, len(bufArray[bufArrayCounter]), bufArrayPos)
				copy(byteArray[bufArrayPos:],bufArray[bufArrayCounter]) // dst,src
				bufArrayPos += len(bufArray[bufArrayCounter])
				bufArray[bufArrayCounter] = nil
				bufArrayCounter++
			}
			myPrint("LoadBook done size=%d\n",bufArrayPos)
			dataLen = bufArrayPos
		}

		if strings.HasSuffix(filename,".md") {
			str := string(byteArray)
			textLen = len(str)
			runeText = make([]rune, textLen+1024)
			copy(runeText,[]rune(str))
			myPrint("LoadBook md %s textLen=%d len(runeText)=%d\n",filename,textLen,len(runeText))
			byteArray = nil
			loadedMd = 3
			return 2
		}

		var err error
		epubReader, err = epub.OpenReaderFromByteArray(byteArray)
		byteArray = nil
		if err != nil {
			myPrint("# err LoadBook: %s\n", err)
			epubReader.Close()
			epubReader = nil
			epubReaderFiles = nil
			return -3
		}

		book = epubReader.Rootfiles[0]
		myPrint("LoadBook book.Title %v\n",book.Title)
		myPrint("LoadBook book.Language %v\n",book.Language)
		myPrint("LoadBook book.FullPath %v\n",book.FullPath)

		if book.Language=="fr" || book.Language=="uk" || book.Language=="ukr" {
			guillemetOpen = '«'
			guillemetClose = '»'
		} else {
			guillemetOpen = '»'
			guillemetClose = '«'
		}

		indexMd := epubReader.Files["index.md"]
		if(indexMd!=nil) {
			loadedMd = 1
		} else {
			indexMdJs := js.Global().Call("loadCachedFile",-1)
			indexMdStr := indexMdJs.String()
			if indexMdStr != "" {
				indexMdLen := len(indexMdStr)
				myPrint("LoadBook indexMd from cache str-len=%d subBookTitle=%s\n",indexMdLen,subBookTitle)
				if indexMdLen>0 {
					if indexMdLen<100 {
						myPrint("LoadBook indexMd from cache str-len=%d indexMdStr=%s\n",indexMdLen,indexMdStr)
					}
					newRune := []rune(indexMdStr)
					textLen = len(newRune)
					runeText = make([]rune, textLen+1024)
					copy(runeText,[]rune(indexMdStr))
					myPrint("LoadBook indexMd from cache textLen=%d len(runeText)=%d\n",textLen,len(runeText))
					loadedMd = 2
					return 2
				}
			}
		}

		if(indexMd!=nil) {
			bookHtmlSb.Reset()
			xhtmlFileMap = make(map[string]int)
			ChapterSlice = make([]ChapterType,0)

			f, err := indexMd.Open()
			if err!=nil {
				myPrint("# LoadBook indexMd.Open err %v\n",err)
				return -1
			}
			myPrint("LoadBook indexMd io.Copy...\n")
			var b bytes.Buffer
			count, err := io.Copy(&b, f)
			f.Close()
			if err != nil {
				myPrint("# LoadBook indexMd io.Copy err %v\n",err)
				textLen = 0
				return -2
			}
			myPrint("LoadBook indexMd count=%d b.Len=%d b.Cap=%d\n",count,b.Len(),b.Cap())
			fromRunes := bytes.Runes(b.Bytes())
			textLen = len(fromRunes)
			runeText = make([]rune, textLen+1024)
			myPrint("LoadBook indexMd textLen=%d len(runeText)=%d\n",textLen,len(runeText))
			copy(runeText,fromRunes)
			return 2
		}

		coverImage = ""
		for _, item := range book.Manifest.Items {
			if item.MediaType == "image/jpeg" || item.MediaType == "image/png" {
				if coverImage=="" {
					coverImage = item.HREF
				}
				if strings.Index(item.HREF,"cover")>=0 {
					coverImage = item.HREF
				}
				if strings.Index(item.HREF,"img1")>=0 {
					coverImage = item.HREF
				}
			}
		}
		if coverImage!="" {
			myPrint("loadBook coverImage=%s\n",coverImage)
		}


		if multiPageMode==1 {
			myPrint("LoadBook multiPageMode==1 -> getReaderForRecourceFile('toc.ncx')\n")
			var subBookNames map[string]bool = make(map[string]bool)
			tocNcxReader,err := getReaderForRecourceFile("toc.ncx")
			if err != nil {
				myPrint("# LoadBook open err=%s\n", err)
			} else if tocNcxReader==nil {
				myPrint("# LoadBook open, no tocNcxReader\n")
			} else {
				myPrint("LoadBook multiPageMode==1 -> xmlpath.Parse()\n")
				tocNcxRoot, err = xmlpath.Parse(tocNcxReader)
				if err != nil {
					myPrint("# LoadBook xmlpath.Parse err=%s\n", err)
				} else if tocNcxRoot == nil {
					myPrint("# LoadBook tocNcxRoo==nil\n")
				} else {
					xpathSelector =
						xmlpath.MustCompile("/ncx/navMap/navPoint/navPoint/content[contains(@src, 'html')]/@src")
					tocNcxIter := xpathSelector.Iter(tocNcxRoot)
					for tocNcxIter.Next() {
						raw := tocNcxIter.Node()
						srcPath := raw.String()
						idxPound := strings.Index(srcPath,"#")
						if idxPound>=0 {
							srcPath = srcPath[:idxPound]
						}
						idxSlash := strings.Index(srcPath,"/")
						if idxSlash>=0 {
							srcPath = srcPath[:idxSlash]
						}
						subBookNames[srcPath]=true
					}
				}
			}

			if len(subBookNames)<=1 {
				multiPageMode = 0
				js.Global().Call("setMode",0)
			}

			myPrint("LoadBook subBookNames len=%d multiPageMode=%d\n", len(subBookNames), multiPageMode)

			booksTitle = nil
			booksStartUrl = nil
			if multiPageMode>0 {
				xpathSelector = xmlpath.MustCompile("/ncx/navMap/navPoint/content[contains(@src, 'html')]/@src")

				tocNcxIter := xpathSelector.Iter(tocNcxRoot)
				for tocNcxIter.Next() {
					raw := tocNcxIter.Node()
					srcPath := raw.String()
					idxRemark := strings.Index(srcPath,"#")
					if idxRemark>=0 {
						srcPath = srcPath[:idxRemark]
					}
					booksStartUrl = append(booksStartUrl,srcPath)
				}
				myPrint("LoadBook multi booksStartUrl %d (%v)\n", len(booksStartUrl), booksStartUrl)

				xpathSelector = xmlpath.MustCompile("/ncx/navMap/navPoint/navLabel/text")
				tocNcxIter = xpathSelector.Iter(tocNcxRoot)
				i := 0
				selectedIdx := -1
				selectedNextUrl := ""
				selectedTitle := ""
				for tocNcxIter.Next() {
					raw := tocNcxIter.Node()
					title := raw.String()
					booksTitle = append(booksTitle,title)

					if subBookTitle!="" && subBookTitle==title {
						nextUrl := ""
						j:=0
						for i+j < len(booksStartUrl)-1 {
							nextUrl = booksStartUrl[i+j+1]
							if nextUrl != booksStartUrl[i] {
								break
							}
							j++
						}

						myPrint("LoadBook subbookSelected startUrl=%s nextUrl=%s title=%s i=%d loadRemote=%v\n",
							booksStartUrl[i], nextUrl, title, i, loadRemote)

						selectedIdx = i
						selectedNextUrl = nextUrl
						selectedTitle = title
					}
					i++
				}
				myPrint("LoadBook multi booksTitle %d (%v)\n", len(booksTitle), booksTitle)
				mainIndexMarkup := getBookIndex(loadRemote)

				if selectedIdx>-1 {
					js.Global().Call("subbookSelected", booksStartUrl[selectedIdx],
						selectedNextUrl, selectedTitle, selectedIdx, loadRemote)
					return 9
				}
				js.Global().Call("showSubbooks",mainIndexMarkup)
				return 0
			}
		}
	}

	myPrint("LoadBook return 1\n")
	return 1
}

func getBookIndex(loadRemote bool) string {
	mainIndexMarkup := ""
	if book!=nil {
		mainIndexMarkup = book.Creator + ": "+book.Title+"<br>"
	}

	lastBookUrl := ""
	for subBookIdx,title := range booksTitle {
		nextUrl := ""
		if booksStartUrl[subBookIdx] != lastBookUrl {
			i:=0
			for subBookIdx + i < len(booksTitle)-1 {
				nextUrl = booksStartUrl[subBookIdx+i+1]
				if nextUrl != booksStartUrl[subBookIdx] {
					break
				}
				i++
				nextUrl = ""
			}

			var encodedTitle = url.QueryEscape(title)
			encodedTitle = strings.Replace(encodedTitle,"/'/g","%27",-1)

			var subbookSelectedStr = "subbookSelected(\""+booksStartUrl[subBookIdx]+"\",\""+
				nextUrl+"\",\""+encodedTitle+"\","+strconv.Itoa(subBookIdx)+","+strconv.FormatBool(loadRemote)+");"
			mainIndexMarkup +=
				"<a onClick='"+subbookSelectedStr+" event.stopPropagation();'>"+title

			if subBookTitle!="" && title==subBookTitle {
				mainIndexMarkup += "&nbsp; 👀"
			}

			mainIndexMarkup += "</a><br>"
			lastBookUrl = booksStartUrl[subBookIdx]
		} else {
			mainIndexMarkup += "<div>"+title+"</div>"
		}
	}
	mainIndexMarkup += "<br>&nbsp;"
	return mainIndexMarkup;
}

var	foundNextXhtml = false
var	processedStartXhtml = false
var	bookHtmlSb strings.Builder
var	timeoutChap = ""

func loadSubBook(loop int, startXhtml string, nextXhtml string, subBookIdx int, setSubbookTitle string) int {
	if loop==0 {
		startTimeReadLoop = time.Now()

		subBookFolder = ""
		slashIdx := strings.Index(startXhtml,"/")
		if slashIdx>=0 {
			subBookFolder = startXhtml[:slashIdx]
		}

		if setSubbookTitle!="" {
			subBookTitle = setSubbookTitle
		}
		myPrint("loadSubBook loop==0 startXhtml=%s nextXhtml=%s subBookFolder=%s subBookTitle=%s subBookIdx=%d\n", startXhtml, nextXhtml, subBookFolder, subBookTitle, subBookIdx)

		loadedMd = 0
		if subBookIdx < 0 {
			myPrint("! loadSubBook try load embedded md but subBookIdx=%d <0\n",subBookIdx)
		} else if epubReaderFiles==nil {
			myPrint("! loadSubBook try load embedded md but epubReaderFiles==nil\n")
		} else {
			indexMdInSubfolder := "index.md"
			if subBookFolder!="" {
				indexMdInSubfolder = subBookFolder + "/" + indexMdInSubfolder
			}
			myPrint("loadSubBook try load embedded md indexMdInSubfolder1=(%s)\n",indexMdInSubfolder)
			indexMd := epubReaderFiles[indexMdInSubfolder]
			if(indexMd==nil) {
				fullPathOpf := book.FullPath
				myIdx := strings.Index(fullPathOpf,"/")
				if myIdx>0 {
					fullPath := fullPathOpf[:myIdx]
					indexMdInSubfolder = fullPath+"/index.md"
					myPrint("loadSubBook try load embedded md fullPath=(%s) subBookFolder=(%s)\n",fullPath,subBookFolder)
					if subBookFolder!="" {
						indexMdInSubfolder = fullPath + "/" + subBookFolder + "/index.md"
					}
					myPrint("loadSubBook try load embedded md indexMdInSubfolder2=(%s)\n",indexMdInSubfolder)
					indexMd = epubReaderFiles[indexMdInSubfolder]
				}
			}
			if(indexMd==nil) {
				myPrint("! loadSubBook try load embedded md indexMdInSubfolder fail (%s)\n",indexMdInSubfolder)
			} else {
				myPrint("loadSubBook try load embedded md indexMdInSubfolder open (%s)\n",indexMdInSubfolder)
				bookHtmlSb.Reset()
				xhtmlFileMap = make(map[string]int)
				ChapterSlice = make([]ChapterType,0)

				f, err := indexMd.Open()
				if err!=nil {
					myPrint("# loadSubBook try load embedded md indexMd.Open err %v\n",err)
					return -1
				}
				myPrint("loadSubBook indexMd io.Copy...\n")
				var b bytes.Buffer
				count, err := io.Copy(&b, f)
				f.Close()
				if err != nil {
					myPrint("# loadSubBook indexMd io.Copy err %v\n",err)
					textLen = 0
					return -2
				}
				myPrint("loadSubBook indexMd count=%d b.Len=%d b.Cap=%d\n",count,b.Len(),b.Cap())
				fromRunes := bytes.Runes(b.Bytes())
				textLen = len(fromRunes)
				runeText = make([]rune, textLen+1024)
				myPrint("loadSubBook indexMd textLen=%d len(runeText)=%d\n",textLen,len(runeText))
				copy(runeText,fromRunes)
				loadedMd = 1
				return 101
			}
		}

		indexMdJs := js.Global().Call("loadCachedFile",subBookIdx)
		indexMdStr := indexMdJs.String()
		if indexMdStr == "" {
			//myPrint("! loadSubBook try load cachedMd but indexMdStr is empty\n")
		} else {
			indexMdLen := len(indexMdStr)
			myPrint("loadSubBook indexMd from cache str-len=%d subBookIdx=%d\n",indexMdLen,subBookIdx)
			if indexMdLen>0 {
				newRune := []rune(indexMdStr)
				textLen = len(newRune)
				runeText = make([]rune, textLen+1024)
				copy(runeText,[]rune(indexMdStr))
				myPrint("loadSubBook indexMd from cache textLen=%d len(runeText)=%d\n",textLen,len(runeText))
				loadedMd = 2
				return 101
			}
		}

		runeTextSizeMax = dataLen*3 // 3 x the size of the zipped epub archive
		if runeTextSizeMax > runeTextMax {
			runeTextSizeMax = runeTextMax
		}
		myPrint("loadSubBook runeText dataLen=%d runes=%d\n",dataLen,runeTextSizeMax)
		runeText = make([]rune, runeTextSizeMax+1024)

		textLen = 0
		foundNextXhtml = false
		processedStartXhtml = false
		bookHtmlSb.Reset()
		xhtmlFileMap = make(map[string]int)
		ChapterSlice = make([]ChapterType,0)
		stateItalic = false
		stateBold = false

		if conv==nil {
			conv = converter.NewConverter(
				converter.WithPlugins(
					base.NewBasePlugin(),
					commonmark.NewCommonmarkPlugin(),
					table.NewTablePlugin(
					),
				),
			)
			conv.Register.Renderer(func(ctx converter.Context, w converter.Writer, node *html.Node) converter.RenderStatus {
				attrId := dom.GetAttributeOr(node, "id", "")
				if strings.HasPrefix(attrId,"pgepub") {
					// do nothing
				} else if strings.HasPrefix(attrId,"pg-") {
					// do nothing
				} else if strings.Index(attrId,"#")>=0 {
					// do nothing
				} else if attrId!="" {
					w.WriteString("[](#"+attrId+")")
				}
				return converter.RenderTryNext
			}, converter.PriorityEarly)
		}
	}

	if loop < len(book.Spine.Itemrefs) {
		itemIdx := loop
		item := *book.Spine.Itemrefs[itemIdx].Item

		return loadSubBookChapter(itemIdx, item, startXhtml, nextXhtml)
	}

	bookHtmlFullStr := bookHtmlSb.String()
	bookHtmlSb.Reset()

	if strings.Index(bookHtmlFullStr,"dir=\"rtl\"")>=0 ||
	   strings.Index(bookHtmlFullStr,"direction:rtl")>=0 ||
	   strings.Index(bookHtmlFullStr,"direction: rtl")>=0 {
		rtlFlag = true
	} else {
		rtlFlag = false
	}

	timeSinceLoad := time.Since(startTimeLoad)
	timeSinceReadLoop := time.Since(startTimeReadLoop)
	myPrint("loadSubBook done bookHtmlFullStrLen=%d textLen=%d time=[%v %v %v] %s\n",
		len(bookHtmlFullStr), textLen, timeSinceLoad-timeSinceReadLoop, timeSinceReadLoop, timeSinceLoad, timeoutChap)

	if textLen >= runeTextSizeMax {
		myPrint("loadSubBook not possible to reduce runeText, textLen=%d >= runeTextSizeMax=%d\n",textLen,runeTextSizeMax)
	} else {
		myPrint("loadSubBook reduce runeText from=%d to=%d\n",len(runeText),textLen+1024)
		runeText = append([]rune(nil), runeText[:textLen+1024]...)
	}
	return 100
}

func loadSubBookChapter(itemIdx int, item epub.Item, startXhtml string, nextXhtml string) int {
	bookTextStr := ""

	if item.MediaType == "image/jpeg" {
		if itemIdx==0 {
			coverImage = item.HREF
			myPrint("loadSubBookChap %d item.MediaType=%s as coverImage=(%s)\n",itemIdx,item.MediaType,coverImage)
		} else {
			myPrint("! loadSubBookChap %d item.MediaType=%s NOT as coverImage(%s)\n",itemIdx,item.MediaType,coverImage)
		}
	} else if item.MediaType == "application/xhtml+xml" {
		if !foundNextXhtml && processedStartXhtml && nextXhtml!="" && strings.HasPrefix(item.HREF,nextXhtml) {
			foundNextXhtml = true
		}
		endCriteria := false
		if foundNextXhtml && processedStartXhtml {
			endCriteria = true
			myPrint("loadSubBookChap endCriteria 1\n")
		}
		if endCriteria {
			myPrint("loadSubBookChap end Items loop on(%s) nextXhtml=(%s)\n",item.HREF,nextXhtml)
			bookHtmlFullStr := bookHtmlSb.String()
			myPrint("loadSubBookChap end Items set bookHtmlFullStrLen=%d\n",len(bookHtmlFullStr))
			bookHtmlSb.Reset()

			if strings.Index(bookHtmlFullStr," dir=\"rtl\"")>=0 {
				rtlFlag = true
			} else {
				rtlFlag = false
			}

			myPrint("loadSubBookChap reduce runeText from=%d to=%d\n",len(runeText),textLen)
			runeText = append([]rune(nil), runeText[:textLen+1024]...)
			return 100
		}

		if !processedStartXhtml {
			if startXhtml!="" && !strings.HasPrefix(item.HREF,startXhtml) {
				return 0
			}

			processedStartXhtml = true
		}

		r, err := item.Open()
		if err != nil {
			myPrint("# loadSubBookChap err create: %s\n", err)
			return 0
		}
		cont, err := io.ReadAll(r)
		if err != nil {
			myPrint("# loadSubBookChap err read: %s\n", err)
			return -10
		}

		bookHtmlStr := prepareHtml(string(cont))
		_,err = bookHtmlSb.WriteString(bookHtmlStr)
		bookHtmlSb.Reset()
		if err!=nil {
			myPrint("# loadSubBookChap bookHtmlSb err=%v\n",err)
			return -30
		}

		bookTextStr, err = conv.ConvertString(bookHtmlStr)

		if err != nil {
			myPrint("# err loadSubBookChap convert: %s\n", err)
			return -11
		}

		if bookTextStr=="" {
			myPrint("! loadSubBookChap bookTextStr empty\n")
		} else {
			bookTextStr = prepareText(bookTextStr)

			xhtmlFileMap[item.HREF] = textLen

			contentAsRunes := []rune(bookTextStr)
			loadCount := 0
			overload := 0
			for i, val := range contentAsRunes {
				if textLen+i < runeTextSizeMax {
					runeText[textLen+i] = val
					loadCount++
				} else {
					overload++
				}
			}
			if overload>0 {
				myPrint("! loadSubBookChap overload=%d textLen=%d+%d >= %d\n",overload,textLen,loadCount,runeTextSizeMax)
			}
			textLen += loadCount
			bookTextStr = "" // free mem
		}
	}
	return 0
}

func prepareHtml(htmlStr string) string {
	idxTitle := strings.Index(htmlStr,"<title>")
	if idxTitle>=0 {
		idxTitleClose := strings.Index(htmlStr[idxTitle:],"</title>")
		if idxTitleClose>=0 {
			htmlStr = htmlStr[:idxTitle] + htmlStr[idxTitle+idxTitleClose+8:]
		}
	}

	htmlStr = strings.Replace(htmlStr,"<pre/>","",-1)

	htmlStr = strings.Replace(htmlStr,"! <a ","!<br><a ",-1)
	htmlStr = strings.Replace(htmlStr,"!<a ","!<br><a ",-1)

	countTitleAttr := 0
	idxAll1 :=0
	idxSpanHtmlonly := strings.Index(htmlStr,"<span class=\"htmlonly\"")
	for idxSpanHtmlonly>=0 {
		idxAll1 += idxSpanHtmlonly
		idxCloseSpan := strings.Index(htmlStr[idxAll1:],"</span>")
		if idxCloseSpan<0 {
			break
		}
		htmlStr = htmlStr[:idxAll1] + htmlStr[idxAll1+idxCloseSpan+7:]
		countTitleAttr++
		idxSpanHtmlonly = strings.Index(htmlStr[idxAll1:],"<span class=\"htmlonly\"")
	}
	if countTitleAttr>0 {
		myPrint("prepareHtml span-htmlonly count=%d\n",countTitleAttr)
	}

	countClosedAnchors := 0
	idxAll := 0
	idxAnchor := strings.Index(htmlStr,"<a ")
	for idxAnchor>=0 {
		idxAll = idxAll + idxAnchor
		idxCloseAnchor := strings.Index(htmlStr[idxAll+3:],">")
		if idxCloseAnchor<0 {
			break
		}

		if strings.HasPrefix(htmlStr[idxAll:],"<a class=\"x-ebookmaker-pageno\" ") {
			if strings.HasPrefix(htmlStr[idxAll+3+idxCloseAnchor+1:],"</a>") {
				idxCloseAnchor += 4
			}
			if htmlStr[idxAll+3+idxCloseAnchor+1] == LFCHAR {
				idxCloseAnchor++
			}
			htmlStr = htmlStr[:idxAll] + htmlStr[idxAll+3+idxCloseAnchor+1:]

		} else if htmlStr[idxAll+3+idxCloseAnchor-1] == '/' {
			htmlStr = htmlStr[:idxAll+3+idxCloseAnchor-1] + "></a>" + htmlStr[idxAll+3+idxCloseAnchor+1:]
			idxAll = idxAll + 3 + idxCloseAnchor + 3
			countClosedAnchors++

		} else {
			idxAll = idxAll + 3 + idxCloseAnchor
		}

		idxAnchor = strings.Index(htmlStr[idxAll:],"<a ")
	}

	idxBIC := strings.Index(htmlStr, " BIC media ")
	for idxBIC>=0 {
		idxPStart := -1
		for i:=0; idxBIC+i >= 0; i-- {
			if htmlStr[idxBIC+i]=='<' && htmlStr[idxBIC+i+1]=='p' {
				idxPStart = idxBIC+i
				break
			}
		}
		if idxPStart>=0 {
			idxPEnd := strings.Index(htmlStr[idxBIC:],"</p>")
			if idxPEnd<0 {
				break
			}
			htmlStr = htmlStr[:idxPStart] + htmlStr[idxBIC+idxPEnd+4:]
		}
		idxBIC = strings.Index(htmlStr, " BIC media ")
	}
	return htmlStr
}

func prepareText(textStr string) string {
	if textStr=="" {
		myPrint("! prepareText textStr empty\n")
	} else {
		textStr = textStr+"\n\n\n\n"
		textStr = cleanupTextFromHtmlConverter(textStr)
		if textLen==0 {
			if coverImage!="" {
				ioR,err := getReaderForRecourceFile(coverImage)
				if err==nil && ioR!=nil {
					idxCover := strings.Index(textStr,"("+coverImage+")")
					if idxCover<0 {
						textStr = "![coverImage]("+coverImage+") " + textStr
						myPrint("prepareText coverImage (%s) added in front of textStr\n",coverImage)
					} else {
						myPrint("prepareText coverImage (%s) NOT added in front - already in\n",coverImage)
					}
				} else {
					myPrint("! prepareText getReader for coverImage (%s) err=%v\n",coverImage,err)
				}
			}
		}
	}
	return textStr
}

var startTimeParse time.Time
func pagination() int {
	myPrint("pagination start maxBlocks=%d textLen=%d/%d pidx=%d\n", maxBlocks,textLen,len(runeText),pidx)
	textLen := len(runeText)
	countLinkIds := 0
	countLink1Ids := 0
	countChars := 0
	for i := range runeText {
		countChars++
		if i+4 < textLen &&
		   runeText[i] == '[' &&
		   runeText[i+1] == ']' &&
		   runeText[i+2] == '(' &&
		   runeText[i+3] == '#' {
			countLink1Ids++
			i2 := i+4
			for i2<textLen && runeText[i2]!= ')' {
				i2++
			}
			if i2-i+4 < 60 {
				linkId := string(runeText[i+4:i2])
				linkIdLen := len(linkId)
				linkIdToIdxMap[linkId] = i + 4 + linkIdLen + 1
				countLinkIds++
				i += 4 + linkIdLen
			} else {
				myPrint("! pagination linkId search i2-i+4=%d >= 60\n", i2-i+4)
			}
		}
		i++
	}
	myPrint("pagination linkId search countLinkIds=%d %d countChars=%d\n",
		countLinkIds, countLink1Ids, countChars)

	ChapterSlice = make([]ChapterType,0)

	startTimeParse = time.Now()
	var pos int
	var runeVal rune
	var countImgReplaced = 0
	for pos = 0; pos < textLen; pos++ {
		runeVal = runeText[pos]
		if runeVal==0 {
			if runeText[pos+1]==0 {
				break
			}
			continue
		}
		if pos+2<textLen && runeVal=='>' && runeText[pos+1]==' ' && runeText[pos+2]=='>' {
			runeText[pos] = ' '
			runeText[pos+1] = LFRUNE
			runeVal = LFRUNE
		}

		if runeVal=='!' && runeText[pos+1]=='[' {
			imgNameIdx := -1
			for pos1 := pos+2; pos1 < pos+800; pos1++ {
				if runeText[pos1]==']' && runeText[pos1+1]=='(' {
					imgNameIdx = pos1 +2
				} else if runeText[pos1]==')' && imgNameIdx>=0 {
					imgLen := pos1 +1 -pos
					replaceStr := ""
					replaceStrLen := 0
					imgName := string(runeText[imgNameIdx:pos1])
					if imgName!="" {
						idxQuote := strings.Index(imgName,"\"")
						if idxQuote>=0 {
							imgName = strings.TrimSpace(imgName[:idxQuote])
						}
						if strings.HasPrefix(imgName,"..") {
							imgName = imgName[2:]
						}
						if strings.HasPrefix(imgName,"/") {
							imgName = imgName[1:]
						}
						imgName = strings.TrimSpace(imgName)
						if strings.Index(imgName,".")>=0 {

								replaceStr += "//!"+imgName
								for i,chr := range []rune(replaceStr) {
									runeText[pos+i] = chr
								}
								replaceStrLen = len(replaceStr)
						} else {
							myPrint("! pagination img5b imgName (%s) has no file extension\n", imgName)
						}
					}

					gapLen := imgLen - replaceStrLen
					if gapLen>0 {
						for i := replaceStrLen; i<imgLen; i++ {
							runeText[pos+i] = ' '
						}
					}
					countImgReplaced++
					pos += imgLen
					break
				}
			} //for
		}
	} //for
	myPrint("pagination countImgReplaced=%d\n",countImgReplaced)

	pidx = 0 // current index of next char in text
	totalBlocks = 0 // growing index of blockArray[] and sentenceLen[]
	for totalBlocks < maxBlocks {
		blockLen, skipEolChars, blockEndCriteria, dbgInfo1, dbgInfo2, linefeedCount := nextTextBlock()
		if blockLen+skipEolChars <= 0 && pidx > textLen-5000 {
			break
		}

		if blockLen > 0 {
			blockArray[totalBlocks] =
				BlockData{	pidx, blockLen, linefeedCount, blockEndCriteria,
							stateItalic, stateBold, false, dbgInfo1, dbgInfo2, ""}
			totalBlocks++
		} else {
			if skipEolChars<=0 {
				myPrint("# pagination blockLen<=0 (%d) and skipEolChars<=0 (%d) (pidx=%d totlen=%d)\n",
					blockLen, skipEolChars, pidx, len(runeText))
				if pidx > runeTextMax {
					break
				}
				blockLen++
			}
		}
		pidx += blockLen + skipEolChars
	}

	myPrint("pagination totalBlocks=%d chapterCount=%d textLen=%d subBookTitle=%s\n",
		totalBlocks,len(ChapterSlice),textLen,subBookTitle)

	var chapterLocalSlice []string
	xpathSelectorString := "/ncx/navMap/navPoint/navLabel/text"
	if subBookTitle!="" {
		xpathSelectorString =
			"/ncx/navMap/navPoint/navLabel/text[text() = '"+subBookTitle+"']/../../navPoint/navLabel/text"
	}
	xpathSelector := xmlpath.MustCompile(xpathSelectorString)
	if xpathSelector == nil {
		myPrint("! pagination xpathSelector==nil\n")
	} else if tocNcxRoot==nil {
		myPrint("! pagination tocNcxRoot==nil\n")
	} else {
		tocNcxIter := xpathSelector.Iter(tocNcxRoot)
		if tocNcxIter==nil {
			myPrint("! pagination tocNcxIter==nil\n")
		} else {
			for tocNcxIter.Next() {
				raw := tocNcxIter.Node()
				chapName := raw.String()
				chapterLocalSlice = append(chapterLocalSlice,chapName)
			}
		}

		xpathSelectorString = "/ncx/navMap/navPoint/content/@src"
		if subBookTitle!="" {
			xpathSelectorString =
				"/ncx/navMap/navPoint/navLabel/text[text() = '"+subBookTitle+"']/../../navPoint/content/@src"
		}
		xpathSelector = xmlpath.MustCompile(xpathSelectorString)
		if xpathSelector == nil {
			myPrint("! pagination xpathSelector==nil\n")
		} else {
			tocNcxIter := xpathSelector.Iter(tocNcxRoot)
			if tocNcxIter==nil {
				myPrint("! pagination tocNcxIter==nil\n")
			} else {
				arrayIdx := 0
				for tocNcxIter.Next() {
					raw := tocNcxIter.Node()
					path := raw.String()
					htmlId := ""

					idxPound := strings.Index(path,"#")
					if idxPound>=0 {
						htmlId = path[idxPound+1:]
						path = path[:idxPound]
					}
					if arrayIdx < len(chapterLocalSlice) {
						pageTextIdx,ok := linkIdToIdxMap[htmlId]
						if !ok {
							pageTextIdx=-1
						}
						if pageTextIdx >= 0 {
							chapName := chapterLocalSlice[arrayIdx]
							chapName = strings.Replace(chapName,"*","",-1)
							chapName = strings.Replace(chapName,"_","",-1)
							if offsetChapName(chapName)<0 {
								foundIdx := -1
								for i, chap := range ChapterSlice {
									if chap.Idx == pageTextIdx || chap.Idx == pageTextIdx+1 {
										foundIdx = i
										break
									}
								}
								if foundIdx >= 0 {
									origChap := ChapterSlice[foundIdx]
									myPrint("pagination chap replace '%s' path=(%v) idx=%d foundIdx=%d\n",
										chapName, path, origChap.Idx, foundIdx)
									ChapterSlice[foundIdx] = ChapterType{chapName,origChap.Idx,origChap.HLevel}
								} else {
									myPrint("pagination chap append '%s' path=(%v) idx=%d\n",
										chapName, path, pageTextIdx)
									ChapterSlice = append(ChapterSlice,ChapterType{chapName,pageTextIdx+1,1})
								}
							}
						} else {
						}
					}
					arrayIdx++
				}

				sort.Slice(ChapterSlice, func(i, j int) bool {
					return ChapterSlice[i].Idx < ChapterSlice[j].Idx
				})
			}
		}
	}

	timeSinceLoad := time.Since(startTimeLoad)
	timeSinceReadLoop := time.Since(startTimeReadLoop)
	timeSinceParse := time.Since(startTimeParse)
	myPrint("pagination done blocks=%d textLen=%d totalTime=[%v %v %v %v %v %v]\n", totalBlocks, textLen,
		startTimeLoad, startTimeReadLoop,
		timeSinceLoad-timeSinceReadLoop, timeSinceReadLoop-timeSinceParse, timeSinceParse, timeSinceLoad)
	return 0
}

func getReaderForRecourceFile(file string) (io.Reader,error) {
	tmp := 0
	myPrint("getReaderForRecourceFile file=(%v) %d\n", file,tmp)
	for itemIdx, item := range book.Manifest.Items {
		tmp = itemIdx
		if item.HREF == file {
			r, err := item.Open()
			return r,err
		}
	}
	return nil,nil
}

func runeSearchString(text []rune, search string) int {
	searchRunes := []rune(search)
	searchRunes0 := searchRunes[0]
	textLen := len(text)
	for i := range text {
	    if text[i] != searchRunes0 {
			continue
	    }
		found := true
		for j := range searchRunes {
			if i+j>=textLen || text[i+j] != searchRunes[j] {
				found = false
				break
			}
		}
		if found {
			return i
		}
    }
    return -1
}

func getTextForLinkId2(linkId string, startIdx int) string {
	search := "[](#"+linkId+")"
	text,ok := linkIdToTextMap[linkId]
	if !ok {
		text = ""
		idxMdID,ok := linkIdToIdxMap[linkId]
		if !ok {
			idxMdID = -1
		}
		if idxMdID < 0 {
			idxMdID = runeSearchString(runeText[startIdx:], search)
			if(idxMdID >= 0) {
				idxMdID = startIdx+idxMdID
				myPrint("getTextForLinkId2 linkIdToIdxMap[linkId=%s] <-- idxMdID=%d\n", linkId, idxMdID)
				linkIdToIdxMap[linkId] = idxMdID
			}
		}
		if(idxMdID >= 0) {
			endIdx := idxMdID + 6000
			if endIdx < textLen {
				text = string(runeText[idxMdID:endIdx])
			} else {
				text = string(runeText[idxMdID:])
			}
		}
		if text!="" {
			text = strings.Replace(text,"[]()","",-1)
			text = strings.Replace(text,search,"",1)
		}
		if text!="" {
			idx := strings.Index(text,"\n#")
			if idx>=0 {
				text = text[0:idx]
			}
		}
		if text!="" {
			linkIdToTextMap[linkId] = text
		}
	}
	return text
}

func mdLinkToLabel(str string, minlen int, doReloops int) string {
	if str!="" {
reloop:
		strRunes := []rune(str)
		squareBracketOpenCount := 0
		roundBracketOpenCount := 0
		labelIdx := -1
		labelLen := -1
		linkIdx := -1
		linkLen := -1
		for idx := range strRunes {
			runeVal := strRunes[idx]
			if runeVal==0 {
				break
			}

			if runeVal=='[' {
				if squareBracketOpenCount==0 {
					labelIdx = idx+1
				}
				squareBracketOpenCount++
			} else if runeVal==']' {
				squareBracketOpenCount--
				if idx+1<len(strRunes) && strRunes[idx+1]=='(' {
					if roundBracketOpenCount==0 {
						labelLen = idx-1-labelIdx
						linkIdx = idx+2
					}
					roundBracketOpenCount++
				}
			}
			if roundBracketOpenCount>0 {
				if runeVal==')' {
					if roundBracketOpenCount>0 {
						linkLen = idx-1-linkIdx
						roundBracketOpenCount--
					}
				}
			}
			if squareBracketOpenCount==0 && roundBracketOpenCount==0 {
				if linkLen>=0 {
					if labelLen>=minlen {
						if doReloops>0 {
							str =	string(strRunes[:labelIdx-1]) +
									string(strRunes[labelIdx:labelIdx+labelLen+1]) +
									string(strRunes[linkIdx+1+linkLen+1:])
							goto reloop
						}
						str =	string(strRunes[:labelIdx-1]) +
								string(strRunes[labelIdx:labelIdx+labelLen+1])
						break
					} else {
						str =	string(strRunes[:labelIdx-1]) +
								string(strRunes[linkIdx+1+linkLen+1:])
						goto reloop
					}
				}
			}
		}
	}
	return str
}

// JS interfacess /////////////////////////////////////////////

func GetCodeTag(this js.Value, args []js.Value) interface{} {
	myPrint("GetCodeTag version codetag=(%v)\n", codetag)
	return codetag
}

func TotalSize(this js.Value, args []js.Value) interface{} {
	return textLen
}

func TotalBlocks(this js.Value, args []js.Value) interface{} {
	return totalBlocks
}

func BlockForIdx(this js.Value, args []js.Value) interface{} {
	var idx int = args[0].Int()
	val := blockForIdx(idx)
	return val
}

func IdxText(this js.Value, args []js.Value) interface{} {
	if len(args)<1 {
		return -1
	}
	searchStr := args[0].String()
	startIdx := 0
	if len(args)>1 {
		startIdx = args[1].Int()
	}
	foundIdx := idxText(runeText[startIdx:],searchStr,len(searchStr))
	if foundIdx >= 0 {
		return startIdx + foundIdx
	}
	return -1
}

func GetChapStartIdxForIdx(this js.Value, args []js.Value) interface{} {
	var idx int = args[0].Int()
	var chapIdx = 0
	for _, chap := range ChapterSlice {
		if idx >= chap.Idx {
			chapIdx = chap.Idx
		}
	}
	return chapIdx
}

func GetChapLenForIdx(this js.Value, args []js.Value) interface{} {
	var idx int = args[0].Int()
	chapLen := 0
	for i, chap := range ChapterSlice {
		if idx >= chap.Idx {
			if i+1 < len(ChapterSlice) {
				chapLen = ChapterSlice[i+1].Idx - chap.Idx
			} else {
				chapLen = textLen - chap.Idx
			}
		}
	}
	return chapLen
}

func GetChapterName(this js.Value, args []js.Value) interface{} {
	if len(args)<1 {
		myPrint("! GetChapterName noarg\n")
		return nil
	}

	var i int = args[0].Int()
	if i>=0 && i<len(ChapterSlice) {
		chap := ChapterSlice[i]
		return chap.Title
	}
	return nil
}

func GetChapNameForIdx(this js.Value, args []js.Value) interface{} {
	if len(args)<1 {
		return ""
	}
	var idx int = args[0].Int()
	var important = false
	if len(args)>=2 {
		important = args[1].Bool()
	}
	chapName := ""
	for i, chap := range ChapterSlice {
		nextChapIdx := 9999999
		if i+1<len(ChapterSlice) && idx < ChapterSlice[i+1].Idx {
			nextChapIdx = ChapterSlice[i+1].Idx
		}
		if idx >= chap.Idx && idx < nextChapIdx {
			chapName = chap.Title
		}
if(important) {
		myPrint("GetChapNameForIdx i=%d idx=%d chap.Idx=%d nextChapIdx=%d '%s'\n",i,idx,chap.Idx,nextChapIdx,chapName)
}
	}
	return chapName
}

func GetChapterIdx(this js.Value, args []js.Value) interface{} {
	if len(args)<1 {
		return -1
	}
	var i int = args[0].Int()
	if i>=0 && i<len(ChapterSlice) {
		chap := ChapterSlice[i]
		return chap.Idx
	}
	return -1
}

func GetChapterLevel(this js.Value, args []js.Value) interface{} {
	if len(args)<1 {
		return -1
	}
	var i int = args[0].Int()
	if i>=0 && i<len(ChapterSlice) {
		chap := ChapterSlice[i]
		return chap.HLevel
	}
	return -1
}

func GetChapterOffset(this js.Value, args []js.Value) interface{} {
	if len(args)<1 {
		return -1
	}
	chapName := args[0].String()
	return offsetChapName(chapName)
}

func IdxForBlockNumber(this js.Value, args []js.Value) interface{} {
	blockNumber := args[0].Int()
	idx := -1
	blockEndCriteria := -1
	italicOnEnd := false
	blockLen := 0
	blockEndInfo1 := -1
	blockEndInfo2 := -1
	if blockNumber >= 0 && blockNumber < totalBlocks {
		idx = blockArray[blockNumber].Idx
		blockEndCriteria = blockArray[blockNumber].BlockEndCriteria
		italicOnEnd = blockArray[blockNumber].italicOnEnd
		blockLen = blockArray[blockNumber].CharLength
		blockEndInfo1 = blockArray[blockNumber].dbgInfo1
		blockEndInfo2 = blockArray[blockNumber].dbgInfo2
	}
	myPrint("IdxForBlockNumber block=%d idx=%d blockEndCriteria=%d len=%d italicOnEnd=%v (%d %d)\n",
		blockNumber, idx, blockEndCriteria, blockLen, italicOnEnd, blockEndInfo1, blockEndInfo2)
	return idx
}

func TextForBlocknumber(this js.Value, args []js.Value) interface{} {
	if len(args)>=1 {
		var blockNumber int = args[0].Int()
		fastmode := false
		if len(args)>=2 {
			fastmode = args[1].Bool()
		}
		return textForBlocknumber(blockNumber,fastmode)
	}
	return ""
}

func TextForIdx(this js.Value, args []js.Value) interface{} {
	if len(args)>=1 {
		idx := args[0].Int()
		leng := 0
		if len(args)>=2 {
			leng = args[1].Int()
		}
		fastmode := false
		if len(args)>=3 {
			fastmode = args[2].Bool()
		}
		myPrint("TextForIdx idx=%d leng=%d\n",idx, leng)
		return textForIdx(idx, leng, -1, fastmode)
	}
	return ""
}

func BlockInfo(this js.Value, args []js.Value) interface{} {
	var blockNumber int = args[0].Int()
	return fmt.Sprintf("%d %d %d %d %d %d",
		blockArray[blockNumber].BlockEndCriteria,
		blockArray[blockNumber].Idx,
		blockArray[blockNumber].CharLength,
		blockArray[blockNumber].LinefeedCount,
		blockArray[blockNumber].dbgInfo1,
		blockArray[blockNumber].dbgInfo2)
}
func BlockCharLen(this js.Value, args []js.Value) interface{} {
	var blockNumber int = args[0].Int()
	return blockArray[blockNumber].CharLength
}
func BlockLFCount(this js.Value, args []js.Value) interface{} {
	var blockNumber int = args[0].Int()
	return blockArray[blockNumber].LinefeedCount
}

func GetTextForLinkId2(this js.Value, args []js.Value) interface{} {
	linkId := args[0].String()
	startIdx := 0
	if len(args)>=2 {
		startIdx = args[1].Int()
	}
	return getTextForLinkId2(linkId, startIdx)
}

func Base64FromByteArray(this js.Value, args []js.Value) interface{} {
	data := args[0] // js.Value of uint8Array (zip-compressed .epub data)
	dataLen := args[1].Int() // file.size (of the zip-compressed .epub file)
	myPrint("Base64FromByteArray input len=%d\n", dataLen)
	byteArray := make([]uint8, dataLen)
	if(byteArray==nil) {
		myPrint("# Base64FromByteArray: out of memory %d\n",dataLen)
		return ""
	}
	js.CopyBytesToGo(byteArray, data) // dst <- src

	b64StrEnc := b64.StdEncoding.EncodeToString(byteArray)
	myPrint("Base64FromByteArray output len=%d\n", len(b64StrEnc))
	return b64StrEnc
}

func Base64Img(this js.Value, args []js.Value) interface{} {
	filename := args[0].String()

	filename2 := "---"
	if len(args)>=2 {
		selectedSubbookFolder := args[1].String()
		if selectedSubbookFolder!="" {
			filename2 = selectedSubbookFolder+"/"+filename
		}
	}

	myPrint("Base64Img (%s)\n", filename)

	imgStrEnc := ""
	if filename!="" {
		for _, item := range book.Manifest.Items {
			if item.HREF == filename || item.HREF==filename2 {
				r, err := item.Open()
				if err != nil {
					myPrint("# Base64Img IndexFile=%s open err=%s\n", item.HREF, err)
					continue
				}
				imgData, err := io.ReadAll(r)
				if err != nil {
					myPrint("# Base64Img IndexFile=%s read err=%s\n", item.HREF, err)
					return ""
				}

				if imgData!=nil {
					myPrint("Base64Img b64 encode %s\n", item.HREF)
					imgStrEnc = b64.StdEncoding.EncodeToString([]byte(imgData))
					break
				}
			}
		}
	}
	return imgStrEnc
}

func IdxForFilename(this js.Value, args []js.Value) interface{} {
	filename := args[0].String()
	xpos,ok := xhtmlFileMap[filename]
	if !ok {
		xpos = -1
		if subBookFolder!="" {
			tmpFilename := subBookFolder+"/"+filename
			xpos,ok = xhtmlFileMap[tmpFilename]
			if !ok {
				xpos = -1
			}
		}
	}
	myPrint("IdxForFilename filename=%s xpos=%d maplen=%d\n", filename, xpos, len(xhtmlFileMap))
	return xpos
}

func LinkIdToIdx(this js.Value, args []js.Value) interface{} {
	if len(args)>=1 {
		linkId := args[0].String()
		idx,ok := linkIdToIdxMap[linkId]
		if !ok {
			idx = -1
		}
		return idx
	}
	return -1
}

func ReplaceItalic(this js.Value, args []js.Value) interface{} {
	str := args[0].String()
	removeItalic := false
	if len(args)>=2 {
		removeItalic = args[1].Bool()
	}
	str = replaceItalic(str,false,removeItalic)
	return str
}

const bufArrayMaxCount = 4096
var bufArray [bufArrayMaxCount][]uint8
var bufArrayCount int
var bufSize int
func PushData(this js.Value, args []js.Value) interface{} {
	loop := args[0].Int()
	epubData := args[1] // js.Value of uint8Array (zip-compressed .epub data)
	dataLen = args[2].Int() // file.size (of the zip-compressed .epub file)
	myPrint("PushData loop=%d len=%d\n", loop, dataLen)
	if(loop==0) {
		bufSize = 0
		bufArrayCount = 0
	}
	byteArray := make([]uint8, dataLen)
	if(byteArray==nil) {
		myPrint("# PushData: out of memory %d\n",dataLen)
		bufArrayCounter := 0
		for bufArrayCounter < bufArrayCount {
			bufArray[bufArrayCounter] = nil
		}
		bufArrayCount = 0
		return -1
	}
	copyCount := js.CopyBytesToGo(byteArray, epubData) // dst <- src
	if(loop<bufArrayMaxCount) {
		bufArray[loop] = byteArray
		bufArrayCount = loop+1
		bufSize += copyCount
		myPrint("PushData loop=%d js.CopyBytesToGo done, copyCount=%d bufSize=%d\n", loop, copyCount, bufSize)
		return copyCount
	}
	return 0
}

func LoadBook(this js.Value, args []js.Value) interface{} {
	filename := args[0].String()
	loop := args[1].Int()
	epubData := args[2] // js.Value of uint8Array (zip-compressed .epub data)
	dataLen = args[3].Int() // file.size (of the zip-compressed .epub file)
	multiPageMode := args[4].Int()

	subbookTitle := ""
	if len(args)>=6 {
		subbookTitle = args[5].String()
	}

	loadRemote := false
	if len(args)>=7 {
		loadRemote = args[6].Bool()
	}

	if(dataLen>0) {
		myPrint("LoadBook loop=%d dataLen=%d js.CopyBytesToGo ...\n", loop, dataLen)
		byteArray := make([]uint8, dataLen)
		copyCount := js.CopyBytesToGo(byteArray, epubData) // dst <- src
		return loadBook(filename, loop, byteArray, dataLen, copyCount, multiPageMode, subbookTitle, loadRemote)
	}
	myPrint("LoadBook loop=%d dataLen=%d loadBook\n", loop, dataLen)
	return loadBook(filename, loop, nil, 0, 0, multiPageMode, subbookTitle, loadRemote)
}

func GetMainIndex(this js.Value, args []js.Value) interface{} {
	return getBookIndex(false)
}

func BookAuthor(this js.Value, args []js.Value) interface{} {
	if book!=nil {
		return book.Creator
	}
	return ""
}
func BookTitle(this js.Value, args []js.Value) interface{} {
	if book!=nil {
		return book.Title
	}
	return ""
}

func SubBookTitle(this js.Value, args []js.Value) interface{} {
	if len(args)>=1 {
		subBookTitle = args[0].String()
	}
	return subBookTitle
}

func Language(this js.Value, args []js.Value) interface{} {
	if book!=nil {
		return book.Language
	}
	return ""
}

func GetRtl(this js.Value, args []js.Value) interface{} {
	return rtlFlag
}

func LoadSubBook(this js.Value, args []js.Value) interface{} {
	var loop = args[0].Int()
	var startUrl string = args[1].String()
	var nextUrl string = args[2].String()
	var subBookIdx int = args[3].Int()
	var subBookName = ""
	if len(args)>=5 {
		subBookName = args[4].String()
	}
	return loadSubBook(loop, startUrl, nextUrl, subBookIdx, subBookName)
}

func Pagination(this js.Value, args []js.Value) interface{} {
	return pagination()
}

func SetDbg(this js.Value, args []js.Value) interface{} {
	dbg = args[0].Bool()
	myPrint("SetDbg=%v\n", dbg)
	return dbg
}

func SearchText(this js.Value, args []js.Value) interface{} {
	search := args[0].String()
	idx := args[1].Int() // start idx for search in runes[]
	if(idx<0) {
		idx = 0
	}
	direc := args[2].Int() // start idx for search in runes[]
	if direc>0 {
		myPrint("SearchText (%s) idx=%d direc>0\n",search,idx)
		return searchRuneArray(runeText[idx:], search, direc)
	}
	myPrint("SearchText (%s) idx=%d direc<0\n",search,idx)
	return searchRuneArray(runeText[:idx], search, direc)
}

func SetBlockSize(this js.Value, args []js.Value) interface{} {
	blockSizeHighMark280 = args[0].Int()                                        // = semicolon, long-dash
	blockSizeHighMark170 = int(float64(blockSizeHighMark280) * float64(0.90))   // = colon
	blockSizeHighMark200 = int(float64(blockSizeHighMark280) * float64(0.93))   // = coma
	blockSizeHighMark240 = int(float64(blockSizeHighMark280) * float64(0.96))   // = blank
	blockSizeHighMark260 = int(float64(blockSizeHighMark280) * float64(0.98))   // = full-end
	myPrint("SetBlockSize blockSizeHigh 170=%d 200=%d 240=%d 260=%d textLen=%d\n",
		blockSizeHighMark170,blockSizeHighMark200,blockSizeHighMark240,blockSizeHighMark260,textLen)
	if textLen>0 {
		pagination()
	}
	return 0
}

func GetMdText(this js.Value, args []js.Value) interface{} {
	myPrint("GetMdText textLen=%d runeTextLen=%d\n",textLen,len(runeText))
	return string(runeText[:textLen])
}

func LoadedMd(this js.Value, args []js.Value) interface{} {
	myPrint("LoadedMd %d\n",loadedMd)
	return loadedMd
}

func LongestBlocks(this js.Value, args []js.Value) interface{} {
	retlen := totalBlocks
	if len(args)>=1 {
		retlen = args[0].Int()
		if retlen > totalBlocks {
			retlen = totalBlocks
		}
	}

	result := longestBlocks()

	jsArray := make([]interface{}, retlen)
	for i := range result {
		if i<retlen {
			jsArray[i] = result[i].Pagenum
		}
	}
	return jsArray
}

func main() {
	c := make(chan int)

	js.Global().Set("PushData", js.FuncOf(PushData))
	js.Global().Set("LoadBook", js.FuncOf(LoadBook))
	js.Global().Set("TextForBlocknumber", js.FuncOf(TextForBlocknumber))
	js.Global().Set("TextForIdx", js.FuncOf(TextForIdx))
	js.Global().Set("BlockForIdx", js.FuncOf(BlockForIdx))
	js.Global().Set("IdxForBlockNumber", js.FuncOf(IdxForBlockNumber))
	js.Global().Set("TotalSize", js.FuncOf(TotalSize))
	js.Global().Set("TotalBlocks", js.FuncOf(TotalBlocks))

	js.Global().Set("GetChapNameForIdx", js.FuncOf(GetChapNameForIdx))
	js.Global().Set("GetChapStartIdxForIdx", js.FuncOf(GetChapStartIdxForIdx))
	js.Global().Set("GetChapLenForIdx", js.FuncOf(GetChapLenForIdx))
	js.Global().Set("GetChapterName", js.FuncOf(GetChapterName))
	js.Global().Set("GetChapterIdx", js.FuncOf(GetChapterIdx))
	js.Global().Set("GetChapterLevel", js.FuncOf(GetChapterLevel))
	js.Global().Set("GetChapterOffset", js.FuncOf(GetChapterOffset))
	js.Global().Set("LinkIdToIdx", js.FuncOf(LinkIdToIdx))

	js.Global().Set("SetDbg", js.FuncOf(SetDbg))
	js.Global().Set("GetCodeTag", js.FuncOf(GetCodeTag))
	js.Global().Set("LoadSubBook", js.FuncOf(LoadSubBook))
	js.Global().Set("Pagination", js.FuncOf(Pagination))

	js.Global().Set("BookAuthor", js.FuncOf(BookAuthor))
	js.Global().Set("BookTitle", js.FuncOf(BookTitle))
	js.Global().Set("SubBookTitle", js.FuncOf(SubBookTitle))
	js.Global().Set("Language", js.FuncOf(Language))
	js.Global().Set("GetRtl", js.FuncOf(GetRtl))

	js.Global().Set("GetMainIndex", js.FuncOf(GetMainIndex))
	js.Global().Set("ReplaceItalic", js.FuncOf(ReplaceItalic))
	js.Global().Set("Base64Img", js.FuncOf(Base64Img))
	js.Global().Set("IdxForFilename", js.FuncOf(IdxForFilename))
	js.Global().Set("IdxText", js.FuncOf(IdxText))
	js.Global().Set("GetTextForLinkId2", js.FuncOf(GetTextForLinkId2))

	js.Global().Set("SearchText", js.FuncOf(SearchText))
	js.Global().Set("SetBlockSize", js.FuncOf(SetBlockSize))

	js.Global().Set("GetMdText", js.FuncOf(GetMdText))
	js.Global().Set("BlockInfo", js.FuncOf(BlockInfo))
	js.Global().Set("LoadedMd", js.FuncOf(LoadedMd))
	js.Global().Set("LongestBlocks", js.FuncOf(LongestBlocks))
	js.Global().Set("BlockCharLen", js.FuncOf(BlockCharLen))
	js.Global().Set("BlockLFCount", js.FuncOf(BlockLFCount))
	js.Global().Set("Base64FromByteArray", js.FuncOf(Base64FromByteArray))


	fmt.Printf("main ready, version %s (set by gobuild)\n",codetag)
	<-c
}

