package tech.lp2p.odin.model

import io.github.remmerw.nott.Store
import io.github.remmerw.saga.createModel
import io.github.remmerw.saga.toValue
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.io.Sink
import kotlinx.io.Source
import tech.lp2p.odin.Entities
import java.net.InetAddress
import java.net.InetSocketAddress


class PeerStore : Store {
    private val peers: MutableSet<InetSocketAddress> = mutableSetOf()
    private val mutex = Mutex()

    override suspend fun addresses(limit: Int): List<InetSocketAddress> {
        mutex.withLock {
            return peers.take(limit).toList()
        }
    }

    override suspend fun store(address: InetSocketAddress) {
        mutex.withLock {
            peers.add(address)

            if (peers.size > 100) { // only 100 elements in memory store
                peers.remove(peers.first())
            }
        }
    }

    suspend fun load(source: Source) {
        mutex.withLock {
            val model = createModel(Entities.PEERS, source)
            model.getChildren().forEach { entity ->

                val port = model.getAttribute(entity, Entities.PORT)!!.toInt()
                val address = model.getAttribute(entity, Entities.ADDRESS)!!.toData()

                peers.add(
                    InetSocketAddress(
                        InetAddress.getByAddress(address), port
                    )
                )
            }
        }
    }

    suspend fun save(sink: Sink) {
        mutex.withLock {
            val model = createModel(Entities.PEERS)
            peers.forEach { inetSocketAddress ->
                model.createEntity(
                    Entities.PEER,
                    mapOf(
                        Entities.PORT to inetSocketAddress.port.toValue(),
                        Entities.ADDRESS to inetSocketAddress.address.address.toValue()
                    )
                )
            }
            model.content(sink)
        }
    }
}