package tech.lp2p.odin

import android.Manifest
import android.annotation.SuppressLint
import android.content.ContentValues
import android.content.Context
import android.content.Context.CONNECTIVITY_SERVICE
import android.content.Intent
import android.content.Intent.CATEGORY_BROWSABLE
import android.database.Cursor
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.net.Uri
import android.provider.MediaStore
import android.provider.OpenableColumns
import androidx.annotation.RequiresPermission
import androidx.core.content.FileProvider
import androidx.core.net.toUri
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.PreferenceDataStoreFactory
import androidx.datastore.preferences.core.Preferences
import io.github.remmerw.borr.Keys
import io.github.remmerw.borr.PeerId
import io.github.remmerw.borr.generateKeys
import io.github.remmerw.dagr.Data
import io.github.remmerw.grid.WorkManager
import io.github.remmerw.idun.Idun
import io.github.remmerw.idun.Storage
import io.github.remmerw.idun.newIdun
import io.github.remmerw.saga.Entity
import io.github.remmerw.saga.Model
import io.github.remmerw.saga.createModel
import io.github.remmerw.saga.toKey
import io.github.remmerw.saga.toTag
import io.github.remmerw.saga.toValue
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.runBlocking
import kotlinx.io.Buffer
import kotlinx.io.asSource
import kotlinx.io.buffered
import kotlinx.io.files.Path
import kotlinx.io.files.SystemFileSystem
import okio.Path.Companion.toPath
import tech.lp2p.odin.data.getPrivateKey
import tech.lp2p.odin.data.getPublicKey
import tech.lp2p.odin.data.setPrivateKey
import tech.lp2p.odin.data.setPublicKey
import tech.lp2p.odin.model.PeerStore
import tech.lp2p.odin.model.PnsWorker
import tech.lp2p.odin.model.directoryContent
import java.io.File
import java.net.InetSocketAddress
import java.net.NetworkInterface
import kotlin.time.measureTime

object Entities {
    val PEERS = "peers".toTag()
    val PEER = "peer".toTag()
    val TASKS = "tasks".toTag()
    val TASK = "task".toTag()
    val FILES = "files".toTag()
    val FILE = "file".toTag()


    val ADDRESS = "address".toKey()
    val PORT = "port".toKey()
    val URI = "uri".toKey()
    val MIME = "mime".toKey()
    val NAME = "name".toKey()
    val SIZE = "size".toKey()
    val WORK = "active".toKey()
    val ACTIVE = "active".toKey()
    val FINISHED = "finished".toKey()
    val PROGRESS = "progress".toKey()

}

class Platform(
    private val context: Context,
    private val peerStore: PeerStore,
    private val tasks: Model,
    private val files: Model,
    private val idun: Idun,
    private val workManager: WorkManager
) {

    fun workManager(): WorkManager {
        return workManager
    }

    fun mimeType(uri: Uri): String {
        var mimeType = context.contentResolver.getType(uri)
        if (mimeType == null) {
            mimeType = "application/octet-stream"
        }
        return mimeType
    }

    fun fileName(uri: Uri): String {
        var filename: String? = null

        val contentResolver = context.contentResolver

        contentResolver.query(
            uri,
            null, null, null, null
        ).use { cursor ->
            cursor!!.moveToFirst()
            val nameIndex = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME)
            filename = cursor.getString(nameIndex)
        }


        if (filename == null) {
            filename = uri.lastPathSegment
        }

        if (filename == null) {
            filename = "file_name_not_detected"
        }

        return filename
    }

    fun fileSize(uri: Uri): Long {
        val contentResolver = context.contentResolver

        contentResolver.openFileDescriptor(uri, "r").use { fd ->
            return fd!!.statSize
        }
    }


    @RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
    fun isNetworkConnected(): Boolean {
        val connectivityManager = context.getSystemService(
            CONNECTIVITY_SERVICE
        ) as ConnectivityManager
        val nw = connectivityManager.activeNetwork ?: return false
        val actNw = connectivityManager.getNetworkCapabilities(nw)
        return actNw != null && (actNw.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)
                || actNw.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)
                || actNw.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET))
    }

    fun files(): Model {
        return files
    }

    fun idun(): Idun {
        return idun
    }

    fun peerId(): PeerId {
        return idun().peerId()
    }


    suspend fun loadPeers() {
        val peersDb = Path(context.filesDir.absolutePath, "peers.db")
        if (SystemFileSystem.exists(peersDb)) {
            SystemFileSystem.source(peersDb).buffered().use { source ->
                peerStore.load(source)
            }
        }
    }

    suspend fun savePeers() {
        val peersDb = Path(context.filesDir.absolutePath, "peers.db")
        SystemFileSystem.sink(peersDb).buffered().use { sink ->
            peerStore.save(sink)
        }
    }


    suspend fun startup() {


        idun().startup(port = ODIN_PORT, object : Storage {
            override suspend fun getData(cid: Long, offset: Long): Data {

                val fileInfo = Entity(cid, Entities.FILE)

                val url = files().getAttribute(fileInfo, Entities.URI)

                if (url != null) {
                    val uri = url.toString().toUri()
                    val contentResolver = context.contentResolver
                    checkNotNull(contentResolver)
                    var size = 0L

                    val cursor: Cursor? = contentResolver.query(
                        uri, null, null, null, null
                    )
                    if (cursor != null) {
                        cursor.moveToFirst()
                        val value = cursor.getColumnIndex(OpenableColumns.SIZE)
                        size = cursor.getLong(value)
                        cursor.close()
                    }

                    val length = (size - offset)
                    val source = contentResolver.openInputStream(uri)!!.asSource().buffered()
                    source.skip(offset)
                    return Data(source, length)
                } else {
                    return Data(Buffer(), 0)
                }
            }
        })
    }

    fun numReservations(): Int {
        return idun().numReservations()
    }

    fun numIncomingConnections(): Int {
        return idun().numIncomingConnections()
    }

    fun fileInfos(): Flow<List<Entity>> {
        return files().children
    }

    fun delete(fileInfo: Entity) {
        files().removeEntity(entity = fileInfo)
    }

    fun observedAddresses(): List<InetSocketAddress> {
        return idun().publishedAddresses()
    }


    suspend fun publishPeeraddrs(
        addresses: List<InetSocketAddress>,
        maxPublifications: Int = 60,
        timeout: Int = 60
    ) {
        idun().publishAddresses(addresses, maxPublifications, timeout)
    }

    fun shutdown() {
        idun().shutdown()
    }

    fun downloadsUri(mimeType: String, name: String, path: String): Uri? {
        val contentValues = ContentValues()
        contentValues.put(MediaStore.Downloads.DISPLAY_NAME, name)
        contentValues.put(MediaStore.Downloads.MIME_TYPE, mimeType)
        contentValues.put(MediaStore.Downloads.RELATIVE_PATH, path)

        val contentResolver = context.contentResolver
        return contentResolver.insert(MediaStore.Downloads.EXTERNAL_CONTENT_URI, contentValues)
    }

    fun cancelTask(task: Entity) {
        tasks().setAttribute(task, Entities.ACTIVE, true.toString().toValue())
        val uuid = tasks().getAttribute(task, Entities.WORK)
        if (uuid != null) {
            workManager().cancel(uuid.toString())
        }
    }


    fun pruneWork() {
        workManager().pruneWork()
    }


    fun pnsDownloader(task: Entity) {
        val uuid = PnsWorker.download(context, task)
        tasks.setAttribute(task, Entities.WORK, uuid.toValue())
    }

    fun tasks(): Model {
        return tasks
    }

    fun showTask(task: Entity, onWarningRequest: (String) -> Unit) {
        try {
            val uri = tasks().getAttribute(task, Entities.URI)!!.toString()
            val intent = Intent(Intent.ACTION_VIEW, uri.toUri())
            intent.addCategory(CATEGORY_BROWSABLE)
            intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            context.startActivity(intent)
        } catch (_: Throwable) {
            onWarningRequest.invoke(
                context.getString(R.string.no_app_found_to_handle_uri)
            )
        }
    }

    fun deviceName(): String {
        return android.os.Build.MANUFACTURER + " " + android.os.Build.MODEL
    }


    fun sharePage(onWarningRequest: (String) -> Unit) {
        try {
            val intent = Intent(Intent.ACTION_SEND)
            intent.putExtra(
                Intent.EXTRA_SUBJECT, context.getString(R.string.share_link)
            )
            intent.setType("vnd.android.cursor.dir/email")

            val mails = File(context.filesDir, "mails")
            mails.mkdirs()

            val requestFile = File(mails, "odin.html")
            requestFile.createNewFile()
            requestFile.writeText(
                directoryContent(
                    platform().files(),
                    platform().peerId(),
                    deviceName()
                )
            )

            val attachUri = FileProvider.getUriForFile(
                context,
                "tech.lp2p.odin.fileprovider",
                requestFile
            )


            intent.putExtra(Intent.EXTRA_STREAM, attachUri)
            intent.putExtra(
                Intent.EXTRA_TEXT,
                context.getString(R.string.email_message)
            )
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
            val chooser = Intent.createChooser(
                intent, context.getString(R.string.share)
            )

            chooser.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP)
            chooser.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            context.startActivity(chooser)
        } catch (_: Throwable) {
            onWarningRequest.invoke(context.getString(R.string.no_app_found_to_handle_uri))
        }

    }
}


const val ODIN_PORT: Int = 5001

@SuppressLint("StaticFieldLeak")
internal var platform: Platform? = null

fun platform(): Platform = platform!!


fun initializePlatform(context: Context) {

    val time = measureTime {


        val peerStore = PeerStore()
        val datastore = createDataStore(context)


        val idun = newIdun(
            keys = keys(datastore),
            store = peerStore
        )

        platform = Platform(
            context = context,
            peerStore = peerStore,
            tasks = createModel(Entities.TASKS),
            files = createModel(Entities.FILES),
            idun = idun,
            workManager = WorkManager()
        )
    }

    debug("App", "App started " + time.inWholeMilliseconds)
}


fun siteLocalAddresses(): List<InetSocketAddress> {
    val inetAddresses: MutableList<InetSocketAddress> = ArrayList()

    try {
        val interfaces = NetworkInterface.getNetworkInterfaces()
        for (networkInterface in interfaces) {
            if (networkInterface.isUp()) {
                val addresses = networkInterface.getInetAddresses()
                for (inetAddress in addresses) {
                    if (inetAddress.isSiteLocalAddress) {
                        inetAddresses.add(
                            InetSocketAddress(inetAddress, ODIN_PORT)
                        )
                    }
                }
            }
        }
    } catch (throwable: Throwable) {
        debug("Platform", throwable)
    }
    return inetAddresses
}


private const val dataStoreFileName = "settings.preferences_pb"


private fun keys(datastore: DataStore<Preferences>): Keys {
    return runBlocking {
        val privateKey = getPrivateKey(datastore).first()
        val publicKey = getPublicKey(datastore).first()
        if (privateKey.isNotEmpty() && publicKey.isNotEmpty()) {
            return@runBlocking Keys(PeerId(publicKey), privateKey)
        } else {
            val keys = generateKeys()
            setPrivateKey(datastore, keys.privateKey)
            setPublicKey(datastore, keys.peerId.hash)
            return@runBlocking keys
        }
    }
}

private fun createDataStore(producePath: () -> String): DataStore<Preferences> =
    PreferenceDataStoreFactory.createWithPath(
        produceFile = { producePath().toPath() }
    )

private fun createDataStore(context: Context): DataStore<Preferences> =
    createDataStore(
        producePath = { context.filesDir.resolve(dataStoreFileName).absolutePath }
    )

