package tech.lp2p.odin

import android.Manifest
import android.app.Application
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.util.Log
import androidx.annotation.RequiresPermission
import io.github.vinceglb.filekit.FileKit
import io.github.vinceglb.filekit.manualFileKitCoreInitialization
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.cancel
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.net.InetSocketAddress


class App : Application() {

    private val scope = CoroutineScope(Dispatchers.IO)

    override fun onTerminate() {
        super.onTerminate()
        scope.cancel()
        platform().shutdown()
    }


    @RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
    fun isNetworkConnected(): Boolean {
        val connectivityManager = getSystemService(
            CONNECTIVITY_SERVICE
        ) as ConnectivityManager
        val nw = connectivityManager.activeNetwork ?: return false
        val actNw = connectivityManager.getNetworkCapabilities(nw)
        return actNw != null && (actNw.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)
                || actNw.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)
                || actNw.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET))
    }

    override fun onCreate() {
        super.onCreate()


        try {
            FileKit.manualFileKitCoreInitialization(applicationContext)
            initializePlatform(applicationContext)

            scope.launch {

                platform().loadPeers()
                platform().startup()

                while (true) {

                    val addresses: MutableList<InetSocketAddress> = mutableListOf()

                    addresses.addAll(platform().observedAddresses()) // for WAN
                    addresses.addAll(siteLocalAddresses()) // for LAN

                    if (addresses.isNotEmpty()) {
                        platform().publishPeeraddrs(addresses)

                        platform().savePeers()

                    }

                    val wasOffline = !isNetworkConnected()

                    var counter = 80

                    while (counter > 0) { // 20 minutes in this loop
                        delay(30000) // 30 sec delay

                        if (wasOffline) {
                            if (isNetworkConnected()) {
                                // not connected (just restart publishing addresses)
                                counter = 0
                            }
                        }
                        counter--
                    }
                }

            }

        } catch (throwable: Throwable) {
            debug("App", throwable)
        }
    }
}


@Suppress("SameReturnValue")
private val isDebug: Boolean get() = false


fun debug(tag: String, message: String) {
    if (isDebug) {
        Log.e(tag, message)
    }
}


fun debug(tag: String, throwable: Throwable) {
    if (isDebug) {
        Log.e(tag, throwable.localizedMessage, throwable)
    }
}