import 'dart:async';

import 'package:flutter/material.dart';
import 'package:flutter/rendering.dart';
import 'package:flutter/services.dart';
import 'package:xterm/src/core/buffer/cell_offset.dart';
import 'package:xterm/src/core/input/keys.dart';
import 'package:xterm/src/core/mouse/button.dart';
import 'package:xterm/src/core/mouse/button_state.dart';
import 'package:xterm/src/terminal.dart';
import 'package:xterm/src/ui/controller.dart';
import 'package:xterm/src/ui/cursor_type.dart';
import 'package:xterm/src/ui/custom_text_edit.dart';
import 'package:xterm/src/ui/gesture/gesture_handler.dart';
import 'package:xterm/src/ui/input_map.dart';
import 'package:xterm/src/ui/key_repeat_controller.dart';
import 'package:xterm/src/ui/keyboard_listener.dart';
import 'package:xterm/src/ui/render.dart';
import 'package:xterm/src/ui/scroll_handler.dart';
import 'package:xterm/src/ui/shortcut/actions.dart';
import 'package:xterm/src/ui/shortcut/shortcuts.dart';
import 'package:xterm/src/ui/terminal_text_style.dart';
import 'package:xterm/src/ui/terminal_theme.dart';
import 'package:xterm/src/ui/themes.dart';

const _repeatableTerminalKeys = <TerminalKey>{
  TerminalKey.backspace,
  TerminalKey.delete,
  TerminalKey.arrowUp,
  TerminalKey.arrowDown,
  TerminalKey.arrowLeft,
  TerminalKey.arrowRight,
};

class TerminalView extends StatefulWidget {
  const TerminalView(
    this.terminal, {
    super.key,
    this.controller,
    this.theme = TerminalThemes.defaultTheme,
    this.textStyle = const TerminalStyle(),
    this.textScaler,
    this.padding,
    this.scrollController,
    this.autoResize = true,
    this.backgroundOpacity = 1,
    this.focusNode,
    this.autofocus = false,
    this.onTapUp,
    this.onSecondaryTapDown,
    this.onSecondaryTapUp,
    this.mouseCursor = SystemMouseCursors.text,
    this.keyboardType = TextInputType.emailAddress,
    this.keyboardAppearance = Brightness.dark,
    this.cursorType = TerminalCursorType.block,
    this.cursorBlink = false,
    this.cursorBlinkInterval = const Duration(milliseconds: 530),
    this.alwaysShowCursor = false,
    this.deleteDetection = false,
    this.shortcuts,
    this.onKeyEvent,
    this.readOnly = false,
    this.hardwareKeyboardOnly = false,
    this.simulateScroll = true,
    this.hideScrollBar = true,
    this.viewOffset = Offset.zero,
    this.showToolbar = true,
    this.enableSuggestions = true,
    this.scrollBehavior,
    this.toolbarBuilder,
  });

  /// The underlying terminal that this widget renders.
  final Terminal terminal;

  final TerminalController? controller;

  /// The theme to use for this terminal.
  final TerminalTheme theme;

  /// The style to use for painting characters.
  final TerminalStyle textStyle;

  final TextScaler? textScaler;

  /// Padding around the inner [Scrollable] widget.
  final EdgeInsets? padding;

  /// Scroll controller for the inner [Scrollable] widget.
  final ScrollController? scrollController;

  /// Should this widget automatically notify the underlying terminal when its
  /// size changes. [true] by default.
  final bool autoResize;

  /// Opacity of the terminal background. Set to 0 to make the terminal
  /// background transparent.
  final double backgroundOpacity;

  /// An optional focus node to use as the focus node for this widget.
  final FocusNode? focusNode;

  /// True if this widget will be selected as the initial focus when no other
  /// node in its scope is currently focused.
  final bool autofocus;

  /// Callback for when the user taps on the terminal.
  final void Function(TapUpDetails, CellOffset)? onTapUp;

  /// Function called when the user taps on the terminal with a secondary
  /// button.
  final void Function(TapDownDetails, CellOffset)? onSecondaryTapDown;

  /// Function called when the user stops holding down a secondary button.
  final void Function(TapUpDetails, CellOffset)? onSecondaryTapUp;

  /// The mouse cursor for mouse pointers that are hovering over the terminal.
  /// [SystemMouseCursors.text] by default.
  final MouseCursor mouseCursor;

  /// The type of information for which to optimize the text input control.
  /// [TextInputType.emailAddress] by default.
  final TextInputType keyboardType;

  /// The appearance of the keyboard. [Brightness.dark] by default.
  ///
  /// This setting is only honored on iOS devices.
  final Brightness keyboardAppearance;

  /// The type of cursor to use. [TerminalCursorType.block] by default.
  final TerminalCursorType cursorType;

  /// Whether the cursor should blink. [false] by default to match legacy behavior.
  final bool cursorBlink;

  /// Interval used when [cursorBlink] is enabled.
  final Duration cursorBlinkInterval;

  /// Whether to always show the cursor. This is useful for debugging.
  /// [false] by default.
  final bool alwaysShowCursor;

  /// Workaround to detect delete key for platforms and IMEs that does not
  /// emit hardware delete event. Prefered on mobile platforms. [false] by
  /// default.
  final bool deleteDetection;

  /// Shortcuts for this terminal. This has higher priority than input handler
  /// of the terminal If not provided, [defaultTerminalShortcuts] will be used.
  final Map<ShortcutActivator, Intent>? shortcuts;

  /// Keyboard event handler of the terminal. This has higher priority than
  /// [shortcuts] and input handler of the terminal.
  final FocusOnKeyEventCallback? onKeyEvent;

  /// True if no input should send to the terminal.
  final bool readOnly;

  /// True if only hardware keyboard events should be used as input. This will
  /// also prevent any on-screen keyboard to be shown.
  final bool hardwareKeyboardOnly;

  /// If true, when the terminal is in alternate buffer (for example running
  /// vim, man, etc), if the application does not declare that it can handle
  /// scrolling, the terminal will simulate scrolling by sending up/down arrow
  /// keys to the application. This is standard behavior for most terminal
  /// emulators. True by default.
  final bool simulateScroll;

  final bool hideScrollBar;

  final Offset viewOffset;

  final bool showToolbar;

  /// If this is false, some Chinese Android will open safe keyboard.
  final bool enableSuggestions;

  /// Allows customizing the scroll behavior used by the terminal viewport.
  final ScrollBehavior? scrollBehavior;

  /// Optional builder to customize selection toolbar items shown by the input bridge.
  final CustomTextEditToolbarBuilder? toolbarBuilder;

  @override
  State<TerminalView> createState() => TerminalViewState();
}

class TerminalViewState extends State<TerminalView>
    with TickerProviderStateMixin {
  late FocusNode _focusNode;

  late final ShortcutManager _shortcutManager;
  late final KeyRepeatController _keyRepeatController;

  final _customTextEditKey = GlobalKey<CustomTextEditState>();

  final _scrollableKey = GlobalKey<ScrollableState>();

  final _viewportKey = GlobalKey();

  Timer? _cursorBlinkTimer;
  bool _cursorBlinkVisible = true;
  bool _previousBlinkEnabled = false;

  String? _composingText;

  late TerminalController _controller;

  late ScrollController _scrollController;

  RenderTerminal get renderTerminal =>
      _viewportKey.currentContext!.findRenderObject() as RenderTerminal;

  late final textSizeNoti = ValueNotifier(widget.textStyle.fontSize);

  @override
  void initState() {
    _keyRepeatController = KeyRepeatController();
    _focusNode = widget.focusNode ?? FocusNode();
    _focusNode.addListener(_handleFocusChange);
    _controller = widget.controller ?? TerminalController(vsync: this);
    _scrollController = widget.scrollController ?? ScrollController();
    _shortcutManager = ShortcutManager(
      shortcuts: widget.shortcuts ?? defaultTerminalShortcuts,
    );
    super.initState();
    _updateCursorBlink(scheduleSetState: false);
  }

  @override
  void didUpdateWidget(TerminalView oldWidget) {
    if (oldWidget.focusNode != widget.focusNode) {
      _focusNode.removeListener(_handleFocusChange);
      if (oldWidget.focusNode == null) {
        _focusNode.dispose();
      }
      _focusNode = widget.focusNode ?? FocusNode();
      _focusNode.addListener(_handleFocusChange);
      _updateCursorBlink(resetVisible: true);
    }
    if (oldWidget.controller != widget.controller) {
      if (oldWidget.controller == null) {
        _controller.dispose();
      }
      _controller = widget.controller ?? TerminalController(vsync: this);
    }
    if (oldWidget.scrollController != widget.scrollController) {
      if (oldWidget.scrollController == null) {
        _scrollController.dispose();
      }
      _scrollController = widget.scrollController ?? ScrollController();
    }
    _shortcutManager.shortcuts = widget.shortcuts ?? defaultTerminalShortcuts;
    if (oldWidget.textStyle.fontSize != widget.textStyle.fontSize) {
      textSizeNoti.value = widget.textStyle.fontSize;
    }
    if (oldWidget.cursorBlink != widget.cursorBlink ||
        oldWidget.cursorBlinkInterval != widget.cursorBlinkInterval ||
        oldWidget.alwaysShowCursor != widget.alwaysShowCursor) {
      _updateCursorBlink(resetVisible: true);
    }
    super.didUpdateWidget(oldWidget);
  }

  @override
  void dispose() {
    _focusNode.removeListener(_handleFocusChange);
    if (widget.focusNode == null) {
      _focusNode.dispose();
    }
    if (widget.controller == null) {
      _controller.dispose();
    }
    if (widget.scrollController == null) {
      _scrollController.dispose();
    }
    _keyRepeatController.dispose();
    _shortcutManager.dispose();
    textSizeNoti.dispose();
    _cursorBlinkTimer?.cancel();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    Widget child = ScrollConfiguration(
      behavior: widget.scrollBehavior ?? const _TerminalScrollBehavior(),
      child: Scrollable(
        key: _scrollableKey,
        controller: _scrollController,
        physics: const ClampingScrollPhysics(),
        viewportBuilder: (context, offset) {
          return ValueListenableBuilder(
            valueListenable: textSizeNoti,
            builder: (_, textSize, _) {
              return _TerminalView(
                key: _viewportKey,
                terminal: widget.terminal,
                controller: _controller,
                offset: offset,
                padding: MediaQuery.of(context).padding,
                autoResize: widget.autoResize,
                textStyle: widget.textStyle.copyWith(fontSize: textSize),
                textScaler:
                    widget.textScaler ?? MediaQuery.textScalerOf(context),
                theme: widget.theme,
                focusNode: _focusNode,
                cursorType: widget.cursorType,
                cursorBlinkEnabled: _cursorBlinkEnabled,
                cursorBlinkVisible: _cursorBlinkVisible,
                alwaysShowCursor: widget.alwaysShowCursor,
                paintSelectionHandles: widget.showToolbar,
                onEditableRect: _onEditableRect,
                composingText: _composingText,
              );
            },
          );
        },
      ),
    );

    if (!widget.hideScrollBar) {
      child = Scrollbar(controller: _scrollController, child: child);
    }

    child = TerminalScrollGestureHandler(
      terminal: widget.terminal,
      simulateScroll: widget.simulateScroll,
      getCellOffset: (offset) => renderTerminal.getCellOffset(offset),
      getLineHeight: () => renderTerminal.lineHeight,
      child: child,
    );

    if (!widget.hardwareKeyboardOnly) {
      child = CustomTextEdit(
        key: _customTextEditKey,
        focusNode: _focusNode,
        autofocus: widget.autofocus,
        inputType: widget.keyboardType,
        keyboardAppearance: widget.keyboardAppearance,
        deleteDetection: widget.deleteDetection,
        enableSuggestions: widget.enableSuggestions,
        onInsert: _onInsert,
        onDelete: () {
          _scrollToBottom();
          widget.terminal.keyInput(TerminalKey.backspace);
          _updateCursorBlink(resetVisible: true);
        },
        onComposing: _onComposing,
        onAction: (action) {
          _scrollToBottom();
          // Android sends TextInputAction.newline when the user presses the virtual keyboard's enter key.
          if (action == TextInputAction.done ||
              action == TextInputAction.newline) {
            widget.terminal.keyInput(TerminalKey.enter);
            _updateCursorBlink(resetVisible: true);
          }
        },
        onKeyEvent: _handleKeyEvent,
        readOnly: widget.readOnly,
        toolbarBuilder: widget.toolbarBuilder,
        child: child,
      );
    } else if (!widget.readOnly) {
      // Only listen for key input from a hardware keyboard.
      child = CustomKeyboardListener(
        child: child,
        focusNode: _focusNode,
        autofocus: widget.autofocus,
        onInsert: _onInsert,
        onComposing: _onComposing,
        onKeyEvent: _handleKeyEvent,
      );
    }

    child = TerminalActions(
      terminal: widget.terminal,
      controller: _controller,
      child: child,
    );

    child = TerminalGestureHandler(
      viewOffset: widget.viewOffset,
      showToolbar: widget.showToolbar,
      terminalView: this,
      terminalController: _controller,
      onTapUp: _onTapUp,
      onTapDown: _onTapDown,
      onSecondaryTapDown: widget.onSecondaryTapDown != null
          ? _onSecondaryTapDown
          : null,
      onSecondaryTapUp: widget.onSecondaryTapUp != null
          ? _onSecondaryTapUp
          : null,
      readOnly: widget.readOnly,
      scrollController: _scrollController,
      child: child,
    );

    child = MouseRegion(cursor: widget.mouseCursor, child: child);

    child = Container(
      color: widget.theme.background.withValues(
        alpha: widget.backgroundOpacity,
      ),
      padding: widget.padding,
      child: child,
    );

    return child;
  }

  void requestKeyboard() {
    _customTextEditKey.currentState?.requestKeyboard();
  }

  void closeKeyboard() {
    _customTextEditKey.currentState?.closeKeyboard();
  }

  void unFocus() {
    _focusNode.unfocus();
    _customTextEditKey.currentState?.closeKeyboard();
  }

  void showSelectionToolbar(Rect globalSelectionRect) {
    _customTextEditKey.currentState?.showToolbar(
      globalSelectionRect: globalSelectionRect,
    );
  }

  void hideSelectionToolbar() {
    _customTextEditKey.currentState?.hideToolbar();
  }

  bool get isSelectionToolbarShown =>
      _customTextEditKey.currentState?.isToolbarShown ?? false;

  void toggleFocus() {
    _customTextEditKey.currentState?.toggleKeyboard();
    if (_focusNode.hasFocus) {
      _focusNode.unfocus();
    } else {
      _focusNode.requestFocus();
    }
  }

  Rect get cursorRect {
    return renderTerminal.cursorOffset & renderTerminal.cellSize;
  }

  Rect get globalCursorRect {
    return renderTerminal.localToGlobal(renderTerminal.cursorOffset) &
        renderTerminal.cellSize;
  }

  bool get _cursorBlinkEnabled {
    return widget.cursorBlink &&
        _focusNode.hasFocus &&
        !widget.alwaysShowCursor;
  }

  void _handleFocusChange() {
    if (!_focusNode.hasFocus) {
      _keyRepeatController.cancel();
    }
    _updateCursorBlink(resetVisible: true);
  }

  void _updateCursorBlink({
    bool resetVisible = false,
    bool scheduleSetState = true,
  }) {
    final shouldBlink = _cursorBlinkEnabled;
    final blinkChanged = shouldBlink != _previousBlinkEnabled;
    _previousBlinkEnabled = shouldBlink;

    _cursorBlinkTimer?.cancel();

    var shouldNotify = blinkChanged;

    if ((resetVisible || !shouldBlink) && !_cursorBlinkVisible) {
      _cursorBlinkVisible = true;
      shouldNotify = true;
    }

    if (shouldBlink) {
      _cursorBlinkTimer = Timer.periodic(widget.cursorBlinkInterval, (_) {
        if (!mounted) {
          return;
        }
        setState(() {
          _cursorBlinkVisible = !_cursorBlinkVisible;
        });
      });
    }

    if (shouldNotify && scheduleSetState && mounted) {
      setState(() {});
    }
  }

  void _onTapUp(TapUpDetails details) {
    final offset = renderTerminal.getCellOffset(details.localPosition);
    widget.onTapUp?.call(details, offset);
    widget.terminal.mouseInput(
      TerminalMouseButton.left,
      TerminalMouseButtonState.up,
      offset,
    );
  }

  void _onTapDown(TapDownDetails details) {
    if (_controller.selection == null) {
      if (!widget.hardwareKeyboardOnly) {
        _customTextEditKey.currentState?.requestKeyboard();
      } else {
        _focusNode.requestFocus();
      }
    }

    _updateCursorBlink(resetVisible: true);

    widget.terminal.mouseInput(
      TerminalMouseButton.left,
      TerminalMouseButtonState.down,
      renderTerminal.getCellOffset(details.localPosition),
    );
  }

  void _onSecondaryTapDown(TapDownDetails details) {
    final offset = renderTerminal.getCellOffset(details.localPosition);
    widget.onSecondaryTapDown?.call(details, offset);
  }

  void _onSecondaryTapUp(TapUpDetails details) {
    final offset = renderTerminal.getCellOffset(details.localPosition);
    widget.onSecondaryTapUp?.call(details, offset);
  }

  bool get hasInputConnection {
    return _customTextEditKey.currentState?.hasInputConnection == true;
  }

  void _onInsert(String text) {
    final key = charToTerminalKey(text.trim());

    // On mobile platforms there is no guarantee that virtual keyboard will
    // generate hardware key events. So we need first try to send the key
    // as a hardware key event. If it fails, then we send it as a text input.
    final consumed = key == null ? false : widget.terminal.keyInput(key);

    if (!consumed) {
      widget.terminal.textInput(text);
    }

    _scrollToBottom();
    _updateCursorBlink(resetVisible: true);
  }

  void _onComposing(String? text) {
    setState(() => _composingText = text);
    _updateCursorBlink(resetVisible: true);
  }

  KeyEventResult _handleKeyEvent(FocusNode focusNode, KeyEvent event) {
    if (event is KeyUpEvent) {
      _keyRepeatController.handleKeyUp(event);
    }

    final resultOverride = widget.onKeyEvent?.call(focusNode, event);
    if (resultOverride != null && resultOverride != KeyEventResult.ignored) {
      return resultOverride;
    }

    // ignore: invalid_use_of_protected_member
    final shortcutResult = _shortcutManager.handleKeypress(
      focusNode.context!,
      event,
    );

    if (shortcutResult != KeyEventResult.ignored) {
      return shortcutResult;
    }

    if (event is KeyUpEvent) {
      return KeyEventResult.ignored;
    }

    final isKeyDown = event is KeyDownEvent;
    final isKeyRepeat = event is KeyRepeatEvent;

    if (!isKeyDown && !isKeyRepeat) {
      return KeyEventResult.ignored;
    }

    final key = keyToTerminalKey(event.logicalKey);
    if (key == null) {
      return KeyEventResult.ignored;
    }

    final repeatable = _repeatableTerminalKeys.contains(key);

    if (isKeyRepeat && repeatable) {
      _keyRepeatController.handleKeyRepeat(event as KeyRepeatEvent);
    }

    final handled = _sendTerminalKey(key);

    if (!handled) {
      if (isKeyDown && repeatable) {
        _keyRepeatController.cancel();
      }
      return KeyEventResult.ignored;
    }

    if (isKeyDown && repeatable) {
      _keyRepeatController.handleKeyDown(
        event as KeyDownEvent,
        onRepeat: () {
          _sendTerminalKey(key);
        },
      );
    }

    return KeyEventResult.handled;
  }

  bool _sendTerminalKey(TerminalKey key) {
    final handled = widget.terminal.keyInput(
      key,
      ctrl: HardwareKeyboard.instance.isControlPressed,
      alt: HardwareKeyboard.instance.isAltPressed,
      shift: HardwareKeyboard.instance.isShiftPressed,
    );

    if (handled) {
      _scrollToBottom();
      _updateCursorBlink(resetVisible: true);
    }

    return handled;
  }

  void _onEditableRect(Rect rect, Rect caretRect) {
    _customTextEditKey.currentState?.setEditableRect(rect, caretRect);
  }

  void _scrollToBottom() {
    final position = _scrollableKey.currentState?.position;
    if (position != null) {
      position.animateTo(
        position.maxScrollExtent,
        duration: const Duration(milliseconds: 377),
        curve: Curves.fastEaseInToSlowEaseOut,
      );
    }
  }

  void autoScrollDown(Offset localPointerPosition) {
    final scrollThrshold = renderTerminal.lineHeight * 3;
    final position = _scrollableKey.currentState?.position;
    if (position == null) return;
    final notBottom = position.pixels < position.maxScrollExtent;
    final shouldScrollDown =
        localPointerPosition.dy > renderTerminal.size.height - scrollThrshold;
    if (shouldScrollDown && notBottom) {
      position.animateTo(
        position.pixels + scrollThrshold,
        duration: const Duration(milliseconds: 177),
        curve: Curves.fastEaseInToSlowEaseOut,
      );
    }
    final notTop = position.pixels > 0;
    final shouldScrollUp = localPointerPosition.dy < scrollThrshold;
    if (shouldScrollUp && notTop) {
      position.animateTo(
        position.pixels - scrollThrshold,
        duration: const Duration(milliseconds: 177),
        curve: Curves.fastEaseInToSlowEaseOut,
      );
    }
  }
}

class _TerminalView extends LeafRenderObjectWidget {
  const _TerminalView({
    super.key,
    required this.terminal,
    required this.controller,
    required this.offset,
    required this.padding,
    required this.autoResize,
    required this.textStyle,
    required this.textScaler,
    required this.theme,
    required this.focusNode,
    required this.cursorType,
    required this.cursorBlinkEnabled,
    required this.cursorBlinkVisible,
    required this.alwaysShowCursor,
    required this.paintSelectionHandles,
    this.onEditableRect,
    this.composingText,
  });

  final Terminal terminal;

  final TerminalController controller;

  final ViewportOffset offset;

  final EdgeInsets padding;

  final bool autoResize;

  final TerminalStyle textStyle;

  final TextScaler textScaler;

  final TerminalTheme theme;

  final FocusNode focusNode;

  final TerminalCursorType cursorType;

  final bool cursorBlinkEnabled;

  final bool cursorBlinkVisible;

  final bool alwaysShowCursor;

  final bool paintSelectionHandles;

  final EditableRectCallback? onEditableRect;

  final String? composingText;

  @override
  RenderTerminal createRenderObject(BuildContext context) {
    return RenderTerminal(
      terminal: terminal,
      controller: controller,
      offset: offset,
      padding: padding,
      autoResize: autoResize,
      textStyle: textStyle,
      textScaler: textScaler,
      theme: theme,
      focusNode: focusNode,
      cursorType: cursorType,
      cursorBlinkEnabled: cursorBlinkEnabled,
      cursorBlinkVisible: cursorBlinkVisible,
      alwaysShowCursor: alwaysShowCursor,
      paintSelectionHandles: paintSelectionHandles,
      onEditableRect: onEditableRect,
      composingText: composingText,
    );
  }

  @override
  void updateRenderObject(BuildContext context, RenderTerminal renderObject) {
    renderObject
      ..terminal = terminal
      ..controller = controller
      ..offset = offset
      ..padding = padding
      ..autoResize = autoResize
      ..textStyle = textStyle
      ..textScaler = textScaler
      ..theme = theme
      ..focusNode = focusNode
      ..cursorType = cursorType
      ..cursorBlinkEnabled = cursorBlinkEnabled
      ..cursorBlinkVisible = cursorBlinkVisible
      ..alwaysShowCursor = alwaysShowCursor
      ..paintSelectionHandles = paintSelectionHandles
      ..onEditableRect = onEditableRect
      ..composingText = composingText;
  }
}

class _TerminalScrollBehavior extends ScrollBehavior {
  const _TerminalScrollBehavior();

  @override
  Widget buildOverscrollIndicator(
    BuildContext context,
    Widget child,
    ScrollableDetails details,
  ) {
    return child;
  }
}
