/*
 * Copyright 2008 ZXing authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import '../../result.dart';
import 'abstract_do_co_mo_result_parser.dart';
import 'address_book_parsed_result.dart';
import 'result_parser.dart';

/// Implements the "BIZCARD" address book entry format, though this has been
/// largely reverse-engineered from examples observed in the wild -- still
/// looking for a definitive reference.
///
/// @author Sean Owen
class BizcardResultParser extends AbstractDoCoMoResultParser {
  // Yes, we extend AbstractDoCoMoResultParser since the format is very much
  // like the DoCoMo MECARD format, but this is not technically one of
  // DoCoMo's proposed formats

  @override
  AddressBookParsedResult? parse(Result result) {
    final rawText = ResultParser.getMassagedText(result);
    if (!rawText.startsWith('BIZCARD:')) {
      return null;
    }
    final firstName = matchSingleDoCoMoPrefixedField('N:', rawText, true);
    final lastName = matchSingleDoCoMoPrefixedField('X:', rawText, true);
    final fullName = _buildName(firstName, lastName);
    final title = matchSingleDoCoMoPrefixedField('T:', rawText, true);
    final org = matchSingleDoCoMoPrefixedField('C:', rawText, true);
    final addresses = matchDoCoMoPrefixedField('A:', rawText);
    final phoneNumber1 = matchSingleDoCoMoPrefixedField('B:', rawText, true);
    final phoneNumber2 = matchSingleDoCoMoPrefixedField('M:', rawText, true);
    final phoneNumber3 = matchSingleDoCoMoPrefixedField('F:', rawText, true);
    final email = matchSingleDoCoMoPrefixedField('E:', rawText, true);

    return AddressBookParsedResult.full(
      maybeWrap(fullName),
      null,
      null,
      _buildPhoneNumbers(phoneNumber1, phoneNumber2, phoneNumber3),
      null,
      maybeWrap(email),
      null,
      null,
      null,
      addresses,
      null,
      org,
      null,
      title,
      null,
      null,
    );
  }

  static List<String>? _buildPhoneNumbers(
    String? number1,
    String? number2,
    String? number3,
  ) {
    final List<String> numbers = [];
    if (number1 != null) {
      numbers.add(number1);
    }
    if (number2 != null) {
      numbers.add(number2);
    }
    if (number3 != null) {
      numbers.add(number3);
    }
    final int size = numbers.length;
    if (size == 0) {
      return null;
    }
    return numbers.toList();
  }

  static String? _buildName(String? firstName, String? lastName) {
    if (firstName == null) {
      return lastName;
    } else {
      return lastName == null ? firstName : ('$firstName $lastName');
    }
  }
}
