import 'dart:typed_data';

import 'package:dartssh2/src/ssh_kex.dart';
import 'package:dartssh2/src/utils/bigint.dart';
import 'package:dartssh2/src/utils/list.dart';

/// The Diffie-Hellman (DH) key exchange provides a shared secret that
/// cannot be determined by either party alone.
/// https://tools.ietf.org/html/rfc4253#section-8
class SSHKexDH implements SSHKex {
  static const gexMin = 1024;
  static const gexPref = 2048;
  static const gexMax = 8192;

  /// p is a large safe prime
  final BigInt p;

  /// g is a generator for a subgroup of GF(p)
  final BigInt g;

  /// Length of the random number x
  final int secretBits;

  /// Random number, used as private key
  late final BigInt x;

  /// e = g^x mod p, used as public key
  late final BigInt e;

  SSHKexDH({required this.p, required this.g, required this.secretBits}) {
    if (secretBits % 8 != 0) {
      throw ArgumentError('secretBits must be a multiple of 8');
    }
    x = decodeBigIntWithSign(1, randomBytes(secretBits ~/ 8));
    e = g.modPow(x, p);
  }

  /// https://tools.ietf.org/html/rfc2409 Second Oakley Group
  factory SSHKexDH.group1() {
    return SSHKexDH(p: _group1Prime, g: BigInt.from(2), secretBits: 160);
  }

  /// https://tools.ietf.org/html/rfc3526 Oakley Group 14
  factory SSHKexDH.group14() {
    return SSHKexDH(p: _group14Prime, g: BigInt.from(2), secretBits: 224);
  }

  /// https://tools.ietf.org/html/rfc3526 Oakley Group 16
  factory SSHKexDH.group16() {
    return SSHKexDH(p: _group16Prime, g: BigInt.from(2), secretBits: 256);
  }

  /// Compute the shared secret K
  BigInt computeSecret(BigInt f) {
    return f.modPow(x, p);
  }
}

final _group1Prime = decodeBigIntWithSign(
  1,
  Uint8List.fromList([
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc9, 0x0f, //
    0xda, 0xa2, 0x21, 0x68, 0xc2, 0x34, 0xc4, 0xc6, 0x62, 0x8b, //
    0x80, 0xdc, 0x1c, 0xd1, 0x29, 0x02, 0x4e, 0x08, 0x8a, 0x67, //
    0xcc, 0x74, 0x02, 0x0b, 0xbe, 0xa6, 0x3b, 0x13, 0x9b, 0x22, //
    0x51, 0x4a, 0x08, 0x79, 0x8e, 0x34, 0x04, 0xdd, 0xef, 0x95, //
    0x19, 0xb3, 0xcd, 0x3a, 0x43, 0x1b, 0x30, 0x2b, 0x0a, 0x6d, //
    0xf2, 0x5f, 0x14, 0x37, 0x4f, 0xe1, 0x35, 0x6d, 0x6d, 0x51, //
    0xc2, 0x45, 0xe4, 0x85, 0xb5, 0x76, 0x62, 0x5e, 0x7e, 0xc6, //
    0xf4, 0x4c, 0x42, 0xe9, 0xa6, 0x37, 0xed, 0x6b, 0x0b, 0xff, //
    0x5c, 0xb6, 0xf4, 0x06, 0xb7, 0xed, 0xee, 0x38, 0x6b, 0xfb, //
    0x5a, 0x89, 0x9f, 0xa5, 0xae, 0x9f, 0x24, 0x11, 0x7c, 0x4b, //
    0x1f, 0xe6, 0x49, 0x28, 0x66, 0x51, 0xec, 0xe6, 0x53, 0x81, //
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  ]),
);

/// RFC 3526 - Group 16: 4096-bit MODP Group
/// https://tools.ietf.org/html/rfc3526#section-5
final _group16Prime = decodeBigIntWithSign(
  1,
  Uint8List.fromList([
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc9, 0x0f, //
    0xda, 0xa2, 0x21, 0x68, 0xc2, 0x34, 0xc4, 0xc6, 0x62, 0x8b, //
    0x80, 0xdc, 0x1c, 0xd1, 0x29, 0x02, 0x4e, 0x08, 0x8a, 0x67, //
    0xcc, 0x74, 0x02, 0x0b, 0xbe, 0xa6, 0x3b, 0x13, 0x9b, 0x22, //
    0x51, 0x4a, 0x08, 0x79, 0x8e, 0x34, 0x04, 0xdd, 0xef, 0x95, //
    0x19, 0xb3, 0xcd, 0x3a, 0x43, 0x1b, 0x30, 0x2b, 0x0a, 0x6d, //
    0xf2, 0x5f, 0x14, 0x37, 0x4f, 0xe1, 0x35, 0x6d, 0x6d, 0x51, //
    0xc2, 0x45, 0xe4, 0x85, 0xb5, 0x76, 0x62, 0x5e, 0x7e, 0xc6, //
    0xf4, 0x4c, 0x42, 0xe9, 0xa6, 0x37, 0xed, 0x6b, 0x0b, 0xff, //
    0x5c, 0xb6, 0xf4, 0x06, 0xb7, 0xed, 0xee, 0x38, 0x6b, 0xfb, //
    0x5a, 0x89, 0x9f, 0xa5, 0xae, 0x9f, 0x24, 0x11, 0x7c, 0x4b, //
    0x1f, 0xe6, 0x49, 0x28, 0x66, 0x51, 0xec, 0xe4, 0x5b, 0x3d, //
    0xc2, 0x00, 0x7c, 0xb8, 0xa1, 0x63, 0xbf, 0x05, 0x98, 0xda, //
    0x48, 0x36, 0x1c, 0x55, 0xd3, 0x9a, 0x69, 0x16, 0x3f, 0xa8, //
    0xfd, 0x24, 0xcf, 0x5f, 0x83, 0x65, 0x5d, 0x23, 0xdc, 0xa3, //
    0xad, 0x96, 0x1c, 0x62, 0xf3, 0x56, 0x20, 0x85, 0x52, 0xbb, //
    0x9e, 0xd5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6d, 0x67, 0x0c, //
    0x35, 0x4e, 0x4a, 0xbc, 0x98, 0x04, 0xf1, 0x74, 0x6c, 0x08, //
    0xca, 0x18, 0x21, 0x7c, 0x32, 0x90, 0x5e, 0x46, 0x2e, 0x36, //
    0xce, 0x3b, 0xe3, 0x9e, 0x77, 0x2c, 0x18, 0x0e, 0x86, 0x03, //
    0x9b, 0x27, 0x83, 0xa2, 0xec, 0x07, 0xa2, 0x8f, 0xb5, 0xc5, //
    0x5d, 0xf0, 0x6f, 0x4c, 0x52, 0xc9, 0xde, 0x2b, 0xcb, 0xf6, //
    0x95, 0x58, 0x17, 0x18, 0x39, 0x95, 0x49, 0x7c, 0xea, 0x95, //
    0x6a, 0xe5, 0x15, 0xd2, 0x26, 0x18, 0x98, 0xfa, 0x05, 0x10, //
    0x15, 0x72, 0x8e, 0x5a, 0x8a, 0xac, 0xaa, 0x68, 0xff, 0xff, //
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x20, 0x05, 0xeb, 0xe0, //
    0xb1, 0x62, 0x7b, 0x4d, 0x44, 0x90, 0x4d, 0xb3, 0x1b, 0x73, //
    0xcc, 0x76, 0xfc, 0x6e, 0xce, 0x01, 0x5c, 0x90, 0x9b, 0x02, //
    0x86, 0x67, 0xde, 0x1d, 0xda, 0x25, 0x86, 0xa7, 0xf9, 0x1a, //
    0x29, 0xd2, 0x98, 0x58, 0x89, 0x65, 0xcb, 0x9c, 0x96, 0x1a, //
    0x1d, 0x1e, 0x8e, 0x1e, 0xa4, 0x60, 0xed, 0xbe, 0xdb, 0x24, //
    0x6e, 0xf6, 0x26, 0x7f, 0xa0, 0x80, 0x70, 0x8b, 0x58, 0xb1, //
    0x01, 0x8c, 0xfb, 0x5e, 0x25, 0xaf, 0x8a, 0xee, 0x40, 0xfa, //
    0xd0, 0x04, 0xb2, 0x51, 0x9a, 0x02, 0x99, 0x06, 0xe1, 0x5b, //
    0xd6, 0xb6, 0xf0, 0x8f, 0x5f, 0x92, 0x26, 0x8d, 0xd1, 0xcc, //
    0x73, 0xbc, 0xf9, 0xe1, 0x87, 0xb2, 0xaf, 0xd9, 0x1b, 0x74, //
    0x18, 0x85, 0x3c, 0x7c, 0x15, 0x7d, 0x2e, 0x7a, 0x40, 0x12, //
    0x20, 0x09, 0x9a, 0x46, 0x2e, 0x9e, 0xb7, 0xfb, 0x1c, 0x7d, //
    0x8b, 0x7b, 0xbc, 0x2d, 0x3e, 0x1e, 0xac, 0xd8, 0x0e, 0x91, //
    0x7c, 0xf5, 0x97, 0xdb, 0x5a, 0x0a, 0x8b, 0x5b, 0x14, 0x8c, //
    0xf5, 0xaa, 0xd7, 0x16, 0x86, 0x2f, 0x9e, 0xa2, 0x35, 0x34, //
    0x9f, 0xc7, 0xa3, 0x2b, 0xa1, 0x13, 0xc2, 0x7e, 0x60, 0x5c, //
    0x9e, 0x5f, 0x94, 0x24, 0x5d, 0x24, 0x70, 0xcc, 0x24, 0x11, //
    0x27, 0x04, 0x31, 0xd7, 0xa8, 0xcc, 0x8e, 0xe4, 0xd8, 0x43, //
    0x91, 0x3e, 0x94, 0x9d, 0xe1, 0x33, 0x70, 0x01, 0x15, 0x93, //
    0xc6, 0x7f, 0x39, 0x86, 0x43, 0x4b, 0x4b, 0x1b, 0xa8, 0x96, //
    0x83, 0x52, 0x4e, 0x30, 0xa8, 0xb5, 0xcb, 0x5e, 0xcc, 0x74, //
    0x52, 0xf6, 0xea, 0x0a, 0x59, 0xbd, 0x5b, 0x52, 0x6e, 0x76, //
    0x89, 0xfb, 0x0a, 0x80, 0x72, 0x1e, 0x55, 0x5a, 0x0e, 0x73, //
    0x39, 0xd0, 0x9a, 0x81, 0xc3, 0x8f, 0x20, 0x7f, 0x0f, 0x4f, //
    0x35, 0x27, 0x4c, 0x3d, 0x17, 0x25, 0x78, 0x5f, 0x58, 0xe7, //
    0x76, 0x89, 0xf5, 0x30, 0x8d, 0xf5, 0x69, 0x17, 0xb1, 0x2e, //
    0xdf, 0x07, 0x9e, 0x12, 0x5e, 0x54, 0x73, 0xd0, 0x3f, 0x59, //
    0xcb, 0xd1, 0x8e, 0x17, 0x32, 0x8d, 0xd1, 0xa8, 0xff, 0xff, //
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  ]),
);

final _group14Prime = decodeBigIntWithSign(
  1,
  Uint8List.fromList([
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc9, 0x0f, //
    0xda, 0xa2, 0x21, 0x68, 0xc2, 0x34, 0xc4, 0xc6, 0x62, 0x8b, //
    0x80, 0xdc, 0x1c, 0xd1, 0x29, 0x02, 0x4e, 0x08, 0x8a, 0x67, //
    0xcc, 0x74, 0x02, 0x0b, 0xbe, 0xa6, 0x3b, 0x13, 0x9b, 0x22, //
    0x51, 0x4a, 0x08, 0x79, 0x8e, 0x34, 0x04, 0xdd, 0xef, 0x95, //
    0x19, 0xb3, 0xcd, 0x3a, 0x43, 0x1b, 0x30, 0x2b, 0x0a, 0x6d, //
    0xf2, 0x5f, 0x14, 0x37, 0x4f, 0xe1, 0x35, 0x6d, 0x6d, 0x51, //
    0xc2, 0x45, 0xe4, 0x85, 0xb5, 0x76, 0x62, 0x5e, 0x7e, 0xc6, //
    0xf4, 0x4c, 0x42, 0xe9, 0xa6, 0x37, 0xed, 0x6b, 0x0b, 0xff, //
    0x5c, 0xb6, 0xf4, 0x06, 0xb7, 0xed, 0xee, 0x38, 0x6b, 0xfb, //
    0x5a, 0x89, 0x9f, 0xa5, 0xae, 0x9f, 0x24, 0x11, 0x7c, 0x4b, //
    0x1f, 0xe6, 0x49, 0x28, 0x66, 0x51, 0xec, 0xe4, 0x5b, 0x3d, //
    0xc2, 0x00, 0x7c, 0xb8, 0xa1, 0x63, 0xbf, 0x05, 0x98, 0xda, //
    0x48, 0x36, 0x1c, 0x55, 0xd3, 0x9a, 0x69, 0x16, 0x3f, 0xa8, //
    0xfd, 0x24, 0xcf, 0x5f, 0x83, 0x65, 0x5d, 0x23, 0xdc, 0xa3, //
    0xad, 0x96, 0x1c, 0x62, 0xf3, 0x56, 0x20, 0x85, 0x52, 0xbb, //
    0x9e, 0xd5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6d, 0x67, 0x0c, //
    0x35, 0x4e, 0x4a, 0xbc, 0x98, 0x04, 0xf1, 0x74, 0x6c, 0x08, //
    0xca, 0x18, 0x21, 0x7c, 0x32, 0x90, 0x5e, 0x46, 0x2e, 0x36, //
    0xce, 0x3b, 0xe3, 0x9e, 0x77, 0x2c, 0x18, 0x0e, 0x86, 0x03, //
    0x9b, 0x27, 0x83, 0xa2, 0xec, 0x07, 0xa2, 0x8f, 0xb5, 0xc5, //
    0x5d, 0xf0, 0x6f, 0x4c, 0x52, 0xc9, 0xde, 0x2b, 0xcb, 0xf6, //
    0x95, 0x58, 0x17, 0x18, 0x39, 0x95, 0x49, 0x7c, 0xea, 0x95, //
    0x6a, 0xe5, 0x15, 0xd2, 0x26, 0x18, 0x98, 0xfa, 0x05, 0x10, //
    0x15, 0x72, 0x8e, 0x5a, 0x8a, 0xac, 0xaa, 0x68, 0xff, 0xff, //
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  ]),
);
