/*
 * Copyright 2008 ZXing authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import 'dart:convert';
import 'dart:typed_data';

import 'package:test/expect.dart';
import 'package:test/scaffolding.dart';
import 'package:zxing_lib/common.dart';
import 'package:zxing_lib/qrcode.dart';
import 'package:zxing_lib/zxing.dart';

void main() {
  test('testGetAlphanumericCode', () {
    // The first ten code points are numbers.
    for (int i = 0; i < 10; ++i) {
      expect(i, Encoder.getAlphanumericCode(48 /* 0 */ + i));
    }

    // The next 26 code points are capital alphabet letters.
    for (int i = 10; i < 36; ++i) {
      expect(i, Encoder.getAlphanumericCode(65 /* A */ + i - 10));
    }

    // Others are symbol letters
    expect(36, Encoder.getAlphanumericCode(32 /*   */));
    expect(37, Encoder.getAlphanumericCode(36 /* $ */));
    expect(38, Encoder.getAlphanumericCode(37 /* % */));
    expect(39, Encoder.getAlphanumericCode(42 /* * */));
    expect(40, Encoder.getAlphanumericCode(43 /* + */));
    expect(41, Encoder.getAlphanumericCode(45 /* - */));
    expect(42, Encoder.getAlphanumericCode(46 /* . */));
    expect(43, Encoder.getAlphanumericCode(47 /* / */));
    expect(44, Encoder.getAlphanumericCode(58 /* : */));

    // Should return -1 for other letters;
    expect(-1, Encoder.getAlphanumericCode(97 /* a */));
    expect(-1, Encoder.getAlphanumericCode(35 /* # */));
    expect(-1, Encoder.getAlphanumericCode(0));
  });

  test('testChooseMode', () {
    // Numeric mode.
    expect(Mode.numeric, Encoder.chooseMode('0'));
    expect(Mode.numeric, Encoder.chooseMode('0123456789'));
    // Alphanumeric mode.
    expect(Mode.alphanumeric, Encoder.chooseMode('A'));
    expect(
      Mode.alphanumeric,
      Encoder.chooseMode(r'0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ $%*+-./:'),
    );
    // 8-bit byte mode.
    expect(Mode.byte, Encoder.chooseMode('a'));
    expect(Mode.byte, Encoder.chooseMode('#'));
    expect(Mode.byte, Encoder.chooseMode(''));
    // Kanji mode.  We used to use MODE_KANJI for these, but we stopped
    // doing that as we cannot distinguish Shift_JIS from other encodings
    // from data bytes alone.  See also comments in qrcode_encoder.h.

    // AIUE in Hiragana in Shift_JIS
    expect(
      Mode.byte,
      Encoder.chooseMode(
        shiftJISString(bytes([0x8, 0xa, 0x8, 0xa, 0x8, 0xa, 0x8, 0xa6])),
      ),
    );

    // Nihon in Kanji in Shift_JIS.
    expect(
      Mode.byte,
      Encoder.chooseMode(shiftJISString(bytes([0x9, 0xf, 0x9, 0x7b]))),
    );

    // Sou-Utsu-Byou in Kanji in Shift_JIS.
    expect(
      Mode.byte,
      Encoder.chooseMode(
        shiftJISString(bytes([0xe, 0x4, 0x9, 0x5, 0x9, 0x61])),
      ),
    );
  });

  test('testEncodeDefault', () {
    final qrCode = Encoder.encode('ABCDEF', ErrorCorrectionLevel.H);
    final expected = '<<\n'
        ' mode: ALPHANUMERIC\n'
        ' ecLevel: H\n'
        ' version: 1\n'
        ' maskPattern: 0\n'
        ' matrix:\n'
        ' 1 1 1 1 1 1 1 0 1 1 1 1 0 0 1 1 1 1 1 1 1\n'
        ' 1 0 0 0 0 0 1 0 0 1 1 1 0 0 1 0 0 0 0 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 1 0 1 1 0 1 0 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 1 1 1 0 1 0 1 0 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 1 1 1 0 0 1 0 1 1 1 0 1\n'
        ' 1 0 0 0 0 0 1 0 0 1 0 0 0 0 1 0 0 0 0 0 1\n'
        ' 1 1 1 1 1 1 1 0 1 0 1 0 1 0 1 1 1 1 1 1 1\n'
        ' 0 0 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0 0 0 0 0\n'
        ' 0 0 1 0 1 1 1 0 1 1 0 0 1 1 0 0 0 1 0 0 1\n'
        ' 1 0 1 1 1 0 0 1 0 0 0 1 0 1 0 0 0 0 0 0 0\n'
        ' 0 0 1 1 0 0 1 0 1 0 0 0 1 0 1 0 1 0 1 1 0\n'
        ' 1 1 0 1 0 1 0 1 1 1 0 1 0 1 0 0 0 0 0 1 0\n'
        ' 0 0 1 1 0 1 1 1 1 0 0 0 1 0 1 0 1 1 1 1 0\n'
        ' 0 0 0 0 0 0 0 0 1 0 0 1 1 1 0 1 0 1 0 0 0\n'
        ' 1 1 1 1 1 1 1 0 0 0 1 0 1 0 1 1 0 0 0 0 1\n'
        ' 1 0 0 0 0 0 1 0 1 1 1 1 0 1 0 1 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 1 0 1 1 0 1 0 1 0 0 0 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 1 1 0 1 1 1 1 0 1 0 1 0\n'
        ' 1 0 1 1 1 0 1 0 1 0 0 0 1 0 1 0 1 1 1 0 1\n'
        ' 1 0 0 0 0 0 1 0 0 1 1 0 1 1 0 1 0 0 0 1 1\n'
        ' 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 1 0 1 0 1\n'
        '>>\n';
    expect(expected, qrCode.toString());
  });

  test('testEncodeWithVersion', () {
    final hints = EncodeHint(qrVersion: 7);
    final qrCode = Encoder.encode('ABCDEF', ErrorCorrectionLevel.H, hints);
    assert(qrCode.toString().contains(' version: 7\n'));
  });

  //@Test(expected = WriterException.class)
  test('testEncodeWithVersionTooSmall', () {
    expect(
      () => Encoder.encode(
        'THISMESSAGEISTOOLONGFORAQRCODEVERSION3',
        ErrorCorrectionLevel.H,
        EncodeHint(qrVersion: 3),
      ),
      throwsA(TypeMatcher<WriterException>()),
    );
  });

  test('testSimpleUTF8ECI', () {
    final qrCode = Encoder.encode(
      'hello',
      ErrorCorrectionLevel.H,
      EncodeHint(characterSet: 'UTF8'),
    );
    final expected = '<<\n'
        ' mode: BYTE\n'
        ' ecLevel: H\n'
        ' version: 1\n'
        ' maskPattern: 3\n'
        ' matrix:\n'
        ' 1 1 1 1 1 1 1 0 0 0 0 0 0 0 1 1 1 1 1 1 1\n'
        ' 1 0 0 0 0 0 1 0 0 0 1 0 1 0 1 0 0 0 0 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 1 0 1 0 0 1 0 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 1 1 0 1 0 1 0 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 1 0 1 0 1 0 1 0 1 1 1 0 1\n'
        ' 1 0 0 0 0 0 1 0 0 0 0 0 1 0 1 0 0 0 0 0 1\n'
        ' 1 1 1 1 1 1 1 0 1 0 1 0 1 0 1 1 1 1 1 1 1\n'
        ' 0 0 0 0 0 0 0 0 1 1 1 0 0 0 0 0 0 0 0 0 0\n'
        ' 0 0 1 1 0 0 1 1 1 1 0 0 0 1 1 0 1 0 0 0 0\n'
        ' 0 0 1 1 1 0 0 0 0 0 1 1 0 0 0 1 0 1 1 1 0\n'
        ' 0 1 0 1 0 1 1 1 0 1 0 1 0 0 0 0 0 1 1 1 1\n'
        ' 1 1 0 0 1 0 0 1 1 0 0 1 1 1 1 0 1 0 1 1 0\n'
        ' 0 0 0 0 1 0 1 1 1 1 0 0 0 0 0 1 0 0 1 0 0\n'
        ' 0 0 0 0 0 0 0 0 1 1 1 1 0 0 1 1 1 0 0 0 1\n'
        ' 1 1 1 1 1 1 1 0 1 1 1 0 1 0 1 1 0 0 1 0 0\n'
        ' 1 0 0 0 0 0 1 0 0 0 1 0 0 1 1 1 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 1 0 0 0 0 1 1 0 0 0 0 0\n'
        ' 1 0 1 1 1 0 1 0 1 1 1 0 1 0 0 0 1 1 0 0 0\n'
        ' 1 0 1 1 1 0 1 0 1 1 0 0 0 1 0 0 1 0 0 0 0\n'
        ' 1 0 0 0 0 0 1 0 0 0 0 1 1 0 1 0 1 0 1 1 0\n'
        ' 1 1 1 1 1 1 1 0 0 1 0 1 1 1 0 1 1 0 0 0 0\n'
        '>>\n';
    expect(expected, qrCode.toString());
  });

  test('testEncodeKanjiMode', () {
    final hints = EncodeHint(characterSet: 'Shift_JIS');
    // Nihon in Kanji
    final qrCode =
        Encoder.encode('\u65e5\u672c', ErrorCorrectionLevel.M, hints);
    final expected = '<<\n'
        ' mode: KANJI\n'
        ' ecLevel: M\n'
        ' version: 1\n'
        ' maskPattern: 4\n'
        ' matrix:\n'
        ' 1 1 1 1 1 1 1 0 1 1 1 1 0 0 1 1 1 1 1 1 1\n'
        ' 1 0 0 0 0 0 1 0 0 0 0 1 1 0 1 0 0 0 0 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 0 1 0 0 0 1 0 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 1 0 1 0 1 0 1 0 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 1 1 0 1 1 0 1 0 1 1 1 0 1\n'
        ' 1 0 0 0 0 0 1 0 1 0 1 0 1 0 1 0 0 0 0 0 1\n'
        ' 1 1 1 1 1 1 1 0 1 0 1 0 1 0 1 1 1 1 1 1 1\n'
        ' 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0\n'
        ' 1 0 0 0 1 0 1 1 1 0 0 0 1 1 1 1 1 1 0 0 1\n'
        ' 0 1 1 0 0 1 0 1 1 0 1 0 1 1 1 0 0 0 1 0 1\n'
        ' 1 1 1 1 0 1 1 1 0 0 1 0 1 1 0 0 0 0 1 1 1\n'
        ' 1 0 1 0 1 1 0 0 0 0 1 1 1 0 0 1 0 0 1 1 0\n'
        ' 0 0 1 0 1 1 1 1 1 1 1 1 0 0 1 1 1 1 0 1 1\n'
        ' 0 0 0 0 0 0 0 0 1 1 1 1 1 0 0 1 0 1 0 0 0\n'
        ' 1 1 1 1 1 1 1 0 1 1 0 1 0 0 1 1 1 1 1 1 0\n'
        ' 1 0 0 0 0 0 1 0 0 0 0 0 0 1 1 0 1 0 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 1 0 1 0 1 1 1 0 0 0 1 1 1\n'
        ' 1 0 1 1 1 0 1 0 0 1 0 0 1 1 1 0 0 0 1 1 1\n'
        ' 1 0 1 1 1 0 1 0 0 1 1 0 1 1 0 0 0 1 0 0 0\n'
        ' 1 0 0 0 0 0 1 0 0 0 1 1 1 0 0 1 0 1 0 0 0\n'
        ' 1 1 1 1 1 1 1 0 1 1 1 1 0 0 1 1 1 0 1 1 0\n'
        '>>\n';
    expect(expected, qrCode.toString());
  });

  test('testEncodeShiftjisNumeric', () {
    final hints = EncodeHint(characterSet: 'Shift_JIS');
    final qrCode = Encoder.encode('0123', ErrorCorrectionLevel.M, hints);
    final expected = '<<\n'
        ' mode: NUMERIC\n'
        ' ecLevel: M\n'
        ' version: 1\n'
        ' maskPattern: 0\n'
        ' matrix:\n'
        ' 1 1 1 1 1 1 1 0 0 0 0 0 1 0 1 1 1 1 1 1 1\n'
        ' 1 0 0 0 0 0 1 0 1 1 0 1 0 0 1 0 0 0 0 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 1 1 0 0 0 1 0 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 0 1 0 0 0 1 0 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 1 0 1 1 1 0 1 0 1 1 1 0 1\n'
        ' 1 0 0 0 0 0 1 0 0 1 0 1 0 0 1 0 0 0 0 0 1\n'
        ' 1 1 1 1 1 1 1 0 1 0 1 0 1 0 1 1 1 1 1 1 1\n'
        ' 0 0 0 0 0 0 0 0 0 1 1 0 0 0 0 0 0 0 0 0 0\n'
        ' 1 0 1 0 1 0 1 0 0 0 0 0 1 0 0 0 1 0 0 1 0\n'
        ' 0 0 0 0 0 0 0 1 1 0 1 1 0 1 0 1 0 1 0 1 0\n'
        ' 0 1 0 1 0 1 1 1 1 0 0 1 0 1 1 1 0 1 0 1 0\n'
        ' 0 1 1 1 0 0 0 0 0 0 1 1 1 1 0 1 1 1 0 1 0\n'
        ' 0 0 0 1 1 1 1 1 1 1 1 1 0 1 1 1 0 0 1 0 1\n'
        ' 0 0 0 0 0 0 0 0 1 1 0 0 0 0 1 0 0 0 1 1 0\n'
        ' 1 1 1 1 1 1 1 0 0 1 0 0 1 0 0 0 1 0 0 0 1\n'
        ' 1 0 0 0 0 0 1 0 0 1 0 0 0 0 1 0 0 0 1 0 0\n'
        ' 1 0 1 1 1 0 1 0 1 1 0 0 1 0 1 0 1 0 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 1 1 1 0 1 0 1 0 1 0 1 0\n'
        ' 1 0 1 1 1 0 1 0 1 0 1 1 0 1 1 1 0 1 1 0 1\n'
        ' 1 0 0 0 0 0 1 0 0 0 1 1 1 1 0 1 1 1 0 0 0\n'
        ' 1 1 1 1 1 1 1 0 1 0 1 1 0 1 1 1 0 1 1 0 1\n'
        '>>\n';
    expect(expected, qrCode.toString());
  });

  test('testEncodeGS1WithStringTypeHint', () {
    final hints = EncodeHint(gs1Format: true);

    final qrCode = Encoder.encode(
      '100001%11171218',
      ErrorCorrectionLevel.H,
      hints,
    );
    verifyGS1EncodedData(qrCode);
  });

  test('testEncodeGS1WithBooleanTypeHint', () {
    final hints = EncodeHint(gs1Format: true);
    final qrCode =
        Encoder.encode('100001%11171218', ErrorCorrectionLevel.H, hints);
    verifyGS1EncodedData(qrCode);
  });

  test('testDoesNotEncodeGS1WhenBooleanTypeHintExplicitlyFalse', () {
    final hints = EncodeHint(gs1Format: false);
    final qrCode = Encoder.encode('ABCDEF', ErrorCorrectionLevel.H, hints);
    verifyNotGS1EncodedData(qrCode);
  });

  test('testDoesNotEncodeGS1WhenStringTypeHintExplicitlyFalse', () {
    final hints = EncodeHint(gs1Format: false);
    final qrCode = Encoder.encode('ABCDEF', ErrorCorrectionLevel.H, hints);
    verifyNotGS1EncodedData(qrCode);
  });

  test('testGS1ModeHeaderWithECI', () {
    final hints = EncodeHint(gs1Format: true, characterSet: 'UTF8');

    final qrCode = Encoder.encode('hello', ErrorCorrectionLevel.H, hints);
    final expected = '<<\n'
        ' mode: BYTE\n'
        ' ecLevel: H\n'
        ' version: 1\n'
        ' maskPattern: 6\n'
        ' matrix:\n'
        ' 1 1 1 1 1 1 1 0 0 0 1 1 0 0 1 1 1 1 1 1 1\n'
        ' 1 0 0 0 0 0 1 0 0 1 1 0 0 0 1 0 0 0 0 0 1\n'
        ' 1 0 1 1 1 0 1 0 1 1 0 0 0 0 1 0 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 1 1 0 1 0 0 1 0 1 1 1 0 1\n'
        ' 1 0 1 1 1 0 1 0 0 0 1 1 0 0 1 0 1 1 1 0 1\n'
        ' 1 0 0 0 0 0 1 0 0 1 0 0 1 0 1 0 0 0 0 0 1\n'
        ' 1 1 1 1 1 1 1 0 1 0 1 0 1 0 1 1 1 1 1 1 1\n'
        ' 0 0 0 0 0 0 0 0 0 0 1 1 1 0 0 0 0 0 0 0 0\n'
        ' 0 0 0 1 1 0 1 1 0 1 0 0 0 0 0 0 0 1 1 0 0\n'
        ' 0 1 0 1 1 0 0 1 0 1 1 1 1 1 1 0 1 1 1 0 1\n'
        ' 0 1 1 1 1 0 1 0 0 1 0 1 0 1 1 1 0 0 1 0 1\n'
        ' 1 1 1 1 1 0 0 1 0 0 0 1 1 0 0 1 0 0 1 0 0\n'
        ' 1 0 0 1 0 0 1 1 0 1 1 0 1 0 1 0 0 1 0 0 1\n'
        ' 0 0 0 0 0 0 0 0 1 1 1 1 1 1 0 0 1 0 0 0 1\n'
        ' 1 1 1 1 1 1 1 0 1 0 0 1 0 1 1 0 1 0 1 0 0\n'
        ' 1 0 0 0 0 0 1 0 0 1 0 0 0 0 1 0 1 1 1 0 0\n'
        ' 1 0 1 1 1 0 1 0 1 1 0 1 1 0 0 0 1 1 0 0 0\n'
        ' 1 0 1 1 1 0 1 0 1 0 1 1 1 1 1 0 0 0 1 1 0\n'
        ' 1 0 1 1 1 0 1 0 0 0 1 0 0 1 0 0 1 0 1 1 1\n'
        ' 1 0 0 0 0 0 1 0 0 1 0 0 0 0 0 0 0 1 1 0 0\n'
        ' 1 1 1 1 1 1 1 0 0 1 0 1 0 0 1 0 0 0 0 0 0\n'
        '>>\n';
    expect(expected, qrCode.toString());
  });

  test('testAppendModeInfo', () {
    final bits = BitArray();
    Encoder.appendModeInfo(Mode.numeric, bits);
    expect(' ...X', bits.toString());
  });

  test('testAppendLengthInfo', () {
    BitArray bits = BitArray();
    Encoder.appendLengthInfo(
      1, // 1 letter (1/1).
      Version.getVersionForNumber(1),
      Mode.numeric,
      bits,
    );
    expect(' ........ .X', bits.toString()); // 10 bits.
    bits = BitArray();
    Encoder.appendLengthInfo(
      2, // 2 letters (2/1).
      Version.getVersionForNumber(10),
      Mode.alphanumeric,
      bits,
    );
    expect(' ........ .X.', bits.toString()); // 11 bits.
    bits = BitArray();
    Encoder.appendLengthInfo(
      255, // 255 letter (255/1).
      Version.getVersionForNumber(27),
      Mode.byte,
      bits,
    );
    expect(' ........ XXXXXXXX', bits.toString()); // 16 bits.
    bits = BitArray();
    Encoder.appendLengthInfo(
      512, // 512 letters (1024/2).
      Version.getVersionForNumber(40),
      Mode.kanji,
      bits,
    );
    expect(' ..X..... ....', bits.toString()); // 12 bits.
  });

  test('testAppendBytes', () {
    // Should use appendNumericBytes.
    // 1 = 01 = 0001 in 4 bits.
    BitArray bits = BitArray();
    Encoder.appendBytes(
      '1',
      Mode.numeric,
      bits,
      Encoder.defaultByteModeEncoding,
    );
    expect(' ...X', bits.toString());
    // Should use appendAlphanumericBytes.
    // A = 10 = 0xa = 001010 in 6 bits
    bits = BitArray();
    Encoder.appendBytes(
      'A',
      Mode.alphanumeric,
      bits,
      Encoder.defaultByteModeEncoding,
    );
    expect(' ..X.X.', bits.toString());
    // Lower letters such as 'a' cannot be encoded in MODE_ALPHANUMERIC.

    expect(
      () => Encoder.appendBytes(
        'a',
        Mode.alphanumeric,
        bits,
        Encoder.defaultByteModeEncoding,
      ),
      throwsA(TypeMatcher<WriterException>()),
    );

    // Should use append8BitBytes.
    // 0x61, 0x62, 0x63
    bits = BitArray();
    Encoder.appendBytes(
      'abc',
      Mode.byte,
      bits,
      Encoder.defaultByteModeEncoding,
    );
    expect(' .XX....X .XX...X. .XX...XX', bits.toString());
    // Anything can be encoded in QRCode.MODE_8BIT_BYTE.
    Encoder.appendBytes(
      '\x00',
      Mode.byte,
      bits,
      Encoder.defaultByteModeEncoding,
    );
    // Should use appendKanjiBytes.
    // 0x93, 0x5f
    bits = BitArray();
    Encoder.appendBytes(
      shiftJISString(bytes([0x93, 0x5f])),
      Mode.kanji,
      bits,
      Encoder.defaultByteModeEncoding,
    );
    expect(' .XX.XX.. XXXXX', bits.toString());
  });

  test('testTerminateBits', () {
    BitArray v = BitArray();
    Encoder.terminateBits(0, v);
    expect('', v.toString());
    v = BitArray();
    Encoder.terminateBits(1, v);
    expect(' ........', v.toString());
    v = BitArray();
    v.appendBits(0, 3); // Append 000
    Encoder.terminateBits(1, v);
    expect(' ........', v.toString());
    v = BitArray();
    v.appendBits(0, 5); // Append 00000
    Encoder.terminateBits(1, v);
    expect(' ........', v.toString());
    v = BitArray();
    v.appendBits(0, 8); // Append 00000000
    Encoder.terminateBits(1, v);
    expect(' ........', v.toString());
    v = BitArray();
    Encoder.terminateBits(2, v);
    expect(' ........ XXX.XX..', v.toString());
    v = BitArray();
    v.appendBits(0, 1); // Append 0
    Encoder.terminateBits(3, v);
    expect(' ........ XXX.XX.. ...X...X', v.toString());
  });

  test('testGetNumDataBytesAndNumECBytesForBlockID', () {
    final numDataBytes = [0];
    final numEcBytes = [0];
    // Version 1-H.
    Encoder.getNumDataBytesAndNumECBytesForBlockID(
      26,
      9,
      1,
      0,
      numDataBytes,
      numEcBytes,
    );
    expect(9, numDataBytes[0]);
    expect(17, numEcBytes[0]);

    // Version 3-H.  2 blocks.
    Encoder.getNumDataBytesAndNumECBytesForBlockID(
      70,
      26,
      2,
      0,
      numDataBytes,
      numEcBytes,
    );
    expect(13, numDataBytes[0]);
    expect(22, numEcBytes[0]);
    Encoder.getNumDataBytesAndNumECBytesForBlockID(
      70,
      26,
      2,
      1,
      numDataBytes,
      numEcBytes,
    );
    expect(13, numDataBytes[0]);
    expect(22, numEcBytes[0]);

    // Version 7-H. (4 + 1) blocks.
    Encoder.getNumDataBytesAndNumECBytesForBlockID(
      196,
      66,
      5,
      0,
      numDataBytes,
      numEcBytes,
    );
    expect(13, numDataBytes[0]);
    expect(26, numEcBytes[0]);
    Encoder.getNumDataBytesAndNumECBytesForBlockID(
      196,
      66,
      5,
      4,
      numDataBytes,
      numEcBytes,
    );
    expect(14, numDataBytes[0]);
    expect(26, numEcBytes[0]);

    // Version 40-H. (20 + 61) blocks.
    Encoder.getNumDataBytesAndNumECBytesForBlockID(
      3706,
      1276,
      81,
      0,
      numDataBytes,
      numEcBytes,
    );
    expect(15, numDataBytes[0]);
    expect(30, numEcBytes[0]);
    Encoder.getNumDataBytesAndNumECBytesForBlockID(
      3706,
      1276,
      81,
      20,
      numDataBytes,
      numEcBytes,
    );
    expect(16, numDataBytes[0]);
    expect(30, numEcBytes[0]);
    Encoder.getNumDataBytesAndNumECBytesForBlockID(
      3706,
      1276,
      81,
      80,
      numDataBytes,
      numEcBytes,
    );
    expect(16, numDataBytes[0]);
    expect(30, numEcBytes[0]);
  });

  test('testInterleaveWithECBytes', () {
    Uint8List dataBytes =
        Uint8List.fromList([32, 65, 205, 69, 41, 220, 46, 128, 236]);
    BitArray inArr = BitArray();
    for (int dataByte in dataBytes) {
      inArr.appendBits(dataByte, 8);
    }
    BitArray out = Encoder.interleaveWithECBytes(inArr, 26, 9, 1);
    Uint8List expected = Uint8List.fromList([
      // Data bytes.
      32, 65, 205, 69, 41, 220, 46, 128, 236,
      // Error correction bytes.
      42, 159, 74, 221, 244, 169, 239, 150, 138, 70,
      237, 85, 224, 96, 74, 219, 61,
    ]);
    expect(expected.length, out.sizeInBytes);
    Uint8List outArray = Uint8List(expected.length);
    out.toBytes(0, outArray, 0, expected.length);
    // Can't use Arrays.equals(), because outArray may be longer than out.sizeInBytes()
    for (int x = 0; x < expected.length; x++) {
      expect(expected[x], outArray[x]);
    }
    // Numbers are from http://www.swetake.com/qr/qr8.html
    dataBytes = Uint8List.fromList([
      67, 70, 22, 38, 54, 70, 86, 102, 118, 134, 150, 166, 182, //
      198, 214, 230, 247, 7, 23, 39, 55, 71, 87, 103, 119, 135,
      151, 166, 22, 38, 54, 70, 86, 102, 118, 134, 150, 166,
      182, 198, 214, 230, 247, 7, 23, 39, 55, 71, 87, 103, 119,
      135, 151, 160, 236, 17, 236, 17, 236, 17, 236,
      17,
    ]);
    inArr = BitArray();
    for (int dataByte in dataBytes) {
      inArr.appendBits(dataByte, 8);
    }

    out = Encoder.interleaveWithECBytes(inArr, 134, 62, 4);
    expected = bytes([
      // Data bytes.
      67, 230, 54, 55, 70, 247, 70, 71, 22, 7, 86, 87, 38, 23, 102, 103,
      54, 39, 118, 119, 70, 55, 134, 135, 86, 71, 150, 151, 102, 87, 166,
      160, 118, 103, 182, 236, 134, 119, 198, 17, 150,
      135, 214, 236, 166, 151, 230, 17, 182,
      166, 247, 236, 198, 22, 7, 17, 214, 38, 23, 236, 39,
      17,
      // Error correction bytes.
      175, 155, 245, 236, 80, 146, 56, 74, 155, 165,
      133, 142, 64, 183, 132, 13, 178, 54, 132, 108, 45,
      113, 53, 50, 214, 98, 193, 152, 233, 147, 50, 71, 65,
      190, 82, 51, 209, 199, 171, 54, 12, 112, 57, 113, 155, 117,
      211, 164, 117, 30, 158, 225, 31, 190, 242, 38,
      140, 61, 179, 154, 214, 138, 147, 87, 27, 96, 77, 47,
      187, 49, 156, 214,
    ]);
    expect(expected.length, out.sizeInBytes);
    outArray = Uint8List(expected.length);
    out.toBytes(0, outArray, 0, expected.length);
    for (int x = 0; x < expected.length; x++) {
      expect(expected[x], outArray[x]);
    }
  });

  test('testAppendNumericBytes', () {
    // 1 = 01 = 0001 in 4 bits.
    BitArray bits = BitArray();
    Encoder.appendNumericBytes('1', bits);
    expect(' ...X', bits.toString());
    // 12 = 0xc = 0001100 in 7 bits.
    bits = BitArray();
    Encoder.appendNumericBytes('12', bits);
    expect(' ...XX..', bits.toString());
    // 123 = 0x7b = 0001111011 in 10 bits.
    bits = BitArray();
    Encoder.appendNumericBytes('123', bits);
    expect(' ...XXXX. XX', bits.toString());
    // 1234 = "123" + "4" = 0001111011 + 0100
    bits = BitArray();
    Encoder.appendNumericBytes('1234', bits);
    expect(' ...XXXX. XX.X..', bits.toString());
    // Empty.
    bits = BitArray();
    Encoder.appendNumericBytes('', bits);
    expect('', bits.toString());
  });

  test('testAppendAlphanumericBytes', () {
    // A = 10 = 0xa = 001010 in 6 bits
    BitArray bits = BitArray();
    Encoder.appendAlphanumericBytes('A', bits);
    expect(' ..X.X.', bits.toString());
    // AB = 10 * 45 + 11 = 461 = 0x1cd = 00111001101 in 11 bits
    bits = BitArray();
    Encoder.appendAlphanumericBytes('AB', bits);
    expect(' ..XXX..X X.X', bits.toString());
    // ABC = "AB" + "C" = 00111001101 + 001100
    bits = BitArray();
    Encoder.appendAlphanumericBytes('ABC', bits);
    expect(' ..XXX..X X.X..XX. .', bits.toString());
    // Empty.
    bits = BitArray();
    Encoder.appendAlphanumericBytes('', bits);
    expect('', bits.toString());

    expect(
      () => Encoder.appendAlphanumericBytes('abc', BitArray()),
      throwsA(
        TypeMatcher<WriterException>(),
      ),
    );
  });

  test('testAppend8BitBytes', () {
    // 0x61, 0x62, 0x63
    BitArray bits = BitArray();
    Encoder.append8BitBytes('abc', bits, Encoder.defaultByteModeEncoding);
    expect(' .XX....X .XX...X. .XX...XX', bits.toString());
    // Empty.
    bits = BitArray();
    Encoder.append8BitBytes('', bits, Encoder.defaultByteModeEncoding);
    expect('', bits.toString());
  });

  // Numbers are from page 21 of JISX0510:2004
  test('testAppendKanjiBytes', () {
    final bits = BitArray();
    Encoder.appendKanjiBytes(shiftJISString(bytes([0x93, 0x5f])), bits);
    expect(' .XX.XX.. XXXXX', bits.toString());
    Encoder.appendKanjiBytes(shiftJISString(bytes([0xe4, 0xaa])), bits);
    expect(' .XX.XX.. XXXXXXX. X.X.X.X. X.', bits.toString());
  });

  // Numbers are from http://www.swetake.com/qr/qr3.html and
  // http://www.swetake.com/qr/qr9.html
  test('testGenerateECBytes', () {
    Uint8List dataBytes = bytes([32, 65, 205, 69, 41, 220, 46, 128, 236]);
    Uint8List ecBytes = Encoder.generateECBytes(dataBytes, 17);
    List<int> expected = [
      42, 159, 74, 221, 244, 169, 239, 150, 138, //
      70, 237, 85, 224, 96, 74, 219, 61,
    ];
    expect(expected.length, ecBytes.length);
    for (int x = 0; x < expected.length; x++) {
      expect(expected[x], ecBytes[x] & 0xFF);
    }
    dataBytes = bytes(
      [67, 70, 22, 38, 54, 70, 86, 102, 118, 134, 150, 166, 182, 198, 214],
    );
    ecBytes = Encoder.generateECBytes(dataBytes, 18);
    expected = [
      175, 80, 155, 64, 178, 45, 214, 233, 65, //
      209, 12, 155, 117, 31, 140, 214, 27, 187,
    ];

    expect(expected.length, ecBytes.length);
    for (int x = 0; x < expected.length; x++) {
      expect(expected[x], ecBytes[x] & 0xFF);
    }
    // High-order zero coefficient case.
    dataBytes = bytes([32, 49, 205, 69, 42, 20, 0, 236, 17]);
    ecBytes = Encoder.generateECBytes(dataBytes, 17);
    expected = [
      0, 3, 130, 179, 194, 0, 55, 211, 110, //
      79, 98, 72, 170, 96, 211, 137, 213,
    ];
    expect(expected.length, ecBytes.length);
    for (int x = 0; x < expected.length; x++) {
      expect(expected[x], ecBytes[x] & 0xFF);
    }
  });

  test('testBugInBitVectorNumBytes', () {
    // There was a bug in BitVector.sizeInBytes() that caused it to return a
    // smaller-by-one value (ex. 1465 instead of 1466) if the number of bits
    // in the vector is not 8-bit aligned.  In QRCodeEncoder::InitQRCode(),
    // BitVector::sizeInBytes() is used for finding the smallest QR Code
    // version that can fit the given data.  Hence there were corner cases
    // where we chose a wrong QR Code version that cannot fit the given
    // data.  Note that the issue did not occur with MODE_8BIT_BYTE, as the
    // bits in the bit vector are always 8-bit aligned.
    //
    // Before the bug was fixed, the following test didn't pass, because:
    //
    // - MODE_NUMERIC is chosen as all bytes in the data are '0'
    // - The 3518-byte numeric data needs 1466 bytes
    //   - 3518 / 3 * 10 + 7 = 11727 bits = 1465.875 bytes
    //   - 3 numeric bytes are encoded in 10 bits, hence the first
    //     3516 bytes are encoded in 3516 / 3 * 10 = 11720 bits.
    //   - 2 numeric bytes can be encoded in 7 bits, hence the last
    //     2 bytes are encoded in 7 bits.
    // - The version 27 QR Code with the EC level L has 1468 bytes for data.
    //   - 1828 - 360 = 1468
    // - In InitQRCode(), 3 bytes are reserved for a header.  Hence 1465 bytes
    //   (1468 -3) are left for data.
    // - Because of the bug in BitVector::sizeInBytes(), InitQRCode() determines
    //   the given data can fit in 1465 bytes, despite it needs 1466 bytes.
    // - Hence QRCodeEncoder.encode() failed and returned false.
    //   - To be precise, it needs 11727 + 4 (getMode info) + 14 (length info) =
    //     11745 bits = 1468.125 bytes are needed (i.e. cannot fit in 1468
    //     bytes).
    final builder = StringBuilder();
    for (int x = 0; x < 3518; x++) {
      builder.write('0');
    }
    Encoder.encode(builder.toString(), ErrorCorrectionLevel.L);
  });

  test('testMinimalEncoder1', () {
    verifyMinimalEncoding('A', 'ALPHANUMERIC(A)', null, false);
  });

  test('testMinimalEncoder2', () {
    verifyMinimalEncoding('AB', 'ALPHANUMERIC(AB)', null, false);
  });

  test('testMinimalEncoder3', () {
    verifyMinimalEncoding('ABC', 'ALPHANUMERIC(ABC)', null, false);
  });

  test('testMinimalEncoder4', () {
    verifyMinimalEncoding('ABCD', 'ALPHANUMERIC(ABCD)', null, false);
  });

  test('testMinimalEncoder5', () {
    verifyMinimalEncoding('ABCDE', 'ALPHANUMERIC(ABCDE)', null, false);
  });

  test('testMinimalEncoder6', () {
    verifyMinimalEncoding('ABCDEF', 'ALPHANUMERIC(ABCDEF)', null, false);
  });

  test('testMinimalEncoder7', () {
    verifyMinimalEncoding('ABCDEFG', 'ALPHANUMERIC(ABCDEFG)', null, false);
  });

  test('testMinimalEncoder8', () {
    verifyMinimalEncoding('1', 'NUMERIC(1)', null, false);
  });

  test('testMinimalEncoder9', () {
    verifyMinimalEncoding('12', 'NUMERIC(12)', null, false);
  });

  test('testMinimalEncoder10', () {
    verifyMinimalEncoding('123', 'NUMERIC(123)', null, false);
  });

  test('testMinimalEncoder11', () {
    verifyMinimalEncoding('1234', 'NUMERIC(1234)', null, false);
  });

  test('testMinimalEncoder12', () {
    verifyMinimalEncoding('12345', 'NUMERIC(12345)', null, false);
  });

  test('testMinimalEncoder13', () {
    verifyMinimalEncoding('123456', 'NUMERIC(123456)', null, false);
  });

  test('testMinimalEncoder14', () {
    verifyMinimalEncoding('123A', 'ALPHANUMERIC(123A)', null, false);
  });

  test('testMinimalEncoder15', () {
    verifyMinimalEncoding('A1', 'ALPHANUMERIC(A1)', null, false);
  });

  test('testMinimalEncoder16', () {
    verifyMinimalEncoding('A12', 'ALPHANUMERIC(A12)', null, false);
  });

  test('testMinimalEncoder17', () {
    verifyMinimalEncoding('A123', 'ALPHANUMERIC(A123)', null, false);
  });

  test('testMinimalEncoder18', () {
    verifyMinimalEncoding('A1234', 'ALPHANUMERIC(A1234)', null, false);
  });

  test('testMinimalEncoder19', () {
    verifyMinimalEncoding('A12345', 'ALPHANUMERIC(A12345)', null, false);
  });

  test('testMinimalEncoder20', () {
    verifyMinimalEncoding('A123456', 'ALPHANUMERIC(A123456)', null, false);
  });

  test('testMinimalEncoder21', () {
    verifyMinimalEncoding('A1234567', 'ALPHANUMERIC(A1234567)', null, false);
  });

  test('testMinimalEncoder22', () {
    verifyMinimalEncoding(
      'A12345678',
      'BYTE(A),NUMERIC(12345678)',
      null,
      false,
    );
  });

  test('testMinimalEncoder23', () {
    verifyMinimalEncoding(
      'A123456789',
      'BYTE(A),NUMERIC(123456789)',
      null,
      false,
    );
  });

  test('testMinimalEncoder24', () {
    verifyMinimalEncoding(
      'A1234567890',
      'ALPHANUMERIC(A1),NUMERIC(234567890)',
      null,
      false,
    );
  });

  test('testMinimalEncoder25', () {
    verifyMinimalEncoding('AB1', 'ALPHANUMERIC(AB1)', null, false);
  });

  test('testMinimalEncoder26', () {
    verifyMinimalEncoding('AB12', 'ALPHANUMERIC(AB12)', null, false);
  });

  test('testMinimalEncoder27', () {
    verifyMinimalEncoding('AB123', 'ALPHANUMERIC(AB123)', null, false);
  });

  test('testMinimalEncoder28', () {
    verifyMinimalEncoding('AB1234', 'ALPHANUMERIC(AB1234)', null, false);
  });

  test('testMinimalEncoder29', () {
    verifyMinimalEncoding('ABC1', 'ALPHANUMERIC(ABC1)', null, false);
  });

  test('testMinimalEncoder30', () {
    verifyMinimalEncoding('ABC12', 'ALPHANUMERIC(ABC12)', null, false);
  });

  test('testMinimalEncoder31', () {
    verifyMinimalEncoding('ABC1234', 'ALPHANUMERIC(ABC1234)', null, false);
  });

  test('testMinimalEncoder32', () {
    verifyMinimalEncoding(
      'http://foo.com',
      'BYTE(http://foo.com)',
      null,
      false,
    );
  });

  test('testMinimalEncoder33', () {
    verifyMinimalEncoding(
      'HTTP://FOO.COM',
      'ALPHANUMERIC(HTTP://FOO.COM' ')',
      null,
      false,
    );
  });

  test('testMinimalEncoder34', () {
    verifyMinimalEncoding(
      '1001114670010%01201220%107211220%140045003267781',
      'NUMERIC(1001114670010),ALPHANUMERIC(%01201220%107211220%),NUMERIC(140045003267781)',
      null,
      false,
    );
  });

  test('testMinimalEncoder35', () {
    verifyMinimalEncoding('\u0150', 'ECI(ISO-8859-2),BYTE(.)', null, false);
  });

  test('testMinimalEncoder36', () {
    verifyMinimalEncoding('\u015C', 'ECI(ISO-8859-3),BYTE(.)', null, false);
  });

  test('testMinimalEncoder37', () {
    verifyMinimalEncoding('\u0150\u015C', 'ECI(UTF-8),BYTE(..)', null, false);
  });

  test('testMinimalEncoder38', () {
    verifyMinimalEncoding(
      '\u0150\u0150\u015C\u015C',
      'ECI(ISO-8859-2),BYTE(..),ECI(ISO-8859-3),BYTE(..)',
      null,
      false,
    );
  });

  test('testMinimalEncoder39', () {
    verifyMinimalEncoding(
      'abcdef\u0150ghij',
      'ECI(ISO-8859-2),BYTE(abcdef.ghij)',
      null,
      false,
    );
  });

  test('testMinimalEncoder40', () {
    verifyMinimalEncoding(
      '2938928329832983\u01502938928329832983\u015C2938928329832983',
      'NUMERIC(2938928329832983),ECI(ISO-8859-2),BYTE(.),NUMERIC(2938928329832983),ECI(ISO-8'
          '859-3),BYTE(.),NUMERIC(2938928329832983)',
      null,
      false,
    );
  });

  test('testMinimalEncoder41', () {
    verifyMinimalEncoding(
      '1001114670010%01201220%107211220%140045003267781',
      'FNC1_FIRST_POSITION(),NUMERIC(100111'
          '4670010),ALPHANUMERIC(%01201220%107211220%),NUMERIC(140045003267781)',
      null,
      true,
    );
  });

  test('testMinimalEncoder42', () {
    // test halfwidth Katakana character (they are single byte encoded in Shift_JIS)
    verifyMinimalEncoding(
      'Katakana:\uFF66\uFF66\uFF66\uFF66\uFF66\uFF66',
      'ECI(shift-jis),BYTE(Katakana:......)',
      null,
      false,
    );
  });

  test('testMinimalEncoder43', () {
    // The character \u30A2 encodes as double byte in Shift_JIS so KANJI is more compact in this case
    verifyMinimalEncoding(
      'Katakana:\u30A2\u30A2\u30A2\u30A2\u30A2\u30A2',
      'BYTE(Katakana:),KANJI(......)',
      null,
      false,
    );
  });

  test('testMinimalEncoder44', () {
    // The character \u30A2 encodes as double byte in Shift_JIS but KANJI is not more compact in this case because
    // KANJI is only more compact when it encodes pairs of characters. In the case of mixed text it can however be
    // that Shift_JIS encoding is more compact as in this example
    verifyMinimalEncoding(
      'Katakana:\u30A2a\u30A2a\u30A2a\u30A2a\u30A2a\u30A2',
      'ECI(shift-jis),BYTE(Katakana:.a.a.a' '.a.a.)',
      null,
      false,
    );
  });
}

Uint8List bytes(List<int> ints) {
  return Uint8List.fromList(ints);
}

void verifyMinimalEncoding(
  String input,
  String expectedResult,
  Encoding? priorityCharset,
  bool isGS1,
) {
  final result = MinimalEncoder.encode(
    input,
    null,
    priorityCharset,
    isGS1,
    ErrorCorrectionLevel.L,
  );
  expect(result.toString(), expectedResult);
}

void verifyGS1EncodedData(QRCode qrCode) {
  final expected = '<<\n'
      ' mode: ALPHANUMERIC\n'
      ' ecLevel: H\n'
      ' version: 2\n'
      ' maskPattern: 2\n'
      ' matrix:\n'
      ' 1 1 1 1 1 1 1 0 1 0 1 1 1 1 0 1 1 0 1 1 1 1 1 1 1\n'
      ' 1 0 0 0 0 0 1 0 1 0 0 0 0 1 1 0 1 0 1 0 0 0 0 0 1\n'
      ' 1 0 1 1 1 0 1 0 1 0 1 1 0 1 1 0 0 0 1 0 1 1 1 0 1\n'
      ' 1 0 1 1 1 0 1 0 0 1 1 0 1 0 1 1 1 0 1 0 1 1 1 0 1\n'
      ' 1 0 1 1 1 0 1 0 0 1 1 1 1 1 1 1 1 0 1 0 1 1 1 0 1\n'
      ' 1 0 0 0 0 0 1 0 1 0 0 1 1 1 0 0 0 0 1 0 0 0 0 0 1\n'
      ' 1 1 1 1 1 1 1 0 1 0 1 0 1 0 1 0 1 0 1 1 1 1 1 1 1\n'
      ' 0 0 0 0 0 0 0 0 1 1 1 0 0 0 1 1 1 0 0 0 0 0 0 0 0\n'
      ' 0 0 1 1 1 0 1 0 1 1 1 1 0 1 1 0 1 1 1 1 0 0 1 1 1\n'
      ' 0 0 0 1 1 1 0 1 0 0 1 0 0 1 0 0 1 1 1 0 0 1 0 0 1\n'
      ' 1 0 1 1 0 0 1 0 1 1 0 0 0 0 1 0 1 1 1 0 0 1 0 0 1\n'
      ' 0 0 1 1 0 1 0 1 1 1 1 0 0 1 1 1 1 0 0 0 1 1 0 1 1\n'
      ' 0 0 1 0 0 0 1 0 0 0 1 1 0 1 0 0 0 1 0 1 1 1 0 1 0\n'
      ' 1 1 1 0 1 1 0 1 0 0 0 0 0 0 0 1 1 0 1 1 0 1 0 0 0\n'
      ' 1 0 1 0 1 0 1 1 0 1 0 1 0 1 1 0 0 0 0 0 1 1 0 0 1\n'
      ' 1 0 0 1 0 1 0 1 0 0 0 1 1 1 1 0 1 0 1 0 0 1 0 0 1\n'
      ' 1 0 1 0 0 1 1 1 0 1 1 0 0 1 0 0 1 1 1 1 1 1 0 0 0\n'
      ' 0 0 0 0 0 0 0 0 1 0 0 1 0 1 1 0 1 0 0 0 1 0 0 1 0\n'
      ' 1 1 1 1 1 1 1 0 0 0 0 1 0 0 1 1 1 0 1 0 1 0 1 1 1\n'
      ' 1 0 0 0 0 0 1 0 0 1 1 1 1 1 0 1 1 0 0 0 1 0 0 0 1\n'
      ' 1 0 1 1 1 0 1 0 1 0 1 0 0 1 1 1 1 1 1 1 1 0 0 0 1\n'
      ' 1 0 1 1 1 0 1 0 1 1 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0\n'
      ' 1 0 1 1 1 0 1 0 1 0 0 0 1 1 0 1 0 0 1 1 1 0 1 0 1\n'
      ' 1 0 0 0 0 0 1 0 0 1 0 1 0 1 1 1 0 1 0 0 1 1 1 1 1\n'
      ' 1 1 1 1 1 1 1 0 0 1 1 0 0 1 1 0 1 0 0 0 0 1 0 1 1\n'
      '>>\n';
  expect(expected, qrCode.toString());
}

void verifyNotGS1EncodedData(QRCode qrCode) {
  final expected = '<<\n'
      ' mode: ALPHANUMERIC\n'
      ' ecLevel: H\n'
      ' version: 1\n'
      ' maskPattern: 0\n'
      ' matrix:\n'
      ' 1 1 1 1 1 1 1 0 1 1 1 1 0 0 1 1 1 1 1 1 1\n'
      ' 1 0 0 0 0 0 1 0 0 1 1 1 0 0 1 0 0 0 0 0 1\n'
      ' 1 0 1 1 1 0 1 0 0 1 0 1 1 0 1 0 1 1 1 0 1\n'
      ' 1 0 1 1 1 0 1 0 1 1 1 0 1 0 1 0 1 1 1 0 1\n'
      ' 1 0 1 1 1 0 1 0 0 1 1 1 0 0 1 0 1 1 1 0 1\n'
      ' 1 0 0 0 0 0 1 0 0 1 0 0 0 0 1 0 0 0 0 0 1\n'
      ' 1 1 1 1 1 1 1 0 1 0 1 0 1 0 1 1 1 1 1 1 1\n'
      ' 0 0 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0 0 0 0 0\n'
      ' 0 0 1 0 1 1 1 0 1 1 0 0 1 1 0 0 0 1 0 0 1\n'
      ' 1 0 1 1 1 0 0 1 0 0 0 1 0 1 0 0 0 0 0 0 0\n'
      ' 0 0 1 1 0 0 1 0 1 0 0 0 1 0 1 0 1 0 1 1 0\n'
      ' 1 1 0 1 0 1 0 1 1 1 0 1 0 1 0 0 0 0 0 1 0\n'
      ' 0 0 1 1 0 1 1 1 1 0 0 0 1 0 1 0 1 1 1 1 0\n'
      ' 0 0 0 0 0 0 0 0 1 0 0 1 1 1 0 1 0 1 0 0 0\n'
      ' 1 1 1 1 1 1 1 0 0 0 1 0 1 0 1 1 0 0 0 0 1\n'
      ' 1 0 0 0 0 0 1 0 1 1 1 1 0 1 0 1 1 1 1 0 1\n'
      ' 1 0 1 1 1 0 1 0 1 0 1 1 0 1 0 1 0 0 0 0 1\n'
      ' 1 0 1 1 1 0 1 0 0 1 1 0 1 1 1 1 0 1 0 1 0\n'
      ' 1 0 1 1 1 0 1 0 1 0 0 0 1 0 1 0 1 1 1 0 1\n'
      ' 1 0 0 0 0 0 1 0 0 1 1 0 1 1 0 1 0 0 0 1 1\n'
      ' 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 1 0 1 0 1\n'
      '>>\n';
  expect(expected, qrCode.toString());
}

String shiftJISString(Uint8List bytes) {
  return StringUtils.shiftJisCharset.decode(bytes);
}
