/*
 * Copyright 2008 ZXing authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/// Represents some type of metadata about the result of the decoding that the decoder
/// wishes to communicate back to the caller.
enum ResultMetadataType {
  /// Unspecified, application-specific metadata. Maps to an unspecified [Object].
  other('OTHER'),

  /// Denotes the likely approximate orientation of the barcode in the image. This value
  /// is given as degrees rotated clockwise from the normal, upright orientation.
  /// For example a 1D barcode which was found by reading top-to-bottom would be
  /// said to have orientation "90". This key maps to an [int] whose
  /// value is in the range [0,360).
  orientation('ORIENTATION'),

  /// <p>2D barcode formats typically encode text, but allow for a sort of 'byte mode'
  /// which is sometimes used to encode binary data. While [Result] makes available
  /// the complete raw bytes in the barcode for these formats, it does not offer the bytes
  /// from the byte segments alone.</p>
  ///
  /// <p>This maps to a {@link java.util.List} of byte arrays corresponding to the
  /// raw bytes in the byte segments in the barcode, in order.</p>
  byteSegments('BYTE_SEGMENTS'),

  /// Error correction level used, if applicable. The value type depends on the
  /// format, but is typically a String.
  errorCorrectionLevel('ERROR_CORRECTION_LEVEL'),

  /// The number of errors corrected. If applicable, maps to an [int] of value
  /// greater than or equal to zero.
  errorsCorrected('ERRORS_CORRECTED'),

  /// The number of erasures corrected. If applicable, maps to an [int] of value
  /// greater than or equal to zero.
  erasuresCorrected('ERASURES_CORRECTED'),

  /// For some periodicals, indicates the issue number as an [int].
  issueNumber('ISSUE_NUMBER'),

  /// For some products, indicates the suggested retail price in the barcode as a
  /// formatted [String].
  suggestedPrice('SUGGESTED_PRICE'),

  /// For some products, the possible country of manufacture as a [String] denoting the
  /// ISO country code. Some map to multiple possible countries, like "US/CA".
  possibleCountry('POSSIBLE_COUNTRY'),

  /// For some products, the extension text.
  upcEanExtension('UPC_EAN_EXTENSION'),

  /// PDF417-specific metadata.
  pdf417ExtraMetadata('PDF417_EXTRA_METADATA'),

  /// If the code format supports structured append and the current scanned code is part of one then the
  /// sequence number is given with it.
  structuredAppendSequence('STRUCTURED_APPEND_SEQUENCE'),

  /// If the code format supports structured append and the current scanned code is part of one then the
  /// parity is given with it.
  structuredAppendParity('STRUCTURED_APPEND_PARITY'),

  /// Barcode Symbology Identifier.
  /// Note: According to the GS1 specification the identifier may have to replace a leading FNC1/GS character when prepending to the barcode content.
  symbologyIdentifier('SYMBOLOGY_IDENTIFIER');

  final String identifier;
  const ResultMetadataType(this.identifier);
}
