/*
 * Copyright (c) 2019 Hocuri
 *
 * This file is part of SuperFreezZ.
 *
 * SuperFreezZ is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * SuperFreezZ is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SuperFreezZ.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package superfreeze.tool.android.userInterface.settingsActivity

import android.annotation.TargetApi
import android.app.AlertDialog
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.Intent
import android.content.res.Configuration
import android.media.RingtoneManager
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.preference.*
import android.provider.Settings
import android.text.TextUtils
import android.util.Log
import android.view.MenuItem
import android.widget.Toast
import superfreeze.tool.android.BuildConfig
import superfreeze.tool.android.R
import superfreeze.tool.android.backend.FreezerService
import superfreeze.tool.android.backend.usageStatsPermissionGranted
import superfreeze.tool.android.userInterface.intro.INTRO_SHOW_ACCESSIBILITY_SERVICE_CHOOSER
import superfreeze.tool.android.userInterface.intro.IntroActivity
import superfreeze.tool.android.userInterface.showUsageStatsSettings
import java.io.BufferedReader
import java.io.IOException
import java.io.InputStreamReader

const val MAX_LOG_LENGTH = 100000
private const val TAG = "SF-SettingsActivity"

/**
 * A [PreferenceActivity] that presents a set of application settings. On
 * handset devices, settings are presented as a single list. On tablets,
 * settings are split by category, with category headers shown to the left of
 * the list of settings.
 *
 * See [Android Design: Settings](http://developer.android.com/design/patterns/settings.html)
 * for design guidelines and the [Settings API Guide](http://developer.android.com/guide/topics/ui/settings.html)
 * for more information on developing a Settings UI.
 */
class SettingsActivity : AppCompatPreferenceActivity() {

	override fun onCreate(savedInstanceState: Bundle?) {
		super.onCreate(savedInstanceState)
		setupActionBar()
	}

	/**
	 * Set up the [android.app.ActionBar], if the API is available.
	 */
	private fun setupActionBar() {
		supportActionBar?.setDisplayHomeAsUpEnabled(true)
	}

	override fun onMenuItemSelected(featureId: Int, item: MenuItem): Boolean {
		val id = item.itemId
		if (id == android.R.id.home) {
			if (!super.onMenuItemSelected(featureId, item)) {
				finishAndRestartMain()
			}
			return true
		}
		return super.onMenuItemSelected(featureId, item)
	}

	private fun finishAndRestartMain() {
		// val intent = Intent(this, MainActivity::class.java)
		// intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK)
		// startActivity(intent)
		// Main will recreate itself
		finish()
		// This was:
		// NavUtils.navigateUpFromSameTask(this)
		// On newer Android versions the following might be possible:
		// navigateUpTo(Intent(this, MainActivity::class.java))
	}

	/**
	 * {@inheritDoc}
	 */
	override fun onIsMultiPane(): Boolean {
		return isXLargeTablet(this)
	}

	/**
	 * {@inheritDoc}
	 */
	@TargetApi(Build.VERSION_CODES.HONEYCOMB)
	override fun onBuildHeaders(target: List<Header>) {
		loadHeadersFromResource(R.xml.pref_headers, target)
	}

	/**
	 * This method stops fragment injection in malicious applications.
	 * Make sure to deny any unknown fragments here.
	 */
	override fun isValidFragment(fragmentName: String): Boolean {
		return PreferenceFragment::class.java.name == fragmentName
				|| AppsListPreferenceFragment::class.java.name == fragmentName
				|| AboutPreferenceFragment::class.java.name == fragmentName
				|| FreezingAppsPreferenceFragment::class.java.name == fragmentName
	}


	open class MyPreferenceFragment : PreferenceFragment() {
		final override fun onOptionsItemSelected(item: MenuItem): Boolean {
			val id = item.itemId
			if (id == android.R.id.home) {
				/*val intent = Intent(activity, SettingsActivity::class.java)
				intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TOP)
				activity.startActivity(intent)
				activity.finish()*/
				activity.onBackPressed()
				return true
			}
			return super.onOptionsItemSelected(item)
		}
	}

	@TargetApi(Build.VERSION_CODES.HONEYCOMB)
	class AppsListPreferenceFragment : MyPreferenceFragment() {
		override fun onCreate(savedInstanceState: Bundle?) {
			super.onCreate(savedInstanceState)
			addPreferencesFromResource(R.xml.pref_appslist)
			setHasOptionsMenu(true)

			// Bind the summaries of EditText/List/Dialog/Ringtone preferences
			// to their values. When their values change, their summaries are
			// updated to reflect the new value, per the Android Design
			// guidelines.
			bindPreferenceSummaryToValue(findPreference("standard_freeze_mode"))
			bindPreferenceSummaryToValue(findPreference("standard_freeze_mode_system"))
		}
	}

	@TargetApi(Build.VERSION_CODES.HONEYCOMB)
	class FreezingAppsPreferenceFragment : MyPreferenceFragment() {
		private lateinit var useAccessibilityServicePreference: SwitchPreference
		private lateinit var useUsagestatsPreference: SwitchPreference
		override fun onCreate(savedInstanceState: Bundle?) {
			super.onCreate(savedInstanceState)
			addPreferencesFromResource(R.xml.pref_freezing)
			setHasOptionsMenu(true)

			bindPreferenceSummaryToValue(findPreference("autofreeze_delay"))

			useAccessibilityServicePreference =
				findPreference("use_accessibility_service") as SwitchPreference
			useAccessibilityServicePreference.setOnPreferenceClickListener {
				if (FreezerService.isEnabled) {
					Toast.makeText(
						activity,
						"Please disable accessibility service for SuperFreezZ",
						Toast.LENGTH_LONG
					).show()
					val intent = Intent(Settings.ACTION_ACCESSIBILITY_SETTINGS)
					startActivity(intent)
				} else {
					val intent = Intent(INTRO_SHOW_ACCESSIBILITY_SERVICE_CHOOSER)
					intent.setClass(activity, IntroActivity::class.java)
					startActivity(intent)
				}
				false // Do not change the preference yet
			}

			useUsagestatsPreference = findPreference("use_usagestats") as SwitchPreference
			if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
				useUsagestatsPreference.setOnPreferenceClickListener {
					showUsageStatsSettings(it.context, show_toast = useUsagestatsPreference.isChecked)

					false
				}
			} else {
				useUsagestatsPreference.isEnabled = false
			}

			if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
				findPreference("freeze_on_screen_off").setOnPreferenceChangeListener { _, newValue ->
					if (newValue == true)
						AlertDialog.Builder(context)
							.setTitle(R.string.warning)
							.setMessage(getString(R.string.warning_long_1) + "\n\n" + getString(R.string.warning_long_2))
							.setCancelable(false)
							.setPositiveButton(android.R.string.ok) { _, _ ->

								if (!Settings.System.canWrite(context)) {
									AlertDialog.Builder(context)
										.setTitle("Modify settings")
										.setMessage("SuperFreezZ needs to modify the settings in order to turn off the screen after freezing.")
										.setCancelable(false)
										.setPositiveButton("Ok") { _, _ ->
											val intent =
												Intent(Settings.ACTION_MANAGE_WRITE_SETTINGS)
											intent.data =
												Uri.parse("package:${BuildConfig.APPLICATION_ID}")
											startActivity(intent)
										}
										.show()
								}

							}
							.show()

					true
				}
			}
		}

		override fun onResume() {
			super.onResume()
			updatePreferenceStates()
		}

		private fun updatePreferenceStates() {
			useAccessibilityServicePreference.isChecked = FreezerService.isEnabled
			useUsagestatsPreference.isChecked = usageStatsPermissionGranted(activity)
		}
	}

	@TargetApi(Build.VERSION_CODES.HONEYCOMB)
	class AboutPreferenceFragment : SettingsActivity.MyPreferenceFragment() {
		override fun onCreate(savedInstanceState: Bundle?) {
			super.onCreate(savedInstanceState)
			addPreferencesFromResource(R.xml.pref_about)
			setHasOptionsMenu(true)
			activity.title =
				getString(R.string.app_name) + " " + getString(R.string.version_abbrev) + BuildConfig.VERSION_NAME

			findPreference("send_logs").setOnPreferenceClickListener {
				//Share info about the exception so that it can be viewed or sent to someone else
				val sharingIntent = Intent(Intent.ACTION_SEND)
				sharingIntent.type = "text/plain"
				sharingIntent.putExtra(Intent.EXTRA_EMAIL, arrayOf("superfreezz-automated@gmx.de"))
				val logs = getLogs()
				sharingIntent.putExtra(Intent.EXTRA_TEXT, logs)
				startActivity(Intent.createChooser(sharingIntent, getString(R.string.share_logs)))
				if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
					// Copy to clipboard:
					Toast.makeText(
						context ?: activity,
						getString(R.string.logs_copied),
						Toast.LENGTH_LONG
					).show()
					val clipboard =
						context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager?
					val clip = ClipData.newPlainText("logs", logs)
					clipboard?.setPrimaryClip(clip)
				}
				true
			}

			findPreference("view_intro_again").setOnPreferenceClickListener {
				startActivity(Intent(activity, IntroActivity::class.java))
				true
			}
		}

		private fun getLogs(): String {
			val logs = try {
				val process = Runtime.getRuntime().exec("logcat -d")
				val bufferedReader = BufferedReader(
					InputStreamReader(process.inputStream)
				)

				"Version " + BuildConfig.VERSION_NAME + bufferedReader.use { it.readText() }

			} catch (e: IOException) {
				Log.e(TAG, "Could not get logs (???)")
				""
			}

			return if (logs.length > MAX_LOG_LENGTH) { // An intent must not contain more than 1MB
				"Truncated\n" + logs.substring(logs.length - MAX_LOG_LENGTH, logs.length - 1)
			} else logs

		}
	}


	companion object {

		/**
		 * A preference value change listener that updates the preference's summary
		 * to reflect its new value.
		 */
		private val sBindPreferenceSummaryToValueListener =
			Preference.OnPreferenceChangeListener { preference, value ->
				val stringValue = value.toString()

				if (preference is ListPreference) {
					// For list preferences, look up the correct display value in
					// the preference's 'entries' list.
					val index = preference.findIndexOfValue(stringValue)

					// Set the summary to reflect the new value.
					preference.setSummary(
						if (index >= 0)
							preference.entries[index]
						else
							null
					)

				} else {
					// For all other preferences, set the summary to the value's
					// simple string representation.
					preference.summary = stringValue
				}
				true
			}

		/**
		 * Helper method to determine if the device has an extra-large screen. For
		 * example, 10" tablets are extra-large.
		 */
		private fun isXLargeTablet(context: Context): Boolean {
			return context.resources.configuration.screenLayout and Configuration.SCREENLAYOUT_SIZE_MASK >= Configuration.SCREENLAYOUT_SIZE_XLARGE
		}

		/**
		 * Binds a preference's summary to its value. More specifically, when the
		 * preference's value is changed, its summary (line of text below the
		 * preference title) is updated to reflect the value. The summary is also
		 * immediately updated upon calling this method. The exact display format is
		 * dependent on the type of preference.

		 * @see .sBindPreferenceSummaryToValueListener
		 */
		private fun bindPreferenceSummaryToValue(preference: Preference) {
			// Set the listener to watch for value changes.
			preference.onPreferenceChangeListener = sBindPreferenceSummaryToValueListener

			// Trigger the listener immediately with the preference's
			// current value.
			sBindPreferenceSummaryToValueListener.onPreferenceChange(
				preference,
				PreferenceManager
					.getDefaultSharedPreferences(preference.context)
					.getString(preference.key, "")
			)
		}

	}
}
