/*
 * Husky -- A Pleroma client for Android
 *
 * Copyright (C) 2023  The Husky Developers
 * Copyright (C) 2019  Conny Duck
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package com.keylesspalace.tusky.adapter

import android.content.res.ColorStateList
import android.view.LayoutInflater
import android.view.MotionEvent
import android.view.ViewGroup
import androidx.core.view.size
import androidx.recyclerview.widget.RecyclerView
import androidx.viewbinding.ViewBinding
import com.google.android.material.chip.Chip
import com.keylesspalace.tusky.HASHTAG
import com.keylesspalace.tusky.LIST
import com.keylesspalace.tusky.R
import com.keylesspalace.tusky.TabData
import com.keylesspalace.tusky.databinding.ItemTabPreferenceBinding
import com.keylesspalace.tusky.databinding.ItemTabPreferenceSmallBinding
import com.keylesspalace.tusky.util.BindingViewHolder
import com.keylesspalace.tusky.util.ThemeUtils
import com.keylesspalace.tusky.util.hide
import com.keylesspalace.tusky.util.show

interface ItemInteractionListener {
    fun onTabAdded(tab: TabData)
    fun onTabRemoved(position: Int)
    fun onStartDelete(viewHolder: RecyclerView.ViewHolder)
    fun onStartDrag(viewHolder: RecyclerView.ViewHolder)
    fun onActionChipClicked(tab: TabData, tabPosition: Int)
    fun onChipClicked(tab: TabData, tabPosition: Int, chipPosition: Int)
}

class TabAdapter(
    private var data: List<TabData>,
    private val small: Boolean,
    private val listener: ItemInteractionListener,
    private var removeButtonEnabled: Boolean = false
) : RecyclerView.Adapter<BindingViewHolder<ViewBinding>>() {

    fun updateData(newData: List<TabData>) {
        this.data = newData
        notifyDataSetChanged()
    }

    override fun onCreateViewHolder(
        parent: ViewGroup,
        viewType: Int
    ): BindingViewHolder<ViewBinding> {
        val binding = if (small) {
            ItemTabPreferenceSmallBinding.inflate(
                LayoutInflater.from(parent.context),
                parent,
                false
            )
        } else {
            ItemTabPreferenceBinding.inflate(LayoutInflater.from(parent.context), parent, false)
        }

        return BindingViewHolder(binding)
    }

    override fun onBindViewHolder(holder: BindingViewHolder<ViewBinding>, position: Int) {
        val tab = data[position]

        if (small) {
            val binding = holder.binding as ItemTabPreferenceSmallBinding

            with(binding.textView) {
                setText(tab.text)
                setCompoundDrawablesRelativeWithIntrinsicBounds(tab.icon, 0, 0, 0)
                setOnClickListener {
                    listener.onTabAdded(tab)
                }
            }
        } else {
            val binding = holder.binding as ItemTabPreferenceBinding

            if (tab.id == LIST) {
                binding.textView.text = tab.arguments.getOrNull(1).orEmpty()
            } else {
                binding.textView.setText(tab.text)
            }
            binding.textView.setCompoundDrawablesRelativeWithIntrinsicBounds(tab.icon, 0, 0, 0)

            binding.imageView.setOnTouchListener { _, event ->
                if (event.action == MotionEvent.ACTION_DOWN) {
                    listener.onStartDrag(holder)
                    true
                } else {
                    false
                }
            }

            binding.removeButton.setOnClickListener {
                listener.onTabRemoved(holder.bindingAdapterPosition)
            }
            binding.removeButton.isEnabled = removeButtonEnabled

            ThemeUtils.setDrawableTint(
                binding.root.context,
                binding.removeButton.drawable,
                (if (removeButtonEnabled) android.R.attr.textColorTertiary else R.attr.textColorDisabled)
            )

            if (tab.id == HASHTAG) {
                binding.chipGroup.show()

                /*
                 * The chip group will always contain the actionChip (it is defined in the xml layout).
                 * The other dynamic chips are inserted in front of the actionChip.
                 * This code tries to reuse already added chips to reduce the number of Views created.
                 */
                tab.arguments.forEachIndexed { i, arg ->
                    val chip = binding.chipGroup.getChildAt(i)
                        .takeUnless { it.id == R.id.actionChip } as Chip?
                        ?: Chip(binding.root.context).apply {
                            binding.chipGroup.addView(this, binding.chipGroup.size - 1)
                            chipIconTint = ColorStateList.valueOf(
                                ThemeUtils.getColor(
                                    context,
                                    android.R.attr.textColorPrimary
                                )
                            )
                        }

                    chip.text = arg

                    if (tab.arguments.size <= 1) {
                        chip.chipIcon = null
                        chip.setOnClickListener(null)
                    } else {
                        chip.setChipIconResource(R.drawable.ic_cancel_24dp)
                        chip.setOnClickListener {
                            listener.onChipClicked(tab, holder.adapterPosition, i)
                        }
                    }
                }

                while (binding.chipGroup.size - 1 > tab.arguments.size) {
                    binding.chipGroup.removeViewAt(tab.arguments.size)
                }

                binding.actionChip.setOnClickListener {
                    listener.onActionChipClicked(tab, holder.adapterPosition)
                }
            } else {
                binding.chipGroup.hide()
            }
        }
    }

    override fun getItemCount() = data.size

    fun setRemoveButtonVisible(enabled: Boolean) {
        if (removeButtonEnabled != enabled) {
            removeButtonEnabled = enabled
            notifyDataSetChanged()
        }
    }
}
