/*
 * TheXTech - A platform game engine ported from old source code for VB6
 *
 * Copyright (c) 2009-2011 Andrew Spinks, original VB6 code
 * Copyright (c) 2020-2025 Vitaly Novichkov <admin@wohlnet.ru>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "sdl_proxy/sdl_timer.h"
#include "sdl_proxy/sdl_atomic.h"

#ifndef PGE_NO_THREADING
#include <SDL2/SDL_thread.h>
#endif

#include <fmt_format_ne.h>
#include <array>

#include <AppPath/app_path.h>
#include <DirManager/dirman.h>
#include <Utils/files.h>
#include <PGE_File_Formats/file_formats.h>
#include <Integrator/integrator.h>

#include "menu_main.h"
#include "game_info.h"
#include "../version.h"
#include "../gfx.h"
#include "main/screen_connect.h"
#include "main/screen_options.h"
#include "main/menu_controls.h"
#include "main/translate_episode.h"

#include "speedrunner.h"
#include "main/gameplay_timer.h"
#include "../game_main.h"
#include "../sound.h"
#include "../player.h"
#include "../collision.h"
#include "../graphics.h"
#include "../core/render.h"
#include "../core/window.h"
#include "../core/events.h"
#include "../controls.h"
#include "../config.h"
#include "level_file.h"
#include "world_file.h"
#include "pge_delay.h"
#include "change_res.h"
#include "game_globals.h"
#include "core/language.h"
#include "main/translate.h"
#include "change_res.h"
#include "game_globals.h"
#include "npc_id.h"
#include "fontman/font_manager.h"
#include "custom.h"

#include "editor.h"
#include "frm_main.h"
#include "load_gfx.h"

#include "screen_textentry.h"
#include "main/asset_pack.h"
#include "main/screen_asset_pack.h"
#include "editor/new_editor.h"
#include "editor/write_level.h"
#include "editor/write_world.h"
#include "editor/editor_custom.h"
#include "script/luna/luna.h"

MainMenuContent g_mainMenu;

#ifndef PGE_NO_THREADING
static bool                 s_atomicsInited = false;
static SDL_atomic_t         loading = {};
static SDL_atomic_t         loadingProgrss = {};
static SDL_atomic_t         loadingProgrssMax = {};

static SDL_Thread*          loadingThread = nullptr;
#endif

static constexpr int c_menuSavesLength = maxSaveSlots + 2;
static constexpr int c_menuSavesFooterHint = (c_menuSavesLength * 30) - 30;
static constexpr int c_menuItemSavesEndList = maxSaveSlots - 1;
static constexpr int c_menuItemSavesCopy = maxSaveSlots;
static constexpr int c_menuItemSavesDelete = maxSaveSlots + 1;
static constexpr int c_menuSavesOffsetY = (maxSaveSlots - 3) * 30;

static uint8_t s_episode_playstyle = 0;
static uint8_t s_episode_speedrun_mode = 0;
static bool s_editor_target_thextech = true;

int NumSelectWorld = 0;
int NumSelectWorldEditable = 0;
int NumSelectBattle = 0;
std::vector<SelectWorld_t> SelectWorld;
std::vector<SelectWorld_t> SelectWorldEditable;
std::vector<SelectWorld_t> SelectBattle;


void initMainMenu()
{
#ifndef PGE_NO_THREADING
    if(!s_atomicsInited)
    {
        SDL_AtomicSet(&loading, 0);
        SDL_AtomicSet(&loadingProgrss, 0);
        SDL_AtomicSet(&loadingProgrssMax, 0);
        s_atomicsInited = true;
    }
#endif

    g_mainMenu.introPressStart = "Press Start";

    g_mainMenu.mainPlayEpisode = "Play Episode";
    g_mainMenu.main1PlayerGame = "1 Player Game";
    g_mainMenu.mainMultiplayerGame = "2 Player Game";
    g_mainMenu.mainBattleGame = "Battle Game";
    g_mainMenu.mainEditor = "Editor";
    g_mainMenu.mainOptions = "Options";
    g_mainMenu.mainExit = "Exit";

    g_mainMenu.loading = "Loading...";

    g_mainMenu.languageName = "English";
    g_mainMenu.pluralRules = "one-is-singular";

    g_mainMenu.selectCharacter = "{0} game";

    g_mainMenu.editorBattles = "<Battle Levels>";
    g_mainMenu.editorNewWorld = "<New World>";
    g_mainMenu.editorMakeFor = "Make for:";
    g_mainMenu.editorErrorMissingResources = "Sorry! You are missing {0}, required for the in-game editor.";
    g_mainMenu.editorPromptNewWorldName = "New world name";

    g_mainMenu.gameNoEpisodesToPlay = "<No episodes to play>";
    g_mainMenu.gameNoBattleLevels = "<No battle levels>";
    g_mainMenu.gameBattleRandom = "Random Level";

    g_mainMenu.warnEpCompat = "This episode was made for a different branch of SMBX and may not work properly.";

    g_mainMenu.gameSlotContinue = "SLOT {0} ... {1}%";
    g_mainMenu.gameSlotNew = "SLOT {0} ... NEW GAME";
    g_mainMenu.gameCopySave = "Copy save";
    g_mainMenu.gameEraseSave = "Erase save";
    g_mainMenu.gameSourceSlot = "Select the source slot";
    g_mainMenu.gameTargetSlot = "Now select the target";
    g_mainMenu.gameEraseSlot = "Select the slot to erase";

    g_mainMenu.phraseScore = "Score: {0}";
    g_mainMenu.phraseTime = "Time: {0}";

    g_mainMenu.errorBattleNoLevels = "Can't start battle because of no levels available";

    g_mainMenu.optionsRestartEngine = "Restart engine for changes to take effect.";

    g_mainMenu.wordPlayer = "Player";
    g_mainMenu.wordProfile = "Profile";
    g_mainMenu.wordBack = "Back";
    g_mainMenu.wordResume = "Resume";
    g_mainMenu.wordWaiting = "Waiting";

    g_mainMenu.controlsTitle = "Controls";
    g_mainMenu.controlsConnected = "Connected:";
    g_mainMenu.controlsDeleteKey = "(Alt Run to Delete)";
    g_mainMenu.controlsDeviceTypes = "Device Types";
    g_mainMenu.controlsInUse = "(In Use)";
    g_mainMenu.controlsNotInUse = "(Not In Use)";

    g_mainMenu.controlsActivateProfile = "Activate profile";
    g_mainMenu.controlsRenameProfile = "Rename profile";
    g_mainMenu.controlsDeleteProfile = "Delete profile";
    g_mainMenu.controlsPlayerControls = "Player controls";
    g_mainMenu.controlsCursorControls = "Cursor controls";
    g_mainMenu.controlsEditorControls = "Editor controls";
    g_mainMenu.controlsHotkeys = "Hotkeys";

    g_mainMenu.controlsOptionRumble = "Rumble";
    // g_mainMenu.controlsOptionGroundPoundButton = "Ground Pound Button";
    g_mainMenu.controlsOptionBatteryStatus = "Battery Status";
    g_mainMenu.controlsOptionAltMenuControls = "Alt Menu Controls";

    g_mainMenu.wordProfiles = "Profiles";
    g_mainMenu.wordButtons  = "Buttons";

    g_mainMenu.controlsReallyDeleteProfile = "Really delete profile?";
    g_mainMenu.controlsNewProfile = "<New Profile>";

    g_mainMenu.wordNo   = "No";
    g_mainMenu.wordYes  = "Yes";
    g_mainMenu.wordOkay = "Okay";
    g_mainMenu.caseNone = "<None>";
    g_mainMenu.wordOn   = "On";
    g_mainMenu.wordOff  = "Off";
    g_mainMenu.wordShow = "Show";
    g_mainMenu.wordHide = "Hide";
    g_mainMenu.abbrevMilliseconds = "MS";

    // g_mainMenu.promptDeprecatedSetting = "This file uses a deprecated compatibility flag that will be removed in version 1.3.7.\n\nOld flag: \"{0}\"\nNew flag: \"{1}\"\n\n\nReplace it with the updated flag for version 1.3.6 and newer?";
    // g_mainMenu.promptDeprecatedSettingUnwritable = "An unwritable file ({0}) uses a deprecated compatibility flag that will be removed in version 1.3.7.\n\nSection: [{1}]\nOld flag: \"{2}\"\nNew flag: \"{3}\"\n\n\nPlease update it manually and copy to your device.";
}


static int ScrollDelay = 0;
static int menuPlayersNum = 0;
static int menuBattleMode = false;

static int menuCopySaveSrc = 0;
static int menuCopySaveDst = 0;
static int menuRecentEpisode = -1;

static int listMenuLastScroll = 0;
static int listMenuLastCursor = 0;

static int s_startAssetPackTimer = 0;

static const int TinyScreenH = 400;
static const int SmallScreenH = 500;
static const int TinyScreenW = 600;

static bool s_prefer_modern_char_sel()
{
    return (g_config.compatibility_mode != Config_t::COMPAT_SMBX13) && (s_episode_speedrun_mode != 3);
}

static bool s_show_separate_2P()
{
    return g_config.compatibility_mode == Config_t::COMPAT_SMBX13 && !g_gameInfo.disableTwoPlayer;
}

// export it, let's hope for some nice LTO here
int mainMenuPlaystyle()
{
    return s_episode_playstyle;
}

static void s_change_save_item()
{
    if(MenuCursor < 0 || MenuCursor >= maxSaveSlots)
        return;

    int save_configs = SaveSlotInfo[MenuCursor + 1].ConfigDefaults;

    if(save_configs > 0)
    {
        s_episode_playstyle = save_configs - 1;
        s_episode_speedrun_mode = 0;
    }
    else if(save_configs < 0)
    {
        s_episode_speedrun_mode = -save_configs;
        s_episode_playstyle = s_episode_speedrun_mode - 1;
    }
    else
    {
        s_episode_playstyle = SelectWorld[selWorld].bugfixes_on_by_default ? Config_t::MODE_MODERN : Config_t::MODE_CLASSIC;
        s_episode_speedrun_mode = g_config.speedrun_mode.m_value;
    }
}

static void s_draw_infobox_switch_arrows(int infobox_x, int infobox_y)
{
    // XRender::renderRect(infobox_x, infobox_y, 480, 68, {0, 0, 0, 192});

    if(CommonFrame % 90 < 45)
        return;

    if(GFX.CharSelIcons.inited)
    {
        XRender::renderTextureFL(infobox_x + 8, infobox_y + 34 - 24 / 2, 24, 24, GFX.CharSelIcons, 72, 0, 0.0, nullptr, X_FLIP_HORIZONTAL);
        XRender::renderTexture(infobox_x + 480 - 8 - 24, infobox_y + 34 - 24 / 2, 24, 24, GFX.CharSelIcons, 72, 0);
    }
    else
    {
        XRender::renderTextureFL(infobox_x + 8, infobox_y + 34 - GFX.MCursor[1].w / 2, GFX.MCursor[1].w, GFX.MCursor[1].h, GFX.MCursor[1], 0, 0, -90.0);
        XRender::renderTextureFL(infobox_x + 480 - 8 - GFX.MCursor[1].h, infobox_y + 34 - GFX.MCursor[2].w / 2, GFX.MCursor[2].w, GFX.MCursor[2].h, GFX.MCursor[2], 0, 0, -90.0);
    }
}

void GetMenuPos(int* MenuX, int* MenuY)
{
    if(MenuX)
        *MenuX = XRender::TargetW / 2 - 100;

    if(MenuY)
        *MenuY = XRender::TargetH - 250;

    // tweaks for MenuX
    if(MenuX && XRender::TargetW < TinyScreenW)
    {
        *MenuX = XRender::TargetW / 2 - 240;
        if(*MenuX < 24 + XRender::TargetOverscanX)
            *MenuX = 24 + XRender::TargetOverscanX;
    }

    // the rest is tweaks for MenuY
    if(!MenuY)
        return;

    if(XRender::TargetH < TinyScreenH)
        *MenuY = 100;
    else if(XRender::TargetH < SmallScreenH)
        *MenuY = XRender::TargetH - 220;

    if(MenuMode >= MENU_SELECT_SLOT_BASE && MenuMode < MENU_SELECT_SLOT_END)
        *MenuY -= c_menuSavesOffsetY;
}

static void s_findRecentEpisode()
{
    menuRecentEpisode = -1;

    std::vector<SelectWorld_t>& SelectorList
        = (MenuMode == MENU_EDITOR)
        ? SelectWorldEditable
        : SelectWorld;

    for(size_t i = 1; i < SelectorList.size(); ++i)
    {
        // special editor entry for battle levels
        bool is_battle = (MenuMode == MENU_EDITOR && i == SelectorList.size() - 2);

        auto &w = SelectorList[i];
        const std::string wPath = (!is_battle) ? w.WorldPath + w.WorldFile : "battle";

        if((MenuMode == MENU_1PLAYER_GAME && wPath == g_recentWorld1p) ||
           (MenuMode == MENU_2PLAYER_GAME && wPath == g_recentWorld2p) ||
           (MenuMode == MENU_EDITOR && wPath == g_recentWorldEditor))
        {
            menuRecentEpisode = (int)i - 1;
            w.highlight = true;
        }
        else
        {
            w.highlight = false;
        }
    }

    if(menuRecentEpisode >= 0)
        worldCurs = menuRecentEpisode - 3;

    MenuCursor = (menuRecentEpisode < 0) ? 0 : menuRecentEpisode;
}

#if !defined(PGE_NO_THREADING)
static int FindWorldsThread(void *)
{
    FindWorlds();
    return 0;
}
#endif

#if (defined(__APPLE__) && defined(USE_BUNDLED_ASSETS)) || defined(FIXED_ASSETS_PATH) || defined(__EMSCRIPTEN__)
#   define CAN_WRITE_APPPATH_WORLDS false
#else
#   define CAN_WRITE_APPPATH_WORLDS true
#endif

struct WorldRoot_t
{
    std::string path;
    bool editable;
};

// helper functions used by FindWorlds() and LoadSingleWorld()
static void s_LoadSingleWorld(const std::string& epDir, const std::string& fName, WorldData& head, TranslateEpisode& tr, bool editable);
static void s_FinishFindWorlds();

void FindWorlds()
{
    TranslateEpisode tr;
    WorldData head;
    NumSelectWorld = 0;

    std::vector<WorldRoot_t> worldRoots =
    {
        {AppPath + "worlds/", CAN_WRITE_APPPATH_WORLDS}
    };

    if(AppPathManager::userDirIsAvailable())
    {
        // assume that assets are not writable
        worldRoots.push_back({AppPathManager::userWorldsRootDir(), true});
        worldRoots[0].editable = false;
    }

#ifdef APP_PATH_HAS_EXTRA_WORLDS
    // add worlds from additional romfs packages
    for(const std::string& root : AppPathManager::worldRootDirs())
        worldRoots.push_back({root, false});
#endif

    SelectWorld.clear();
    SelectWorld.emplace_back(SelectWorld_t()); // Dummy entry
    SelectWorldEditable.clear();
    SelectWorldEditable.push_back(SelectWorld_t()); // Dummy entry

#ifndef PGE_NO_THREADING
    if(SDL_AtomicGet(&loading))
    {
        SDL_AtomicSet(&loadingProgrss, 0);
        SDL_AtomicSet(&loadingProgrssMax, 0);

        for(const auto &worldsRoot : worldRoots)
        {
            std::vector<std::string> dirs;
            DirMan episodes(worldsRoot.path);
            episodes.getListOfFolders(dirs);
            SDL_AtomicAdd(&loadingProgrssMax, (int)dirs.size());
        }
    }
#endif

    for(const auto &worldsRoot : worldRoots)
    {
        DirMan episodes(worldsRoot.path);

        std::vector<std::string> dirs;
        std::vector<std::string> files;
        episodes.getListOfFolders(dirs);

        for(const auto &dir : dirs)
        {
            std::string epDir = worldsRoot.path + dir + "/";
            DirMan episode(epDir);
            episode.getListOfFiles(files, {".wld", ".wldx"});

            for(std::string &fName : files)
                s_LoadSingleWorld(epDir, fName, head, tr, worldsRoot.editable);

#ifdef THEXTECH_PRELOAD_LEVELS
            if(LoadingInProcess)
                UpdateLoad();
#endif

#ifndef PGE_NO_THREADING
            SDL_AtomicAdd(&loadingProgrss, 1);
#endif
        }
    }

    s_FinishFindWorlds();
}

static void s_LoadSingleWorld(const std::string& epDir, const std::string& fName, WorldData& head, TranslateEpisode& tr, bool editable)
{
    std::string wPath = epDir + fName;

    PGE_FileFormats_misc::RWopsTextInput in(Files::open_file(wPath, "r"), wPath);

    if(FileFormats::OpenWorldFileHeaderT(in, head))
    {
        SelectWorld_t w;
        w.WorldName = head.EpisodeTitle;
        head.charactersToS64();
        w.WorldPath = epDir;
        w.WorldFile = fName;
        if(w.WorldName.empty())
            w.WorldName = fName;

        bool is_wldx = (head.meta.RecentFormat == WorldData::PGEX);
        bool is_wld38a = (head.meta.RecentFormat == WorldData::SMBX38A);
        if(is_wldx && (head.meta.configPackId == "SMBX2" || head.meta.engineFeatureLevel > g_gameInfo.contentFeatureLevel))
            w.probably_incompatible = true;
        else if(is_wld38a)
            w.probably_incompatible = true;
        else
            w.bugfixes_on_by_default = is_wldx;

        w.blockChar[1] = head.nocharacter1;
        w.blockChar[2] = head.nocharacter2;
        w.blockChar[3] = head.nocharacter3;
        w.blockChar[4] = head.nocharacter4;
        w.blockChar[5] = head.nocharacter5;

        w.editable = editable;

        if(tr.tryTranslateTitle(epDir, fName, w.WorldName))
            pLogDebug("Translated world title: %s", w.WorldName.c_str());

        SelectWorld.push_back(w);
        if(editable)
            SelectWorldEditable.push_back(w);
    }
}

static void s_FinishFindWorlds()
{
    if(SelectWorld.size() <= 1) // No available worlds in the list
    {
        SelectWorld.clear();
        SelectWorld.emplace_back(SelectWorld_t()); // Dummy entry
        SelectWorld.emplace_back(SelectWorld_t()); // "no episodes to play" entry
        SelectWorld[1].WorldName = g_mainMenu.gameNoEpisodesToPlay;
        SelectWorld[1].disabled = true;
    }

    // Sort all worlds by alphabetical order
    std::sort(SelectWorld.begin(), SelectWorld.end(),
              [](const SelectWorld_t& a, const SelectWorld_t& b)
    {
        return a.WorldName < b.WorldName;
    });

    std::sort(SelectWorldEditable.begin(), SelectWorldEditable.end(),
              [](const SelectWorld_t& a, const SelectWorld_t& b)
    {
        return a.WorldName < b.WorldName;
    });

    NumSelectWorld = (int)(SelectWorld.size() - 1);

    if(!g_gameInfo.disableBattleMode)
    {
        SelectWorld_t battles = SelectWorld_t();
        battles.WorldName = g_mainMenu.editorBattles;
        SelectWorldEditable.push_back(battles);
    }

    SelectWorld_t createWorld = SelectWorld_t();
    createWorld.WorldName = g_mainMenu.editorNewWorld;
    SelectWorldEditable.push_back(createWorld);

    NumSelectWorldEditable = ((int)SelectWorldEditable.size() - 1);

    s_findRecentEpisode();

#ifndef PGE_NO_THREADING
    SDL_AtomicSet(&loading, 0);
#endif
}

void LoadSingleWorld(const std::string wPath)
{
    TranslateEpisode tr;
    WorldData head;
    NumSelectWorld = 0;

    SelectWorld.clear();
    SelectWorld.emplace_back(SelectWorld_t()); // Dummy entry
    SelectWorldEditable.clear();
    SelectWorldEditable.push_back(SelectWorld_t()); // Dummy entry

    std::string fName = Files::basename(wPath);
    std::string epDir = wPath.substr(0, wPath.size() - fName.size());

    if(epDir.empty())
        epDir = "./";

    epDir = DirMan(epDir).absolutePath() + "/";

    s_LoadSingleWorld(epDir, fName, head, tr, true);

    s_FinishFindWorlds();
}

#if !defined(THEXTECH_PRELOAD_LEVELS) && !defined(PGE_NO_THREADING)
static int FindLevelsThread(void *)
{
    FindLevels();
    return 0;
}
#endif

void FindLevels()
{
    std::vector<WorldRoot_t> battleRoots =
    {
        {AppPath + "battle/", CAN_WRITE_APPPATH_WORLDS}
    };

    if(AppPathManager::userDirIsAvailable())
    {
        // assume that assets are not writable
        battleRoots.push_back({AppPathManager::userBattleRootDir(), true});
        battleRoots[0].editable = false;
    }

    SelectBattle.clear();
    SelectBattle.emplace_back(SelectWorld_t()); // Dummy entry

    NumSelectBattle = 1;
    SelectBattle.emplace_back(SelectWorld_t()); // "random level" entry
    SelectBattle[1].WorldName = g_mainMenu.gameBattleRandom;
    LevelData head;

#ifndef PGE_NO_THREADING
    if(SDL_AtomicGet(&loading))
    {
        SDL_AtomicSet(&loadingProgrss, 0);
        SDL_AtomicSet(&loadingProgrssMax, 0);

        for(const auto &battleRoot : battleRoots)
        {
            std::vector<std::string> files;
            DirMan battleLvls(battleRoot.path);
            battleLvls.getListOfFiles(files, {".lvl", ".lvlx"});
            SDL_AtomicAdd(&loadingProgrssMax, (int)files.size());
        }
    }
#endif

    for(const auto &battleRoot : battleRoots)
    {
        std::vector<std::string> files;
        DirMan battleLvls(battleRoot.path);
        battleLvls.getListOfFiles(files, {".lvl", ".lvlx"});
        for(std::string &fName : files)
        {
            std::string wPath = battleRoot.path + fName;

            PGE_FileFormats_misc::RWopsTextInput in(Files::open_file(wPath, "r"), wPath);
            if(FileFormats::OpenLevelFileHeaderT(in, head))
            {
                SelectWorld_t w;
                w.WorldPath = battleRoot.path;
                w.WorldFile = fName;
                w.WorldName = head.LevelName;
                if(w.WorldName.empty())
                    w.WorldName = fName;
                w.editable = battleRoot.editable;
                SelectBattle.push_back(w);

#ifdef THEXTECH_PRELOAD_LEVELS
                if(LoadingInProcess)
                    UpdateLoad();
#endif
            }
#ifndef PGE_NO_THREADING
            SDL_AtomicAdd(&loadingProgrss, 1);
#endif
        }
    }

    NumSelectBattle = ((int)SelectBattle.size() - 1);

    if(SelectBattle.size() <= 2) // No available levels in the list
    {
        SelectBattle.clear();
        SelectBattle.emplace_back(SelectWorld_t()); // Dummy entry

        NumSelectBattle = 1;
        SelectBattle.emplace_back(SelectWorld_t()); // "no battle levels" entry
        SelectBattle[1].WorldName = g_mainMenu.gameNoBattleLevels;
        SelectBattle[1].disabled = true;
    }

#ifndef PGE_NO_THREADING
    SDL_AtomicSet(&loading, 0);
#endif
}


static void s_handleMouseMove(int items, int x, int y, int maxWidth, int itemHeight)
{
    For(A, 0, items)
    {
        if(SharedCursor.Y >= y + A * itemHeight && SharedCursor.Y <= y + 16 + A * itemHeight)
        {
            if(SharedCursor.X >= x && SharedCursor.X <= x + maxWidth)
            {
                if(MenuMouseRelease && SharedCursor.Primary)
                    MenuMouseClick = true;
                if(MenuCursor != A)
                {
                    PlaySoundMenu(SFX_Slide);
                    MenuCursor = A;
                    break;
                }
            }
        }
    }
}


static bool s_can_enter_ap_screen()
{
    return MenuMode == MENU_INTRO || (XRender::TargetH >= TinyScreenH && MenuMode == MENU_MAIN);
}



bool mainMenuUpdate()
{
    int MenuX, MenuY;
    GetMenuPos(&MenuX, &MenuY);

    int B;
    // Location_t tempLocation;
    int menuLen;
    // Player_t blankPlayer;

    MenuControls_t menuControls = Controls::GetMenuControls();

    menuControls.Back |= (SharedCursor.Secondary && MenuMouseRelease);

    if(menuControls.Back && menuControls.Do)
        menuControls.Do = false;

    bool should_enter_ap_screen = menuControls.Home && s_can_enter_ap_screen() && MenuCursorCanMove;

    if(should_enter_ap_screen && GetAssetPacks().size() <= 1)
    {
        PlaySoundMenu(SFX_BlockHit);
        MenuCursorCanMove = false;
    }

    if(should_enter_ap_screen && GetAssetPacks().size() > 1)
    {
        s_startAssetPackTimer++;
        if(s_startAssetPackTimer == 60)
        {
            PlaySoundMenu(SFX_Do);
            FadeOutMusic(500);
            ScreenAssetPack::g_LoopActive = true;
            GameMenu = false;
            return true;
        }
    }
    else if(s_can_enter_ap_screen() && s_startAssetPackTimer > 0)
        s_startAssetPackTimer -= 2;
    else
        s_startAssetPackTimer = 0;

    {
        {
            bool k = false;
            k |= menuControls.Back;
            k |= menuControls.Do;
            k |= menuControls.Up;
            k |= menuControls.Down;
            k |= menuControls.Left;
            k |= menuControls.Right;
            k |= menuControls.Home;
            k |= menuControls.Erase;

            if(!k)
                MenuCursorCanMove = true;
        }

        if(!g_pollingInput && (MenuMode != MENU_CHARACTER_SELECT_NEW && MenuMode != MENU_CHARACTER_SELECT_NEW_BM && MenuMode != MENU_NEW_OPTIONS && MenuMode != MENU_INTRO))
        {
            int cursorDelta = 0;

            if(menuControls.Up)
            {
                if(MenuCursorCanMove)
                {
                    MenuCursor -= 1;
                    cursorDelta = -1;
                }

                MenuCursorCanMove = false;
            }
            else if(menuControls.Down)
            {
                if(MenuCursorCanMove)
                {
                    MenuCursor += 1;
                    cursorDelta = +1;
                }

                MenuCursorCanMove = false;
            }

            if(cursorDelta != 0)
            {
#if 0
                if(MenuMode >= MENU_CHARACTER_SELECT_BASE && MenuMode <= MENU_CHARACTER_SELECT_BASE_END)
                {
                    while((MenuCursor == (PlayerCharacter - 1) &&
                          (MenuMode == MENU_CHARACTER_SELECT_2P_S2 || MenuMode == MENU_CHARACTER_SELECT_BM_S2)) ||
                           blockCharacter[MenuCursor + 1])
                    {
                        MenuCursor += cursorDelta;
                        if(MenuCursor < 0)
                            MenuCursor = numCharacters - 1;
                        else if(MenuCursor >= numCharacters)
                            MenuCursor = 0;
                    }
                }
#endif

                if(MenuMode >= MENU_SELECT_SLOT_BASE && MenuMode < MENU_SELECT_SLOT_END)
                    s_change_save_item();

                PlaySoundMenu(SFX_Slide);
            }

        } // No keyboard/Joystick grabbing active

        if(MenuMode == MENU_INTRO && XRender::TargetH >= TinyScreenH)
            MenuMode = MENU_MAIN;

#ifndef PGE_NO_THREADING
        if(SDL_AtomicGet(&loading))
        {
            if((menuControls.Do && MenuCursorCanMove) || MenuMouseClick)
                PlaySoundMenu(SFX_BlockHit);

            if(MenuCursor != 0)
                MenuCursor = 0;
        }
        else
#endif

        // Menu Intro
        if(MenuMode == MENU_INTRO)
        {
            if(MenuMouseRelease && SharedCursor.Primary)
                MenuMouseClick = true;

            if(menuControls.Back && MenuCursorCanMove)
            {
                int quitKeyPos = 2;

                if(s_show_separate_2P())
                    quitKeyPos++;

                if(!g_gameInfo.disableBattleMode)
                    quitKeyPos++;

                if(g_config.enable_editor)
                    quitKeyPos++;

                MenuMode = MENU_MAIN;
                MenuCursor = quitKeyPos;
                MenuCursorCanMove = false;
                PlaySoundMenu(SFX_Slide);
            }

            if((menuControls.Do && MenuCursorCanMove) || MenuMouseClick)
            {
                MenuCursorCanMove = false;
                MenuMode = MENU_MAIN;
                MenuCursor = 0;
                PlaySoundMenu(SFX_Do);
            }
        }
        // Main Menu
        else if(MenuMode == MENU_MAIN)
        {
            if(SharedCursor.Move)
            {
                For(A, 0, 10)
                {
                    if(SharedCursor.Y >= MenuY + A * 30 && SharedCursor.Y <= MenuY + A * 30 + 16)
                    {
                        int i = 0;
                        if(A == i++)
                            menuLen = 18 * (s_show_separate_2P() ? (int)g_mainMenu.main1PlayerGame.size() : (int)g_mainMenu.mainPlayEpisode.size()) - 2;
                        else if(s_show_separate_2P() && A == i++)
                            menuLen = 18 * (int)g_mainMenu.mainMultiplayerGame.size() - 2;
                        else if(!g_gameInfo.disableBattleMode && A == i++)
                            menuLen = 18 * (int)g_mainMenu.mainBattleGame.size();
                        else if(g_config.enable_editor && A == i++)
                            menuLen = 18 * (int)g_mainMenu.mainEditor.size();
                        else if(A == i++)
                            menuLen = 18 * (int)g_mainMenu.mainOptions.size();
                        else if(A == i++)
                            menuLen = 18 * (int)g_mainMenu.mainExit.size();
                        else
                            break;

                        if(SharedCursor.X >= MenuX && SharedCursor.X <= MenuX + menuLen)
                        {
                            if(MenuMouseRelease && SharedCursor.Primary)
                                MenuMouseClick = true;

                            if(MenuCursor != A)
                            {
                                PlaySoundMenu(SFX_Slide);
                                MenuCursor = A;
                            }
                        }
                    }
                }
            }

            if(menuControls.Back && MenuCursorCanMove)
            {
                int quitKeyPos = 2;

                if(s_show_separate_2P())
                    quitKeyPos++;

                if(!g_gameInfo.disableBattleMode)
                    quitKeyPos++;

                if(g_config.enable_editor)
                    quitKeyPos++;

                if(XRender::TargetH < TinyScreenH)
                {
                    MenuCursorCanMove = false;
                    MenuMode = MENU_INTRO;
                    PlaySoundMenu(SFX_Slide);
                }
                else if(MenuCursor != quitKeyPos)
                {
                    MenuCursor = quitKeyPos;
                    MenuCursorCanMove = false;
                    PlaySoundMenu(SFX_Slide);
                }
            }
            else if((menuControls.Do && MenuCursorCanMove) || MenuMouseClick)
            {
                MenuCursorCanMove = false;
                PlayerCharacter = 0;
                PlayerCharacter2 = 0;

                int i = 0;
                if(MenuCursor == i++)
                {
                    PlaySoundMenu(SFX_Do);
                    MenuMode = MENU_1PLAYER_GAME;
                    menuPlayersNum = 1;
                    menuBattleMode = false;
                    MenuCursor = 0;
#ifdef THEXTECH_PRELOAD_LEVELS
                    s_findRecentEpisode();
#elif defined(PGE_NO_THREADING)
                    FindWorlds();
#else
                    SDL_AtomicSet(&loading, 1);
                    loadingThread = SDL_CreateThread(FindWorldsThread, "FindWorlds", nullptr);
                    SDL_DetachThread(loadingThread);
#endif
                }
                else if(s_show_separate_2P() && MenuCursor == i++)
                {
                    PlaySoundMenu(SFX_Do);
                    MenuMode = MENU_2PLAYER_GAME;
                    menuPlayersNum = 2;
                    menuBattleMode = false;
                    MenuCursor = 0;
#ifdef THEXTECH_PRELOAD_LEVELS
                    s_findRecentEpisode();
#elif defined(PGE_NO_THREADING)
                    FindWorlds();
#else
                    SDL_AtomicSet(&loading, 1);
                    loadingThread = SDL_CreateThread(FindWorldsThread, "FindWorlds", nullptr);
                    SDL_DetachThread(loadingThread);
#endif
                }
                else if(!g_gameInfo.disableBattleMode && MenuCursor == i++)
                {
                    PlaySoundMenu(SFX_Do);
                    MenuMode = MENU_BATTLE_MODE;
                    menuPlayersNum = 2;
                    menuBattleMode = true;
#if !defined(THEXTECH_PRELOAD_LEVELS) && defined(PGE_NO_THREADING)
                    FindLevels();
#elif !defined(THEXTECH_PRELOAD_LEVELS)
                    SDL_AtomicSet(&loading, 1);
                    loadingThread = SDL_CreateThread(FindLevelsThread, "FindLevels", nullptr);
                    SDL_DetachThread(loadingThread);
#endif
                    MenuCursor = 0;
                }
                else if(g_config.enable_editor && MenuCursor == i++)
                {
#if 0
                    if(XRender::TargetW < 640 || XRender::TargetH < 480)
                    {
                        PlaySoundMenu(SFX_BlockHit);
                        MessageText = g_mainMenu.editorErrorResolution;
                        PauseGame(PauseCode::Message);
                    }
                    else
#endif
                    if(!GFX.EIcons.inited)
                    {
                        PlaySoundMenu(SFX_BlockHit);
                        g_MessageType = MESSAGE_TYPE_SYS_WARNING;
                        MessageText = fmt::format_ne(g_mainMenu.editorErrorMissingResources, "EditorIcons.png");
                        PauseGame(PauseCode::Message);
                    }
                    else if(!EditorCustom::loaded)
                    {
                        PlaySoundMenu(SFX_BlockHit);
                        g_MessageType = MESSAGE_TYPE_SYS_WARNING;
                        MessageText = fmt::format_ne(g_mainMenu.editorErrorMissingResources, "editor.ini");
                        PauseGame(PauseCode::Message);
                    }
                    else
                    {
                        PlaySoundMenu(SFX_Do);
                        MenuMode = MENU_EDITOR;
                        MenuCursor = 0;

#ifdef THEXTECH_PRELOAD_LEVELS
                        s_findRecentEpisode();
#elif defined(PGE_NO_THREADING)
                        FindWorlds();
#else
                        SDL_AtomicSet(&loading, 1);
                        loadingThread = SDL_CreateThread(FindWorldsThread, "FindWorlds", nullptr);
                        SDL_DetachThread(loadingThread);
#endif
                    }
                }
                else if(MenuCursor == i++)
                {
                    PlaySoundMenu(SFX_Do);
                    MenuMode = MENU_NEW_OPTIONS;
                    OptionsScreen::Init();
                }
                else if(MenuCursor == i++)
                {
                    PlaySoundMenu(SFX_Do);
                    GracefulQuit(true);
                    return true;
                }

            }


            if(MenuMode == MENU_MAIN)
            {
                int quitKeyPos = 2;

                if(s_show_separate_2P())
                    quitKeyPos++;

                if(!g_gameInfo.disableBattleMode)
                    quitKeyPos++;

                if(g_config.enable_editor)
                    quitKeyPos++;


                if(MenuCursor > quitKeyPos)
                    MenuCursor = 0;

                if(MenuCursor < 0)
                    MenuCursor = quitKeyPos;
            }
        } // Main Menu

        // Character Select
        else if(MenuMode == MENU_CHARACTER_SELECT_NEW ||
                MenuMode == MENU_CHARACTER_SELECT_NEW_BM)
        {
            int ret = ConnectScreen::Logic();
            if(ret == -1)
            {
                // disconnect input methods for convenience
                Controls::ClearInputMethods();

                if(MenuMode == MENU_CHARACTER_SELECT_NEW_BM)
                {
                    MenuCursor = selWorld - 1;
                    MenuMode = MENU_BATTLE_MODE;
                }
                else
                {
                    UnloadCustomPlayerPreviews();

                    MenuCursor = selSave - 1;
                    if(menuPlayersNum == 1)
                        MenuMode = MENU_SELECT_SLOT_1P;
                    else
                        MenuMode = MENU_SELECT_SLOT_2P;
                }

                MenuCursorCanMove = false;
            }
            else if(ret == 1)
            {
                if(MenuMode == MENU_CHARACTER_SELECT_NEW)
                {
                    // writing to m_value to avoid extra UpdateConfig hook
                    g_config.playstyle.m_value = s_episode_playstyle;
                    g_config.playstyle.m_set = ConfigSetLevel::ep_config;

                    if(g_config.speedrun_mode.m_set != ConfigSetLevel::cmdline)
                    {
                        g_config.speedrun_mode.m_value = s_episode_speedrun_mode;
                        g_config.speedrun_mode.m_set = ConfigSetLevel::ep_config;
                    }

                    MenuCursor = 0;
                    StartEpisode();
                    return true;
                }
                else if(MenuMode == MENU_CHARACTER_SELECT_NEW_BM)
                {
                    MenuCursor = 0;
                    StartBattleMode();
                    return true;
                }
            }
        }
#if 0 // old code, no longer used
        // Character Select
        else if(MenuMode == MENU_CHARACTER_SELECT_1P ||
                MenuMode == MENU_CHARACTER_SELECT_2P_S1 ||
                MenuMode == MENU_CHARACTER_SELECT_2P_S2 ||
                MenuMode == MENU_CHARACTER_SELECT_BM_S1 ||
                MenuMode == MENU_CHARACTER_SELECT_BM_S2)
        {
            if(SharedCursor.Move)
            {
                B = 0;
                For(A, 0, 4)
                {
                    if(blockCharacter[A + 1])
                    {
                        B -= 30;
                    }
                    else
                    {
                        if(SharedCursor.Y >= MenuY + A * 30 + B && SharedCursor.Y <= 366 + A * 30 + B)
                        {
                            if(A >= 0 && A < numCharacters)
                            {
                                menuLen = 18 * g_mainMenu.selectPlayer[A + 1].size();
                                if(A == 0)
                                    menuLen += 2;
                            }
                            else
                            {
                                menuLen = 180;
                            }

                            if(SharedCursor.X >= MenuX && SharedCursor.X <= MenuX + menuLen)
                            {
                                if(MenuMouseRelease && SharedCursor.Primary)
                                    MenuMouseClick = true;
                                if(MenuCursor != A)
                                {
                                    if(
                                        ((MenuMode == MENU_CHARACTER_SELECT_2P_S2 || MenuMode == MENU_CHARACTER_SELECT_BM_S2) && PlayerCharacter - 1 == A) ||
                                        ((blockCharacter[A + 1]) && MenuMouseClick)
                                    )
                                    {
                                        MenuMouseClick = false;
                                    }
                                    else
                                    {
                                        PlaySoundMenu(SFX_Slide);
                                        MenuCursor = A;
                                    }
                                }
                            }
                        }
                    }
                }
            }

            if(MenuCursorCanMove || MenuMouseClick)
            {
                if(menuBackPress)
                {
                    if(MenuMode == MENU_CHARACTER_SELECT_2P_S2)
                    {
                        MenuMode = MENU_CHARACTER_SELECT_2P_S1;
                        MenuCursor = PlayerCharacter - 1;
                    }
                    else if(MenuMode == MENU_CHARACTER_SELECT_BM_S2)
                    {
                        MenuMode = MENU_CHARACTER_SELECT_BM_S1;
                        MenuCursor = PlayerCharacter - 1;
                    }
                    else
                    {
                        // MenuCursor = selWorld - 1;
                        MenuMode /= MENU_CHARACTER_SELECT_BASE;
                        // Restore menu state
                        worldCurs = listMenuLastScroll;
                        MenuCursor = listMenuLastCursor;
                    }

                    MenuCursorCanMove = false;
                    PlaySoundMenu(SFX_Slide);
                }
                else if(menuDoPress || MenuMouseClick)
                {
                    PlaySoundMenu(SFX_Do);

                    if(MenuMode == MENU_CHARACTER_SELECT_1P)
                    {
                        PlayerCharacter = MenuCursor + 1;
                        MenuMode = MENU_SELECT_SLOT_1P;
                        MenuCursor = 0;
                    }
                    else if(MenuMode == MENU_CHARACTER_SELECT_2P_S1)
                    {
                        PlayerCharacter = MenuCursor + 1;
                        MenuMode = MENU_CHARACTER_SELECT_2P_S2;
                        MenuCursor = PlayerCharacter2;
                    }
                    else if(MenuMode == MENU_CHARACTER_SELECT_2P_S2)
                    {
                        PlayerCharacter2 = MenuCursor + 1;
                        MenuMode = MENU_SELECT_SLOT_2P;
                        MenuCursor = 0;
                    }
                    else if(MenuMode == MENU_CHARACTER_SELECT_BM_S1)
                    {
                        PlayerCharacter = MenuCursor + 1;
                        MenuMode = MENU_CHARACTER_SELECT_BM_S2;
                        MenuCursor = PlayerCharacter2 - 1;
                        if(MenuCursor < 0)
                            MenuCursor = 0;
                    }
                    else if(MenuMode == MENU_CHARACTER_SELECT_BM_S2)
                    {
                        PlayerCharacter2 = MenuCursor + 1;
                        MenuCursor = 0;
                        StartBattleMode();
                        return true;
                    }

                    MenuCursorCanMove = false;
                }
            }

            bool isListMenu = (MenuMode == MENU_1PLAYER_GAME || MenuMode == MENU_2PLAYER_GAME || MenuMode == MENU_BATTLE_MODE);

            if(MenuMode > MENU_MAIN && !isListMenu)
            {
                if(MenuCursor > numCharacters - 1)
                {
                    MenuCursor = 0;

                    while((MenuCursor == PlayerCharacter - 1 && (MenuMode == MENU_CHARACTER_SELECT_2P_S2 || MenuMode == MENU_CHARACTER_SELECT_BM_S2)) ||
                          blockCharacter[MenuCursor + 1])
                    {
                        MenuCursor += 1;
                    }

                }

                if(MenuCursor < 0)
                {
                    MenuCursor = numCharacters - 1;

                    while((MenuCursor == PlayerCharacter - 1 && (MenuMode == MENU_CHARACTER_SELECT_2P_S2 || MenuMode == MENU_CHARACTER_SELECT_BM_S2)) ||
                          blockCharacter[MenuCursor + 1])
                    {
                        MenuCursor -= 1;
                    }
                }
            }

            if(!isListMenu)
            {
                while(((MenuMode == MENU_CHARACTER_SELECT_2P_S2 || MenuMode == MENU_CHARACTER_SELECT_BM_S2) && MenuCursor == PlayerCharacter - 1) ||
                       blockCharacter[MenuCursor + 1])
                {
                    MenuCursor += 1;
                }
            }

            if(MenuMode >= MENU_CHARACTER_SELECT_BASE && MenuMode <= MENU_CHARACTER_SELECT_BASE_END)
            {
                if(MenuCursor >= numCharacters)
                {
                    MenuCursor = 0;
                }
                else
                {
                    For(A, 1, numPlayers)
                    {
                        Player[A].Character = MenuCursor + 1;
                        SizeCheck(A);
                    }

                    For(A, 1, numNPCs)
                    {
                        if(NPC[A].Type == NPCID_PLR_FIREBALL)
                            NPC[A].Special = MenuCursor + 1;
                    }
                }
            }
        } // Character Select
#endif

        // World Select
        else if(MenuMode == MENU_1PLAYER_GAME || MenuMode == MENU_2PLAYER_GAME
            || MenuMode == MENU_BATTLE_MODE || MenuMode == MENU_EDITOR)
        {
            const std::vector<SelectWorld_t>& SelectorList
                = (MenuMode == MENU_BATTLE_MODE) ? SelectBattle :
                    ((MenuMode == MENU_EDITOR) ? SelectWorldEditable :
                        SelectWorld);

            if(ScrollDelay > 0)
            {
                SharedCursor.Move = true;
                ScrollDelay -= 1;
            }

            if(SharedCursor.Move)
            {
                B = 0;

                For(A, minShow - 1, maxShow - 1)
                {
                    if(SharedCursor.Y >= MenuY + B * 30 && SharedCursor.Y <= MenuY + B * 30 + 16)
                    {
                        menuLen = 19 * static_cast<int>(SelectorList[A + 1].WorldName.size());

                        if(SharedCursor.X >= MenuX && SharedCursor.X <= MenuX + menuLen)
                        {
                            if(MenuMouseRelease && SharedCursor.Primary)
                                MenuMouseClick = true;

                            if(MenuCursor != A && ScrollDelay == 0)
                            {
                                ScrollDelay = 10;
                                PlaySoundMenu(SFX_Slide);
                                MenuCursor = A;
                            }
                        }
                    }

                    B += 1;
                }
            }

            if(MenuCursorCanMove || MenuMouseClick)
            {
                int first_new_content = (g_gameInfo.disableBattleMode) ? NumSelectWorldEditable : NumSelectWorldEditable - 1;

                if(menuControls.Back)
                {
                    MenuCursor = MenuMode - 1;

                    if(MenuMode == MENU_BATTLE_MODE)
                    {
                        MenuCursor = !s_show_separate_2P() ? 1 : 2;
                    }
                    else if(MenuMode == MENU_EDITOR)
                    {
                        MenuCursor = 3;
                        if(!s_show_separate_2P())
                            MenuCursor--;
                        if(g_gameInfo.disableBattleMode)
                            MenuCursor--;
                    }

                    MenuMode = MENU_MAIN;
//'world select back

                    PlaySoundMenu(SFX_Slide);
                    MenuCursorCanMove = false;
                }
                else if((menuControls.Left || menuControls.Right) && MenuMode == MENU_EDITOR && MenuCursor + 1 >= first_new_content)
                {
                    s_editor_target_thextech = !s_editor_target_thextech;
                    PlaySoundMenu(SFX_Climbing);
                    MenuCursorCanMove = false;
                    ScrollDelay = -1;
                }
                else if(menuControls.Do || MenuMouseClick)
                {
                    bool disabled = false;
                    // Save menu state
                    listMenuLastScroll = worldCurs;
                    listMenuLastCursor = MenuCursor;

                    selWorld = MenuCursor + 1;

                    if((MenuMode == MENU_BATTLE_MODE && SelectBattle[selWorld].disabled) ||
                       ((MenuMode == MENU_1PLAYER_GAME || MenuMode == MENU_2PLAYER_GAME) && SelectWorld[selWorld].disabled))
                        disabled = true;

                    if(!disabled)
                        PlaySoundMenu(SFX_Do);

                    if(disabled)
                    {
                        PlaySoundMenu(SFX_BlockHit);
                        // Do nothing. stay at menu
                    }
                    // level editor
                    else if(MenuMode == MENU_EDITOR)
                    {
                        if(selWorld == NumSelectWorldEditable)
                        {
                            ClearWorld(true);
                            WorldName = TextEntryScreen::Run(g_mainMenu.editorPromptNewWorldName);
                            if(!WorldName.empty())
                            {
                                std::string fn = WorldName;
                                // eliminate bad characters
                                std::replace(fn.begin(), fn.end(), '/', '_');
                                std::replace(fn.begin(), fn.end(), '\\', '_');
                                std::replace(fn.begin(), fn.end(), '.', '_');
                                std::replace(fn.begin(), fn.end(), ':', '_');
                                std::replace(fn.begin(), fn.end(), '<', '_');
                                std::replace(fn.begin(), fn.end(), '>', '_');
                                std::replace(fn.begin(), fn.end(), '"', '_');
                                std::replace(fn.begin(), fn.end(), '|', '_');
                                std::replace(fn.begin(), fn.end(), '?', '_');
                                std::replace(fn.begin(), fn.end(), '*', '_');
                                // ensure uniqueness (but still case-sensitive for now)
                                while(DirMan::exists(AppPathManager::userWorldsRootDir() + fn))
                                    fn += "2";
                                DirMan::mkAbsPath(AppPathManager::userWorldsRootDir() + fn);

                                std::string wPath = AppPathManager::userWorldsRootDir() + fn + "/world.wld";
                                if(s_editor_target_thextech)
                                    wPath += "x";
                                g_recentWorldEditor = wPath;

                                SaveWorld(wPath, (s_editor_target_thextech) ? FileFormats::WLD_PGEX : FileFormats::WLD_SMBX64);

#ifdef PGE_NO_THREADING
                                FindWorlds();
#else
                                SDL_AtomicSet(&loading, 1);
                                loadingThread = SDL_CreateThread(FindWorldsThread, "FindWorlds", NULL);
                                SDL_DetachThread(loadingThread);
#endif
                            }
                        }
                        else if(!g_gameInfo.disableBattleMode && selWorld == NumSelectWorldEditable - 1)
                        {
                            ClearWorld(true);
                            GameMenu = false;
                            LevelSelect = false;
                            BattleMode = true;
                            LevelEditor = true;
                            WorldEditor = false;
                            ClearLevel();
                            ClearGame();
                            SetupPlayers();

                            std::string lPath = AppPathManager::userBattleRootDir();
                            {
                                // find a single battle level to open first
                                std::vector<std::string> files;
                                DirMan battleLvls(lPath);
                                battleLvls.getListOfFiles(files, {(s_editor_target_thextech) ? ".lvlx" : ".lvl"});
                                if(files.empty())
                                    lPath += "newbattle.lvl";
                                else
                                    lPath += files[0];
                            }

                            if(!Files::fileExists(lPath))
                            {
                                if(s_editor_target_thextech)
                                    lPath += "x";

                                SaveLevel(lPath, (s_editor_target_thextech) ? FileFormats::LVL_PGEX : FileFormats::LVL_SMBX64);
                            }

                            OpenLevel(lPath);
                            EditorBackup(); // EditorRestore() gets called when not in world editor

                            if(g_recentWorldEditor != "battle")
                            {
                                g_recentWorldEditor = "battle";
                                SaveConfig();
                            }

                            Integrator::setEditorFile(FileName);
                            editorScreen.ResetCursor();
                            editorScreen.active = false;
                            MouseRelease = false;
                            return true;
                        }
                        else
                        {
                            GameMenu = false;
                            LevelSelect = false;
                            BattleMode = false;
                            LevelEditor = true;
                            WorldEditor = true;
                            ClearLevel();
                            ClearGame();
                            SetupPlayers();
                            std::string wPath = SelectWorldEditable[selWorld].WorldPath
                                + SelectWorldEditable[selWorld].WorldFile;
                            OpenWorld(wPath);
                            if(g_recentWorldEditor != wPath)
                            {
                                g_recentWorldEditor = wPath;
                                SaveConfig();
                            }
                            Integrator::setEditorFile(FileName);
                            editorScreen.ResetCursor();
                            editorScreen.active = false;
                            MouseRelease = false;
                            return true;
                        }
                    }
                    // battle mode
                    else if(MenuMode == MENU_BATTLE_MODE)
                    {
                        MenuMode = MENU_CHARACTER_SELECT_NEW_BM;
                        ConnectScreen::MainMenu_Start(2);
                    }
                    // enter save select
                    else
                    {
                        FindSaves();
                        MenuMode *= MENU_SELECT_SLOT_BASE;
                        MenuCursor = 0;

                        s_change_save_item();
                    }

                    MenuCursorCanMove = false;
                }
            }

            // New world select scroll options!
            // Based on Wohlstand's but somewhat simpler and less keyboard-specific.
            // Left and right are -/+ 3 (repeatable, so they also provide a quick-first/quick-last function).
            // DropItem / Tertiary cursor button is return to last episode.
            bool dontWrap = false;

            if(menuControls.Left && (MenuCursorCanMove || ScrollDelay == 0))
            {
                PlaySoundMenu(SFX_Saw);
                MenuCursor -= 3;
                MenuCursorCanMove = false;
                ScrollDelay = 15;
                dontWrap = true;
            }
            else if(menuControls.Right && (MenuCursorCanMove || ScrollDelay == 0))
            {
                PlaySoundMenu(SFX_Saw);
                MenuCursor += 3;
                MenuCursorCanMove = false;
                ScrollDelay = 15;
                dontWrap = true;
            }
            else if((menuControls.Left || menuControls.Right) && !SharedCursor.Move)
            {
                ScrollDelay -= 1;
            }
            else if(SharedCursor.ScrollUp)
            {
                PlaySoundMenu(SFX_Saw);
                MenuCursor -= 1;
                dontWrap = true;
            }
            else if(SharedCursor.ScrollDown)
            {
                PlaySoundMenu(SFX_Saw);
                MenuCursor += 1;
                dontWrap = true;
            }

            if(menuControls.Home && MenuCursorCanMove && menuRecentEpisode >= 0)
            {
                PlaySoundMenu(SFX_Camera);
                MenuCursor = menuRecentEpisode;
                MenuCursorCanMove = false;
                dontWrap = true;
            }

            if(MenuMode == MENU_1PLAYER_GAME || MenuMode == MENU_2PLAYER_GAME
                || MenuMode == MENU_BATTLE_MODE || MenuMode == MENU_EDITOR)
            {
                maxShow = (MenuMode == MENU_BATTLE_MODE) ? NumSelectBattle :
                    ((MenuMode == MENU_EDITOR) ? NumSelectWorldEditable :
                        NumSelectWorld);

                if(dontWrap)
                {
                    if(MenuCursor >= maxShow)
                        MenuCursor = maxShow - 1;
                    if(MenuCursor < 0)
                        MenuCursor = 0;
                }
                else
                {
                    if(MenuCursor >= maxShow)
                        MenuCursor = 0;
                    if(MenuCursor < 0)
                        MenuCursor = maxShow - 1;
                }
            }
        } // World select

        // Save Select
        else if(MenuMode == MENU_SELECT_SLOT_1P || MenuMode == MENU_SELECT_SLOT_2P)
        {
            if(SharedCursor.Move)
            {
                int old_item = MenuCursor;
                s_handleMouseMove(c_menuItemSavesDelete, MenuX, MenuY, 300, 30);

                if(MenuCursor != old_item)
                    s_change_save_item();
            }

            // new mode selection logic
            if(MenuCursor >= 0 && MenuCursor < maxSaveSlots && SaveSlotInfo[MenuCursor + 1].ConfigDefaults == 0)
            {
                // switch mode
                if(MenuCursorCanMove && (menuControls.Left || menuControls.Right) && s_episode_speedrun_mode == 0)
                {
                    if(s_episode_playstyle == Config_t::MODE_MODERN)
                        s_episode_playstyle = Config_t::MODE_CLASSIC;
                    else
                        s_episode_playstyle = Config_t::MODE_MODERN;

                    PlaySoundMenu(SFX_Climbing);
                    MenuCursorCanMove = false;
                }

                // only enter the menu if allowed by the global speedrun mode
                if(MenuCursorCanMove && menuControls.Home && g_config.speedrun_mode.m_set == ConfigSetLevel::cmdline)
                {
                    PlaySoundMenu(SFX_BlockHit);
                }
                // enter vanilla mode if an existing save slot
                else if(MenuCursorCanMove && menuControls.Home && SaveSlotInfo[MenuCursor + 1].Progress >= 0)
                {
                    int target_bugfixes = (SelectWorld[selWorld].bugfixes_on_by_default) ? Config_t::MODE_MODERN : Config_t::MODE_CLASSIC;

                    if(s_episode_playstyle == Config_t::MODE_VANILLA)
                        s_episode_playstyle = target_bugfixes;
                    else
                        s_episode_playstyle = Config_t::MODE_VANILLA;

                    PlaySoundMenu(SFX_Climbing);
                    MenuCursorCanMove = false;
                }
                // go to speedrun menu otherwise
                else if(MenuCursorCanMove && menuControls.Home)
                {
                    PlaySoundMenu(SFX_PlayerGrow);
                    selSave = MenuCursor + 1;

                    if(s_episode_speedrun_mode == 0)
                        MenuCursor = s_episode_playstyle;
                    else
                        MenuCursor = s_episode_speedrun_mode + 2;

                    MenuMode += MENU_SELECT_SLOT_ADVMODE_ADD;
                    MenuCursorCanMove = false;
                    MenuMouseClick = false;
                }
            }
            else if(MenuCursor >= 0 && MenuCursor < maxSaveSlots)
            {
                int save_configs = SaveSlotInfo[MenuCursor + 1].ConfigDefaults;

                if(save_configs > 0)
                {
                    s_episode_playstyle = save_configs - 1;
                    s_episode_speedrun_mode = 0;
                }
                else if(save_configs < 0)
                {
                    s_episode_speedrun_mode = -save_configs;
                    s_episode_playstyle = s_episode_speedrun_mode - 1;
                }
            }

            if(MenuCursorCanMove || MenuMouseClick)
            {
                if(menuControls.Back)
                {
//'save select back
                    MenuMode /= MENU_SELECT_SLOT_BASE;

                    // Restore menu state
                    worldCurs = listMenuLastScroll;
                    MenuCursor = listMenuLastCursor;

                    MenuCursorCanMove = false;
                    PlaySoundMenu(SFX_Slide);
                }
                else if(menuControls.Do || MenuMouseClick)
                {
                    if(MenuCursor == c_menuItemSavesCopy) // Copy the gamesave
                    {
                        PlaySoundMenu(SFX_Do);
                        MenuCursor = 0;
                        MenuMode += MENU_SELECT_SLOT_COPY_S1_ADD;
                        MenuCursorCanMove = false;
                    }
                    else if(MenuCursor == c_menuItemSavesDelete) // Delete the gamesave
                    {
                        PlaySoundMenu(SFX_Do);
                        MenuCursor = 0;
                        MenuMode += MENU_SELECT_SLOT_DELETE_ADD;
                        MenuCursorCanMove = false;
                    }
                    // block invalid speedrun continuation
                    else if(MenuCursor >= 0 && MenuCursor <= c_menuItemSavesEndList
                        && (int)g_config.speedrun_mode != 0 && g_config.speedrun_mode != -SaveSlotInfo[MenuCursor + 1].ConfigDefaults
                        && (SaveSlotInfo[MenuCursor + 1].ConfigDefaults != 0 || SaveSlotInfo[MenuCursor + 1].Progress >= 0))
                    {
                        PlaySoundMenu(SFX_BlockHit);
                        MenuCursorCanMove = false;
                    }
                    else if(MenuCursor >= 0 && MenuCursor <= c_menuItemSavesEndList) // Select the save slot, but still need to select players
                    {
                        selSave = MenuCursor + 1;

                        // warn the user of incompatibility if present
                        if(SaveSlotInfo[selSave].Progress < 0 && SelectWorld[selWorld].probably_incompatible)
                        {
                            PlaySoundMenu(SFX_Message);

                            g_MessageType = MESSAGE_TYPE_SYS_WARNING;
                            MessageText = g_mainMenu.warnEpCompat;
                            PauseGame(PauseCode::Message);
                        }

                        PlaySoundMenu(SFX_Do);

                        LoadCustomPlayerPreviews(SelectWorld[selWorld].WorldPath.c_str());

                        if(MenuMode == MENU_SELECT_SLOT_2P)
                            ConnectScreen::MainMenu_Start(2);
                        else if(s_prefer_modern_char_sel())
                            ConnectScreen::MainMenu_Start(1);
                        else
                            ConnectScreen::LegacyMenu_Start();

                        MenuMode = MENU_CHARACTER_SELECT_NEW;
                        MenuCursorCanMove = false;
                    }
                }
            }

            if(MenuMode == MENU_SELECT_SLOT_1P || MenuMode == MENU_SELECT_SLOT_2P)
            {
                if(MenuCursor > c_menuItemSavesDelete) MenuCursor = 0;
                if(MenuCursor < 0) MenuCursor = c_menuItemSavesDelete;
            }
        } // Save Slot Select

        // Save Select
        else if(MenuMode == MENU_SELECT_SLOT_1P_COPY_S1 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S1 ||
                MenuMode == MENU_SELECT_SLOT_1P_COPY_S2 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S2)
        {
            if(SharedCursor.Move)
                s_handleMouseMove(c_menuItemSavesEndList, MenuX, MenuY, 300, 30);

            if(MenuCursorCanMove || MenuMouseClick)
            {
                if(menuControls.Back)
                {
//'save select back
                    if(MenuMode == MENU_SELECT_SLOT_1P_COPY_S2 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S2)
                    {
                        MenuMode -= MENU_SELECT_SLOT_COPY_S1_ADD;
                    }
                    else
                    {
                        MenuMode -= MENU_SELECT_SLOT_COPY_S1_ADD;
                        MenuCursor = c_menuItemSavesCopy;
                    }

                    MenuCursorCanMove = false;
                    PlaySoundMenu(SFX_Do);
                }
                else if(menuControls.Do || MenuMouseClick)
                {
                    SDL_assert_release(IF_INRANGE(MenuCursor, 0, maxSaveSlots - 1));
                    int slot = MenuCursor + 1;

                    if(MenuMode == MENU_SELECT_SLOT_1P_COPY_S1 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S1)
                    {
                        if(SaveSlotInfo[slot].Progress < 0)
                            PlaySoundMenu(SFX_BlockHit);
                        else
                        {
                            PlaySoundMenu(SFX_Do);
                            menuCopySaveSrc = slot;
                            MenuMode += MENU_SELECT_SLOT_COPY_S1_ADD;
                        }

                        MenuCursorCanMove = false;
                    }
                    else if(menuCopySaveSrc == slot)
                    {
                        PlaySoundMenu(SFX_BlockHit);
                        MenuCursorCanMove = false;
                    }
                    else
                    {
                        PlaySoundMenu(SFX_Transform);
                        menuCopySaveDst = slot;
                        CopySave(selWorld, menuCopySaveSrc, menuCopySaveDst);
                        FindSaves();
                        MenuMode -= MENU_SELECT_SLOT_COPY_S2_ADD;
                        MenuCursor = c_menuItemSavesCopy;
                        MenuCursorCanMove = false;
                    }
                }
            }

            if(MenuMode == MENU_SELECT_SLOT_1P_COPY_S1 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S1 ||
               MenuMode == MENU_SELECT_SLOT_1P_COPY_S2 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S2)
            {
                if(MenuCursor > c_menuItemSavesEndList) MenuCursor = 0;
                if(MenuCursor < 0) MenuCursor = c_menuItemSavesEndList;
            }
        }

        // Delete gamesave
        else if(MenuMode == MENU_SELECT_SLOT_1P_DELETE || MenuMode == MENU_SELECT_SLOT_2P_DELETE)
        {
            if(SharedCursor.Move)
                s_handleMouseMove(c_menuItemSavesEndList, MenuX, MenuY, 300, 30);

            if(MenuCursorCanMove || MenuMouseClick)
            {
                if(menuControls.Back)
                {
//'save select back
                    MenuMode -= MENU_SELECT_SLOT_DELETE_ADD;
                    MenuCursor = c_menuItemSavesDelete;
                    PlaySoundMenu(SFX_Do);
                    MenuCursorCanMove = false;
                }
                else if(menuControls.Do || MenuMouseClick)
                {
                    MenuMode -= MENU_SELECT_SLOT_DELETE_ADD;
                    DeleteSave(selWorld, (MenuCursor + 1));
                    FindSaves();
                    MenuCursor = c_menuItemSavesDelete;
                    PlaySoundMenu(SFX_LavaMonster);
                    MenuCursorCanMove = false;
                }
            }

            if(MenuMode == MENU_SELECT_SLOT_1P_DELETE || MenuMode == MENU_SELECT_SLOT_2P_DELETE)
            {
                if(MenuCursor > c_menuItemSavesEndList) MenuCursor = 0;
                if(MenuCursor < 0) MenuCursor = c_menuItemSavesEndList;
            }
        }

        // Advanced mode selection
        else if(MenuMode == MENU_SELECT_SLOT_1P_ADVMODE || MenuMode == MENU_SELECT_SLOT_2P_ADVMODE)
        {
            if(SharedCursor.Move)
                s_handleMouseMove(5, MenuX, MenuY, MenuX, 30);

            if(MenuCursorCanMove || MenuMouseClick)
            {
                if(menuControls.Back)
                {
//'save select back
                    MenuMode -= MENU_SELECT_SLOT_ADVMODE_ADD;
                    MenuCursor = selSave - 1;
                    selSave = 0;
                    PlaySoundMenu(SFX_Slide);
                    MenuCursorCanMove = false;
                }
                else if(menuControls.Do || MenuMouseClick)
                {
                    if(MenuCursor < 3)
                    {
                        s_episode_playstyle = MenuCursor;
                        s_episode_speedrun_mode = 0;
                    }
                    else
                    {
                        s_episode_speedrun_mode = MenuCursor - 2;
                        s_episode_playstyle = s_episode_speedrun_mode - 1;
                    }

                    PlaySoundMenu(SFX_Do);
                    MenuMode -= MENU_SELECT_SLOT_ADVMODE_ADD;
                    MenuCursor = selSave - 1;
                    selSave = 0;
                    MenuCursorCanMove = false;
                }
            }

            if(MenuMode == MENU_SELECT_SLOT_1P_ADVMODE || MenuMode == MENU_SELECT_SLOT_2P_ADVMODE)
            {
                if(MenuCursor > 5) MenuCursor = 0;
                if(MenuCursor < 0) MenuCursor = 5;
            }
        }

        // Input Settings
        else if(MenuMode == MENU_INPUT_SETTINGS)
        {
            int ret = menuControls_Logic();
            if(ret == -1)
            {
                Controls::SaveConfig();
                MenuMode = MENU_NEW_OPTIONS;
                MenuCursorCanMove = false;
            }
        }

        // New Settings
        else if(MenuMode == MENU_NEW_OPTIONS)
        {
            if(OptionsScreen::Logic())
            {
                int optionsIndex = 1;
                if(s_show_separate_2P())
                    optionsIndex++;
                if(!g_gameInfo.disableBattleMode)
                    optionsIndex++;
                if(g_config.enable_editor)
                    optionsIndex++;
                MenuMode = MENU_MAIN;
                MenuCursor = optionsIndex;
                MenuCursorCanMove = false;
                PlaySoundMenu(SFX_Slide);
            }
        }
    }

    return false;
}

static constexpr int find_in_string(const char* haystack, const char* haystack_start, char needle)
{
    return (*haystack == '\0') ? -1 :
        ((*haystack == needle) ? haystack - haystack_start : find_in_string(haystack + 1, haystack_start, needle));
}

static constexpr bool in_string(const char* haystack, char needle)
{
    return find_in_string(haystack, haystack, needle) != -1;
}

static constexpr bool str_prefix(const char* string, const char* prefix)
{
    return (*prefix == '\0' || (*string == *prefix && str_prefix(string + 1, prefix + 1)));
}

static constexpr int find_in_string(const char* haystack, char needle)
{
    return find_in_string(haystack, haystack, needle);
}

void drawGameVersion(bool disable_git, bool git_only)
{
    constexpr bool is_release = !in_string(V_LATEST_STABLE, '-');
    constexpr bool is_main = str_prefix(V_BUILD_BRANCH, "main");
    constexpr bool is_stable = str_prefix(V_BUILD_BRANCH, "stable");
    constexpr bool is_head = str_prefix(V_BUILD_BRANCH, "HEAD");
    constexpr bool is_wip = str_prefix(V_BUILD_BRANCH, "wip-");

    constexpr bool is_dirty = in_string(V_BUILD_VER, '-');

    constexpr bool show_branch = (!is_main && !(is_release && is_stable) && !is_head);
    constexpr bool show_commit = (!is_release || (!is_main && !is_stable && !is_head));

    // show version
    if(!git_only)
        SuperPrintRightAlign("v" V_LATEST_STABLE, 5, XRender::TargetW - XRender::TargetOverscanX - 2, 2);

    // show branch
    if(show_branch && !disable_git)
    {
        int y = show_commit ? XRender::TargetH - 36 : XRender::TargetH - 18;

        if(is_wip)
        {
            // strip the "wip-"
            SuperPrintRightAlign(&V_BUILD_BRANCH[find_in_string(V_BUILD_BRANCH, '-') + 1], 5, XRender::TargetW - XRender::TargetOverscanX - 2, y);
        }
        else
            SuperPrintRightAlign(V_BUILD_BRANCH, 5, XRender::TargetW - XRender::TargetOverscanX - 2, y);
    }

    // show git commit
    if(show_commit && !disable_git)
    {
        if(is_dirty)
        {
            // only show -d, not -dirty
            SuperPrintRightAlign(find_in_string(V_BUILD_VER, '-') + 2 + 1, "#" V_BUILD_VER, 5, XRender::TargetW - XRender::TargetOverscanX - 2, XRender::TargetH - 18);
        }
        else
            SuperPrintRightAlign("#" V_BUILD_VER, 5, XRender::TargetW - XRender::TargetOverscanX - 2, XRender::TargetH - 18);
    }
}

static void s_drawGameTypeTitle(int x, int y)
{
    if(MenuMode == MENU_EDITOR)
        SuperPrint(g_mainMenu.mainEditor, 3, x, y, XTColorF(0.8f, 0.8f, 0.3f));
    else if(menuBattleMode)
        SuperPrint(g_mainMenu.mainBattleGame, 3, x, y, XTColorF(0.3f, 0.3f, 1.0f));
    else
    {
        // show "Play Episode" when at the save slot selecting and adjusting settings (not when the game was started in compat / speedrun mode)
        if(!s_show_separate_2P())
            SuperPrint(g_mainMenu.mainPlayEpisode, 3, x, y, XTColorF(1.0f, 0.3f, 0.3f));
        else if(menuPlayersNum == 1)
            SuperPrint(g_mainMenu.main1PlayerGame, 3, x, y, XTColorF(1.0f, 0.3f, 0.3f));
        else
            SuperPrint(g_mainMenu.mainMultiplayerGame, 3, x, y, XTColorF(0.3f, 1.0f, 0.3f));
    }
}

static void s_drawGameSaves(int MenuX, int MenuY)
{
    int A;

    for(A = 1; A <= maxSaveSlots; A++)
    {
        int posY = MenuY - 30 + (A * 30);
        const auto& info = SaveSlotInfo[A];

        if(info.Progress >= 0)
        {
            // "SLOT {0} ... {1}%"
            std::string p = fmt::format_ne(g_mainMenu.gameSlotContinue, A, SaveSlotInfo[A].Progress);
            int len = SuperTextPixLen(p, 3);

            SuperPrint(p, 3, MenuX, posY);

            if(info.Stars > 0)
            {
                len += 4;
                XRender::renderTexture(MenuX + len, posY + 1,
                                      GFX.Interface[5].w, GFX.Interface[5].h,
                                      GFX.Interface[5], 0, 0);

                len += GFX.Interface[5].w + 4;
                XRender::renderTexture(MenuX + len, posY + 2,
                                      GFX.Interface[1].w, GFX.Interface[1].h,
                                      GFX.Interface[1], 0, 0);

                len += GFX.Interface[1].w + 4;
                SuperPrint(fmt::format_ne("{0}", info.Stars), 3, MenuX + len, posY);
            }
        }
        else
        {
            // "SLOT {0} ... NEW GAME"
            SuperPrint(fmt::format_ne(g_mainMenu.gameSlotNew, A), 3, MenuX, posY);
        }

        int mode_icon_X = MenuX + 340;
        int mode_icon_Y = MenuY - 34 + (A * 30);

        int save_configs = info.ConfigDefaults;
        if(save_configs == 0 && A == MenuCursor + 1 && (MenuMode == MENU_SELECT_SLOT_1P || MenuMode == MENU_SELECT_SLOT_2P))
        {
            save_configs = s_episode_playstyle + 1;
            if((int)s_episode_speedrun_mode != 0)
                save_configs = -s_episode_speedrun_mode;
            if((int)g_config.speedrun_mode != 0)
                save_configs = -g_config.speedrun_mode;
        }

        if(save_configs != 0)
        {
            float rot = 0;
            float op = 1.0f;
            if(A == MenuCursor + 1)
            {
                rot = std::abs((int)(CommonFrame % 90) - 45);
                rot = rot - 22.5;
            }

            if(info.ConfigDefaults == 0)
            {
                op = std::abs((int)(CommonFrame % 30) - 15);
                op /= 100.0f;
                op += 0.5f;
            }

            if(save_configs == Config_t::MODE_CLASSIC + 1)
            {
                XRender::renderTextureScaleEx(mode_icon_X, mode_icon_Y, 24, 24, GFXNPC[NPCID_POWER_S3], 0, 0, 32, 32, rot, nullptr, X_FLIP_NONE, XTAlphaF(op));
            }
            else if(save_configs == Config_t::MODE_VANILLA + 1)
            {
                XRender::renderTextureScaleEx(mode_icon_X, mode_icon_Y, 24, 24, GFXNPC[NPCID_FODDER_S3], 0, 0, 32, 32, rot, nullptr, X_FLIP_NONE, XTAlphaF(op));
            }
            else if(save_configs < 0)
            {
                XRender::renderTextureScaleEx(mode_icon_X, mode_icon_Y, 24, 24, GFXNPC[NPCID_TIMER_S2], 0, 0, 32, 32, rot, nullptr, X_FLIP_NONE, XTAlphaF(op));
                if(save_configs > -10)
                    XRender::renderTexture(mode_icon_X + 12, mode_icon_Y + 12, GFX.Font1[-save_configs]);
            }
            else
            {
                if(GFX.EIcons.inited)
                    XRender::renderTextureScaleEx(mode_icon_X, mode_icon_Y, 24, 24, GFX.EIcons, 0, 32*Icon::thextech, 32, 32, rot, nullptr, X_FLIP_NONE, XTAlphaF(op));
                else
                    XRender::renderTextureScaleEx(mode_icon_X, mode_icon_Y, 24, 24, GFXNPC[NPCID_ICE_POWER_S3], 0, 0, 32, 32, rot, nullptr, X_FLIP_NONE, XTAlphaF(op));
            }
        }
    }

    if(MenuMode == MENU_SELECT_SLOT_1P || MenuMode == MENU_SELECT_SLOT_2P)
    {
        SuperPrint(g_mainMenu.gameCopySave, 3, MenuX, MenuY - 30 + (A * 30));
        A++;
        SuperPrint(g_mainMenu.gameEraseSave, 3, MenuX, MenuY - 30 + (A * 30));
    }


    if(MenuCursor < 0 || MenuCursor >= maxSaveSlots || (MenuMode != MENU_SELECT_SLOT_1P && MenuMode != MENU_SELECT_SLOT_2P))
        return;

    const auto& info = SaveSlotInfo[MenuCursor + 1];

    int infobox_x = XRender::TargetW / 2 - 240;
    int infobox_y = MenuY + 145 + c_menuSavesOffsetY;

    // forbid incompatible speedrun
    if((int)g_config.speedrun_mode != 0 && g_config.speedrun_mode != -info.ConfigDefaults
        && (info.ConfigDefaults != 0 || info.Progress >= 0))
    {
        XRender::renderRect(XRender::TargetW / 2 - 240, infobox_y, 480, 68, XTColorF(0, 0, 0, 1.0f));
        SuperPrintScreenCenter("SAVE INCOMPATIBLE", 3, infobox_y + 4, XTColorF(1.0f, 0.0f, 0.0f, 1.0f));
        SuperPrintScreenCenter(fmt::format_ne("WITH SPEEDRUN MODE {0}", (int)g_config.speedrun_mode), 3, infobox_y + 24, XTColorF(1.0f, 0.0f, 0.0f, 1.0f));
        SuperPrintScreenCenter("(COMMAND LINE FLAG)", 3, infobox_y + 44, XTColorF(0.8f, 0.8f, 0.8f, 1.0f));
    }
    // initialize config
    else if(info.ConfigDefaults == 0)
    {
        XRender::renderRect(infobox_x, infobox_y, 480, 68, {0, 0, 0, 192});

        XTColor color;

        if(s_episode_playstyle == Config_t::MODE_MODERN)
            color = XTColorF(0.5f, 0.8f, 1.0f);
        else if(s_episode_playstyle == Config_t::MODE_CLASSIC)
            color = XTColorF(1.0f, 0.5f, 0.5f);
        else
            color = XTColorF(0.8f, 0.5f, 0.2f);

        if(s_episode_speedrun_mode != 0)
        {
            SuperPrintScreenCenter("SPEEDRUN MODE " + std::to_string(s_episode_speedrun_mode), 3, infobox_y + 14, color);
        }
        else
        {
            // int target_bugfixes = (SelectWorld[selWorld].bugfixes_on_by_default) ? Config_t::MODE_MODERN : Config_t::MODE_CLASSIC;

            std::string playstyle_string;
            playstyle_string += g_options.playstyle.m_display_name;
            playstyle_string += ": ";

            if(s_episode_playstyle == Config_t::MODE_MODERN)
                playstyle_string += g_options.playstyle.m_enum_values[0].m_display_name;
            else if(s_episode_playstyle == Config_t::MODE_CLASSIC)
                playstyle_string += g_options.playstyle.m_enum_values[1].m_display_name;
            else
                playstyle_string += g_options.playstyle.m_enum_values[2].m_display_name;

            // if(target_bugfixes == g_config.playstyle)
            //     playstyle_string += " (Recommended)";

            SuperPrintScreenCenter(playstyle_string, 3, infobox_y + 14, color);

            // switch arrows
            s_draw_infobox_switch_arrows(infobox_x, infobox_y);
        }

        const std::string& playstyle_description
            = (s_episode_playstyle == Config_t::MODE_MODERN) ?
                g_options.playstyle.m_enum_values[0].m_display_tooltip
            : (s_episode_playstyle == Config_t::MODE_CLASSIC) ?
                g_options.playstyle.m_enum_values[1].m_display_tooltip
            :
                g_options.playstyle.m_enum_values[2].m_display_tooltip;

        SuperPrintScreenCenter(playstyle_description, 5, infobox_y + 34, color);
    }
    // display fun save slot info
    else if(info.Progress >= 0)
    {
        int row_1 = infobox_y + 10;
        int row_2 = infobox_y + 42;
        int row_c = infobox_y + 26;

        bool hasFails = (g_config.enable_fails_tracking || info.FailsEnabled);
        bool show_timer = info.Time > 0 && (g_config.enable_playtime_tracking || info.ConfigDefaults < 0);
        bool show_score = (s_episode_playstyle != Config_t::MODE_VANILLA);

        // recenter if single row
        if(!show_timer && !hasFails)
            row_1 = infobox_y + 26;

        XRender::renderRect(infobox_x, infobox_y, 480, 68, {0, 0, 0, 127});

        // Temp string
        std::string t;

        // Score
        if(show_score)
        {
            int row_score = show_timer ? row_1 : row_c;
            SuperPrint(t = fmt::format_ne(g_mainMenu.phraseScore, info.Score), 3, infobox_x + 10, row_score);
        }

        // Gameplay Timer
        if(show_timer)
        {
            std::string t = GameplayTimer::formatTime(info.Time);

            if(t.size() > 9)
                t = t.substr(0, t.size() - 4);

            SuperPrint(fmt::format_ne(g_mainMenu.phraseTime, t), 3, infobox_x + 10, row_2);
        }

        // If demos off, put (l)ives and (c)oins on center
        int row_lc = (hasFails) ? row_1 : row_c;

        // Print lives on the screen (from gfx_update2.cpp)
        DrawLives(infobox_x + 272 + 32, row_lc, info.Lives, info.Hundreds, s_episode_playstyle == Config_t::MODE_VANILLA);

        // Print coins on the screen (from gfx_update2.cpp)
        int coins_x = infobox_x + 480 - 10 - 36 - 62;
        XRender::renderTexture(coins_x + 16, row_lc, GFX.Interface[2]);
        XRender::renderTexture(coins_x + 40, row_lc + 2, GFX.Interface[1]);
        SuperPrintRightAlign(t = std::to_string(info.Coins), 1, coins_x + 62 + 36, row_lc + 2);

        // Fails Counter
        if(hasFails)
            SuperPrintRightAlign(fmt::format_ne("{0}: {1}", g_gameInfo.fails_counter_title, info.Fails), 3, infobox_x + 480 - 10, row_2);
    }
}

void mainMenuDraw()
{
    int MenuX, MenuY;
    GetMenuPos(&MenuX, &MenuY);

    int B = 0;

    if(MenuMode != MENU_1PLAYER_GAME && MenuMode != MENU_2PLAYER_GAME && MenuMode != MENU_BATTLE_MODE && MenuMode != MENU_EDITOR)
        worldCurs = 0;

#ifdef __3DS__
    XRender::setTargetLayer(2);
#endif

    // Render the permanent menu graphics (curtain, URL, logo)

    // URL
    if(XRender::TargetH >= SmallScreenH)
        XRender::renderTexture(XRender::TargetW / 2 - GFX.MenuGFX[3].w / 2, XRender::TargetH - 24, GFX.MenuGFX[3]);

    bool at_main_title = (MenuMode == MENU_MAIN || MenuMode == MENU_INTRO);
    bool draw_in_asset_pack = at_main_title && s_startAssetPackTimer >= 2;

    if(!draw_in_asset_pack)
    {
        // Curtain
        // correction to loop the original asset properly
        int curtain_draw_w = GFX.MenuGFX[1].w;
        if(curtain_draw_w == 800)
            curtain_draw_w = 768;
        int curtain_horiz_reps = XRender::TargetW / curtain_draw_w + 2;

        for(int i = 0; i < curtain_horiz_reps; i++)
            XRender::renderTexture(curtain_draw_w * i, 0, curtain_draw_w, GFX.MenuGFX[1].h, GFX.MenuGFX[1], 0, 0);

#ifdef __3DS__
        if(at_main_title)
            XRender::setTargetLayer(3);
#endif

        // game logo
        int LogoMode = 0;
        if(XRender::TargetH >= TinyScreenH || MenuMode == MENU_INTRO)
            LogoMode = 1;
        else if(MenuMode == MENU_MAIN)
            LogoMode = 2;

        if(LogoMode == 1)
        {
            // show at half opacity if not at main menu on a small screen
            XTColor logo_tint = (XRender::TargetH < SmallScreenH && MenuMode != MENU_INTRO && MenuMode != MENU_MAIN) ? XTAlpha(127) : XTColor();

            int logo_y = XRender::TargetH / 2 - 230;

            // place manually on small screens
            if(XRender::TargetH < TinyScreenH)
                logo_y = 16 + (XRender::TargetH - 320) / 2;
            else if(XRender::TargetH < SmallScreenH)
                logo_y = 16;
            else if(XRender::TargetH <= 600)
                logo_y = 40;

            XRender::renderTexture(XRender::TargetW / 2 - GFX.MenuGFX[2].w / 2, logo_y, GFX.MenuGFX[2], logo_tint);
        }
        else if(LogoMode == 2)
        {
            SuperPrint(g_gameInfo.title, 3, XRender::TargetW / 2 - g_gameInfo.title.length()*9, 30);
        }
    }

#ifdef __3DS__
    if(MenuMode != MENU_NEW_OPTIONS)
        XRender::setTargetLayer(3);
#endif

    drawGameVersion(false, draw_in_asset_pack);


    // Menu Intro
    if(MenuMode == MENU_INTRO)
    {
        if((CommonFrame % 90) < 45)
            SuperPrintScreenCenter(g_mainMenu.introPressStart, 3, XRender::TargetH - 48 - (XRender::TargetH - 320) / 4);
    }

#ifndef PGE_NO_THREADING
    // loading (can't safely render)
    if(SDL_AtomicGet(&loading))
    {
        if(SDL_AtomicGet(&loadingProgrssMax) <= 0)
            SuperPrint(g_mainMenu.loading, 3, MenuX, MenuY);
        else
        {
            int progress = (SDL_AtomicGet(&loadingProgrss) * 100) / SDL_AtomicGet(&loadingProgrssMax);
            SuperPrint(fmt::format_ne("{0} {1}%", g_mainMenu.loading, progress), 3, MenuX, MenuY);
        }
    }
    else
#endif
    // DO NOT DETACH THE ABOVE ELSE STATEMENT FROM THE FOLLOWING SERIES OF IF CLAUSES

    // Main menu
    if(MenuMode == MENU_MAIN)
    {
        int i = 0;
        SuperPrint(s_show_separate_2P() ? g_mainMenu.main1PlayerGame : g_mainMenu.mainPlayEpisode, 3, MenuX, MenuY+30*(i++));
        if(s_show_separate_2P())
            SuperPrint(g_mainMenu.mainMultiplayerGame, 3, MenuX, MenuY+30*(i++));
        if(!g_gameInfo.disableBattleMode)
            SuperPrint(g_mainMenu.mainBattleGame, 3, MenuX, MenuY+30*(i++));
        if(g_config.enable_editor)
            SuperPrint(g_mainMenu.mainEditor, 3, MenuX, MenuY+30*(i++));
        SuperPrint(g_mainMenu.mainOptions, 3, MenuX, MenuY+30*(i++));
        SuperPrint(g_mainMenu.mainExit, 3, MenuX, MenuY+30*(i++));
        XRender::renderTexture(MenuX - 20, MenuY + (MenuCursor * 30), 16, 16, GFX.MCursor[0], 0, 0);
    }

    // Character select
    else if(MenuMode == MENU_CHARACTER_SELECT_NEW ||
            MenuMode == MENU_CHARACTER_SELECT_NEW_BM)
    {
        s_drawGameTypeTitle(MenuX, MenuY - 70);
        const auto& world_list = (MenuMode == MENU_CHARACTER_SELECT_NEW) ? SelectWorld : SelectBattle;
        SuperPrint(world_list[selWorld].WorldName, 3, MenuX, MenuY - 40, XTColorF(0.6f, 1.f, 1.f));

        ConnectScreen::Render();
    }
#if 0 // dead now
    else if(MenuMode == MENU_CHARACTER_SELECT_1P ||
            MenuMode == MENU_CHARACTER_SELECT_2P_S1 ||
            MenuMode == MENU_CHARACTER_SELECT_2P_S2 ||
            MenuMode == MENU_CHARACTER_SELECT_BM_S1 ||
            MenuMode == MENU_CHARACTER_SELECT_BM_S2)
    {
        A = 0;
        B = 0;
        C = 0;

        s_drawGameTypeTitle(MenuX, 280);
        SuperPrint(SelectWorld[selWorld].WorldName, 3, MenuX, 310, XTColorF(0.6f, 1.f, 1.f));

        // TODO: Make a custom playable character names print here
        if(!blockCharacter[1])
            SuperPrint(g_mainMenu.selectPlayer[1], 3, MenuX, MenuY);
        else
        {
            A -= 30;
            if(MenuCursor + 1 >= 1)
                B -= 30;
            if(PlayerCharacter >= 1)
                C -= 30;
        }

        if(!blockCharacter[2])
            SuperPrint(g_mainMenu.selectPlayer[2], 3, MenuX, 380 + A);
        else
        {
            A -= 30;
            if(MenuCursor + 1 >= 2)
                B -= 30;
            if(PlayerCharacter >= 2)
                C -= 30;
        }

        if(!blockCharacter[3])
            SuperPrint(g_mainMenu.selectPlayer[3], 3, MenuX, 410 + A);
        else
        {
            A -= 30;
            if(MenuCursor + 1 >= 3)
                B -= 30;
            if(PlayerCharacter >= 3)
                C -= 30;
        }

        if(!blockCharacter[4])
            SuperPrint(g_mainMenu.selectPlayer[4], 3, MenuX, 440 + A);
        else
        {
            A -= 30;
            if(MenuCursor + 1 >= 4)
                B -= 30;
            if(PlayerCharacter >= 4)
                C -= 30;
        }

        if(!blockCharacter[5])
            SuperPrint(g_mainMenu.selectPlayer[5], 3, MenuX, 470 + A);
        else
        {
            A -= 30;
            if(MenuCursor + 1 >= 5)
                B -= 30;
            if(PlayerCharacter >= 5)
                C -= 30;
        }

        if(MenuMode == MENU_CHARACTER_SELECT_2P_S2 || MenuMode == MENU_CHARACTER_SELECT_BM_S2)
        {
            XRender::renderTexture(MenuX - 20, B + MenuY + (MenuCursor * 30), GFX.MCursor[3]);
            XRender::renderTexture(MenuX - 20, B + MenuY + ((PlayerCharacter - 1) * 30), GFX.MCursor[0]);
        }
        else
        {
            XRender::renderTexture(MenuX - 20, B + MenuY + (MenuCursor * 30), GFX.MCursor[0]);
        }
    }
#endif

    // Episode / Level selection
    else if(MenuMode == MENU_1PLAYER_GAME || MenuMode == MENU_2PLAYER_GAME || MenuMode == MENU_BATTLE_MODE || MenuMode == MENU_EDITOR)
    {
        s_drawGameTypeTitle(MenuX, MenuY - 70);
        // std::string tempStr;

        minShow = 1;
        maxShow = (MenuMode == MENU_BATTLE_MODE) ? NumSelectBattle :
            ((MenuMode == MENU_EDITOR) ? NumSelectWorldEditable :
                NumSelectWorld);
        const std::vector<SelectWorld_t>& SelectorList
            = (MenuMode == MENU_BATTLE_MODE) ? SelectBattle :
                ((MenuMode == MENU_EDITOR) ? SelectWorldEditable :
                    SelectWorld);

        int original_maxShow = maxShow;
        if(maxShow > 5)
        {
            minShow = worldCurs;
            maxShow = minShow + 4;

            if(MenuCursor <= minShow - 1)
                worldCurs -= 1;

            if(MenuCursor >= maxShow - 1)
                worldCurs += 1;

            if(worldCurs < 1)
                worldCurs = 1;

            if(worldCurs > original_maxShow - 4)
                worldCurs = original_maxShow - 4;

            minShow = worldCurs;
            maxShow = minShow + 4;
        }

        for(auto A = minShow; A <= maxShow; A++)
        {
            auto w = SelectorList[A];

            XTColor color;
            color.r = w.highlight ? 0 : 255;

            if(w.disabled)
                color = {127, 127, 127};

            if(w.probably_incompatible)
                color = {255, 127, 127};

            B = A - minShow + 1;

            SuperPrint(w.WorldName, 3, MenuX, MenuY - 30 + (B * 30), color);
        }

        // preview type of content being created
        int first_new_content = (g_gameInfo.disableBattleMode) ? NumSelectWorldEditable : NumSelectWorldEditable - 1;
        if(MenuMode == MENU_EDITOR && MenuCursor + 1 >= first_new_content)
        {
            B = MenuCursor + 1 - minShow + 1;

            // draw mode icon
            int mode_icon_X = MenuX + 340;
            int mode_icon_Y = MenuY - 34 + (B * 30);

            float rot = 0;
            rot = std::abs((int)(CommonFrame % 90) - 45);
            rot = rot - 22.5;

            if(!s_editor_target_thextech)
                XRender::renderTextureScaleEx(mode_icon_X, mode_icon_Y, 24, 24, GFXNPC[NPCID_POWER_S3], 0, 0, 32, 32, rot, nullptr, X_FLIP_NONE);
            else if(GFX.EIcons.inited)
                XRender::renderTextureScaleEx(mode_icon_X, mode_icon_Y, 24, 24, GFX.EIcons, 0, 32*Icon::thextech, 32, 32, rot, nullptr, X_FLIP_NONE);
            else
                XRender::renderTextureScaleEx(mode_icon_X, mode_icon_Y, 24, 24, GFXNPC[NPCID_ICE_POWER_S3], 0, 0, 32, 32, rot, nullptr, X_FLIP_NONE);

            // draw infobox
            int infobox_x = XRender::TargetW / 2 - 240;
            int infobox_y = MenuY + 145;

            XRender::renderRect(infobox_x, infobox_y, 480, 68, {0, 0, 0, 192});

            XTColor color;

            if(s_editor_target_thextech)
                color = XTColorF(0.5f, 0.8f, 1.0f);
            else
                color = XTColorF(1.0f, 0.5f, 0.5f);

            SuperPrintScreenCenter(g_mainMenu.editorMakeFor, 3, infobox_y + 14, color);

            if(s_editor_target_thextech)
                SuperPrintScreenCenter("TheXTech", 3, infobox_y + 34, color);
            else
                SuperPrintScreenCenter("SMBX 1.3", 3, infobox_y + 34, color);

            s_draw_infobox_switch_arrows(infobox_x, infobox_y);
        }

        // render the scroll indicators
        if(minShow > 1)
            XRender::renderTexture(XRender::TargetW/2 - 8, MenuY - 20, GFX.MCursor[1]);

        if(maxShow < original_maxShow)
            XRender::renderTexture(XRender::TargetW/2 - 8, MenuY + 140, GFX.MCursor[2]);

        B = MenuCursor - minShow + 1;

        if(B >= 0 && B < 5)
            XRender::renderTexture(MenuX - 20, MenuY + (B * 30), GFX.MCursor[0].w, GFX.MCursor[0].h, GFX.MCursor[0], 0, 0);
    }

    else if(MenuMode == MENU_SELECT_SLOT_1P || MenuMode == MENU_SELECT_SLOT_2P) // Save Select
    {
        s_drawGameTypeTitle(MenuX, MenuY - 70);
        SuperPrint(SelectWorld[selWorld].WorldName, 3, MenuX, MenuY - 40, XTColorF(0.6f, 1.f, 1.f));
        s_drawGameSaves(MenuX, MenuY);
        XRender::renderTexture(MenuX - 20, MenuY + (MenuCursor * 30), GFX.MCursor[0]);
    }

    else if(MenuMode == MENU_SELECT_SLOT_1P_COPY_S1 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S1 ||
            MenuMode == MENU_SELECT_SLOT_1P_COPY_S2 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S2) // Copy save
    {
        s_drawGameTypeTitle(MenuX, MenuY - 70);
        SuperPrint(SelectWorld[selWorld].WorldName, 3, MenuX, MenuY - 40, XTColorF(0.6f, 1.f, 1.f));
        s_drawGameSaves(MenuX, MenuY);

        if(MenuMode == MENU_SELECT_SLOT_1P_COPY_S1 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S1)
            SuperPrint(g_mainMenu.gameSourceSlot, 3, MenuX, MenuY + c_menuSavesFooterHint, XTColorF(0.7f, 0.7f, 1.0f));
        else if(MenuMode == MENU_SELECT_SLOT_1P_COPY_S2 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S2)
            SuperPrint(g_mainMenu.gameTargetSlot, 3, MenuX, MenuY + c_menuSavesFooterHint, XTColorF(0.7f, 1.0f, 0.7f));

        if(MenuMode == MENU_SELECT_SLOT_1P_COPY_S2 || MenuMode == MENU_SELECT_SLOT_2P_COPY_S2)
        {
            XRender::renderTexture(MenuX - 20, MenuY + ((menuCopySaveSrc - 1) * 30), GFX.MCursor[0]);
            XRender::renderTexture(MenuX - 20, MenuY + (MenuCursor * 30), GFX.MCursor[3]);
        }
        else
            XRender::renderTexture(MenuX - 20, MenuY + (MenuCursor * 30), GFX.MCursor[0]);
    }

    else if(MenuMode == MENU_SELECT_SLOT_1P_DELETE || MenuMode == MENU_SELECT_SLOT_2P_DELETE) // Delete save
    {
        s_drawGameTypeTitle(MenuX, MenuY - 70);
        SuperPrint(SelectWorld[selWorld].WorldName, 3, MenuX, MenuY - 40, XTColorF(0.6f, 1.f, 1.f));
        s_drawGameSaves(MenuX, MenuY);

        SuperPrint(g_mainMenu.gameEraseSlot, 3, MenuX, MenuY + c_menuSavesFooterHint, XTColorF(1.0f, 0.7f, 0.7f));

        XRender::renderTexture(MenuX - 20, MenuY + (MenuCursor * 30), GFX.MCursor[0]);
    }

    else if(MenuMode == MENU_SELECT_SLOT_1P_ADVMODE || MenuMode == MENU_SELECT_SLOT_2P_ADVMODE) // Advanced mode select
    {
        s_drawGameTypeTitle(MenuX, MenuY - 70);
        SuperPrint(SelectWorld[selWorld].WorldName, 3, MenuX, MenuY - 40, XTColorF(0.6f, 1.f, 1.f));

        int A = 0;

        SuperPrint("Modern Game", 3, MenuX, MenuY + (A++ * 30), XTColorF(0.5f, 0.7f, 1.0f));
        SuperPrint("Classic Game", 3, MenuX, MenuY + (A++ * 30), XTColorF(1.0f, 0.7f, 0.7f));
        SuperPrint("Vanilla Game", 3, MenuX, MenuY + (A++ * 30), XTColorF(0.8f, 0.5f, 0.2f));
        SuperPrint("Speedrun Mode 1", 3, MenuX, MenuY + (A++ * 30), XTColorF(0.5f, 0.7f, 1.0f));
        SuperPrint("Speedrun Mode 2", 3, MenuX, MenuY + (A++ * 30), XTColorF(1.0f, 0.7f, 0.7f));
        SuperPrint("Speedrun Mode 3", 3, MenuX, MenuY + (A++ * 30), XTColorF(0.8f, 0.5f, 0.2f));

        XRender::renderTexture(MenuX - 20, MenuY + (MenuCursor * 30), GFX.MCursor[0]);
    }

    // Player controls setup
    else if(MenuMode == MENU_INPUT_SETTINGS)
    {
        menuControls_Render();
    }

    // New options screen
    else if(MenuMode == MENU_NEW_OPTIONS)
    {
#ifdef __3DS__
        // draw options screen (only) in top draw plane
        XRender::setTargetLayer(3);
#endif

        OptionsScreen::Render();
    }

    // fade to / from asset pack screen
    if(s_can_enter_ap_screen() && s_startAssetPackTimer > 0)
    {
        ScreenAssetPack::DrawBackground(s_startAssetPackTimer / 60.0);
        g_levelScreenFader.clearFader();
    }

    // Mouse cursor
    XRender::renderTexture(int(SharedCursor.X), int(SharedCursor.Y), GFX.ECursor[2]);
}
