/*
 * TheXTech - A platform game engine ported from old source code for VB6
 *
 * Copyright (c) 2009-2011 Andrew Spinks, original VB6 code
 * Copyright (c) 2020-2025 Vitaly Novichkov <admin@wohlnet.ru>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <algorithm>

#include <AppPath/app_path.h>

#include <Utils/files.h>
#include <Utils/files_ini.h>
#include <DirManager/dirman.h>
#include <Logger/logger.h>
#include <IniProcessor/ini_processing.h>

#include "main/asset_pack.h"
#include "main/translate.h"
#include "core/render.h"
#include "core/msgbox.h"
#include "core/language.h"

#include "sound.h"
#include "gfx.h"
#include "load_gfx.h"
#include "game_main.h"

static std::vector<AssetPack_t> s_asset_packs;
static bool s_found_asset_packs = false;

std::string g_AssetPackID;
bool g_AssetsLoaded = false;

static void appendSlash(std::string &path)
{
#if defined(__EMSCRIPTEN__)
    // fix emscripten bug of duplicated worlds
    if(path.empty() || path.back() != '/')
        path.push_back('/');
#else
    if(!path.empty() && path.back() != '/')
        path.push_back('/');
#endif
}

static inline void s_load_image(StdPicture& dest, std::string nameWithPng)
{
    dest.reset();

#if defined(X_IMG_EXT)
    if(nameWithPng.size() > 4)
    {
        std::string orig_ext;
        size_t base_size = nameWithPng.find_last_of('.');

        if(base_size == std::string::npos)
            base_size = nameWithPng.size();
        else
        {
            orig_ext = nameWithPng.substr(base_size);
            nameWithPng.resize(base_size);
        }

        nameWithPng += X_IMG_EXT;

        if(Files::fileExists(nameWithPng))
        {
            XRender::lazyLoadPicture(dest, nameWithPng);

            if(dest.inited)
                return; // Success
        }

#   if !defined(X_NO_PNG_GIF)
        nameWithPng.resize(base_size);

        if(!orig_ext.empty())
            nameWithPng += orig_ext;
#   endif // !defined(X_NO_PNG_GIF)
    }
#endif // defined(X_IMG_EXT)

#if !defined(X_NO_PNG_GIF)
    if(!dest.inited && Files::fileExists(nameWithPng))
        XRender::lazyLoadPicture(dest, nameWithPng);
#endif
}

static AssetPack_t s_scan_asset_pack(const std::string& path, bool skip_graphics = false)
{
    AssetPack_t ret;
    ret.path = path;
    ret.gfx.reset(new AssetPack_t::Gfx());

    appendSlash(ret.path);

    // based on Launcher.java:updateOverlook()
    const char* assets_icon = "graphics/ui/icon/thextech_128.png";
    const char* logo_image_path_default = "graphics/ui/MenuGFX2.png";
    const char* bg_image_path_default = "graphics/background2/background2-2.png";

    std::string logo_image_path;
    std::string bg_image_path;

    ret.gfx->bg_frames = 1;
    ret.gfx->bg_frame_ticks = 125;

    std::string gi_path = ret.path + "gameinfo.ini";

    if(Files::fileExists(gi_path))
    {
        IniProcessing gameinfo = Files::load_ini(gi_path);

        gameinfo.beginGroup("game");
        gameinfo.read("id", ret.id, ret.id);
        gameinfo.read("version", ret.version, ret.version);
        gameinfo.read("show-id", ret.show_id, ret.show_id);
        gameinfo.endGroup();

        gameinfo.beginGroup("android");
        gameinfo.read("logo", logo_image_path, logo_image_path_default);
        gameinfo.read("background", bg_image_path, bg_image_path_default);
        gameinfo.read("background-frames", ret.gfx->bg_frames, ret.gfx->bg_frames);
        gameinfo.read("background-delay", ret.gfx->bg_frame_ticks, ret.gfx->bg_frame_ticks);
        gameinfo.endGroup();
    }

    if(logo_image_path != logo_image_path_default)
        ret.logo_override = true;

    // don't allow any dirsep characters in asset pack ID
    for(char& c : ret.id)
    {
        if(c == '/' || c == '\\')
            c = '_';
    }

    // ms to frames (approximately)
    ret.gfx->bg_frame_ticks /= 15;

    // load graphics
    if(!skip_graphics)
    {
        s_load_image(ret.gfx->icon, ret.path + assets_icon);
        s_load_image(ret.gfx->logo, ret.path + logo_image_path);
        s_load_image(ret.gfx->background, ret.path + bg_image_path);

        if(!ret.gfx->logo.inited && logo_image_path != logo_image_path_default)
            s_load_image(ret.gfx->logo, ret.path + logo_image_path_default);

        if(!ret.gfx->background.inited && bg_image_path != bg_image_path_default)
            s_load_image(ret.gfx->background, ret.path + bg_image_path_default);
    }

    return ret;
}

static void s_strip_id(AssetPack_t& pack)
{
    if(!pack.id.empty() && !pack.version.empty())
        pack.version = pack.id + '-' + pack.version;
    else if(!pack.id.empty())
        pack.version = pack.id;

    pack.id.clear();
}

static void s_find_asset_packs()
{
    pLogDebug("Searching for asset packs in...");

    DirMan assets;
    std::vector<std::string> subdirList;
    std::string subdir;

    for(const auto& root_ : AppPathManager::assetsSearchPath())
    {
        std::string root = root_.first;
        AssetsPathType type = root_.second;

        // check for a root passed via `-c`
        bool is_modern_root = (type == AssetsPathType::Single);
        bool is_multiple_root = (type == AssetsPathType::Multiple);

        // Normally, root is a legacy asset pack, and <root>/assets/ contains modern asset packs.
        // If passed via -c, root must be a modern asset pack also.

        // check for legacy debug assets
        if(!is_multiple_root)
            pLogDebug("- %s%s", root.c_str(), (is_modern_root) ? "" : " (legacy)");

        if(!is_multiple_root && DirMan::exists(root + "graphics/ui/"))
        {
            AssetPack_t pack = s_scan_asset_pack(root);

            // strip the ID from a legacy asset pack
            if(!is_modern_root)
                s_strip_id(pack);

            if(!pack.gfx || !pack.gfx->logo.inited)
                pLogWarning("  Could not load UI assets from %s asset pack [%s], ignoring", (is_modern_root) ? "user-specified" : "possible legacy", pack.path.c_str());
            else if(is_modern_root && pack.id.empty())
                pLogWarning("  Could not read ID of modern asset pack [%s], ignoring", pack.path.c_str());
            else
                s_asset_packs.push_back(std::move(pack));
        }

        if(type == AssetsPathType::Legacy)
            root += "assets/";

        if(!is_modern_root)
            pLogDebug("- %s*", root.c_str());

        if(!is_modern_root && DirMan::exists(root))
        {
            assets.setPath(root);
            assets.getListOfFolders(subdirList);

            for(const std::string& sub : subdirList)
            {
                subdir = root + sub;

                D_pLogDebug("  Checking %s", subdir.c_str());

                if(DirMan::exists(subdir + "/graphics/ui/"))
                {
                    AssetPack_t pack = s_scan_asset_pack(subdir);

                    if(!pack.gfx || !pack.gfx->logo.inited)
                        pLogWarning("  Could not load UI assets from possible asset pack [%s], ignoring", pack.path.c_str());
                    else if(pack.id.empty())
                        pLogWarning("  Could not read ID of possible asset pack [%s], ignoring", pack.path.c_str());
                    else
                        s_asset_packs.push_back(std::move(pack));
                }
            }
        }
    }

    if(!s_asset_packs.empty())
    {
        pLogDebug("Found asset packs:");

        for(const AssetPack_t& pack : s_asset_packs)
            pLogDebug("- %s (%s)", pack.full_id().c_str(), pack.path.c_str());
    }
    else
        pLogCritical("Could not find any asset packs.");

    // check for duplicates
    for(size_t i = 0; i < s_asset_packs.size(); i++)
    {
        for(size_t j = i + 1; j < s_asset_packs.size();)
        {
            if(s_asset_packs[i].id == s_asset_packs[j].id)
            {
                if(s_asset_packs[i].version == s_asset_packs[j].version)
                {
                    // erase a case with same ID and version
                    s_asset_packs.erase(s_asset_packs.begin() + j);
                    continue;
                }

                // if version is the distinguishing factor, display it
                s_asset_packs[i].show_version = true;
                s_asset_packs[j].show_version = true;
            }

            j++;
        }
    }

    s_found_asset_packs = true;
}

static bool s_try_load_gfx(const AssetPack_t& pack, bool skip_gfx)
{
    pLogDebug("= Trying to load asset pack \"%s/%s\" from [%s]", pack.id.c_str(), pack.version.c_str(), pack.path.c_str());

    AppPathManager::setCurrentAssetPack(pack.id, pack.path);
    AppPath = AppPathManager::assetsRoot();
    g_AssetPackID = pack.full_id();

    XLanguage::findLanguages(); // find present translations
    ReloadTranslations(); // load translations

    if(!skip_gfx && !GFX.load())
    {
        GFX.unLoad();
        pLogWarning("Failed to load GFX from %s", AppPath.c_str());
        return false;
    }

    return true;
}


bool ReloadAssetsFrom(const AssetPack_t& pack)
{
    const AssetPack_t* fallback_pack = nullptr;
    for(const AssetPack_t& pack : s_asset_packs)
    {
        if(pack.path == AppPath && pack.full_id() == g_AssetPackID)
        {
            fallback_pack = &pack;
            break;
        }
    }

    UnloadCustomGFX();
    UnloadWorldCustomGFX();

    GFX.unLoad();
    StopAllSounds();
    StopMusic();
    UnloadSound();

    if(!s_try_load_gfx(pack, false))
    {
        pLogWarning("Failed to load UI assets");

        // restore original GFX
        if(!fallback_pack || !s_try_load_gfx(*fallback_pack, false))
            pLogCritical("Could not reload graphics from previous path.");
        else
            InitSound(); // Restore sound effects

        // also, remove from list of valid asset packs
        for(auto it = s_asset_packs.begin(); it != s_asset_packs.end(); ++it)
        {
            // find the requested pack exactly (by identity, not contents)
            if(&*it != &pack)
                continue;

            pLogInfo("Removed %s from asset pack list because it is missing UI assets", pack.path.c_str());
            s_asset_packs.erase(it);
            break;
        }

        return false;
    }

    pLogDebug("Successfully loaded UI assets; now loading all other assets from [%s]", AppPath.c_str());

    MainLoadAll();
    return true;
}


bool InitUIAssetsFrom(const std::string& cmdline_id, bool skip_gfx)
{
    if(!s_found_asset_packs)
        s_find_asset_packs();

    // figure out exactly what is requested.
    std::string full_id = cmdline_id;
    std::string id_as_path = cmdline_id;
    if(full_id.empty())
    {
        full_id = g_recentAssetPack;
        pLogDebug("Trying to load most recent asset pack [ID: %s]", full_id.c_str());
    }
    else
    {
        appendSlash(id_as_path);
        pLogDebug("Trying to load CLI-specified asset pack [%s]", full_id.c_str());
    }

    // split full ID into ID and version
    std::string id = full_id;
    std::string version;

    auto slash_pos = id.find('/');

    if(slash_pos != std::string::npos)
    {
        version = id.substr(slash_pos + 1);
        id.resize(slash_pos);
    }

    // path-match, full-match, id-match, any
    for(int round = 0; round < 4; round++)
    {
        for(auto it = s_asset_packs.begin(); it != s_asset_packs.end(); /* ++it at bottom of loop */)
        {
            bool matched = false;
            if(round == 0)
                matched = (!cmdline_id.empty() && it->path == cmdline_id);
            else if(round == 1)
                matched = (it->id == id && it->version == version);
            else if(round == 2 && (cmdline_id.empty() || slash_pos == std::string::npos))
                matched = (it->id == id);
            else
                matched = cmdline_id.empty();

            if(matched)
            {
                if(s_try_load_gfx(*it, skip_gfx))
                    return true;

                // failed to load, erase the failed asset pack and move on to next loop
                it = s_asset_packs.erase(it);
                continue;
            }

            ++it;
        }
    }

    if(!skip_gfx && !GFX.load())
        return false;

    return true;
}


const std::vector<AssetPack_t>& GetAssetPacks()
{
    if(!s_found_asset_packs)
        s_find_asset_packs();

    return s_asset_packs;
}
