/*
 * OPN2/OPNA models library - a set of various conversion models for OPL-family chips
 *
 * Copyright (c) 2025-2025 Vitaly Novichkov <admin@wohlnet.ru>
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <stddef.h>
#include "opn_models.h"


/***************************************************************
 *               Standard frequency formula                    *
 ***************************************************************/

/***************************************************************
 * To compute frequency from MIDI tone used the next formula:  *
 * hz = 309.12412 * exp(0.057762265 * tone)                    *
 *    tone - MIDI tone with bend detune as decimals            *
 *                                                             *
 *  The highest tone possible in the range is 30.823808        *
 *  (I mean, without going to higher octaves)                  *
 *                                                             *
 * This table is built using "freq-opna.c" file at the "util"  *
 * directory.                                                  *
 ***************************************************************/


static const double s_expTableDividorOPNA = 0.0084873220;
static const double s_maxToneValueOPNA = 32.693164;


#define OPNA_EXP_TABLE_SIZE  3853

static const uint16_t s_OPNAgenericExpTable[OPNA_EXP_TABLE_SIZE] =
{
    0x135, 0x135, 0x135, 0x135, 0x135, 0x135, 0x136, 0x136, 0x136, 0x136,
    0x136, 0x136, 0x136, 0x137, 0x137, 0x137, 0x137, 0x137, 0x137, 0x138,
    0x138, 0x138, 0x138, 0x138, 0x138, 0x138, 0x139, 0x139, 0x139, 0x139,
    0x139, 0x139, 0x13A, 0x13A, 0x13A, 0x13A, 0x13A, 0x13A, 0x13A, 0x13B,
    0x13B, 0x13B, 0x13B, 0x13B, 0x13B, 0x13C, 0x13C, 0x13C, 0x13C, 0x13C,
    0x13C, 0x13C, 0x13D, 0x13D, 0x13D, 0x13D, 0x13D, 0x13D, 0x13E, 0x13E,
    0x13E, 0x13E, 0x13E, 0x13E, 0x13E, 0x13F, 0x13F, 0x13F, 0x13F, 0x13F,
    0x13F, 0x140, 0x140, 0x140, 0x140, 0x140, 0x140, 0x141, 0x141, 0x141,
    0x141, 0x141, 0x141, 0x141, 0x142, 0x142, 0x142, 0x142, 0x142, 0x142,
    0x143, 0x143, 0x143, 0x143, 0x143, 0x143, 0x144, 0x144, 0x144, 0x144,
    0x144, 0x144, 0x144, 0x145, 0x145, 0x145, 0x145, 0x145, 0x145, 0x146,
    0x146, 0x146, 0x146, 0x146, 0x146, 0x147, 0x147, 0x147, 0x147, 0x147,
    0x147, 0x148, 0x148, 0x148, 0x148, 0x148, 0x148, 0x148, 0x149, 0x149,
    0x149, 0x149, 0x149, 0x149, 0x14A, 0x14A, 0x14A, 0x14A, 0x14A, 0x14A,
    0x14B, 0x14B, 0x14B, 0x14B, 0x14B, 0x14B, 0x14C, 0x14C, 0x14C, 0x14C,
    0x14C, 0x14C, 0x14D, 0x14D, 0x14D, 0x14D, 0x14D, 0x14D, 0x14E, 0x14E,
    0x14E, 0x14E, 0x14E, 0x14E, 0x14F, 0x14F, 0x14F, 0x14F, 0x14F, 0x14F,
    0x14F, 0x150, 0x150, 0x150, 0x150, 0x150, 0x150, 0x151, 0x151, 0x151,
    0x151, 0x151, 0x151, 0x152, 0x152, 0x152, 0x152, 0x152, 0x152, 0x153,
    0x153, 0x153, 0x153, 0x153, 0x153, 0x154, 0x154, 0x154, 0x154, 0x154,
    0x154, 0x155, 0x155, 0x155, 0x155, 0x155, 0x155, 0x156, 0x156, 0x156,
    0x156, 0x156, 0x156, 0x157, 0x157, 0x157, 0x157, 0x157, 0x157, 0x158,
    0x158, 0x158, 0x158, 0x158, 0x159, 0x159, 0x159, 0x159, 0x159, 0x159,
    0x15A, 0x15A, 0x15A, 0x15A, 0x15A, 0x15A, 0x15B, 0x15B, 0x15B, 0x15B,
    0x15B, 0x15B, 0x15C, 0x15C, 0x15C, 0x15C, 0x15C, 0x15C, 0x15D, 0x15D,
    0x15D, 0x15D, 0x15D, 0x15D, 0x15E, 0x15E, 0x15E, 0x15E, 0x15E, 0x15E,
    0x15F, 0x15F, 0x15F, 0x15F, 0x15F, 0x160, 0x160, 0x160, 0x160, 0x160,
    0x160, 0x161, 0x161, 0x161, 0x161, 0x161, 0x161, 0x162, 0x162, 0x162,
    0x162, 0x162, 0x162, 0x163, 0x163, 0x163, 0x163, 0x163, 0x164, 0x164,
    0x164, 0x164, 0x164, 0x164, 0x165, 0x165, 0x165, 0x165, 0x165, 0x165,
    0x166, 0x166, 0x166, 0x166, 0x166, 0x166, 0x167, 0x167, 0x167, 0x167,
    0x167, 0x168, 0x168, 0x168, 0x168, 0x168, 0x168, 0x169, 0x169, 0x169,
    0x169, 0x169, 0x169, 0x16A, 0x16A, 0x16A, 0x16A, 0x16A, 0x16B, 0x16B,
    0x16B, 0x16B, 0x16B, 0x16B, 0x16C, 0x16C, 0x16C, 0x16C, 0x16C, 0x16D,
    0x16D, 0x16D, 0x16D, 0x16D, 0x16D, 0x16E, 0x16E, 0x16E, 0x16E, 0x16E,
    0x16E, 0x16F, 0x16F, 0x16F, 0x16F, 0x16F, 0x170, 0x170, 0x170, 0x170,
    0x170, 0x170, 0x171, 0x171, 0x171, 0x171, 0x171, 0x172, 0x172, 0x172,
    0x172, 0x172, 0x172, 0x173, 0x173, 0x173, 0x173, 0x173, 0x174, 0x174,
    0x174, 0x174, 0x174, 0x174, 0x175, 0x175, 0x175, 0x175, 0x175, 0x176,
    0x176, 0x176, 0x176, 0x176, 0x176, 0x177, 0x177, 0x177, 0x177, 0x177,
    0x178, 0x178, 0x178, 0x178, 0x178, 0x179, 0x179, 0x179, 0x179, 0x179,
    0x179, 0x17A, 0x17A, 0x17A, 0x17A, 0x17A, 0x17B, 0x17B, 0x17B, 0x17B,
    0x17B, 0x17B, 0x17C, 0x17C, 0x17C, 0x17C, 0x17C, 0x17D, 0x17D, 0x17D,
    0x17D, 0x17D, 0x17E, 0x17E, 0x17E, 0x17E, 0x17E, 0x17E, 0x17F, 0x17F,
    0x17F, 0x17F, 0x17F, 0x180, 0x180, 0x180, 0x180, 0x180, 0x181, 0x181,
    0x181, 0x181, 0x181, 0x181, 0x182, 0x182, 0x182, 0x182, 0x182, 0x183,
    0x183, 0x183, 0x183, 0x183, 0x184, 0x184, 0x184, 0x184, 0x184, 0x185,
    0x185, 0x185, 0x185, 0x185, 0x185, 0x186, 0x186, 0x186, 0x186, 0x186,
    0x187, 0x187, 0x187, 0x187, 0x187, 0x188, 0x188, 0x188, 0x188, 0x188,
    0x189, 0x189, 0x189, 0x189, 0x189, 0x18A, 0x18A, 0x18A, 0x18A, 0x18A,
    0x18A, 0x18B, 0x18B, 0x18B, 0x18B, 0x18B, 0x18C, 0x18C, 0x18C, 0x18C,
    0x18C, 0x18D, 0x18D, 0x18D, 0x18D, 0x18D, 0x18E, 0x18E, 0x18E, 0x18E,
    0x18E, 0x18F, 0x18F, 0x18F, 0x18F, 0x18F, 0x190, 0x190, 0x190, 0x190,
    0x190, 0x191, 0x191, 0x191, 0x191, 0x191, 0x192, 0x192, 0x192, 0x192,
    0x192, 0x193, 0x193, 0x193, 0x193, 0x193, 0x194, 0x194, 0x194, 0x194,
    0x194, 0x194, 0x195, 0x195, 0x195, 0x195, 0x195, 0x196, 0x196, 0x196,
    0x196, 0x196, 0x197, 0x197, 0x197, 0x197, 0x197, 0x198, 0x198, 0x198,
    0x198, 0x198, 0x199, 0x199, 0x199, 0x199, 0x199, 0x19A, 0x19A, 0x19A,
    0x19A, 0x19A, 0x19B, 0x19B, 0x19B, 0x19B, 0x19C, 0x19C, 0x19C, 0x19C,
    0x19C, 0x19D, 0x19D, 0x19D, 0x19D, 0x19D, 0x19E, 0x19E, 0x19E, 0x19E,
    0x19E, 0x19F, 0x19F, 0x19F, 0x19F, 0x19F, 0x1A0, 0x1A0, 0x1A0, 0x1A0,
    0x1A0, 0x1A1, 0x1A1, 0x1A1, 0x1A1, 0x1A1, 0x1A2, 0x1A2, 0x1A2, 0x1A2,
    0x1A2, 0x1A3, 0x1A3, 0x1A3, 0x1A3, 0x1A3, 0x1A4, 0x1A4, 0x1A4, 0x1A4,
    0x1A4, 0x1A5, 0x1A5, 0x1A5, 0x1A5, 0x1A6, 0x1A6, 0x1A6, 0x1A6, 0x1A6,
    0x1A7, 0x1A7, 0x1A7, 0x1A7, 0x1A7, 0x1A8, 0x1A8, 0x1A8, 0x1A8, 0x1A8,
    0x1A9, 0x1A9, 0x1A9, 0x1A9, 0x1A9, 0x1AA, 0x1AA, 0x1AA, 0x1AA, 0x1AB,
    0x1AB, 0x1AB, 0x1AB, 0x1AB, 0x1AC, 0x1AC, 0x1AC, 0x1AC, 0x1AC, 0x1AD,
    0x1AD, 0x1AD, 0x1AD, 0x1AD, 0x1AE, 0x1AE, 0x1AE, 0x1AE, 0x1AF, 0x1AF,
    0x1AF, 0x1AF, 0x1AF, 0x1B0, 0x1B0, 0x1B0, 0x1B0, 0x1B0, 0x1B1, 0x1B1,
    0x1B1, 0x1B1, 0x1B1, 0x1B2, 0x1B2, 0x1B2, 0x1B2, 0x1B3, 0x1B3, 0x1B3,
    0x1B3, 0x1B3, 0x1B4, 0x1B4, 0x1B4, 0x1B4, 0x1B4, 0x1B5, 0x1B5, 0x1B5,
    0x1B5, 0x1B6, 0x1B6, 0x1B6, 0x1B6, 0x1B6, 0x1B7, 0x1B7, 0x1B7, 0x1B7,
    0x1B7, 0x1B8, 0x1B8, 0x1B8, 0x1B8, 0x1B9, 0x1B9, 0x1B9, 0x1B9, 0x1B9,
    0x1BA, 0x1BA, 0x1BA, 0x1BA, 0x1BB, 0x1BB, 0x1BB, 0x1BB, 0x1BB, 0x1BC,
    0x1BC, 0x1BC, 0x1BC, 0x1BC, 0x1BD, 0x1BD, 0x1BD, 0x1BD, 0x1BE, 0x1BE,
    0x1BE, 0x1BE, 0x1BE, 0x1BF, 0x1BF, 0x1BF, 0x1BF, 0x1C0, 0x1C0, 0x1C0,
    0x1C0, 0x1C0, 0x1C1, 0x1C1, 0x1C1, 0x1C1, 0x1C2, 0x1C2, 0x1C2, 0x1C2,
    0x1C2, 0x1C3, 0x1C3, 0x1C3, 0x1C3, 0x1C4, 0x1C4, 0x1C4, 0x1C4, 0x1C4,
    0x1C5, 0x1C5, 0x1C5, 0x1C5, 0x1C5, 0x1C6, 0x1C6, 0x1C6, 0x1C6, 0x1C7,
    0x1C7, 0x1C7, 0x1C7, 0x1C8, 0x1C8, 0x1C8, 0x1C8, 0x1C8, 0x1C9, 0x1C9,
    0x1C9, 0x1C9, 0x1CA, 0x1CA, 0x1CA, 0x1CA, 0x1CA, 0x1CB, 0x1CB, 0x1CB,
    0x1CB, 0x1CC, 0x1CC, 0x1CC, 0x1CC, 0x1CC, 0x1CD, 0x1CD, 0x1CD, 0x1CD,
    0x1CE, 0x1CE, 0x1CE, 0x1CE, 0x1CE, 0x1CF, 0x1CF, 0x1CF, 0x1CF, 0x1D0,
    0x1D0, 0x1D0, 0x1D0, 0x1D1, 0x1D1, 0x1D1, 0x1D1, 0x1D1, 0x1D2, 0x1D2,
    0x1D2, 0x1D2, 0x1D3, 0x1D3, 0x1D3, 0x1D3, 0x1D4, 0x1D4, 0x1D4, 0x1D4,
    0x1D4, 0x1D5, 0x1D5, 0x1D5, 0x1D5, 0x1D6, 0x1D6, 0x1D6, 0x1D6, 0x1D7,
    0x1D7, 0x1D7, 0x1D7, 0x1D7, 0x1D8, 0x1D8, 0x1D8, 0x1D8, 0x1D9, 0x1D9,
    0x1D9, 0x1D9, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DA, 0x1DB, 0x1DB, 0x1DB,
    0x1DB, 0x1DC, 0x1DC, 0x1DC, 0x1DC, 0x1DD, 0x1DD, 0x1DD, 0x1DD, 0x1DD,
    0x1DE, 0x1DE, 0x1DE, 0x1DE, 0x1DF, 0x1DF, 0x1DF, 0x1DF, 0x1E0, 0x1E0,
    0x1E0, 0x1E0, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E1, 0x1E2, 0x1E2, 0x1E2,
    0x1E2, 0x1E3, 0x1E3, 0x1E3, 0x1E3, 0x1E4, 0x1E4, 0x1E4, 0x1E4, 0x1E5,
    0x1E5, 0x1E5, 0x1E5, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E6, 0x1E7, 0x1E7,
    0x1E7, 0x1E7, 0x1E8, 0x1E8, 0x1E8, 0x1E8, 0x1E9, 0x1E9, 0x1E9, 0x1E9,
    0x1EA, 0x1EA, 0x1EA, 0x1EA, 0x1EB, 0x1EB, 0x1EB, 0x1EB, 0x1EC, 0x1EC,
    0x1EC, 0x1EC, 0x1EC, 0x1ED, 0x1ED, 0x1ED, 0x1ED, 0x1EE, 0x1EE, 0x1EE,
    0x1EE, 0x1EF, 0x1EF, 0x1EF, 0x1EF, 0x1F0, 0x1F0, 0x1F0, 0x1F0, 0x1F1,
    0x1F1, 0x1F1, 0x1F1, 0x1F2, 0x1F2, 0x1F2, 0x1F2, 0x1F3, 0x1F3, 0x1F3,
    0x1F3, 0x1F4, 0x1F4, 0x1F4, 0x1F4, 0x1F5, 0x1F5, 0x1F5, 0x1F5, 0x1F5,
    0x1F6, 0x1F6, 0x1F6, 0x1F6, 0x1F7, 0x1F7, 0x1F7, 0x1F7, 0x1F8, 0x1F8,
    0x1F8, 0x1F8, 0x1F9, 0x1F9, 0x1F9, 0x1F9, 0x1FA, 0x1FA, 0x1FA, 0x1FA,
    0x1FB, 0x1FB, 0x1FB, 0x1FB, 0x1FC, 0x1FC, 0x1FC, 0x1FC, 0x1FD, 0x1FD,
    0x1FD, 0x1FD, 0x1FE, 0x1FE, 0x1FE, 0x1FE, 0x1FF, 0x1FF, 0x1FF, 0x1FF,
    0x200, 0x200, 0x200, 0x200, 0x201, 0x201, 0x201, 0x201, 0x202, 0x202,
    0x202, 0x202, 0x203, 0x203, 0x203, 0x203, 0x204, 0x204, 0x204, 0x204,
    0x205, 0x205, 0x205, 0x205, 0x206, 0x206, 0x206, 0x207, 0x207, 0x207,
    0x207, 0x208, 0x208, 0x208, 0x208, 0x209, 0x209, 0x209, 0x209, 0x20A,
    0x20A, 0x20A, 0x20A, 0x20B, 0x20B, 0x20B, 0x20B, 0x20C, 0x20C, 0x20C,
    0x20C, 0x20D, 0x20D, 0x20D, 0x20D, 0x20E, 0x20E, 0x20E, 0x20E, 0x20F,
    0x20F, 0x20F, 0x20F, 0x210, 0x210, 0x210, 0x211, 0x211, 0x211, 0x211,
    0x212, 0x212, 0x212, 0x212, 0x213, 0x213, 0x213, 0x213, 0x214, 0x214,
    0x214, 0x214, 0x215, 0x215, 0x215, 0x215, 0x216, 0x216, 0x216, 0x217,
    0x217, 0x217, 0x217, 0x218, 0x218, 0x218, 0x218, 0x219, 0x219, 0x219,
    0x219, 0x21A, 0x21A, 0x21A, 0x21A, 0x21B, 0x21B, 0x21B, 0x21C, 0x21C,
    0x21C, 0x21C, 0x21D, 0x21D, 0x21D, 0x21D, 0x21E, 0x21E, 0x21E, 0x21E,
    0x21F, 0x21F, 0x21F, 0x220, 0x220, 0x220, 0x220, 0x221, 0x221, 0x221,
    0x221, 0x222, 0x222, 0x222, 0x222, 0x223, 0x223, 0x223, 0x224, 0x224,
    0x224, 0x224, 0x225, 0x225, 0x225, 0x225, 0x226, 0x226, 0x226, 0x227,
    0x227, 0x227, 0x227, 0x228, 0x228, 0x228, 0x228, 0x229, 0x229, 0x229,
    0x229, 0x22A, 0x22A, 0x22A, 0x22B, 0x22B, 0x22B, 0x22B, 0x22C, 0x22C,
    0x22C, 0x22C, 0x22D, 0x22D, 0x22D, 0x22E, 0x22E, 0x22E, 0x22E, 0x22F,
    0x22F, 0x22F, 0x22F, 0x230, 0x230, 0x230, 0x231, 0x231, 0x231, 0x231,
    0x232, 0x232, 0x232, 0x233, 0x233, 0x233, 0x233, 0x234, 0x234, 0x234,
    0x234, 0x235, 0x235, 0x235, 0x236, 0x236, 0x236, 0x236, 0x237, 0x237,
    0x237, 0x238, 0x238, 0x238, 0x238, 0x239, 0x239, 0x239, 0x239, 0x23A,
    0x23A, 0x23A, 0x23B, 0x23B, 0x23B, 0x23B, 0x23C, 0x23C, 0x23C, 0x23D,
    0x23D, 0x23D, 0x23D, 0x23E, 0x23E, 0x23E, 0x23F, 0x23F, 0x23F, 0x23F,
    0x240, 0x240, 0x240, 0x240, 0x241, 0x241, 0x241, 0x242, 0x242, 0x242,
    0x242, 0x243, 0x243, 0x243, 0x244, 0x244, 0x244, 0x244, 0x245, 0x245,
    0x245, 0x246, 0x246, 0x246, 0x246, 0x247, 0x247, 0x247, 0x248, 0x248,
    0x248, 0x248, 0x249, 0x249, 0x249, 0x24A, 0x24A, 0x24A, 0x24A, 0x24B,
    0x24B, 0x24B, 0x24C, 0x24C, 0x24C, 0x24C, 0x24D, 0x24D, 0x24D, 0x24E,
    0x24E, 0x24E, 0x24F, 0x24F, 0x24F, 0x24F, 0x250, 0x250, 0x250, 0x251,
    0x251, 0x251, 0x251, 0x252, 0x252, 0x252, 0x253, 0x253, 0x253, 0x253,
    0x254, 0x254, 0x254, 0x255, 0x255, 0x255, 0x256, 0x256, 0x256, 0x256,
    0x257, 0x257, 0x257, 0x258, 0x258, 0x258, 0x258, 0x259, 0x259, 0x259,
    0x25A, 0x25A, 0x25A, 0x25B, 0x25B, 0x25B, 0x25B, 0x25C, 0x25C, 0x25C,
    0x25D, 0x25D, 0x25D, 0x25D, 0x25E, 0x25E, 0x25E, 0x25F, 0x25F, 0x25F,
    0x260, 0x260, 0x260, 0x260, 0x261, 0x261, 0x261, 0x262, 0x262, 0x262,
    0x263, 0x263, 0x263, 0x263, 0x264, 0x264, 0x264, 0x265, 0x265, 0x265,
    0x266, 0x266, 0x266, 0x266, 0x267, 0x267, 0x267, 0x268, 0x268, 0x268,
    0x269, 0x269, 0x269, 0x269, 0x26A, 0x26A, 0x26A, 0x26B, 0x26B, 0x26B,
    0x26C, 0x26C, 0x26C, 0x26D, 0x26D, 0x26D, 0x26D, 0x26E, 0x26E, 0x26E,
    0x26F, 0x26F, 0x26F, 0x270, 0x270, 0x270, 0x270, 0x271, 0x271, 0x271,
    0x272, 0x272, 0x272, 0x273, 0x273, 0x273, 0x274, 0x274, 0x274, 0x274,
    0x275, 0x275, 0x275, 0x276, 0x276, 0x276, 0x277, 0x277, 0x277, 0x278,
    0x278, 0x278, 0x279, 0x279, 0x279, 0x279, 0x27A, 0x27A, 0x27A, 0x27B,
    0x27B, 0x27B, 0x27C, 0x27C, 0x27C, 0x27D, 0x27D, 0x27D, 0x27D, 0x27E,
    0x27E, 0x27E, 0x27F, 0x27F, 0x27F, 0x280, 0x280, 0x280, 0x281, 0x281,
    0x281, 0x282, 0x282, 0x282, 0x283, 0x283, 0x283, 0x283, 0x284, 0x284,
    0x284, 0x285, 0x285, 0x285, 0x286, 0x286, 0x286, 0x287, 0x287, 0x287,
    0x288, 0x288, 0x288, 0x289, 0x289, 0x289, 0x289, 0x28A, 0x28A, 0x28A,
    0x28B, 0x28B, 0x28B, 0x28C, 0x28C, 0x28C, 0x28D, 0x28D, 0x28D, 0x28E,
    0x28E, 0x28E, 0x28F, 0x28F, 0x28F, 0x290, 0x290, 0x290, 0x291, 0x291,
    0x291, 0x292, 0x292, 0x292, 0x292, 0x293, 0x293, 0x293, 0x294, 0x294,
    0x294, 0x295, 0x295, 0x295, 0x296, 0x296, 0x296, 0x297, 0x297, 0x297,
    0x298, 0x298, 0x298, 0x299, 0x299, 0x299, 0x29A, 0x29A, 0x29A, 0x29B,
    0x29B, 0x29B, 0x29C, 0x29C, 0x29C, 0x29D, 0x29D, 0x29D, 0x29E, 0x29E,
    0x29E, 0x29F, 0x29F, 0x29F, 0x2A0, 0x2A0, 0x2A0, 0x2A1, 0x2A1, 0x2A1,
    0x2A2, 0x2A2, 0x2A2, 0x2A2, 0x2A3, 0x2A3, 0x2A3, 0x2A4, 0x2A4, 0x2A4,
    0x2A5, 0x2A5, 0x2A5, 0x2A6, 0x2A6, 0x2A6, 0x2A7, 0x2A7, 0x2A7, 0x2A8,
    0x2A8, 0x2A8, 0x2A9, 0x2A9, 0x2A9, 0x2AA, 0x2AA, 0x2AA, 0x2AB, 0x2AB,
    0x2AB, 0x2AC, 0x2AC, 0x2AC, 0x2AD, 0x2AD, 0x2AE, 0x2AE, 0x2AE, 0x2AF,
    0x2AF, 0x2AF, 0x2B0, 0x2B0, 0x2B0, 0x2B1, 0x2B1, 0x2B1, 0x2B2, 0x2B2,
    0x2B2, 0x2B3, 0x2B3, 0x2B3, 0x2B4, 0x2B4, 0x2B4, 0x2B5, 0x2B5, 0x2B5,
    0x2B6, 0x2B6, 0x2B6, 0x2B7, 0x2B7, 0x2B7, 0x2B8, 0x2B8, 0x2B8, 0x2B9,
    0x2B9, 0x2B9, 0x2BA, 0x2BA, 0x2BA, 0x2BB, 0x2BB, 0x2BB, 0x2BC, 0x2BC,
    0x2BC, 0x2BD, 0x2BD, 0x2BD, 0x2BE, 0x2BE, 0x2BF, 0x2BF, 0x2BF, 0x2C0,
    0x2C0, 0x2C0, 0x2C1, 0x2C1, 0x2C1, 0x2C2, 0x2C2, 0x2C2, 0x2C3, 0x2C3,
    0x2C3, 0x2C4, 0x2C4, 0x2C4, 0x2C5, 0x2C5, 0x2C5, 0x2C6, 0x2C6, 0x2C7,
    0x2C7, 0x2C7, 0x2C8, 0x2C8, 0x2C8, 0x2C9, 0x2C9, 0x2C9, 0x2CA, 0x2CA,
    0x2CA, 0x2CB, 0x2CB, 0x2CB, 0x2CC, 0x2CC, 0x2CC, 0x2CD, 0x2CD, 0x2CE,
    0x2CE, 0x2CE, 0x2CF, 0x2CF, 0x2CF, 0x2D0, 0x2D0, 0x2D0, 0x2D1, 0x2D1,
    0x2D1, 0x2D2, 0x2D2, 0x2D2, 0x2D3, 0x2D3, 0x2D4, 0x2D4, 0x2D4, 0x2D5,
    0x2D5, 0x2D5, 0x2D6, 0x2D6, 0x2D6, 0x2D7, 0x2D7, 0x2D7, 0x2D8, 0x2D8,
    0x2D9, 0x2D9, 0x2D9, 0x2DA, 0x2DA, 0x2DA, 0x2DB, 0x2DB, 0x2DB, 0x2DC,
    0x2DC, 0x2DC, 0x2DD, 0x2DD, 0x2DE, 0x2DE, 0x2DE, 0x2DF, 0x2DF, 0x2DF,
    0x2E0, 0x2E0, 0x2E0, 0x2E1, 0x2E1, 0x2E1, 0x2E2, 0x2E2, 0x2E3, 0x2E3,
    0x2E3, 0x2E4, 0x2E4, 0x2E4, 0x2E5, 0x2E5, 0x2E5, 0x2E6, 0x2E6, 0x2E7,
    0x2E7, 0x2E7, 0x2E8, 0x2E8, 0x2E8, 0x2E9, 0x2E9, 0x2E9, 0x2EA, 0x2EA,
    0x2EB, 0x2EB, 0x2EB, 0x2EC, 0x2EC, 0x2EC, 0x2ED, 0x2ED, 0x2EE, 0x2EE,
    0x2EE, 0x2EF, 0x2EF, 0x2EF, 0x2F0, 0x2F0, 0x2F0, 0x2F1, 0x2F1, 0x2F2,
    0x2F2, 0x2F2, 0x2F3, 0x2F3, 0x2F3, 0x2F4, 0x2F4, 0x2F5, 0x2F5, 0x2F5,
    0x2F6, 0x2F6, 0x2F6, 0x2F7, 0x2F7, 0x2F8, 0x2F8, 0x2F8, 0x2F9, 0x2F9,
    0x2F9, 0x2FA, 0x2FA, 0x2FB, 0x2FB, 0x2FB, 0x2FC, 0x2FC, 0x2FC, 0x2FD,
    0x2FD, 0x2FE, 0x2FE, 0x2FE, 0x2FF, 0x2FF, 0x2FF, 0x300, 0x300, 0x301,
    0x301, 0x301, 0x302, 0x302, 0x302, 0x303, 0x303, 0x304, 0x304, 0x304,
    0x305, 0x305, 0x305, 0x306, 0x306, 0x307, 0x307, 0x307, 0x308, 0x308,
    0x308, 0x309, 0x309, 0x30A, 0x30A, 0x30A, 0x30B, 0x30B, 0x30C, 0x30C,
    0x30C, 0x30D, 0x30D, 0x30D, 0x30E, 0x30E, 0x30F, 0x30F, 0x30F, 0x310,
    0x310, 0x311, 0x311, 0x311, 0x312, 0x312, 0x312, 0x313, 0x313, 0x314,
    0x314, 0x314, 0x315, 0x315, 0x316, 0x316, 0x316, 0x317, 0x317, 0x317,
    0x318, 0x318, 0x319, 0x319, 0x319, 0x31A, 0x31A, 0x31B, 0x31B, 0x31B,
    0x31C, 0x31C, 0x31D, 0x31D, 0x31D, 0x31E, 0x31E, 0x31E, 0x31F, 0x31F,
    0x320, 0x320, 0x320, 0x321, 0x321, 0x322, 0x322, 0x322, 0x323, 0x323,
    0x324, 0x324, 0x324, 0x325, 0x325, 0x326, 0x326, 0x326, 0x327, 0x327,
    0x328, 0x328, 0x328, 0x329, 0x329, 0x32A, 0x32A, 0x32A, 0x32B, 0x32B,
    0x32C, 0x32C, 0x32C, 0x32D, 0x32D, 0x32E, 0x32E, 0x32E, 0x32F, 0x32F,
    0x330, 0x330, 0x330, 0x331, 0x331, 0x332, 0x332, 0x332, 0x333, 0x333,
    0x334, 0x334, 0x334, 0x335, 0x335, 0x336, 0x336, 0x336, 0x337, 0x337,
    0x338, 0x338, 0x338, 0x339, 0x339, 0x33A, 0x33A, 0x33A, 0x33B, 0x33B,
    0x33C, 0x33C, 0x33C, 0x33D, 0x33D, 0x33E, 0x33E, 0x33E, 0x33F, 0x33F,
    0x340, 0x340, 0x340, 0x341, 0x341, 0x342, 0x342, 0x343, 0x343, 0x343,
    0x344, 0x344, 0x345, 0x345, 0x345, 0x346, 0x346, 0x347, 0x347, 0x347,
    0x348, 0x348, 0x349, 0x349, 0x34A, 0x34A, 0x34A, 0x34B, 0x34B, 0x34C,
    0x34C, 0x34C, 0x34D, 0x34D, 0x34E, 0x34E, 0x34E, 0x34F, 0x34F, 0x350,
    0x350, 0x351, 0x351, 0x351, 0x352, 0x352, 0x353, 0x353, 0x353, 0x354,
    0x354, 0x355, 0x355, 0x356, 0x356, 0x356, 0x357, 0x357, 0x358, 0x358,
    0x359, 0x359, 0x359, 0x35A, 0x35A, 0x35B, 0x35B, 0x35B, 0x35C, 0x35C,
    0x35D, 0x35D, 0x35E, 0x35E, 0x35E, 0x35F, 0x35F, 0x360, 0x360, 0x361,
    0x361, 0x361, 0x362, 0x362, 0x363, 0x363, 0x364, 0x364, 0x364, 0x365,
    0x365, 0x366, 0x366, 0x366, 0x367, 0x367, 0x368, 0x368, 0x369, 0x369,
    0x369, 0x36A, 0x36A, 0x36B, 0x36B, 0x36C, 0x36C, 0x36C, 0x36D, 0x36D,
    0x36E, 0x36E, 0x36F, 0x36F, 0x370, 0x370, 0x370, 0x371, 0x371, 0x372,
    0x372, 0x373, 0x373, 0x373, 0x374, 0x374, 0x375, 0x375, 0x376, 0x376,
    0x376, 0x377, 0x377, 0x378, 0x378, 0x379, 0x379, 0x379, 0x37A, 0x37A,
    0x37B, 0x37B, 0x37C, 0x37C, 0x37D, 0x37D, 0x37D, 0x37E, 0x37E, 0x37F,
    0x37F, 0x380, 0x380, 0x380, 0x381, 0x381, 0x382, 0x382, 0x383, 0x383,
    0x384, 0x384, 0x384, 0x385, 0x385, 0x386, 0x386, 0x387, 0x387, 0x388,
    0x388, 0x388, 0x389, 0x389, 0x38A, 0x38A, 0x38B, 0x38B, 0x38C, 0x38C,
    0x38C, 0x38D, 0x38D, 0x38E, 0x38E, 0x38F, 0x38F, 0x390, 0x390, 0x390,
    0x391, 0x391, 0x392, 0x392, 0x393, 0x393, 0x394, 0x394, 0x395, 0x395,
    0x395, 0x396, 0x396, 0x397, 0x397, 0x398, 0x398, 0x399, 0x399, 0x399,
    0x39A, 0x39A, 0x39B, 0x39B, 0x39C, 0x39C, 0x39D, 0x39D, 0x39E, 0x39E,
    0x39E, 0x39F, 0x39F, 0x3A0, 0x3A0, 0x3A1, 0x3A1, 0x3A2, 0x3A2, 0x3A3,
    0x3A3, 0x3A3, 0x3A4, 0x3A4, 0x3A5, 0x3A5, 0x3A6, 0x3A6, 0x3A7, 0x3A7,
    0x3A8, 0x3A8, 0x3A8, 0x3A9, 0x3A9, 0x3AA, 0x3AA, 0x3AB, 0x3AB, 0x3AC,
    0x3AC, 0x3AD, 0x3AD, 0x3AE, 0x3AE, 0x3AE, 0x3AF, 0x3AF, 0x3B0, 0x3B0,
    0x3B1, 0x3B1, 0x3B2, 0x3B2, 0x3B3, 0x3B3, 0x3B4, 0x3B4, 0x3B5, 0x3B5,
    0x3B5, 0x3B6, 0x3B6, 0x3B7, 0x3B7, 0x3B8, 0x3B8, 0x3B9, 0x3B9, 0x3BA,
    0x3BA, 0x3BB, 0x3BB, 0x3BC, 0x3BC, 0x3BC, 0x3BD, 0x3BD, 0x3BE, 0x3BE,
    0x3BF, 0x3BF, 0x3C0, 0x3C0, 0x3C1, 0x3C1, 0x3C2, 0x3C2, 0x3C3, 0x3C3,
    0x3C4, 0x3C4, 0x3C4, 0x3C5, 0x3C5, 0x3C6, 0x3C6, 0x3C7, 0x3C7, 0x3C8,
    0x3C8, 0x3C9, 0x3C9, 0x3CA, 0x3CA, 0x3CB, 0x3CB, 0x3CC, 0x3CC, 0x3CD,
    0x3CD, 0x3CD, 0x3CE, 0x3CE, 0x3CF, 0x3CF, 0x3D0, 0x3D0, 0x3D1, 0x3D1,
    0x3D2, 0x3D2, 0x3D3, 0x3D3, 0x3D4, 0x3D4, 0x3D5, 0x3D5, 0x3D6, 0x3D6,
    0x3D7, 0x3D7, 0x3D8, 0x3D8, 0x3D9, 0x3D9, 0x3DA, 0x3DA, 0x3DA, 0x3DB,
    0x3DB, 0x3DC, 0x3DC, 0x3DD, 0x3DD, 0x3DE, 0x3DE, 0x3DF, 0x3DF, 0x3E0,
    0x3E0, 0x3E1, 0x3E1, 0x3E2, 0x3E2, 0x3E3, 0x3E3, 0x3E4, 0x3E4, 0x3E5,
    0x3E5, 0x3E6, 0x3E6, 0x3E7, 0x3E7, 0x3E8, 0x3E8, 0x3E9, 0x3E9, 0x3EA,
    0x3EA, 0x3EB, 0x3EB, 0x3EC, 0x3EC, 0x3ED, 0x3ED, 0x3EE, 0x3EE, 0x3EF,
    0x3EF, 0x3F0, 0x3F0, 0x3F0, 0x3F1, 0x3F1, 0x3F2, 0x3F2, 0x3F3, 0x3F3,
    0x3F4, 0x3F4, 0x3F5, 0x3F5, 0x3F6, 0x3F6, 0x3F7, 0x3F7, 0x3F8, 0x3F8,
    0x3F9, 0x3F9, 0x3FA, 0x3FA, 0x3FB, 0x3FB, 0x3FC, 0x3FC, 0x3FD, 0x3FD,
    0x3FE, 0x3FE, 0x3FF, 0x3FF, 0x400, 0x400, 0x401, 0x401, 0x402, 0x402,
    0x403, 0x403, 0x404, 0x404, 0x405, 0x405, 0x406, 0x406, 0x407, 0x408,
    0x408, 0x409, 0x409, 0x40A, 0x40A, 0x40B, 0x40B, 0x40C, 0x40C, 0x40D,
    0x40D, 0x40E, 0x40E, 0x40F, 0x40F, 0x410, 0x410, 0x411, 0x411, 0x412,
    0x412, 0x413, 0x413, 0x414, 0x414, 0x415, 0x415, 0x416, 0x416, 0x417,
    0x417, 0x418, 0x418, 0x419, 0x419, 0x41A, 0x41A, 0x41B, 0x41B, 0x41C,
    0x41C, 0x41D, 0x41D, 0x41E, 0x41F, 0x41F, 0x420, 0x420, 0x421, 0x421,
    0x422, 0x422, 0x423, 0x423, 0x424, 0x424, 0x425, 0x425, 0x426, 0x426,
    0x427, 0x427, 0x428, 0x428, 0x429, 0x429, 0x42A, 0x42A, 0x42B, 0x42C,
    0x42C, 0x42D, 0x42D, 0x42E, 0x42E, 0x42F, 0x42F, 0x430, 0x430, 0x431,
    0x431, 0x432, 0x432, 0x433, 0x433, 0x434, 0x434, 0x435, 0x436, 0x436,
    0x437, 0x437, 0x438, 0x438, 0x439, 0x439, 0x43A, 0x43A, 0x43B, 0x43B,
    0x43C, 0x43C, 0x43D, 0x43D, 0x43E, 0x43F, 0x43F, 0x440, 0x440, 0x441,
    0x441, 0x442, 0x442, 0x443, 0x443, 0x444, 0x444, 0x445, 0x446, 0x446,
    0x447, 0x447, 0x448, 0x448, 0x449, 0x449, 0x44A, 0x44A, 0x44B, 0x44B,
    0x44C, 0x44C, 0x44D, 0x44E, 0x44E, 0x44F, 0x44F, 0x450, 0x450, 0x451,
    0x451, 0x452, 0x452, 0x453, 0x454, 0x454, 0x455, 0x455, 0x456, 0x456,
    0x457, 0x457, 0x458, 0x458, 0x459, 0x45A, 0x45A, 0x45B, 0x45B, 0x45C,
    0x45C, 0x45D, 0x45D, 0x45E, 0x45E, 0x45F, 0x460, 0x460, 0x461, 0x461,
    0x462, 0x462, 0x463, 0x463, 0x464, 0x465, 0x465, 0x466, 0x466, 0x467,
    0x467, 0x468, 0x468, 0x469, 0x469, 0x46A, 0x46B, 0x46B, 0x46C, 0x46C,
    0x46D, 0x46D, 0x46E, 0x46E, 0x46F, 0x470, 0x470, 0x471, 0x471, 0x472,
    0x472, 0x473, 0x473, 0x474, 0x475, 0x475, 0x476, 0x476, 0x477, 0x477,
    0x478, 0x479, 0x479, 0x47A, 0x47A, 0x47B, 0x47B, 0x47C, 0x47C, 0x47D,
    0x47E, 0x47E, 0x47F, 0x47F, 0x480, 0x480, 0x481, 0x482, 0x482, 0x483,
    0x483, 0x484, 0x484, 0x485, 0x486, 0x486, 0x487, 0x487, 0x488, 0x488,
    0x489, 0x48A, 0x48A, 0x48B, 0x48B, 0x48C, 0x48C, 0x48D, 0x48D, 0x48E,
    0x48F, 0x48F, 0x490, 0x490, 0x491, 0x492, 0x492, 0x493, 0x493, 0x494,
    0x494, 0x495, 0x496, 0x496, 0x497, 0x497, 0x498, 0x498, 0x499, 0x49A,
    0x49A, 0x49B, 0x49B, 0x49C, 0x49C, 0x49D, 0x49E, 0x49E, 0x49F, 0x49F,
    0x4A0, 0x4A1, 0x4A1, 0x4A2, 0x4A2, 0x4A3, 0x4A3, 0x4A4, 0x4A5, 0x4A5,
    0x4A6, 0x4A6, 0x4A7, 0x4A8, 0x4A8, 0x4A9, 0x4A9, 0x4AA, 0x4AA, 0x4AB,
    0x4AC, 0x4AC, 0x4AD, 0x4AD, 0x4AE, 0x4AF, 0x4AF, 0x4B0, 0x4B0, 0x4B1,
    0x4B1, 0x4B2, 0x4B3, 0x4B3, 0x4B4, 0x4B4, 0x4B5, 0x4B6, 0x4B6, 0x4B7,
    0x4B7, 0x4B8, 0x4B9, 0x4B9, 0x4BA, 0x4BA, 0x4BB, 0x4BC, 0x4BC, 0x4BD,
    0x4BD, 0x4BE, 0x4BF, 0x4BF, 0x4C0, 0x4C0, 0x4C1, 0x4C1, 0x4C2, 0x4C3,
    0x4C3, 0x4C4, 0x4C4, 0x4C5, 0x4C6, 0x4C6, 0x4C7, 0x4C7, 0x4C8, 0x4C9,
    0x4C9, 0x4CA, 0x4CA, 0x4CB, 0x4CC, 0x4CC, 0x4CD, 0x4CD, 0x4CE, 0x4CF,
    0x4CF, 0x4D0, 0x4D1, 0x4D1, 0x4D2, 0x4D2, 0x4D3, 0x4D4, 0x4D4, 0x4D5,
    0x4D5, 0x4D6, 0x4D7, 0x4D7, 0x4D8, 0x4D8, 0x4D9, 0x4DA, 0x4DA, 0x4DB,
    0x4DB, 0x4DC, 0x4DD, 0x4DD, 0x4DE, 0x4DE, 0x4DF, 0x4E0, 0x4E0, 0x4E1,
    0x4E2, 0x4E2, 0x4E3, 0x4E3, 0x4E4, 0x4E5, 0x4E5, 0x4E6, 0x4E6, 0x4E7,
    0x4E8, 0x4E8, 0x4E9, 0x4EA, 0x4EA, 0x4EB, 0x4EB, 0x4EC, 0x4ED, 0x4ED,
    0x4EE, 0x4EE, 0x4EF, 0x4F0, 0x4F0, 0x4F1, 0x4F2, 0x4F2, 0x4F3, 0x4F3,
    0x4F4, 0x4F5, 0x4F5, 0x4F6, 0x4F7, 0x4F7, 0x4F8, 0x4F8, 0x4F9, 0x4FA,
    0x4FA, 0x4FB, 0x4FC, 0x4FC, 0x4FD, 0x4FD, 0x4FE, 0x4FF, 0x4FF, 0x500,
    0x501, 0x501, 0x502, 0x502, 0x503, 0x504, 0x504, 0x505, 0x506, 0x506,
    0x507, 0x508, 0x508, 0x509, 0x509, 0x50A, 0x50B, 0x50B, 0x50C, 0x50D,
    0x50D, 0x50E, 0x50E, 0x50F, 0x510, 0x510, 0x511, 0x512, 0x512, 0x513,
    0x514, 0x514, 0x515, 0x515, 0x516, 0x517, 0x517, 0x518, 0x519, 0x519,
    0x51A, 0x51B, 0x51B, 0x51C, 0x51D, 0x51D, 0x51E, 0x51E, 0x51F, 0x520,
    0x520, 0x521, 0x522, 0x522, 0x523, 0x524, 0x524, 0x525, 0x526, 0x526,
    0x527, 0x527, 0x528, 0x529, 0x529, 0x52A, 0x52B, 0x52B, 0x52C, 0x52D,
    0x52D, 0x52E, 0x52F, 0x52F, 0x530, 0x531, 0x531, 0x532, 0x533, 0x533,
    0x534, 0x534, 0x535, 0x536, 0x536, 0x537, 0x538, 0x538, 0x539, 0x53A,
    0x53A, 0x53B, 0x53C, 0x53C, 0x53D, 0x53E, 0x53E, 0x53F, 0x540, 0x540,
    0x541, 0x542, 0x542, 0x543, 0x544, 0x544, 0x545, 0x546, 0x546, 0x547,
    0x548, 0x548, 0x549, 0x54A, 0x54A, 0x54B, 0x54C, 0x54C, 0x54D, 0x54E,
    0x54E, 0x54F, 0x550, 0x550, 0x551, 0x552, 0x552, 0x553, 0x554, 0x554,
    0x555, 0x556, 0x556, 0x557, 0x558, 0x558, 0x559, 0x55A, 0x55A, 0x55B,
    0x55C, 0x55C, 0x55D, 0x55E, 0x55E, 0x55F, 0x560, 0x560, 0x561, 0x562,
    0x562, 0x563, 0x564, 0x564, 0x565, 0x566, 0x566, 0x567, 0x568, 0x568,
    0x569, 0x56A, 0x56A, 0x56B, 0x56C, 0x56D, 0x56D, 0x56E, 0x56F, 0x56F,
    0x570, 0x571, 0x571, 0x572, 0x573, 0x573, 0x574, 0x575, 0x575, 0x576,
    0x577, 0x577, 0x578, 0x579, 0x57A, 0x57A, 0x57B, 0x57C, 0x57C, 0x57D,
    0x57E, 0x57E, 0x57F, 0x580, 0x580, 0x581, 0x582, 0x582, 0x583, 0x584,
    0x585, 0x585, 0x586, 0x587, 0x587, 0x588, 0x589, 0x589, 0x58A, 0x58B,
    0x58B, 0x58C, 0x58D, 0x58E, 0x58E, 0x58F, 0x590, 0x590, 0x591, 0x592,
    0x592, 0x593, 0x594, 0x595, 0x595, 0x596, 0x597, 0x597, 0x598, 0x599,
    0x599, 0x59A, 0x59B, 0x59C, 0x59C, 0x59D, 0x59E, 0x59E, 0x59F, 0x5A0,
    0x5A1, 0x5A1, 0x5A2, 0x5A3, 0x5A3, 0x5A4, 0x5A5, 0x5A5, 0x5A6, 0x5A7,
    0x5A8, 0x5A8, 0x5A9, 0x5AA, 0x5AA, 0x5AB, 0x5AC, 0x5AD, 0x5AD, 0x5AE,
    0x5AF, 0x5AF, 0x5B0, 0x5B1, 0x5B2, 0x5B2, 0x5B3, 0x5B4, 0x5B4, 0x5B5,
    0x5B6, 0x5B7, 0x5B7, 0x5B8, 0x5B9, 0x5B9, 0x5BA, 0x5BB, 0x5BC, 0x5BC,
    0x5BD, 0x5BE, 0x5BF, 0x5BF, 0x5C0, 0x5C1, 0x5C1, 0x5C2, 0x5C3, 0x5C4,
    0x5C4, 0x5C5, 0x5C6, 0x5C6, 0x5C7, 0x5C8, 0x5C9, 0x5C9, 0x5CA, 0x5CB,
    0x5CC, 0x5CC, 0x5CD, 0x5CE, 0x5CE, 0x5CF, 0x5D0, 0x5D1, 0x5D1, 0x5D2,
    0x5D3, 0x5D4, 0x5D4, 0x5D5, 0x5D6, 0x5D7, 0x5D7, 0x5D8, 0x5D9, 0x5D9,
    0x5DA, 0x5DB, 0x5DC, 0x5DC, 0x5DD, 0x5DE, 0x5DF, 0x5DF, 0x5E0, 0x5E1,
    0x5E2, 0x5E2, 0x5E3, 0x5E4, 0x5E4, 0x5E5, 0x5E6, 0x5E7, 0x5E7, 0x5E8,
    0x5E9, 0x5EA, 0x5EA, 0x5EB, 0x5EC, 0x5ED, 0x5ED, 0x5EE, 0x5EF, 0x5F0,
    0x5F0, 0x5F1, 0x5F2, 0x5F3, 0x5F3, 0x5F4, 0x5F5, 0x5F6, 0x5F6, 0x5F7,
    0x5F8, 0x5F9, 0x5F9, 0x5FA, 0x5FB, 0x5FC, 0x5FC, 0x5FD, 0x5FE, 0x5FF,
    0x5FF, 0x600, 0x601, 0x602, 0x602, 0x603, 0x604, 0x605, 0x605, 0x606,
    0x607, 0x608, 0x608, 0x609, 0x60A, 0x60B, 0x60B, 0x60C, 0x60D, 0x60E,
    0x60F, 0x60F, 0x610, 0x611, 0x612, 0x612, 0x613, 0x614, 0x615, 0x615,
    0x616, 0x617, 0x618, 0x618, 0x619, 0x61A, 0x61B, 0x61B, 0x61C, 0x61D,
    0x61E, 0x61F, 0x61F, 0x620, 0x621, 0x622, 0x622, 0x623, 0x624, 0x625,
    0x625, 0x626, 0x627, 0x628, 0x629, 0x629, 0x62A, 0x62B, 0x62C, 0x62C,
    0x62D, 0x62E, 0x62F, 0x630, 0x630, 0x631, 0x632, 0x633, 0x633, 0x634,
    0x635, 0x636, 0x637, 0x637, 0x638, 0x639, 0x63A, 0x63A, 0x63B, 0x63C,
    0x63D, 0x63E, 0x63E, 0x63F, 0x640, 0x641, 0x642, 0x642, 0x643, 0x644,
    0x645, 0x645, 0x646, 0x647, 0x648, 0x649, 0x649, 0x64A, 0x64B, 0x64C,
    0x64D, 0x64D, 0x64E, 0x64F, 0x650, 0x650, 0x651, 0x652, 0x653, 0x654,
    0x654, 0x655, 0x656, 0x657, 0x658, 0x658, 0x659, 0x65A, 0x65B, 0x65C,
    0x65C, 0x65D, 0x65E, 0x65F, 0x660, 0x660, 0x661, 0x662, 0x663, 0x664,
    0x664, 0x665, 0x666, 0x667, 0x668, 0x668, 0x669, 0x66A, 0x66B, 0x66C,
    0x66C, 0x66D, 0x66E, 0x66F, 0x670, 0x671, 0x671, 0x672, 0x673, 0x674,
    0x675, 0x675, 0x676, 0x677, 0x678, 0x679, 0x679, 0x67A, 0x67B, 0x67C,
    0x67D, 0x67E, 0x67E, 0x67F, 0x680, 0x681, 0x682, 0x682, 0x683, 0x684,
    0x685, 0x686, 0x686, 0x687, 0x688, 0x689, 0x68A, 0x68B, 0x68B, 0x68C,
    0x68D, 0x68E, 0x68F, 0x690, 0x690, 0x691, 0x692, 0x693, 0x694, 0x694,
    0x695, 0x696, 0x697, 0x698, 0x699, 0x699, 0x69A, 0x69B, 0x69C, 0x69D,
    0x69E, 0x69E, 0x69F, 0x6A0, 0x6A1, 0x6A2, 0x6A3, 0x6A3, 0x6A4, 0x6A5,
    0x6A6, 0x6A7, 0x6A8, 0x6A8, 0x6A9, 0x6AA, 0x6AB, 0x6AC, 0x6AD, 0x6AD,
    0x6AE, 0x6AF, 0x6B0, 0x6B1, 0x6B2, 0x6B2, 0x6B3, 0x6B4, 0x6B5, 0x6B6,
    0x6B7, 0x6B8, 0x6B8, 0x6B9, 0x6BA, 0x6BB, 0x6BC, 0x6BD, 0x6BD, 0x6BE,
    0x6BF, 0x6C0, 0x6C1, 0x6C2, 0x6C3, 0x6C3, 0x6C4, 0x6C5, 0x6C6, 0x6C7,
    0x6C8, 0x6C8, 0x6C9, 0x6CA, 0x6CB, 0x6CC, 0x6CD, 0x6CE, 0x6CE, 0x6CF,
    0x6D0, 0x6D1, 0x6D2, 0x6D3, 0x6D4, 0x6D4, 0x6D5, 0x6D6, 0x6D7, 0x6D8,
    0x6D9, 0x6DA, 0x6DA, 0x6DB, 0x6DC, 0x6DD, 0x6DE, 0x6DF, 0x6E0, 0x6E0,
    0x6E1, 0x6E2, 0x6E3, 0x6E4, 0x6E5, 0x6E6, 0x6E7, 0x6E7, 0x6E8, 0x6E9,
    0x6EA, 0x6EB, 0x6EC, 0x6ED, 0x6ED, 0x6EE, 0x6EF, 0x6F0, 0x6F1, 0x6F2,
    0x6F3, 0x6F4, 0x6F4, 0x6F5, 0x6F6, 0x6F7, 0x6F8, 0x6F9, 0x6FA, 0x6FB,
    0x6FB, 0x6FC, 0x6FD, 0x6FE, 0x6FF, 0x700, 0x701, 0x702, 0x702, 0x703,
    0x704, 0x705, 0x706, 0x707, 0x708, 0x709, 0x70A, 0x70A, 0x70B, 0x70C,
    0x70D, 0x70E, 0x70F, 0x710, 0x711, 0x712, 0x712, 0x713, 0x714, 0x715,
    0x716, 0x717, 0x718, 0x719, 0x71A, 0x71A, 0x71B, 0x71C, 0x71D, 0x71E,
    0x71F, 0x720, 0x721, 0x722, 0x722, 0x723, 0x724, 0x725, 0x726, 0x727,
    0x728, 0x729, 0x72A, 0x72B, 0x72B, 0x72C, 0x72D, 0x72E, 0x72F, 0x730,
    0x731, 0x732, 0x733, 0x734, 0x734, 0x735, 0x736, 0x737, 0x738, 0x739,
    0x73A, 0x73B, 0x73C, 0x73D, 0x73E, 0x73E, 0x73F, 0x740, 0x741, 0x742,
    0x743, 0x744, 0x745, 0x746, 0x747, 0x748, 0x748, 0x749, 0x74A, 0x74B,
    0x74C, 0x74D, 0x74E, 0x74F, 0x750, 0x751, 0x752, 0x753, 0x753, 0x754,
    0x755, 0x756, 0x757, 0x758, 0x759, 0x75A, 0x75B, 0x75C, 0x75D, 0x75E,
    0x75F, 0x75F, 0x760, 0x761, 0x762, 0x763, 0x764, 0x765, 0x766, 0x767,
    0x768, 0x769, 0x76A, 0x76B, 0x76C, 0x76C, 0x76D, 0x76E, 0x76F, 0x770,
    0x771, 0x772, 0x773, 0x774, 0x775, 0x776, 0x777, 0x778, 0x779, 0x77A,
    0x77A, 0x77B, 0x77C, 0x77D, 0x77E, 0x77F, 0x780, 0x781, 0x782, 0x783,
    0x784, 0x785, 0x786, 0x787, 0x788, 0x789, 0x78A, 0x78B, 0x78B, 0x78C,
    0x78D, 0x78E, 0x78F, 0x790, 0x791, 0x792, 0x793, 0x794, 0x795, 0x796,
    0x797, 0x798, 0x799, 0x79A, 0x79B, 0x79C, 0x79D, 0x79E, 0x79E, 0x79F,
    0x7A0, 0x7A1, 0x7A2, 0x7A3, 0x7A4, 0x7A5, 0x7A6, 0x7A7, 0x7A8, 0x7A9,
    0x7AA, 0x7AB, 0x7AC, 0x7AD, 0x7AE, 0x7AF, 0x7B0, 0x7B1, 0x7B2, 0x7B3,
    0x7B4, 0x7B5, 0x7B6, 0x7B7, 0x7B8, 0x7B8, 0x7B9, 0x7BA, 0x7BB, 0x7BC,
    0x7BD, 0x7BE, 0x7BF, 0x7C0, 0x7C1, 0x7C2, 0x7C3, 0x7C4, 0x7C5, 0x7C6,
    0x7C7, 0x7C8, 0x7C9, 0x7CA, 0x7CB, 0x7CC, 0x7CD, 0x7CE, 0x7CF, 0x7D0,
    0x7D1, 0x7D2, 0x7D3, 0x7D4, 0x7D5, 0x7D6, 0x7D7, 0x7D8, 0x7D9, 0x7DA,
    0x7DB, 0x7DC, 0x7DD, 0x7DE, 0x7DF, 0x7E0, 0x7E1, 0x7E2, 0x7E3, 0x7E4,
    0x7E5, 0x7E6, 0x7E7, 0x7E8, 0x7E9, 0x7EA, 0x7EB, 0x7EC, 0x7ED, 0x7EE,
    0x7EF, 0x7F0, 0x7F1, 0x7F2, 0x7F3, 0x7F3, 0x7F4, 0x7F5, 0x7F6, 0x7F7,
    0x7F8, 0x7F9, 0x7FA
};

uint16_t opnModel_genericFreqOPNA(double tone, uint32_t *mul_offset)
{
    uint32_t octave = 0;
    uint16_t freq;
    size_t idx;

    *mul_offset = 0;

    if(tone < 0.0)
        tone = 0.0;

    while(tone > s_maxToneValueOPNA)
    {
        tone -= 12.0;
        ++octave;
    }

    idx = (size_t)(tone / s_expTableDividorOPNA);

    if(idx > OPNA_EXP_TABLE_SIZE)
        idx = OPNA_EXP_TABLE_SIZE - 1; /* Out of range! */

    freq = s_OPNAgenericExpTable[idx];

    while(octave > 7)
    {
        ++(*mul_offset);
        --octave;
    }

    return freq | (octave << 11);
}

