/*
 * OPL2/OPL3 models library - a set of various conversion models for OPL-family chips
 *
 * Copyright (c) 2025-2025 Vitaly Novichkov <admin@wohlnet.ru>
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <stddef.h>
#include <math.h>   /* log() */
#include "opl_models.h"

/***************************************************************
 *               Standard frequency formula                    *
 ***************************************************************/

/***************************************************************
 * To compute frequency from MIDI tone used the next formula:  *
 * hz = 172.4387 * exp(0.057762265 * tone)                     *
 *    tone - MIDI tone with bend detune as decimals            *
 *                                                             *
 *  The highest tone possible in the range is 30.823808        *
 *  (I mean, without going to higher octaves)                  *
 *                                                             *
 * This table is built using "freq-opl.c" file at the "util"   *
 * directory.                                                  *
 ***************************************************************/


static const double s_expTableDividor = 0.0169268578;
static const double s_maxToneValue = 30.823808;
#define     EXP_TABLE_SIZE  1822

static const uint16_t s_genericExpTable[EXP_TABLE_SIZE] =
{
    0x0AC, 0x0AC, 0x0AC, 0x0AC, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD, 0x0AD,
    0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AE, 0x0AF, 0x0AF, 0x0AF, 0x0AF,
    0x0AF, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B0, 0x0B1, 0x0B1, 0x0B1,
    0x0B1, 0x0B1, 0x0B1, 0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B2, 0x0B3,
    0x0B3, 0x0B3, 0x0B3, 0x0B3, 0x0B4, 0x0B4, 0x0B4, 0x0B4, 0x0B4, 0x0B4,
    0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B5, 0x0B6, 0x0B6, 0x0B6, 0x0B6,
    0x0B6, 0x0B7, 0x0B7, 0x0B7, 0x0B7, 0x0B7, 0x0B7, 0x0B8, 0x0B8, 0x0B8,
    0x0B8, 0x0B8, 0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0B9, 0x0BA, 0x0BA,
    0x0BA, 0x0BA, 0x0BA, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BB, 0x0BC,
    0x0BC, 0x0BC, 0x0BC, 0x0BC, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BD, 0x0BD,
    0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BE, 0x0BF, 0x0BF, 0x0BF, 0x0BF, 0x0BF,
    0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C0, 0x0C1, 0x0C1, 0x0C1, 0x0C1,
    0x0C1, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C2, 0x0C3, 0x0C3, 0x0C3, 0x0C3,
    0x0C3, 0x0C4, 0x0C4, 0x0C4, 0x0C4, 0x0C4, 0x0C4, 0x0C5, 0x0C5, 0x0C5,
    0x0C5, 0x0C5, 0x0C6, 0x0C6, 0x0C6, 0x0C6, 0x0C6, 0x0C7, 0x0C7, 0x0C7,
    0x0C7, 0x0C7, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C8, 0x0C9, 0x0C9, 0x0C9,
    0x0C9, 0x0C9, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CA, 0x0CB, 0x0CB, 0x0CB,
    0x0CB, 0x0CB, 0x0CC, 0x0CC, 0x0CC, 0x0CC, 0x0CC, 0x0CD, 0x0CD, 0x0CD,
    0x0CD, 0x0CD, 0x0CE, 0x0CE, 0x0CE, 0x0CE, 0x0CE, 0x0CF, 0x0CF, 0x0CF,
    0x0CF, 0x0CF, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D0, 0x0D1, 0x0D1, 0x0D1,
    0x0D1, 0x0D1, 0x0D2, 0x0D2, 0x0D2, 0x0D2, 0x0D2, 0x0D3, 0x0D3, 0x0D3,
    0x0D3, 0x0D3, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D4, 0x0D5, 0x0D5, 0x0D5,
    0x0D5, 0x0D6, 0x0D6, 0x0D6, 0x0D6, 0x0D6, 0x0D7, 0x0D7, 0x0D7, 0x0D7,
    0x0D7, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D8, 0x0D9, 0x0D9, 0x0D9, 0x0D9,
    0x0DA, 0x0DA, 0x0DA, 0x0DA, 0x0DA, 0x0DB, 0x0DB, 0x0DB, 0x0DB, 0x0DB,
    0x0DC, 0x0DC, 0x0DC, 0x0DC, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DD, 0x0DE,
    0x0DE, 0x0DE, 0x0DE, 0x0DF, 0x0DF, 0x0DF, 0x0DF, 0x0DF, 0x0E0, 0x0E0,
    0x0E0, 0x0E0, 0x0E0, 0x0E1, 0x0E1, 0x0E1, 0x0E1, 0x0E2, 0x0E2, 0x0E2,
    0x0E2, 0x0E2, 0x0E3, 0x0E3, 0x0E3, 0x0E3, 0x0E4, 0x0E4, 0x0E4, 0x0E4,
    0x0E4, 0x0E5, 0x0E5, 0x0E5, 0x0E5, 0x0E6, 0x0E6, 0x0E6, 0x0E6, 0x0E6,
    0x0E7, 0x0E7, 0x0E7, 0x0E7, 0x0E8, 0x0E8, 0x0E8, 0x0E8, 0x0E9, 0x0E9,
    0x0E9, 0x0E9, 0x0E9, 0x0EA, 0x0EA, 0x0EA, 0x0EA, 0x0EB, 0x0EB, 0x0EB,
    0x0EB, 0x0EC, 0x0EC, 0x0EC, 0x0EC, 0x0EC, 0x0ED, 0x0ED, 0x0ED, 0x0ED,
    0x0EE, 0x0EE, 0x0EE, 0x0EE, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0EF, 0x0F0,
    0x0F0, 0x0F0, 0x0F0, 0x0F1, 0x0F1, 0x0F1, 0x0F1, 0x0F2, 0x0F2, 0x0F2,
    0x0F2, 0x0F3, 0x0F3, 0x0F3, 0x0F3, 0x0F3, 0x0F4, 0x0F4, 0x0F4, 0x0F4,
    0x0F5, 0x0F5, 0x0F5, 0x0F5, 0x0F6, 0x0F6, 0x0F6, 0x0F6, 0x0F7, 0x0F7,
    0x0F7, 0x0F7, 0x0F8, 0x0F8, 0x0F8, 0x0F8, 0x0F9, 0x0F9, 0x0F9, 0x0F9,
    0x0FA, 0x0FA, 0x0FA, 0x0FA, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FB, 0x0FC,
    0x0FC, 0x0FC, 0x0FC, 0x0FD, 0x0FD, 0x0FD, 0x0FD, 0x0FE, 0x0FE, 0x0FE,
    0x0FE, 0x0FF, 0x0FF, 0x0FF, 0x0FF, 0x100, 0x100, 0x100, 0x100, 0x101,
    0x101, 0x101, 0x101, 0x102, 0x102, 0x102, 0x102, 0x103, 0x103, 0x103,
    0x104, 0x104, 0x104, 0x104, 0x105, 0x105, 0x105, 0x105, 0x106, 0x106,
    0x106, 0x106, 0x107, 0x107, 0x107, 0x107, 0x108, 0x108, 0x108, 0x108,
    0x109, 0x109, 0x109, 0x109, 0x10A, 0x10A, 0x10A, 0x10A, 0x10B, 0x10B,
    0x10B, 0x10C, 0x10C, 0x10C, 0x10C, 0x10D, 0x10D, 0x10D, 0x10D, 0x10E,
    0x10E, 0x10E, 0x10E, 0x10F, 0x10F, 0x10F, 0x10F, 0x110, 0x110, 0x110,
    0x111, 0x111, 0x111, 0x111, 0x112, 0x112, 0x112, 0x112, 0x113, 0x113,
    0x113, 0x113, 0x114, 0x114, 0x114, 0x115, 0x115, 0x115, 0x115, 0x116,
    0x116, 0x116, 0x116, 0x117, 0x117, 0x117, 0x118, 0x118, 0x118, 0x118,
    0x119, 0x119, 0x119, 0x119, 0x11A, 0x11A, 0x11A, 0x11B, 0x11B, 0x11B,
    0x11B, 0x11C, 0x11C, 0x11C, 0x11D, 0x11D, 0x11D, 0x11D, 0x11E, 0x11E,
    0x11E, 0x11E, 0x11F, 0x11F, 0x11F, 0x120, 0x120, 0x120, 0x120, 0x121,
    0x121, 0x121, 0x122, 0x122, 0x122, 0x122, 0x123, 0x123, 0x123, 0x124,
    0x124, 0x124, 0x124, 0x125, 0x125, 0x125, 0x126, 0x126, 0x126, 0x126,
    0x127, 0x127, 0x127, 0x128, 0x128, 0x128, 0x128, 0x129, 0x129, 0x129,
    0x12A, 0x12A, 0x12A, 0x12B, 0x12B, 0x12B, 0x12B, 0x12C, 0x12C, 0x12C,
    0x12D, 0x12D, 0x12D, 0x12D, 0x12E, 0x12E, 0x12E, 0x12F, 0x12F, 0x12F,
    0x130, 0x130, 0x130, 0x130, 0x131, 0x131, 0x131, 0x132, 0x132, 0x132,
    0x133, 0x133, 0x133, 0x133, 0x134, 0x134, 0x134, 0x135, 0x135, 0x135,
    0x136, 0x136, 0x136, 0x136, 0x137, 0x137, 0x137, 0x138, 0x138, 0x138,
    0x139, 0x139, 0x139, 0x139, 0x13A, 0x13A, 0x13A, 0x13B, 0x13B, 0x13B,
    0x13C, 0x13C, 0x13C, 0x13D, 0x13D, 0x13D, 0x13E, 0x13E, 0x13E, 0x13E,
    0x13F, 0x13F, 0x13F, 0x140, 0x140, 0x140, 0x141, 0x141, 0x141, 0x142,
    0x142, 0x142, 0x143, 0x143, 0x143, 0x143, 0x144, 0x144, 0x144, 0x145,
    0x145, 0x145, 0x146, 0x146, 0x146, 0x147, 0x147, 0x147, 0x148, 0x148,
    0x148, 0x149, 0x149, 0x149, 0x14A, 0x14A, 0x14A, 0x14B, 0x14B, 0x14B,
    0x14B, 0x14C, 0x14C, 0x14C, 0x14D, 0x14D, 0x14D, 0x14E, 0x14E, 0x14E,
    0x14F, 0x14F, 0x14F, 0x150, 0x150, 0x150, 0x151, 0x151, 0x151, 0x152,
    0x152, 0x152, 0x153, 0x153, 0x153, 0x154, 0x154, 0x154, 0x155, 0x155,
    0x155, 0x156, 0x156, 0x156, 0x157, 0x157, 0x157, 0x158, 0x158, 0x158,
    0x159, 0x159, 0x159, 0x15A, 0x15A, 0x15A, 0x15B, 0x15B, 0x15B, 0x15C,
    0x15C, 0x15C, 0x15D, 0x15D, 0x15D, 0x15E, 0x15E, 0x15F, 0x15F, 0x15F,
    0x160, 0x160, 0x160, 0x161, 0x161, 0x161, 0x162, 0x162, 0x162, 0x163,
    0x163, 0x163, 0x164, 0x164, 0x164, 0x165, 0x165, 0x165, 0x166, 0x166,
    0x167, 0x167, 0x167, 0x168, 0x168, 0x168, 0x169, 0x169, 0x169, 0x16A,
    0x16A, 0x16A, 0x16B, 0x16B, 0x16B, 0x16C, 0x16C, 0x16D, 0x16D, 0x16D,
    0x16E, 0x16E, 0x16E, 0x16F, 0x16F, 0x16F, 0x170, 0x170, 0x170, 0x171,
    0x171, 0x172, 0x172, 0x172, 0x173, 0x173, 0x173, 0x174, 0x174, 0x174,
    0x175, 0x175, 0x176, 0x176, 0x176, 0x177, 0x177, 0x177, 0x178, 0x178,
    0x178, 0x179, 0x179, 0x17A, 0x17A, 0x17A, 0x17B, 0x17B, 0x17B, 0x17C,
    0x17C, 0x17D, 0x17D, 0x17D, 0x17E, 0x17E, 0x17E, 0x17F, 0x17F, 0x180,
    0x180, 0x180, 0x181, 0x181, 0x181, 0x182, 0x182, 0x183, 0x183, 0x183,
    0x184, 0x184, 0x184, 0x185, 0x185, 0x186, 0x186, 0x186, 0x187, 0x187,
    0x188, 0x188, 0x188, 0x189, 0x189, 0x189, 0x18A, 0x18A, 0x18B, 0x18B,
    0x18B, 0x18C, 0x18C, 0x18D, 0x18D, 0x18D, 0x18E, 0x18E, 0x18E, 0x18F,
    0x18F, 0x190, 0x190, 0x190, 0x191, 0x191, 0x192, 0x192, 0x192, 0x193,
    0x193, 0x194, 0x194, 0x194, 0x195, 0x195, 0x196, 0x196, 0x196, 0x197,
    0x197, 0x198, 0x198, 0x198, 0x199, 0x199, 0x19A, 0x19A, 0x19A, 0x19B,
    0x19B, 0x19C, 0x19C, 0x19C, 0x19D, 0x19D, 0x19E, 0x19E, 0x19E, 0x19F,
    0x19F, 0x1A0, 0x1A0, 0x1A0, 0x1A1, 0x1A1, 0x1A2, 0x1A2, 0x1A2, 0x1A3,
    0x1A3, 0x1A4, 0x1A4, 0x1A5, 0x1A5, 0x1A5, 0x1A6, 0x1A6, 0x1A7, 0x1A7,
    0x1A7, 0x1A8, 0x1A8, 0x1A9, 0x1A9, 0x1AA, 0x1AA, 0x1AA, 0x1AB, 0x1AB,
    0x1AC, 0x1AC, 0x1AC, 0x1AD, 0x1AD, 0x1AE, 0x1AE, 0x1AF, 0x1AF, 0x1AF,
    0x1B0, 0x1B0, 0x1B1, 0x1B1, 0x1B1, 0x1B2, 0x1B2, 0x1B3, 0x1B3, 0x1B4,
    0x1B4, 0x1B4, 0x1B5, 0x1B5, 0x1B6, 0x1B6, 0x1B7, 0x1B7, 0x1B7, 0x1B8,
    0x1B8, 0x1B9, 0x1B9, 0x1BA, 0x1BA, 0x1BA, 0x1BB, 0x1BB, 0x1BC, 0x1BC,
    0x1BD, 0x1BD, 0x1BE, 0x1BE, 0x1BE, 0x1BF, 0x1BF, 0x1C0, 0x1C0, 0x1C1,
    0x1C1, 0x1C1, 0x1C2, 0x1C2, 0x1C3, 0x1C3, 0x1C4, 0x1C4, 0x1C5, 0x1C5,
    0x1C5, 0x1C6, 0x1C6, 0x1C7, 0x1C7, 0x1C8, 0x1C8, 0x1C9, 0x1C9, 0x1C9,
    0x1CA, 0x1CA, 0x1CB, 0x1CB, 0x1CC, 0x1CC, 0x1CD, 0x1CD, 0x1CE, 0x1CE,
    0x1CE, 0x1CF, 0x1CF, 0x1D0, 0x1D0, 0x1D1, 0x1D1, 0x1D2, 0x1D2, 0x1D3,
    0x1D3, 0x1D3, 0x1D4, 0x1D4, 0x1D5, 0x1D5, 0x1D6, 0x1D6, 0x1D7, 0x1D7,
    0x1D8, 0x1D8, 0x1D8, 0x1D9, 0x1D9, 0x1DA, 0x1DA, 0x1DB, 0x1DB, 0x1DC,
    0x1DC, 0x1DD, 0x1DD, 0x1DE, 0x1DE, 0x1DF, 0x1DF, 0x1DF, 0x1E0, 0x1E0,
    0x1E1, 0x1E1, 0x1E2, 0x1E2, 0x1E3, 0x1E3, 0x1E4, 0x1E4, 0x1E5, 0x1E5,
    0x1E6, 0x1E6, 0x1E7, 0x1E7, 0x1E8, 0x1E8, 0x1E8, 0x1E9, 0x1E9, 0x1EA,
    0x1EA, 0x1EB, 0x1EB, 0x1EC, 0x1EC, 0x1ED, 0x1ED, 0x1EE, 0x1EE, 0x1EF,
    0x1EF, 0x1F0, 0x1F0, 0x1F1, 0x1F1, 0x1F2, 0x1F2, 0x1F3, 0x1F3, 0x1F4,
    0x1F4, 0x1F5, 0x1F5, 0x1F6, 0x1F6, 0x1F7, 0x1F7, 0x1F8, 0x1F8, 0x1F9,
    0x1F9, 0x1F9, 0x1FA, 0x1FA, 0x1FB, 0x1FB, 0x1FC, 0x1FC, 0x1FD, 0x1FD,
    0x1FE, 0x1FE, 0x1FF, 0x1FF, 0x200, 0x200, 0x201, 0x201, 0x202, 0x202,
    0x203, 0x203, 0x204, 0x204, 0x205, 0x206, 0x206, 0x207, 0x207, 0x208,
    0x208, 0x209, 0x209, 0x20A, 0x20A, 0x20B, 0x20B, 0x20C, 0x20C, 0x20D,
    0x20D, 0x20E, 0x20E, 0x20F, 0x20F, 0x210, 0x210, 0x211, 0x211, 0x212,
    0x212, 0x213, 0x213, 0x214, 0x214, 0x215, 0x215, 0x216, 0x216, 0x217,
    0x218, 0x218, 0x219, 0x219, 0x21A, 0x21A, 0x21B, 0x21B, 0x21C, 0x21C,
    0x21D, 0x21D, 0x21E, 0x21E, 0x21F, 0x21F, 0x220, 0x221, 0x221, 0x222,
    0x222, 0x223, 0x223, 0x224, 0x224, 0x225, 0x225, 0x226, 0x226, 0x227,
    0x227, 0x228, 0x229, 0x229, 0x22A, 0x22A, 0x22B, 0x22B, 0x22C, 0x22C,
    0x22D, 0x22D, 0x22E, 0x22F, 0x22F, 0x230, 0x230, 0x231, 0x231, 0x232,
    0x232, 0x233, 0x234, 0x234, 0x235, 0x235, 0x236, 0x236, 0x237, 0x237,
    0x238, 0x238, 0x239, 0x23A, 0x23A, 0x23B, 0x23B, 0x23C, 0x23C, 0x23D,
    0x23E, 0x23E, 0x23F, 0x23F, 0x240, 0x240, 0x241, 0x241, 0x242, 0x243,
    0x243, 0x244, 0x244, 0x245, 0x245, 0x246, 0x247, 0x247, 0x248, 0x248,
    0x249, 0x249, 0x24A, 0x24B, 0x24B, 0x24C, 0x24C, 0x24D, 0x24D, 0x24E,
    0x24F, 0x24F, 0x250, 0x250, 0x251, 0x251, 0x252, 0x253, 0x253, 0x254,
    0x254, 0x255, 0x256, 0x256, 0x257, 0x257, 0x258, 0x259, 0x259, 0x25A,
    0x25A, 0x25B, 0x25B, 0x25C, 0x25D, 0x25D, 0x25E, 0x25E, 0x25F, 0x260,
    0x260, 0x261, 0x261, 0x262, 0x263, 0x263, 0x264, 0x264, 0x265, 0x266,
    0x266, 0x267, 0x267, 0x268, 0x269, 0x269, 0x26A, 0x26A, 0x26B, 0x26C,
    0x26C, 0x26D, 0x26D, 0x26E, 0x26F, 0x26F, 0x270, 0x270, 0x271, 0x272,
    0x272, 0x273, 0x274, 0x274, 0x275, 0x275, 0x276, 0x277, 0x277, 0x278,
    0x278, 0x279, 0x27A, 0x27A, 0x27B, 0x27C, 0x27C, 0x27D, 0x27D, 0x27E,
    0x27F, 0x27F, 0x280, 0x281, 0x281, 0x282, 0x282, 0x283, 0x284, 0x284,
    0x285, 0x286, 0x286, 0x287, 0x288, 0x288, 0x289, 0x289, 0x28A, 0x28B,
    0x28B, 0x28C, 0x28D, 0x28D, 0x28E, 0x28F, 0x28F, 0x290, 0x290, 0x291,
    0x292, 0x292, 0x293, 0x294, 0x294, 0x295, 0x296, 0x296, 0x297, 0x298,
    0x298, 0x299, 0x299, 0x29A, 0x29B, 0x29B, 0x29C, 0x29D, 0x29D, 0x29E,
    0x29F, 0x29F, 0x2A0, 0x2A1, 0x2A1, 0x2A2, 0x2A3, 0x2A3, 0x2A4, 0x2A5,
    0x2A5, 0x2A6, 0x2A7, 0x2A7, 0x2A8, 0x2A9, 0x2A9, 0x2AA, 0x2AB, 0x2AB,
    0x2AC, 0x2AD, 0x2AD, 0x2AE, 0x2AF, 0x2AF, 0x2B0, 0x2B1, 0x2B1, 0x2B2,
    0x2B3, 0x2B3, 0x2B4, 0x2B5, 0x2B5, 0x2B6, 0x2B7, 0x2B7, 0x2B8, 0x2B9,
    0x2B9, 0x2BA, 0x2BB, 0x2BC, 0x2BC, 0x2BD, 0x2BE, 0x2BE, 0x2BF, 0x2C0,
    0x2C0, 0x2C1, 0x2C2, 0x2C2, 0x2C3, 0x2C4, 0x2C4, 0x2C5, 0x2C6, 0x2C7,
    0x2C7, 0x2C8, 0x2C9, 0x2C9, 0x2CA, 0x2CB, 0x2CB, 0x2CC, 0x2CD, 0x2CE,
    0x2CE, 0x2CF, 0x2D0, 0x2D0, 0x2D1, 0x2D2, 0x2D2, 0x2D3, 0x2D4, 0x2D5,
    0x2D5, 0x2D6, 0x2D7, 0x2D7, 0x2D8, 0x2D9, 0x2DA, 0x2DA, 0x2DB, 0x2DC,
    0x2DC, 0x2DD, 0x2DE, 0x2DF, 0x2DF, 0x2E0, 0x2E1, 0x2E1, 0x2E2, 0x2E3,
    0x2E4, 0x2E4, 0x2E5, 0x2E6, 0x2E7, 0x2E7, 0x2E8, 0x2E9, 0x2E9, 0x2EA,
    0x2EB, 0x2EC, 0x2EC, 0x2ED, 0x2EE, 0x2EF, 0x2EF, 0x2F0, 0x2F1, 0x2F2,
    0x2F2, 0x2F3, 0x2F4, 0x2F4, 0x2F5, 0x2F6, 0x2F7, 0x2F7, 0x2F8, 0x2F9,
    0x2FA, 0x2FA, 0x2FB, 0x2FC, 0x2FD, 0x2FD, 0x2FE, 0x2FF, 0x300, 0x300,
    0x301, 0x302, 0x303, 0x303, 0x304, 0x305, 0x306, 0x306, 0x307, 0x308,
    0x309, 0x30A, 0x30A, 0x30B, 0x30C, 0x30D, 0x30D, 0x30E, 0x30F, 0x310,
    0x310, 0x311, 0x312, 0x313, 0x313, 0x314, 0x315, 0x316, 0x317, 0x317,
    0x318, 0x319, 0x31A, 0x31A, 0x31B, 0x31C, 0x31D, 0x31E, 0x31E, 0x31F,
    0x320, 0x321, 0x321, 0x322, 0x323, 0x324, 0x325, 0x325, 0x326, 0x327,
    0x328, 0x329, 0x329, 0x32A, 0x32B, 0x32C, 0x32C, 0x32D, 0x32E, 0x32F,
    0x330, 0x330, 0x331, 0x332, 0x333, 0x334, 0x334, 0x335, 0x336, 0x337,
    0x338, 0x339, 0x339, 0x33A, 0x33B, 0x33C, 0x33D, 0x33D, 0x33E, 0x33F,
    0x340, 0x341, 0x341, 0x342, 0x343, 0x344, 0x345, 0x346, 0x346, 0x347,
    0x348, 0x349, 0x34A, 0x34A, 0x34B, 0x34C, 0x34D, 0x34E, 0x34F, 0x34F,
    0x350, 0x351, 0x352, 0x353, 0x354, 0x354, 0x355, 0x356, 0x357, 0x358,
    0x359, 0x359, 0x35A, 0x35B, 0x35C, 0x35D, 0x35E, 0x35E, 0x35F, 0x360,
    0x361, 0x362, 0x363, 0x364, 0x364, 0x365, 0x366, 0x367, 0x368, 0x369,
    0x369, 0x36A, 0x36B, 0x36C, 0x36D, 0x36E, 0x36F, 0x370, 0x370, 0x371,
    0x372, 0x373, 0x374, 0x375, 0x376, 0x376, 0x377, 0x378, 0x379, 0x37A,
    0x37B, 0x37C, 0x37D, 0x37D, 0x37E, 0x37F, 0x380, 0x381, 0x382, 0x383,
    0x384, 0x384, 0x385, 0x386, 0x387, 0x388, 0x389, 0x38A, 0x38B, 0x38B,
    0x38C, 0x38D, 0x38E, 0x38F, 0x390, 0x391, 0x392, 0x393, 0x393, 0x394,
    0x395, 0x396, 0x397, 0x398, 0x399, 0x39A, 0x39B, 0x39C, 0x39C, 0x39D,
    0x39E, 0x39F, 0x3A0, 0x3A1, 0x3A2, 0x3A3, 0x3A4, 0x3A5, 0x3A6, 0x3A6,
    0x3A7, 0x3A8, 0x3A9, 0x3AA, 0x3AB, 0x3AC, 0x3AD, 0x3AE, 0x3AF, 0x3B0,
    0x3B1, 0x3B2, 0x3B2, 0x3B3, 0x3B4, 0x3B5, 0x3B6, 0x3B7, 0x3B8, 0x3B9,
    0x3BA, 0x3BB, 0x3BC, 0x3BD, 0x3BE, 0x3BF, 0x3C0, 0x3C0, 0x3C1, 0x3C2,
    0x3C3, 0x3C4, 0x3C5, 0x3C6, 0x3C7, 0x3C8, 0x3C9, 0x3CA, 0x3CB, 0x3CC,
    0x3CD, 0x3CE, 0x3CF, 0x3D0, 0x3D1, 0x3D2, 0x3D2, 0x3D3, 0x3D4, 0x3D5,
    0x3D6, 0x3D7, 0x3D8, 0x3D9, 0x3DA, 0x3DB, 0x3DC, 0x3DD, 0x3DE, 0x3DF,
    0x3E0, 0x3E1, 0x3E2, 0x3E3, 0x3E4, 0x3E5, 0x3E6, 0x3E7, 0x3E8, 0x3E9,
    0x3EA, 0x3EB, 0x3EC, 0x3ED, 0x3EE, 0x3EF, 0x3F0, 0x3F1, 0x3F2, 0x3F3,
    0x3F4, 0x3F5, 0x3F6, 0x3F7, 0x3F8, 0x3F9, 0x3FA, 0x3FB, 0x3FC, 0x3FD,
    0x3FE, 0x3FF,
};


uint16_t oplModel_genericFreq(double tone, uint32_t *mul_offset)
{
    uint32_t octave = 0;
    uint16_t freq;
    size_t idx;

    *mul_offset = 0;

    if(tone < 0.0)
        tone = 0.0;

    while(tone > s_maxToneValue)
    {
        tone -= 12.0;
        ++octave;
    }

    idx = (size_t)(tone / s_expTableDividor);

    if(idx > EXP_TABLE_SIZE)
        idx = EXP_TABLE_SIZE - 1; /* Out of range! */

    freq = s_genericExpTable[idx];

    while(octave > 7)
    {
        ++(*mul_offset);
        --octave;
    }

    return freq | (octave << 10);
}


/***************************************************************
 *                     Generic volume formula                  *
 ***************************************************************/

void oplModel_genericVolume(struct OPLVolume_t *v)
{
    const double c1 = 11.541560327111707;
    const double c2 = 1.601379199767093e+02;
    const uint_fast32_t minVolume = 1108075; /* 8725 * 127 */
    double lv;
    uint_fast32_t volume = 0;

    volume = v->vel * v->masterVolume * v->chVol * v->chExpr;

    if(volume > minVolume)
    {
        lv = log((double)volume);
        volume = (uint_fast32_t)(lv * c1 - c2);

        if(volume > 63)
            volume = 63;
    }
    else
        volume = 0;

    if(v->doMod)
        v->tlMod = 63 - volume + (volume * v->tlMod) / 63;

    if(v->doCar)
        v->tlCar = 63 - volume + (volume * v->tlCar) / 63;
}

void oplModel_nativeVolume(struct OPLVolume_t *v)
{
    uint_fast32_t volume = 0;

    volume = v->vel * v->chVol * v->chExpr;
    /* 4096766 = (127 * 127 * 127) * 2 */
    volume = (volume * v->masterVolume) / 4096766;

    if(volume > 63)
        volume = 63;

    if(v->doMod)
        v->tlMod = 63 - volume + (volume * v->tlMod) / 63;

    if(v->doCar)
        v->tlCar = 63 - volume + (volume * v->tlCar) / 63;
}

void oplModel_rsxxVolume(struct OPLVolume_t *v)
{
    uint_fast32_t volume = 0;

    volume = v->vel * v->chVol * v->chExpr;
    /* 4096766 = (127 * 127 * 127) * 2 */
    volume = (volume * v->masterVolume) / 4096766;

    if(volume > 63)
        volume = 63;

    v->tlCar -= volume / 2;
}



/***************************************************************
 *                     XG CC74 Brightness                      *
 ***************************************************************/

/*! Pre-computed table of XG Brightness for OPL family chips.
    Result of:
    ```
    double b = (double)(brightness);
    double ret = round(127.0 * sqrt(b * (1.0 / 127.0))) / 2.0;
    return (uint_fast32_t)(ret);
    ```
*/
static uint8_t s_xgBrightness[] =
{
    0,  5,  8,  10, 11, 12, 14, 15, 16, 17,
    18, 18, 19, 20, 21, 22, 22, 23, 24, 24,
    25, 26, 26, 27, 27, 28, 28, 29, 30, 30,
    31, 31, 32, 32, 33, 33, 34, 34, 34, 35,
    35, 36, 36, 37, 37, 38, 38, 38, 39, 39,
    40, 40, 40, 41, 41, 42, 42, 42, 43, 43,
    43, 44, 44, 44, 45, 45, 46, 46, 46, 47,
    47, 47, 48, 48, 48, 49, 49, 49, 50, 50,
    50, 50, 51, 51, 51, 52, 52, 52, 53, 53,
    53, 54, 54, 54, 54, 55, 55, 55, 56, 56,
    56, 56, 57, 57, 57, 57, 58, 58, 58, 59,
    59, 59, 59, 60, 60, 60, 60, 61, 61, 61,
    61, 62, 62, 62, 62, 63, 63, 63
};

uint_fast16_t oplModels_xgBrightnessToOPL(uint_fast16_t brightness)
{
    return s_xgBrightness[brightness & 0xFF];
}
