/*
 * OPL2/OPL3 models library - a set of various conversion models for OPL-family chips
 *
 * Copyright (c) 2025-2025 Vitaly Novichkov <admin@wohlnet.ru>
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "opl_models.h"

/***************************************************************
 *             Apogee Sound System frequency model             *
 ***************************************************************/

static const int_fast32_t s_apogee_freq_table[31 + 1][12] =
{
    {0x157, 0x16b, 0x181, 0x198, 0x1b0, 0x1ca, 0x1e5, 0x202, 0x220, 0x241, 0x263, 0x287},
    {0x157, 0x16b, 0x181, 0x198, 0x1b0, 0x1ca, 0x1e5, 0x202, 0x220, 0x242, 0x264, 0x288},
    {0x158, 0x16c, 0x182, 0x199, 0x1b1, 0x1cb, 0x1e6, 0x203, 0x221, 0x243, 0x265, 0x289},
    {0x158, 0x16c, 0x183, 0x19a, 0x1b2, 0x1cc, 0x1e7, 0x204, 0x222, 0x244, 0x266, 0x28a},
    {0x159, 0x16d, 0x183, 0x19a, 0x1b3, 0x1cd, 0x1e8, 0x205, 0x223, 0x245, 0x267, 0x28b},
    {0x15a, 0x16e, 0x184, 0x19b, 0x1b3, 0x1ce, 0x1e9, 0x206, 0x224, 0x246, 0x268, 0x28c},
    {0x15a, 0x16e, 0x185, 0x19c, 0x1b4, 0x1ce, 0x1ea, 0x207, 0x225, 0x247, 0x269, 0x28e},
    {0x15b, 0x16f, 0x185, 0x19d, 0x1b5, 0x1cf, 0x1eb, 0x208, 0x226, 0x248, 0x26a, 0x28f},
    {0x15b, 0x170, 0x186, 0x19d, 0x1b6, 0x1d0, 0x1ec, 0x209, 0x227, 0x249, 0x26b, 0x290},
    {0x15c, 0x170, 0x187, 0x19e, 0x1b7, 0x1d1, 0x1ec, 0x20a, 0x228, 0x24a, 0x26d, 0x291},
    {0x15d, 0x171, 0x188, 0x19f, 0x1b7, 0x1d2, 0x1ed, 0x20b, 0x229, 0x24b, 0x26e, 0x292},
    {0x15d, 0x172, 0x188, 0x1a0, 0x1b8, 0x1d3, 0x1ee, 0x20c, 0x22a, 0x24c, 0x26f, 0x293},
    {0x15e, 0x172, 0x189, 0x1a0, 0x1b9, 0x1d4, 0x1ef, 0x20d, 0x22b, 0x24d, 0x270, 0x295},
    {0x15f, 0x173, 0x18a, 0x1a1, 0x1ba, 0x1d4, 0x1f0, 0x20e, 0x22c, 0x24e, 0x271, 0x296},
    {0x15f, 0x174, 0x18a, 0x1a2, 0x1bb, 0x1d5, 0x1f1, 0x20f, 0x22d, 0x24f, 0x272, 0x297},
    {0x160, 0x174, 0x18b, 0x1a3, 0x1bb, 0x1d6, 0x1f2, 0x210, 0x22e, 0x250, 0x273, 0x298},
    {0x161, 0x175, 0x18c, 0x1a3, 0x1bc, 0x1d7, 0x1f3, 0x211, 0x22f, 0x251, 0x274, 0x299},
    {0x161, 0x176, 0x18c, 0x1a4, 0x1bd, 0x1d8, 0x1f4, 0x212, 0x230, 0x252, 0x276, 0x29b},
    {0x162, 0x176, 0x18d, 0x1a5, 0x1be, 0x1d9, 0x1f5, 0x212, 0x231, 0x254, 0x277, 0x29c},
    {0x162, 0x177, 0x18e, 0x1a6, 0x1bf, 0x1d9, 0x1f5, 0x213, 0x232, 0x255, 0x278, 0x29d},
    {0x163, 0x178, 0x18f, 0x1a6, 0x1bf, 0x1da, 0x1f6, 0x214, 0x233, 0x256, 0x279, 0x29e},
    {0x164, 0x179, 0x18f, 0x1a7, 0x1c0, 0x1db, 0x1f7, 0x215, 0x235, 0x257, 0x27a, 0x29f},
    {0x164, 0x179, 0x190, 0x1a8, 0x1c1, 0x1dc, 0x1f8, 0x216, 0x236, 0x258, 0x27b, 0x2a1},
    {0x165, 0x17a, 0x191, 0x1a9, 0x1c2, 0x1dd, 0x1f9, 0x217, 0x237, 0x259, 0x27c, 0x2a2},
    {0x166, 0x17b, 0x192, 0x1aa, 0x1c3, 0x1de, 0x1fa, 0x218, 0x238, 0x25a, 0x27e, 0x2a3},
    {0x166, 0x17b, 0x192, 0x1aa, 0x1c3, 0x1df, 0x1fb, 0x219, 0x239, 0x25b, 0x27f, 0x2a4},
    {0x167, 0x17c, 0x193, 0x1ab, 0x1c4, 0x1e0, 0x1fc, 0x21a, 0x23a, 0x25c, 0x280, 0x2a6},
    {0x168, 0x17d, 0x194, 0x1ac, 0x1c5, 0x1e0, 0x1fd, 0x21b, 0x23b, 0x25d, 0x281, 0x2a7},
    {0x168, 0x17d, 0x194, 0x1ad, 0x1c6, 0x1e1, 0x1fe, 0x21c, 0x23c, 0x25e, 0x282, 0x2a8},
    {0x169, 0x17e, 0x195, 0x1ad, 0x1c7, 0x1e2, 0x1ff, 0x21d, 0x23d, 0x260, 0x283, 0x2a9},
    {0x16a, 0x17f, 0x196, 0x1ae, 0x1c8, 0x1e3, 0x1ff, 0x21e, 0x23e, 0x261, 0x284, 0x2ab},
    {0x16a, 0x17f, 0x197, 0x1af, 0x1c8, 0x1e4, 0x200, 0x21f, 0x23f, 0x262, 0x286, 0x2ac}
};

uint16_t oplModel_apogeeFreq(double tone, uint32_t *mul_offset)
{
    uint_fast32_t noteI;
    int_fast32_t bendI = 0, outHz = 0, octave, scaleNote;
    double bendDec;

    *mul_offset = 0;

    noteI = (uint_fast32_t)(tone >= 12 ? tone - 12 : tone);
    bendDec = tone - (int)tone;

    if(bendDec > 0.5)
    {
        noteI += 1;
        bendDec -= 1.0;
    }

    bendI = (int_fast32_t)(bendDec * 32) + 32;

    noteI += (bendI / 32);
    noteI -= 1;

    scaleNote = noteI % 12;
    octave = noteI / 12;

    outHz = s_apogee_freq_table[bendI % 32][scaleNote];

    while(octave > 7)
    {
        ++(*mul_offset);
        --octave;
    }

    return outHz | (octave << 10);
}


/***************************************************************
 *                   Apogee volume formula                     *
 ***************************************************************/

void oplModel_apogeeOrigVolume(struct OPLVolume_t *v)
{
    uint_fast32_t volume;
    uint_fast32_t mod = v->tlMod, car = v->tlCar, tmpMod;

    volume = (v->chVol * v->chExpr * v->masterVolume / 16129);

    if(volume > 127)
        volume = 127;

    if(v->doCar)
    {
        car = 63 - car;
        car *= v->vel + 0x80;
        car = (volume * car) >> 15;
        car = car ^ 63;
        v->tlCar = car;
    }

    if(v->doMod)
    {
        tmpMod = v->tlCar;

        mod = 63 - mod;
        mod *= v->vel + 0x80;

        if(v->voiceMode > OPLVoice_MODE_2op_AM)
            tmpMod = mod; /* Fix the AM voices bug ащк 4-op voices */

        /*
            NOTE: Here is a bug of Apogee Sound System that makes modulator
            to not work properly on AM instruments. The fix of this bug, you
            need to replace the tlCar with tmMod in this formula.
            Don't do the bug on 4-op voices.
        */
        mod = (volume * tmpMod) >> 15;
        mod ^= 63;
        v->tlMod = mod;
    }
}

void oplModel_apogeeFixedVolume(struct OPLVolume_t *v)
{
    uint_fast32_t volume;
    uint_fast32_t mod = v->tlMod, car = v->tlCar, tmpMod;

    volume = (v->chVol * v->chExpr * v->masterVolume / 16129);

    if(volume > 127)
        volume = 127;

    if(v->doCar)
    {
        car = 63 - car;
        car *= v->vel + 0x80;
        car = (volume * car) >> 15;
        car = car ^ 63;
        v->tlCar = car;
    }

    if(v->doMod)
    {
        mod = 63 - mod;
        mod *= v->vel + 0x80;
        /* Fix the AM voices bug */
        tmpMod = mod;

        mod = (volume * tmpMod) >> 15;
        mod ^= 63;
        v->tlMod = mod;
    }
}
