// SPDX-License-Identifier: GPL-3.0-only

import 'dart:math';

import 'package:BendyStraw/app_state.dart';
import 'package:flutter/material.dart';
import 'package:sqlite3/sqlite3.dart' as sqlite3;

import 'bs_row.dart';
import 'db_manager.dart';
import 'enum.dart';
import 'table_header.dart';
import 'constants.dart';
import 'location.dart';
import 'screen.dart';
import 'table_model.dart';

class TableDataWidget extends StatefulWidget {
  final sqlite3.ResultSet resultSet;
  final Location location;
  final TableModel tableNotifier;

  const TableDataWidget({
    super.key,
    required this.resultSet,
    required this.location,
    required this.tableNotifier,
  });

  @override
  State<TableDataWidget> createState() => _TableDataWidgetState();
}

class _TableDataWidgetState extends State<TableDataWidget> {
  late Color rowColorSelected;
  late Color rowColorEven;
  late Color rowColorOdd;
  late Color columnHeaderNonSortableColor;
  late Color columnHeaderSortableColor;

  bool? triStateAllRowsAreSelected = false;

  void _selectAllOrNone(dynamic checkboxState) {
    if (triStateAllRowsAreSelected == true) {
      // Everything is currently selected --- deselect all
      widget.tableNotifier.clearRowSelection();
    } else {
      // Some or none are selected --- select all
      widget.tableNotifier.selectAllRows();
    }

    // Update AppState and tri-state
    _updateTriStateCheckbox();
  }

  /// Update which items are selected.
  void _updateTriStateCheckbox() {
    // Update tri-state checkbox
    if (widget.tableNotifier.selectedRows.length == widget.resultSet.length &&
        widget.resultSet.isNotEmpty) {
      triStateAllRowsAreSelected = true;
    } else if (widget.tableNotifier.selectedRows.isEmpty) {
      triStateAllRowsAreSelected = false;
    } else {
      triStateAllRowsAreSelected = null;
    }
  }

  /// When a row is dragged, [ReorderableListView] gives us data about the movement of the row via
  /// [oldIndex] and [newIndex], but doesn't actually reorder any widgets. Here we use the info we
  /// get to re-order the list of rows and update the real data in the database, which will trigger
  /// a rebuild via [ChangeNotifier].
  void _onReorder(int oldIndex, int newIndex) {
    List<int> backingList = List.generate(
      widget.resultSet.length,
      ((int rowIndex) =>
          widget.resultSet[rowIndex].columnAt(widget.tableNotifier.primaryKeyColumnIndex) as int),
    );
    AppState.debug('$oldIndex -> $newIndex');
    if (oldIndex < newIndex) {
      // removing the item at oldIndex will shorten the list by 1.
      newIndex -= 1;
    }
    final int element = backingList.removeAt(oldIndex);
    backingList.insert(newIndex, element);
    AppState.debug('backingList: $backingList');
    DbManager.reOrderLocalPlaylist(
      dbPath: widget.location.dbPath,
      playlistUid: widget.location.tableId,
      streamUidsInNewOrder: backingList,
    );
  }

  @override
  void didChangeDependencies() {
    super.didChangeDependencies();

    ColorScheme colorScheme = Theme.of(context).colorScheme;

    rowColorSelected = Color.alphaBlend(
      colorScheme.primary.withValues(alpha: 0.2),
      colorScheme.surface,
    );
    rowColorEven = Color.alphaBlend(
      colorScheme.surfaceContainerHighest.withValues(alpha: 0.5),
      colorScheme.surface,
    );
    rowColorOdd = colorScheme.surface;
    columnHeaderNonSortableColor =
        Theme.of(context).textTheme.bodySmall?.color ?? colorScheme.onSurface;
    columnHeaderSortableColor = colorScheme.primary;
  }

  @override
  Widget build(BuildContext context) {
    final bool primaryKeyColumnIsHidden = AppState.getPreference('primaryKeyColumnIsHidden');
    final bool isReorderable =
        AppState.getPreference('allowRowReordering') &&
        widget.location.tableType == TableType.localPlaylist;

    final Map<String, dynamic> cellLayout = Screen.isCompact(context)
        ? BS.tableCellLayoutCompact
        : BS.tableCellLayoutStandard;

    /// Extra padding for reorderable/draggable list is only needed on non-mobile platforms where
    /// the drag handles will be drawn.
    double rightPadding = (isReorderable && AppState.isDesktop)
        ? BS.paddingLarge + BS.paddingSmall
        : BS.paddingMedium;
    final innerHeight = min(widget.resultSet.length * BsRow.rowHeight, BS.tableWidgetHeight);

    _updateTriStateCheckbox();

    /// Outer wrapper
    return Column(
      children: [
        /// Column headers
        TableHeader(
          location: widget.location,
          sortableColor: columnHeaderSortableColor,
          nonSortableColor: columnHeaderNonSortableColor,
          cellLayout: cellLayout,
          triStateAllRowsAreSelected: triStateAllRowsAreSelected,
          primaryKeyColumnIndex: widget.tableNotifier.primaryKeyColumnIndex,
          primaryKeyColumnIsHidden: primaryKeyColumnIsHidden,
          rightPadding: rightPadding,
          selectAllOrNone: _selectAllOrNone,
        ),

        /// Main inner table
        Container(
          constraints: BoxConstraints(maxHeight: innerHeight),
          child: _viewBuilder(
            isReorderable: isReorderable,
            primaryKeyColumnIsHidden: primaryKeyColumnIsHidden,
            cellLayout: cellLayout,
            rightPadding: rightPadding,
          ),
        ),
      ],
    );
  }

  Widget _viewBuilder({
    required bool isReorderable,
    required bool primaryKeyColumnIsHidden,
    required Map<String, dynamic> cellLayout,
    required double rightPadding,
  }) {
    Widget itemBuilder(BuildContext context, int rowIndex) {
      int rowSelectionId = switch (widget.tableNotifier.tableType) {
        TableType.localPlaylist || TableType.search => rowIndex,
        TableType.remotePlaylists || TableType.subscriptions => widget.resultSet[rowIndex].columnAt(
          widget.tableNotifier.primaryKeyColumnIndex,
        ),
      };

      return BsRow(
        key: ValueKey(rowSelectionId),
        data: widget.resultSet[rowIndex],
        columnsInfoList: widget.location.tableType!.columns
            .where(
              (ColumnInfo columnInfo) =>
                  columnInfo.isVisible && !(primaryKeyColumnIsHidden && columnInfo.isPrimaryKey),
            )
            .toList(),
        defaultColor: rowIndex % 2 == 0 ? rowColorEven : rowColorOdd,
        selectedColor: rowColorSelected,
        cellLayout: cellLayout,
        isSelected: widget.tableNotifier.selectedRows.contains(rowSelectionId),
        onSelectChanged: (value) {
          widget.tableNotifier.setRowIsSelected(rowIndexOrUid: rowSelectionId, select: value!);
          _updateTriStateCheckbox();
        },
        rightPadding: rightPadding,
        dragHandle: isReorderable
            ? ReorderableDragStartListener(
                index: rowIndex,
                child: Icon(Icons.drag_handle, color: columnHeaderSortableColor),
              )
            : null,
      );
    }

    return isReorderable
        ? ReorderableListView.builder(
            itemBuilder: itemBuilder,
            itemCount: widget.resultSet.length,
            buildDefaultDragHandles: false,
            onReorder: _onReorder,
          )
        : ListView.builder(itemBuilder: itemBuilder, itemCount: widget.resultSet.length);
  }
}
