import 'package:sqlite3/sqlite3.dart';

// NewPipe's database schema has changed over time. Here we assign arbitrary versions numbers as part of the enum name. The presence or absence of specific columns is used to work out which version of the schema we are dealing with. For each schema definition, each of these important columns must ne in one of `presentColumns' or `absentColumns`.
// Many database transactions in the app will involve reading/writing between different versions/shemas. Knowing which versions we're dealing with helps us write the correct SQL.
enum SchemaVersion {
  newPipeV1(
    description: 'Legacy/initial NewPipe schema version',
    exampleDbFilePath:
        'test/test_data/extracted_zips/newPipeV1-NewPipeData-20220202_092932/newpipe.db',
    exampleZipFilePath: 'test/test_data/newPipeV1-NewPipeData-20220202_092932.zip',
    presentColumns: {
      'playlists': ['thumbnail_url'],
    },
    absentColumns: {
      'subscriptions': ['notification_mode'],
      'playlists': ['is_thumbnail_permanent', 'display_index', 'thumbnail_stream_id'],
      'remote_playlists': ['display_index'],
    },
  ),
  newPipeV2(
    description: 'NewPipe schema with subscription notification settings',
    exampleDbFilePath:
        'test/test_data/extracted_zips/newPipeV2-NewPipeData-20221222_231219/newpipe.db',
    exampleZipFilePath: 'test/test_data/newPipeV2-NewPipeData-20221222_231219.zip',
    presentColumns: {
      'subscriptions': ['notification_mode'],
      'playlists': ['thumbnail_url'],
    },
    absentColumns: {
      'playlists': ['is_thumbnail_permanent', 'display_index', 'thumbnail_stream_id'],
      'remote_playlists': ['display_index'],
    },
  ),
  newPipeV3(
    description:
        'NewPipe schema with subscription notification settings and dedicated playlist thumbnails',
    exampleDbFilePath:
        'test/test_data/extracted_zips/newPipeV3-NewPipeData-20230216_064224/newpipe.db',
    exampleZipFilePath: 'test/test_data/newPipeV3-NewPipeData-20230216_064224.zip',
    presentColumns: {
      'playlists': ['is_thumbnail_permanent', 'thumbnail_url'],
      'subscriptions': ['notification_mode'],
    },
    absentColumns: {
      'playlists': ['display_index', 'thumbnail_stream_id'],
      'remote_playlists': ['display_index'],
    },
  ),
  newPipeV4(
    description: 'NewPipe schema supporting dedicated playlist thumbnails',
    exampleDbFilePath: 'test/test_data/extracted_zips/newPipeV4-NewPipeData-Note3/newpipe.db',
    exampleZipFilePath: 'test/test_data/newPipeV4-NewPipeData-Note3.zip',
    presentColumns: {
      'subscriptions': ['notification_mode'],
      'playlists': ['is_thumbnail_permanent', 'thumbnail_stream_id'],
    },
    absentColumns: {
      'playlists': ['display_index', 'thumbnail_url'],
      'remote_playlists': ['display_index'],
    },
  ),
  newPipeV5(
    description: 'NewPipe schema with playlist ordering and dedicated thumbnails',
    exampleDbFilePath:
        'test/test_data/extracted_zips/newPipeV5-NewPipeData-20240831_064157/newpipe.db',
    exampleZipFilePath: 'test/test_data/newPipeV5-NewPipeData-20240831_064157.zip',
    presentColumns: {
      'subscriptions': ['notification_mode'],
      'playlists': ['is_thumbnail_permanent', 'display_index', 'thumbnail_stream_id'],
      'remote_playlists': ['display_index'],
    },
    absentColumns: {
      'playlists': ['thumbnail_url'],
    },
  ),
  pipePipeV1(
    description: 'PipePipe app schema',
    exampleDbFilePath:
        'test/test_data/extracted_zips/pipePipeV1-PipePipeData-20251023_062353/newpipe.db',
    exampleZipFilePath: 'test/test_data/pipePipeV1-PipePipeData-20251023_062353.zip',
    presentColumns: {
      'subscriptions': ['notification_mode'],
      'playlists': ['display_index', 'thumbnail_url'],
      'remote_playlists': ['display_index'],
    },
    absentColumns: {
      'playlists': ['is_thumbnail_permanent', 'thumbnail_stream_id'],
    },
  ),
  unknown(
    description: 'Unknown or unsupported database version',

    /// For use during testing, these example paths lead to an intentionally broken database
    /// (missing `streams` table).
    exampleDbFilePath: 'test/test_data/extracted_zips/incompatible-db/newpipe.db',
    exampleZipFilePath: 'test/test_data/incompatible-db.zip',
  );

  // --- Enum Implementation Details ---

  final String description;
  final String exampleDbFilePath;
  final String exampleZipFilePath;
  final Map<String, List<String>> presentColumns;
  final Map<String, List<String>> absentColumns;

  const SchemaVersion({
    required this.description,
    this.exampleDbFilePath = '',
    this.exampleZipFilePath = '',
    this.presentColumns = const {},
    this.absentColumns = const {},
  });

  // Method to check if a database's structure matches this schema's rules.
  bool _matches(Map<String, List<String>> allTableColumns) {
    // Check for presence of required columns
    for (final entry in presentColumns.entries) {
      final tableColumns = allTableColumns[entry.key] ?? [];
      if (!entry.value.every((col) => tableColumns.contains(col))) {
        return false;
      }
    }

    // Check for absence of disallowed columns
    for (final entry in absentColumns.entries) {
      final tableColumns = allTableColumns[entry.key] ?? [];
      if (entry.value.any((col) => tableColumns.contains(col))) {
        return false;
      }
    }

    return true;
  }

  /// Checks if a specific column is expected to be present in this schema version.
  bool hasPresentColumn(String tableName, String columnName) {
    return presentColumns[tableName]?.contains(columnName) ?? false;
  }

  /// Checks if a specific column is expected to be absent in this schema version.
  bool hasAbsentColumn(String tableName, String columnName) {
    return absentColumns[tableName]?.contains(columnName) ?? false;
  }
}

/// Detects the schema version by iterating through the enum values.
SchemaVersion detectSchemaVersion(Database db) {
  // Get all table and column info upfront to minimize queries.
  final allTableColumns = <String, List<String>>{};
  final tables = db.select("SELECT name FROM sqlite_master WHERE type='table'");
  for (final row in tables) {
    final tableName = row['name'] as String;
    try {
      final columns = db.select('PRAGMA table_info($tableName)');
      allTableColumns[tableName] = columns.map((c) => c['name'] as String).toList();
    } catch (e) {
      // In case of an error reading table info, treat it as an empty table.
      allTableColumns[tableName] = [];
    }
  }

  // Find the first enum case (other than Unknown) that matches.
  return SchemaVersion.values.firstWhere(
    (schema) => schema != SchemaVersion.unknown && schema._matches(allTableColumns),
    orElse: () => SchemaVersion.unknown,
  );
}
