// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'app_state.dart';
import 'constants.dart';
import 'screen.dart';

class SearchInfoButton extends StatefulWidget {
  const SearchInfoButton({
    super.key,
  });

  @override
  State<SearchInfoButton> createState() => _SearchInfoButtonState();
}

class _SearchInfoButtonState extends State<SearchInfoButton> {
  final ScrollController _scrollController = ScrollController();

  @override
  Widget build(BuildContext context) {
    ColorScheme colorScheme = Theme.of(context).colorScheme;
    return TapRegion(
      onTapInside: (PointerDownEvent event) {
        _showSearchInfoDialog(context);
      },
      child: Icon(Icons.info_outline,
          color: Color.alphaBlend(
              colorScheme.primary.withValues(alpha: BS.tappableMildOpacity), colorScheme.surface)),
    );
  }

  /// Start the copy/move process by displaying a dialog with a confirm button,
  /// which will copy or move the items when pressed.
  Future<void> _showSearchInfoDialog(BuildContext context) async {
    ColorScheme colorScheme = Theme.of(context).colorScheme;
    TextStyle textStyleNormal = TextStyle(
      color: colorScheme.onSurface,
      height: 1.2,
    );
    TextStyle textStyleHighLine = TextStyle(
      color: colorScheme.onSurface,
      height: 2,
    );
    TextStyle textStyleSubHeading = TextStyle(
      fontWeight: FontWeight.bold,
      height: 2.1,
      leadingDistribution: TextLeadingDistribution.proportional,
    );
    TextStyle textStyleExample = TextStyle(
      fontWeight: FontWeight.bold,
      backgroundColor: colorScheme.secondaryContainer,
      color: colorScheme.onSecondaryContainer,
      height: 1.2,
    );

    Map<String, List<TextSpan>> rows = {
      'Similar words match': [
        TextSpan(style: textStyleExample, text: ' restore '),
        TextSpan(text: '  matches `restores`, `restoration`\n'),
      ],
      'Boolean operators': [
        TextSpan(style: textStyleExample, text: ' cat AND dog '),
        TextSpan(text: ',  '),
        TextSpan(style: textStyleExample, text: ' cat OR dog '),
        TextSpan(text: ',  '),
        TextSpan(style: textStyleExample, text: ' cat NOT dog \n'),
      ],
      'Wildcards (*)': [
        TextSpan(style: textStyleExample, text: ' cat* '),
        TextSpan(text: '  matches `catch`\n'),
        TextSpan(style: textStyleExample, text: ' *lly '),
        TextSpan(text: '  matches `silly`\n'),
      ],
      'Exact phrase': [
        TextSpan(style: textStyleExample, text: ' "wrap it in quotes" '),
        TextSpan(text: ' \n'),
      ],
    };

    Table examplesTable = Table(
      defaultVerticalAlignment: TableCellVerticalAlignment.top,
      columnWidths: {
        0: FixedColumnWidth(
          Screen.isCompact(context)
              ? BS.infoTableLabelColumnWidthCompact
              : BS.infoTableLabelColumnWidthStandard,
        ),
      },
      children: [
        ...rows.entries.map(
          (entry) => TableRow(
            children: [
              TableCell(
                child: Padding(
                  padding: const EdgeInsets.only(
                    top: BS.paddingMedium,
                    right: BS.paddingSmall,
                    left: BS.paddingSmall,
                  ),
                  child: Text(
                    entry.key,
                    style: textStyleSubHeading,
                    textAlign: TextAlign.end,
                  ),
                ),
              ),
              TableCell(
                child: Padding(
                  padding: const EdgeInsets.only(
                    top: BS.paddingMedium,
                    left: BS.paddingMedium,
                  ),
                  child: RichText(
                    text: TextSpan(
                      style: textStyleHighLine,
                      children: <TextSpan>[...entry.value],
                    ),
                  ),
                ),
              ),
            ],
          ),
        ),
      ],
    );

    AppState.showAppDialog(
      title: 'Database Search',
      content: Scrollbar(
        thumbVisibility: true,
        controller: _scrollController,
        child: SingleChildScrollView(
          controller: _scrollController,
          padding: EdgeInsets.only(
            right: BS.paddingMedium,
            bottom: BS.paddingLarge,
          ),
          child: Column(
            mainAxisSize: MainAxisSize.min,
            crossAxisAlignment: CrossAxisAlignment.start,
            spacing: BS.paddingMedium,
            children: [
              RichText(
                text: TextSpan(
                  style: textStyleNormal,
                  children: <TextSpan>[
                    TextSpan(
                        style: textStyleSubHeading,
                        text: 'Search through the stream history in this database/zip.\n\n'),
                    TextSpan(
                        text:
                            'Results appear in the Search playlist. From there you can copy streams to other playlists.\n\n'),
                    TextSpan(
                        text:
                            'Search in the titles and/or channel names, use the checkboxes to choose.\n'),
                  ],
                ),
              ),
              Padding(
                padding: const EdgeInsets.only(
                  left: BS.paddingSmall,
                  right: BS.paddingSmall,
                ),
                child: examplesTable,
              ),
            ],
          ),
        ),
      ),
    );
  }
}
