// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';
import 'package:shared_preferences/shared_preferences.dart';

import 'app_state.dart';
import 'constants.dart';

/// Display and control a togglable [SharedPreferencesWithCache] item.
class PreferenceItemBoolean extends StatefulWidget {
  const PreferenceItemBoolean({
    super.key,
    required this.title,
    required this.id,
    required this.screenIsCompact,
    this.description = '',
    this.onChangeRebuild,
  });

  final bool screenIsCompact;
  final String title;
  final String id;
  final String description;
  final String? onChangeRebuild;

  @override
  State<PreferenceItemBoolean> createState() => _PreferenceItemBooleanState();
}

class _PreferenceItemBooleanState extends State<PreferenceItemBoolean> {
  @override
  Widget build(BuildContext context) {
    var layout =
        widget.screenIsCompact ? BS.preferenceItemLayoutCompact : BS.preferenceItemLayoutStandard;
    var defaultValue = BS.defaultPreferences[widget.id];
    var currentValue = AppState.getPreference(widget.id) ?? defaultValue;

    return InkWell(
      onTap: () {
        _setValue(!currentValue);
      },
      child: Padding(
        padding: layout['outerPadding'],
        child: Column(
          children: [
            Row(
              mainAxisSize: MainAxisSize.max,
              mainAxisAlignment: MainAxisAlignment.start,
              children: [
                Expanded(
                  child: Padding(
                    padding: layout['titlePadding'],
                    child: Text(
                      widget.title,
                      style: Theme.of(context).textTheme.titleLarge!,
                    ),
                  ),
                ),
              ],
            ),
            Row(
              mainAxisSize: MainAxisSize.max,
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Checkbox(
                  value: currentValue,
                  onChanged: (value) {
                    _setValue(value ?? defaultValue);
                  },
                ),
                if (widget.description.isNotEmpty)
                  Expanded(
                    child: Row(
                      children: [
                        Expanded(
                          child: Padding(
                            padding: layout['descriptionPadding'],
                            child: Text(
                              widget.description,
                              style: Theme.of(context).textTheme.bodySmall!,
                            ),
                          ),
                        ),
                      ],
                    ),
                  ),
              ],
            ),
          ],
        ),
      ),
    );
  }

  /// Update the state of this preference in the app.
  void _setValue(value) async {
    await AppState.setPreference(widget.id, value, forceRebuild: true);
    if (widget.onChangeRebuild != null) {
      AppState.forceRebuildOn(widget.onChangeRebuild!);
    } else {
      setState(() {});
    }
  }
}
