// SPDX-License-Identifier: GPL-3.0-only

import 'dart:io';

import 'package:flex_color_scheme/flex_color_scheme.dart';
import 'package:flutter/material.dart';

import 'enum.dart';

/// App constants (`BS` = `BendyStraw`).
///
/// Lots of constants related to the app:
/// - UI (eg colours, sizes)
/// - Defaults for user preferences
/// - Important file paths
/// - Some NewPipe-specific info relating to the database, such as table names
class BS {
  static const String appName = 'BendyStraw';

  /// This directory will be created inside the OS temporary directory and used
  /// for extracting databases and working on them
  static const String baseWorkingDirectory = 'bendy-straw-unzipped';

  /// This gets appended to the original filename (before the file extension) on export.
  static const String savedFileSuffix = '-bendy-straw';

  /// This gets appended to the dbPath to name its temp search playlist.
  static const String searchPlaylistSuffix = '-search';
  static const String playlistSelectionSuffix = '-sel';

  // Route names for nested navigation, with 'base_info' and 'tabs' nested inside 'inner'.
  static const routeHome = '/';
  static const routePrefixInner = '/inner/';
  static const routeInnerStart = '/inner/$routeInnerStartPage';
  static const routeInnerStartPage = 'base_info';
  static const routeInnerTabsPage = 'tabs';

  /// Arbitrary choice
  static const playlistNameMaxLength = 255;

  static const hashIdLength = 10;

  // [Expansible] doesn't come with a kConstant default, so add it here (as found in widget definition)
  static const expansibleDefaultDurationMs = 200;

  static const Map<String, dynamic> defaultPreferences = {
    'brightnessOverride': [String, 'system'],
    'darkIsTrueBlack': [bool, false],
    'showInstantYoutubePlaylistButton': [bool, true],
    'showJsonImportButton': [bool, false],
    'showSmallIconButtonLabels': [bool, true],
    'showFloatingIconButtonLabels': [bool, true],
    'showPlaylistShuffleButton': [bool, false],
    'showPlaylistDeduplicateButton': [bool, true],
    'showPlaylistReverseButton': [bool, true],
    'shortenUrlsForDisplay': [bool, true],
    'saveSearchTableToZip': [bool, false],
    'rememberLastOpenedDirectory': [bool, true],
    'primaryKeyColumnIsHidden': [bool, true],
    'allowRowReordering': [bool, true],
    'tableUidIsHidden': [bool, true],
    'disableUiAnimations': [bool, false],
    'lastOpenedDirectory': [String, null],

    /// The values here are from enums, we use their name as our [String]
    /// ([SharedPreferencesWithCache] only allows a few data types).
    'dbInfoToShow': [String, 'filepathOnly'],
  };

  ///
  ///
  ///////////////////////
  ///
  /// UI / Layout
  /// Responsive
  ///
  ///////////////////////

  /*
    From:
    https://developer.android.com/guide/topics/large-screens/support-different-screen-sizes

    Compact width 	width < 600dp 	99.96% of phones in portrait
    Medium width 	600dp ≤ width < 840dp 	93.73% of tablets in portrait,

    Large unfolded inner displays in portrait
    Expanded width 	width ≥ 840dp 	97.22% of tablets in landscape,
  */
  static const double breakpointCompact = 600;

  static const Map dialogLayoutCompact = {
    'screenWidthPercent': .9,
    'screenHeightPercent': .9,
    'topPad': BS.paddingSmall,
    'rightPad': BS.paddingSmall,
    'bottomPad': BS.paddingLarge,
    'leftPad': BS.paddingMedium,
    'innerTopPad': BS.paddingLarge,
    'innerRightPad': BS.paddingMediumLarge,
    'innerBottomPad': BS.paddingLarge,
    'innerLeftPad': BS.paddingLarge,
  };
  static const Map dialogLayoutStandard = {
    'screenWidthPercent': .66,
    'screenHeightPercent': .66,
    'topPad': BS.paddingMedium,
    'rightPad': BS.paddingMedium,
    'bottomPad': BS.paddingLarge,
    'leftPad': BS.paddingLarge,
    'innerTopPad': BS.paddingLarge,
    'innerRightPad': BS.paddingMediumLarge,
    'innerBottomPad': BS.paddingLarge,
    'innerLeftPad': BS.paddingLarge,
  };

  static const Map preferenceItemLayoutCompact = {
    'between': BS.paddingSmall,
    'outerPadding': EdgeInsets.all(BS.paddingSmall),
    'titlePadding': EdgeInsets.only(left: BS.paddingExtraSmall),
    'descriptionPadding': EdgeInsets.only(top: BS.paddingExtraSmall, left: BS.paddingSmall),
  };
  static const Map preferenceItemLayoutStandard = {
    'between': BS.paddingMedium,
    'outerPadding': EdgeInsets.all(BS.paddingMedium),
    'titlePadding': EdgeInsets.only(left: BS.paddingSmall),
    'descriptionPadding': EdgeInsets.only(top: BS.paddingSmall, left: BS.paddingSmall),
  };

  static const Map tableLayoutCompact = {
    'checkboxHorizontalMargin': 0.0,
    'horizontalMargin': BS.paddingSmall,
    'columnSpacing': BS.paddingExtraSmall,
    'headingRowHeight': 30.0,
    'headingTextStyle': TextStyle(height: 1, fontSize: fontSizeExtraSmall),
  };
  static const Map tableLayoutStandard = {
    'checkboxHorizontalMargin': BS.paddingMedium,
    'horizontalMargin': BS.paddingMedium,
    'columnSpacing': BS.paddingSmall,
    'headingRowHeight': 36.0,
    'headingTextStyle': TextStyle(height: 1, fontSize: fontSizeSmall),
  };

  static const Map<String, dynamic> tableCellLayoutCompact = {
    'padding': EdgeInsets.only(
      left: BS.paddingSmall,
      right: BS.paddingSmall,
      //right: BS.paddingSmall,
      top: BS.paddingSuperSmall,
      bottom: BS.paddingSuperSmall,
    ),
    'textStyle': TextStyle(height: 1.2, fontSize: fontSizeExtraSmall, fontWeight: FontWeight.w600),
    'urlButtonPadding': EdgeInsets.symmetric(
      horizontal: BS.paddingSuperSmall,
      vertical: BS.paddingSuperSmall,
    ),
  };
  static const Map<String, dynamic> tableCellLayoutStandard = {
    'padding': EdgeInsets.only(
      left: BS.paddingMedium,
      right: BS.paddingSmall,
      top: BS.paddingExtraSmall,
      bottom: BS.paddingExtraSmall,
    ),
    'textStyle': TextStyle(height: 1.1, fontSize: fontSizeSmall, fontWeight: FontWeight.w400),
    'urlButtonPadding': EdgeInsets.symmetric(
      horizontal: BS.paddingExtraSmall,
      vertical: BS.paddingExtraSmall,
    ),
  };

  static ButtonStyle buttonStyleCompact = ElevatedButton.styleFrom(
    minimumSize: Size.zero,
    visualDensity: VisualDensity.compact,
    padding: const EdgeInsets.all(BS.paddingIconCompact),
    tapTargetSize: MaterialTapTargetSize.shrinkWrap,
  );
  static ButtonStyle buttonStyleStandard = ElevatedButton.styleFrom(
    visualDensity: VisualDensity.compact,
    padding: const EdgeInsets.all(BS.paddingIconStandard),
  );

  static const double scrollBottomBufferStandard = 150;
  static const double scrollBottomBufferCompact = 75;

  static const double paddingIconStandard = 1;
  static const double paddingIconCompact = 14;

  static const double smallIconButtonWidthStandard = 58;
  static const double smallIconButtonWidthCompact = 46;
  static const double smallIconButtonWidthTextOverflow = 100;

  static const double buttonIconSizeCompact = 20;
  static const double buttonIconSizeStandard = 26;

  static const Duration defaultDebounceDuration = Duration(milliseconds: 200);
  static const Duration openYouTubeDebounceDuration = Duration(milliseconds: 700);

  ///
  ///
  ///////////////////////
  ///
  /// UI / Layout
  ///
  ///////////////////////

  static const double paddingSuperSmall = 3;
  static const double paddingExtraSmall = 4;
  static const double paddingSmall = 6;
  static const double paddingMedium = 16;
  static const double paddingMediumLarge = 24;
  static const double paddingLarge = 32;
  static const double paddingExtraLarge = 64;

  /// Used to prevent tabs (especially the [X] button) disappearing underneath
  /// the settings button.
  static const double tabBarRightPadding = 56;

  static const double cornerRadius = 6;
  static const double cornerButtonRadius = 24;

  static const double renamableTextFieldWidth = 300;

  static const double checkboxBorderWidth = 2;
  static const double tableWidgetHeight = 500;
  static const double tabHeight = 22;
  static const double playlistItemCountWidth = 36;
  static const double playlistHeaderTopRowHeight = 48;
  static const double buttonMaxWidth = 180;
  static const double moveModeTextMinWidth = 50;
  static const double moveModeTextMaxHeight = 36;

  /// Used eg for add/export buttons which hover above the rest of the UI
  static const double highButtonElevation = 13;

  static const double disableControlOpacity = 0.3;
  static const double tappableMildOpacity = 0.75;

  /// List of destinations eg playlists that can be copied to.
  static const double destinationSingleItemHeight = 108;
  static const double destinationListHeight = 250;
  static const double destinationListWidth = 200;

  static const double fontSizePageHeading = 28;
  static const double fontSizeSectionHeading = 24;
  static const double fontSizeNumberIcon = 24;
  static const double fontSizeBaseInfo = 18;
  static const double fontSizeHeading = 20;
  static const double fontSizeSmall = 12.5;
  static const double fontSizeExtraSmall = 10.5;
  static const double fontSizePlaylistItemCount = 15;
  static const double fontSizePlaylistName = 15;

  static const double numberIconSize = 48;
  static const double infoRowLineHeight = 1.3;
  static const double infoRowGapLineHeight = 0.6;

  static const double scrollbarThicknessThin = 2;

  static const double accentBorderThickness = 4;

  static const double infoTableLabelColumnWidthCompact = 100;
  static const double infoTableLabelColumnWidthStandard = 100;

  /// Database tabs are coloured according to this list, in sequence, looping
  /// back to the start if more than [flexSchemeColors.length] tabs are opened.
  static const List<FlexScheme> flexSchemeColors = [
    FlexScheme.redM3,
    FlexScheme.greenM3,
    FlexScheme.blueM3,
    FlexScheme.yellowM3,
    FlexScheme.purpleM3,
    FlexScheme.tealM3,
  ];

  ///
  ///
  ///////////////////////
  ///
  /// Paths to show in the file picker
  ///
  ///////////////////////

  static final List<String> filepathsLinux = [Platform.environment['HOME'] ?? '/'];

  static final List<String> filepathsWindows = [Platform.environment['USERPROFILE'] ?? '/'];

  static const List<String> filepathsAndroid = [
    '/sdcard/',
    '/sdcard/Download',
    '/storage/emulated/0/',
    '/storage/emulated/0/Download',
  ];

  ///
  ///
  ///////////////////////
  ///
  /// SQL/database related data below
  ///
  ///////////////////////

  /// Table names as used in NewPipe databases
  static const String streamsTableId = 'streams';

  static const String localPlaylistsTableId = 'playlists';
  static const String remotePlaylistsTableId = 'remote_playlists';
  static const String subscriptionsTableId = 'subscriptions';
  static const String joinTableId = 'playlist_stream_join';

  /// Internal name for the special temporary table used to join streams with
  /// their playlists
  static const String tempPlaylistStreamJoinedTableId = 'tempPlaylistStreamJoinedTable';

  /// Used to check if the database is a compatible version (it must contain
  /// all of these tables).
  static const List<String> requiredTables = [
    'streams',
    'playlists',
    //'nonexistent_table_for_testing',
    'playlist_stream_join',
    'remote_playlists',
    'subscriptions',
  ];

  static const Map<String, Map<String, dynamic>> knownTables = {
    BS.remotePlaylistsTableId: {
      'displayName': 'Bookmarked Playlists',
      'tableType': TableType.remotePlaylists,
      'columnList': RemotePlaylistsColumns.values,
    },
    BS.subscriptionsTableId: {
      'displayName': 'Channels',
      'tableType': TableType.subscriptions,
      'columnList': SubscriptionsColumns.values,
    },
  };

  ///
  ///
  ///////////////////////
  ///
  /// JSON playlist imports
  ///
  ///////////////////////

  static const Map<String, Map> jsonImportColumnSpec = {
    'service_id': {'type': int, 'required': false, 'default': 0},
    'url': {
      'type': String,
      'required': true,
      'error':
          '`url` (eg `https://www.youtube.com/watch?v=3RJ5BfmYT2c`) is required for every stream',
    },
    'title': {
      'type': String,
      'required': true,
      'error': '`title` (eg `Rabbits eating clover`) is required for every stream',
    },
    'stream_type': {
      'type': String,
      'required': true,
      'error': '`stream_type` (`VIDEO_STREAM` or `AUDIO_STREAM`) is required for every stream',
    },
    'duration': {'type': int, 'required': false, 'default': -1},
    'uploader': {
      'type': String,
      'required': true,
      'error': '`uploader` (eg `The Cooking Channel`) is required for every stream',
    },
    'thumbnail_url': {'type': String, 'required': false},
    'view_count': {'type': int, 'required': false},
    'textual_upload_date': {'type': String, 'required': false},
    'upload_date': {'type': int, 'required': false},
    'uploader_url': {'type': String, 'required': false},
  };
}
