// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'constants.dart';
import 'screen.dart';

class CheckboxLabelled extends StatelessWidget {
  const CheckboxLabelled({
    super.key,
    this.labelText = '',
    this.onChanged,
    this.value = false,
  });

  final String labelText;
  final ValueChanged<bool?>? onChanged;
  final bool value;

  @override
  Widget build(BuildContext context) {
    final ColorScheme colorScheme = Theme.of(context).colorScheme;
    //final Color foregroundColor = colorScheme.primary;
    //final Color foregroundColorDisabled = Color.alphaBlend(
    //    colorScheme.primary.withValues(alpha: BS.disableControlOpacity), colorScheme.surface);
    final Color labelColor = Color.alphaBlend(
        colorScheme.primary.withValues(alpha: BS.tappableMildOpacity), colorScheme.surface);
    final Color disabledColor = Color.alphaBlend(
        colorScheme.primary.withValues(alpha: BS.disableControlOpacity), colorScheme.surface);
    final Color mildColor = Color.alphaBlend(
        colorScheme.primary.withValues(alpha: BS.tappableMildOpacity), colorScheme.surface);

    return SizedBox(
      height: BS.playlistHeaderTopRowHeight,

      /// Visually match the [SmallIconButton] width
      width: Screen.isCompact(context)
          ? BS.smallIconButtonWidthCompact
          : BS.smallIconButtonWidthStandard,

      /// Provide larger hit area by wrapping entire widget with [InkWell]. Call the same
      /// [onChanged] function that we use for the [Checkbox], flipping the [value].
      child: OverflowBox(
        maxWidth: BS.smallIconButtonWidthTextOverflow,
        child: InkWell(
          onTap: onChanged == null
              ? null
              : () {
                  onChanged!(!value);
                },
          child: Stack(
            children: [
              Positioned.directional(
                textDirection: TextDirection.ltr,
                top: 0,
                start: 0,
                end: 0,
                child: Padding(
                  padding: const EdgeInsets.only(
                    top: BS.paddingExtraSmall,
                    left: BS.paddingMedium,
                    right: BS.paddingMedium,
                  ),
                  child: CheckboxTheme(
                    data: CheckboxThemeData(
                      fillColor: WidgetStateProperty.resolveWith((states) {
                        return states.contains(WidgetState.selected) ? colorScheme.primary : null;
                      }),
                      checkColor: WidgetStateProperty.resolveWith((states) {
                        return states.contains(WidgetState.selected) ? colorScheme.onPrimary : null;
                      }),
                      side: WidgetStateBorderSide.resolveWith((states) {
                        return states.contains(WidgetState.disabled)
                            ? BorderSide(color: disabledColor, width: BS.checkboxBorderWidth)
                            : BorderSide(color: mildColor, width: BS.checkboxBorderWidth);
                      }),
                    ),
                    child: Padding(
                      padding: const EdgeInsets.only(
                        top: BS.paddingSuperSmall,
                      ),
                      child: Checkbox(
                        visualDensity: const VisualDensity(
                            horizontal: VisualDensity.minimumDensity,
                            vertical: VisualDensity.minimumDensity),
                        value: value,
                        onChanged: onChanged,
                      ),
                    ),
                  ),
                ),
              ),
              Positioned.directional(
                textDirection: TextDirection.ltr,
                bottom: 0,
                start: 0,
                end: 0,
                child: IgnorePointer(
                  child: Text(
                    labelText,
                    textAlign: TextAlign.center,
                    style: TextStyle(
                      fontSize: BS.fontSizeExtraSmall,
                      fontWeight: FontWeight.bold,
                      color: onChanged == null ? disabledColor : labelColor,
                    ),
                  ),
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }
}
