// SPDX-License-Identifier: GPL-3.0-only

import 'package:flex_color_scheme/flex_color_scheme.dart';
import 'package:flutter/material.dart';
import 'package:sqlite3/sqlite3.dart';

import 'app_state.dart';
import 'constants.dart';
import 'db_manager.dart';
import 'enum.dart';
import 'table_model.dart';
import 'sql_helpers.dart';
import 'sql_local_playlists.dart';
import 'sql_schema_logic.dart';
import 'tables_model.dart';
import 'utils.dart';

class BsDatabase with ChangeNotifier {
  late final ThemeData themeData;
  late final ThemeData themeDataDark;
  final String prettyName;
  final String originalDirectoryPath;
  final SchemaVersion schemaVersion;
  final String pathHash;
  final String searchPlaylistName;
  int? searchPlaylistUid;

  /// Path to extracted *.db file
  final String path;
  // Path to zip the *.db was contained within
  final String originalZipPath;

  bool hasUnsavedChanges = false;

  final TablesModel tablesNotifier;

  BsDatabase({required this.path, required this.originalZipPath})
    : prettyName = originalZipPath.split('/').last,
      pathHash = getMd5Hash(path),
      searchPlaylistName = getMd5Hash(path) + BS.searchPlaylistSuffix,
      schemaVersion = SqlHelpers.deduceSchemaVersion(path),
      tablesNotifier = TablesModel(dbPathHash: getMd5Hash(path)),
      originalDirectoryPath = originalZipPath.replaceFirst(originalZipPath.split('/').last, '') {
    ///
    /// Count to ensure consecutive tabs never end up with the same colour
    int totalDbsOpenedSinceSessionStart = AppState.get('totalDbsOpenedSinceSessionStart') ?? 0;

    /// Set theme to be used for the tab which shows this database
    FlexScheme flexScheme =
        BS.flexSchemeColors[totalDbsOpenedSinceSessionStart % BS.flexSchemeColors.length];
    themeData = FlexThemeData.light(scheme: flexScheme, visualDensity: VisualDensity.standard);
    themeDataDark = FlexThemeData.dark(
      scheme: flexScheme,
      darkIsTrueBlack: AppState.getPreference('darkIsTrueBlack'),
      visualDensity: VisualDensity.standard,
    );

    /// Increment count since session start
    AppState.update('totalDbsOpenedSinceSessionStart', totalDbsOpenedSinceSessionStart + 1);

    AppState.debug(
      'schema version of db in zip: ${schemaVersion.toString()} --- ${schemaVersion.description}',
    );
  }

  void initSearchPlaylist() {
    createSearchPlaylist();

    /// Initialise search-related options
    AppState.update('${searchPlaylistName}-terms', '');
    AppState.update('${searchPlaylistName}-includeChannels', true);
    AppState.update('${searchPlaylistName}-includeTitles', true);
  }

  /// Create special local playlist for holding search results
  void createSearchPlaylist() {
    searchPlaylistUid ??= SqlLocalPlaylists.createNewEmptyLocalPlaylist(
      path,
      name: searchPlaylistName,
      skipIfNameExists: true,
    );
  }

  void rebuildRemotePlaylists({bool setDbDirty = true}) {
    AppState.debug('rebuildRemotePlaylists()');

    /// Update the `db` object for use by the rest of the app
    setRemotePlaylists(
      DbManager.getFilteredColumns(
        path,
        BS.remotePlaylistsTableId,
        RemotePlaylistsColumns.values
            .where((e) => e.isVisible)
            .map((e) => '${e.name} AS ${e.displayName}')
            .toList()
            .join(', '),
      ),
    );

    /// We usually want to do this as it announces to any interested widgets that their db has
    /// been written to/changed.
    if (setDbDirty) {
      DbManager.setDbDirtyState(path, true);
    }
  }

  void rebuildChannelSubscriptions({bool setDbDirty = true}) {
    AppState.debug('rebuildChannelSubscriptions()');

    /// Update the `db` object for use by the rest of the app
    setChannelSubscriptions(
      DbManager.getFilteredColumns(
        path,
        BS.subscriptionsTableId,
        SubscriptionsColumns.values
            .where((e) => e.isVisible)
            .map((e) => '${e.name} AS ${e.displayName}')
            .toList()
            .join(', '),
      ),
    );

    /// We usually want to do this as it announces to any interested widgets that their db has
    /// been written to/changed.
    if (setDbDirty) {
      DbManager.setDbDirtyState(path, true);
    }
  }

  void rebuildLocalAndSearchPlaylistMaps({bool setDbDirty = true}) {
    AppState.debug('rebuildLocalAndSearchPlaylistMaps()');
    if (searchPlaylistUid != null) {
      Map<int, TableModel> playlists = SqlLocalPlaylists.getLocalPlaylistsMap(path);

      /// Create a new map containing only the search playlist
      setSearchPlaylists({searchPlaylistUid!: playlists[searchPlaylistUid]!});

      /// Remove the search playlist from the original map, leaving us with only the standard
      /// local playlists.
      playlists.removeWhere((uid, playlist) => uid == searchPlaylistUid);
      setLocalPlaylists(playlists);

      /// We usually want to do this as it announces to any interested widgets that their db has
      /// been written to/changed.
      if (setDbDirty) {
        DbManager.setDbDirtyState(path, true);
      }
    }
  }

  void setRemotePlaylists(ResultSet resultSet) {
    AppState.debug('BsDatabase::setRemotePlaylists');
    //AppState.debug('\tresultSet: ${resultSet.toString()}');
    tablesNotifier.updateRemotePlaylistsResultSet(resultSet);
  }

  void setChannelSubscriptions(ResultSet resultSet) {
    AppState.debug('BsDatabase::setChannelSubscriptions');
    tablesNotifier.updateChannelSubscriptionsResultSet(resultSet);
  }

  TableModel get bookmarkedPlaylistsNotifier {
    return tablesNotifier.bookmarkedPlaylistsNotifier;
  }

  TableModel get channelSubscriptionsNotifier {
    return tablesNotifier.channelSubscriptionsNotifier;
  }

  void setLocalPlaylists(Map<int, TableModel> playlists) {
    tablesNotifier.setLocalPlaylists(playlists);
    //tablesNotifier.setLocalPlaylists(Map.from(playlists));
  }

  void setSearchPlaylists(Map<int, TableModel> playlists) {
    tablesNotifier.setSearchPlaylists(playlists);
    //tablesNotifier.setSearchPlaylists(Map.from(playlists));
  }

  Map<int, TableModel> get localPlaylists {
    return tablesNotifier.localPlaylists;
  }

  Map<int, TableModel> get searchPlaylists {
    return tablesNotifier.searchPlaylists;
  }
}
