// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'constants.dart';
import 'screen.dart';

/// Tab for the TabBar, will be wrapped in a [Theme] by its parent widget, hence the [Builder].
class TabWidget extends StatelessWidget {
  const TabWidget({
    super.key,
    this.isSelected = false,
    this.text = '',
    this.onPressed,
  });

  final bool isSelected;
  final String text;
  final VoidCallback? onPressed;

  @override
  Widget build(BuildContext context) {
    return Builder(builder: (context) {
      ColorScheme colorScheme = Theme.of(context).colorScheme;

      final Color tabFillColor = isSelected
          ? colorScheme.secondaryContainer
          : Color.alphaBlend(
              colorScheme.secondaryContainer.withValues(alpha: 0.5), colorScheme.surface);
      final Color tabTextColor = isSelected
          ? colorScheme.primary
          : Color.alphaBlend(
              colorScheme.primary.withValues(alpha: 0.5), colorScheme.secondaryContainer);

      return Container(
          decoration: BoxDecoration(
            color: tabFillColor,
            borderRadius: const BorderRadius.only(
              topRight: Radius.circular(BS.cornerRadius),
              topLeft: Radius.circular(BS.cornerRadius),
            ),
          ),
          padding: const EdgeInsets.only(
            top: BS.paddingExtraSmall,
            bottom: BS.paddingSmall,
            left: BS.paddingMedium,
            right: BS.paddingSmall,
          ),
          child: Tab(
            height: BS.tabHeight,
            child: Row(
              spacing: Screen.isCompact(context) ? BS.paddingExtraSmall : BS.paddingSmall,
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              crossAxisAlignment: CrossAxisAlignment.end,
              children: [
                Flexible(
                  flex: 1,
                  child: Text(
                    text,
                    overflow: TextOverflow.ellipsis,
                    style: TextStyle(
                      color: tabTextColor,
                    ),
                  ),
                ),
                Flexible(
                  flex: 0,
                  child: IconButton(
                      alignment: Alignment.center,
                      visualDensity: VisualDensity.compact,
                      padding: EdgeInsets.zero,
                      onPressed: onPressed,
                      icon: Icon(
                        Icons.close_rounded,
                        color: colorScheme.primary,
                      )),
                )
              ],
            ),
          ));
    });
  }
}
