import 'package:sqlite3/sqlite3.dart';

import 'app_state.dart';
import 'constants.dart';
import 'db_manager.dart';
import 'enum.dart';
import 'sql_helpers.dart';
import 'sql_schema_logic.dart';
import 'sql_string_maker.dart';
import 'table_model.dart';
import 'utils.dart';

/// Low level SQL functions to operate on local playlists.
///
/// 'Search Playlist' is a special kind of local playlist but in most ways operates identically.
///
/// Only [DbManager] should call functions directly from this class. Other
/// classes should call  wrapper functions in [DbManager].

/// IMPORTANT
/// Local Playlists identify rows using their row index ([rowIndices]). This is because playlists
/// (and subsequently their tables) can be sorted/re-ordered, and can contain duplicates (a playlist
/// can have more than one occurance of a track (stream)).
class SqlLocalPlaylists {
  static void renameLocalPlaylist(String dbPath, int playlistUid, String newName) {
    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);
      db.execute('''
        UPDATE ${BS.localPlaylistsTableId}
          SET name = '${SqlHelpers.escapeQuotes(newName)}'
          WHERE uid = '${playlistUid}';
      ''');
      db.dispose();
    });
  }

  /// Create a new local playlist. Returns the uid of the new playlist. If [skipIfNameExists] is
  /// true and a playlist with [name] already exists, will return -1 to indicate that no new
  /// playlist was created.
  static int createNewEmptyLocalPlaylist(
    String dbPath, {
    String name = 'new playlist',
    bool skipIfNameExists = false,
  }) {
    int newPlaylistUid = -1;

    SchemaVersion schemaVersion = DbManager.getSchemaVersion(dbPath);

    /// Build a [Map] of columns depending on schema, and fill them with default values.
    Map<String, dynamic> insertMap = {'name': name};
    if (schemaVersion.presentColumns['playlists'] != null) {
      for (String columnName in schemaVersion.presentColumns['playlists']!) {
        insertMap[columnName] = switch (columnName) {
          'thumbnail_stream_id' => -1,
          'is_thumbnail_permanent' => 0,
          'display_index' => -1,
          'thumbnail_url' => '',
          _ => null,
        };
      }
    }

    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);

      bool skipInsert = false;
      if (skipIfNameExists) {
        if (getPlaylistUidFromName(db, name) != null) {
          skipInsert == true;
        }
        AppState.debug('skipInsert: $skipInsert');
      }

      if (skipInsert) {
        AppState.debug(
          "Playlist '$name' exists, and 'skipIfNameExists' is true --- SKIPPING INSERT",
        );
      } else {
        /// Do the insert
        String sql = SqlHelpers.getInsertStatement(
          tableName: BS.localPlaylistsTableId,
          map: insertMap,
        );
        AppState.debug(sql);
        db.execute(sql);

        /// Get the uid of the new playlist
        sql =
            '''
          SELECT * FROM ${BS.localPlaylistsTableId}
          WHERE name = '${name}';
        ''';
        //SqlHelpers.logSelect(db, sql);

        newPlaylistUid = db.select(sql)[0]['uid'];
      }

      db.dispose();
    }, SqlHelpers.getTransactionSchemaDetails(toSchemaVersion: DbManager.getSchemaVersion(dbPath)));

    return newPlaylistUid;
  }

  /// Delete a local playlist.
  static void deleteLocalPlaylist(
    String dbPath,
    int playlistUid, {
    bool deleteEntriesOnly = false,
  }) {
    AppState.debug(
      'SqlLocalPlaylists::deleteLocalPlaylist(): \ndbPath: $dbPath\nplaylistUid: $playlistUid\ndeleteEntriesOnly: $deleteEntriesOnly\n',
    );
    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);
      if (!deleteEntriesOnly) {
        // Delete playlist from playlists table
        db.execute('''
          DELETE FROM ${BS.localPlaylistsTableId}
          WHERE uid = ${playlistUid};
        ''');
      }
      // Delete playlist entries from playlists stream join table
      db.execute('''
        DELETE FROM ${BS.joinTableId}
        WHERE playlist_id = ${playlistUid};
      ''');
      db.dispose();
    });
  }

  /// Delete streams from a local playlist.
  /// Identified by their row index or 'track number' (conceptually) in the playlist/UI, which
  /// correlates with their `join_index` in [BS.joinTableId].
  static void deleteItemsFromLocalPlaylist(String dbPath, int playlistUid, List<int> rowIndices) {
    AppState.debug(
      'SqlLocalPlaylists::deleteItemsFromLocalPlaylist(): \ndbPath: $dbPath\nplaylistUid: $playlistUid\nrowIndices: $rowIndices\n',
    );
    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);

      /// Delete all the items from [BS.joinTableId].
      db.execute('''
        DELETE FROM ${BS.joinTableId}
        WHERE playlist_id = ${playlistUid}
          AND join_index IN (${rowIndices.join(", ")})
      ''');

      /// Get lowest join_index of deleted items
      /// For each join_index equal or greater than min, decrement by numOfRowsDeleted
      for (final rowIndex in rowIndices) {
        /// Decrement any higher [join_index] of other streams in this playlist to fill in the gap
        /// it left.
        db.execute('''
          UPDATE ${BS.joinTableId}
            SET join_index = join_index - 1
            WHERE playlist_id = ${playlistUid}
              AND join_index > ${rowIndex}
              AND join_index > 0;
        ''');
      }
      db.dispose();
    });
  }

  /// Reverse the order of streams in a local playlist.
  static void reverseLocalPlaylistOrder(String dbPath, int playlistUid) {
    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);

      ResultSet playlistStreamJoinResultSet = db.select('''
        SELECT stream_id FROM ${BS.joinTableId}
          WHERE playlist_id = ${playlistUid}
          ORDER BY join_index
      ''');

      List<int> streamUidsReversed = playlistStreamJoinResultSet
          .map((row) => row['stream_id'])
          .toList()
          .reversed
          .toList()
          .cast<int>();

      SqlLocalPlaylists.reOrderLocalPlaylist(dbPath, playlistUid, streamUidsReversed);

      db.dispose();
    });
  }

  /// Shuffle the order of streams in a local playlist.
  static void shuffleLocalPlaylistOrder(String dbPath, int playlistUid) {
    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);

      ResultSet playlistStreamJoinResultSet = db.select('''
        SELECT stream_id FROM ${BS.joinTableId}
          WHERE playlist_id = ${playlistUid}
          ORDER BY join_index
      ''');

      List<int> streamUidsShuffled = playlistStreamJoinResultSet
          .map((row) => row['stream_id'])
          .toList()
          .cast<int>();
      streamUidsShuffled.shuffle();

      SqlLocalPlaylists.reOrderLocalPlaylist(dbPath, playlistUid, streamUidsShuffled);

      db.dispose();
    });
  }

  /// Copy streams from one local playlist to another.
  /// Identified by their row index or 'track number' (conceptually) in the playlist/UI, which
  /// correlates with their `join_index` in [BS.joinTableId].
  static void copyItemsFromLocalPlaylist({
    required String dbPathFrom,
    required String dbPathTo,
    required int playlistUidFrom,
    required int playlistUidTo,
    required List<int> rowIndices,
  }) {
    AppState.debug(
      'SqlLocalPlaylists::copyItemsFromLocalPlaylist(): $dbPathFrom, $dbPathTo, $playlistUidFrom, $playlistUidTo, $rowIndices',
    );
    SqlHelpers.tryAndCatchErrors(() {
      final Database dbFrom = sqlite3.open(dbPathFrom);
      final Database dbTo = sqlite3.open(dbPathTo);

      /// Convert row indices to stream_uids (ordered by join_index)
      ResultSet orderedJoinResultSet = dbFrom.select('''
        SELECT stream_id
        FROM ${BS.joinTableId}
        WHERE playlist_id = $playlistUidFrom
        ORDER BY join_index
      ''');

      List<int> streamUids = rowIndices
          .map((rowIndex) => orderedJoinResultSet[rowIndex]['stream_id'] as int)
          .toList();

      /// Get joined streams and playlists
      createPlaylistStreamJoinTable(dbFrom);

      ResultSet playlistStreamJoinResultSet = streamUids.isEmpty
          ? ResultSet([], [], [])
          : dbFrom.select('''
              SELECT *
              FROM ${BS.tempPlaylistStreamJoinedTableId}
              WHERE playlist_uid = $playlistUidFrom
                AND stream_uid IN (${streamUids.join(", ")})
            ''');

      List<int> newStreamUids = insertNewAndGetExistingStreams(
        playlistStreamJoinResultSet: playlistStreamJoinResultSet,
        db: dbTo,
      );

      /// Join index ('track number' in the playlist) increments with each added stream.
      /// !!! first get the highest index for this playlist !!!
      /// Use that as the starting join_index (so we add our streams to the end of the playlist).
      int joinIndex = SqlLocalPlaylists.getNextJoinIndex(dbTo, playlistUidTo);

      /// Finally add the streams to the playlist (by adding them to the join table).
      for (final newStreamUid in newStreamUids) {
        dbTo.execute('''
            INSERT INTO ${BS.joinTableId} 
              ( 'playlist_id', 'stream_id', 'join_index')
            VALUES 
              ( ${playlistUidTo}, ${newStreamUid}, ${joinIndex} );
          ''');
        joinIndex++;
      }

      dbFrom.dispose();
      dbTo.dispose();
    });
  }

  /// Copy a local playlist from one database to another.
  static int copyLocalPlaylist(String dbPathFrom, String dbPathTo, int playlistUid) {
    //AppState.debug('SqlLocalPlaylists.copyLocalPlaylist(): $dbPathFrom, $dbPathTo, $playlistUid');
    int newPlaylistUid = -1;
    SqlHelpers.tryAndCatchErrors(() {
      final Database dbFrom = sqlite3.open(dbPathFrom);
      final Database dbTo = sqlite3.open(dbPathTo);

      /// Get joined streams and playlists into [BS.tempPlaylistStreamJoinedTableId]
      createPlaylistStreamJoinTable(dbFrom);

      /// Then select all the joined columns for all the streams in this playlist
      ResultSet playlistStreamJoinResultSet = dbFrom.select('''
        SELECT *
        FROM ${BS.tempPlaylistStreamJoinedTableId}
        WHERE playlist_uid = ${playlistUid}
      ''');
      //AppState.debug('playlistStreamJoinResultSet : $playlistStreamJoinResultSet');

      List<int> newStreamUids = insertNewAndGetExistingStreams(
        playlistStreamJoinResultSet: playlistStreamJoinResultSet,
        db: dbTo,
      );

      /// Add an entry for the new playlist in [BS.localPlaylistsTableId].
      /// Different schemas have different columns so we need to handle that.
      SqlHelpers.safeCopyBetweenDbs(
        dbPathFrom: dbPathFrom,
        dbPathTo: dbPathTo,
        tableName: BS.localPlaylistsTableId,
        columnToMatch: 'uid',
        valuesToMatch: [playlistUid],
        inconsistentColumns: {
          'is_thumbnail_permanent': {'dataType': 'INTEGER', 'defaultValue': 0},
          'display_index': {'dataType': 'INTEGER', 'defaultValue': -1},
          'thumbnail_stream_id': {'dataType': 'INTEGER', 'defaultValue': -1},
          'thumbnail_url': {'dataType': 'TEXT', 'defaultValue': "''"},
        },
        uidToEnforce: 'uid',
      );

      String sql =
          '''

WITH idmax_cte AS (
    SELECT MAX(uid) AS max_uid FROM ${BS.localPlaylistsTableId}
)
SELECT max_uid FROM idmax_cte

        ''';
      //SqlHelpers.logSelect(dbTo, sql);

      newPlaylistUid = dbTo.select(sql)[0]['max_uid'];
      AppState.debug('newPlaylistUid: $newPlaylistUid');

      // Fill in the join table. Join index increments with each added stream.
      int joinIndex = 0;
      for (final newStreamUid in newStreamUids) {
        dbTo.execute('''
          INSERT INTO ${BS.joinTableId} 
            ( 'playlist_id', 'stream_id', 'join_index' )
          VALUES ( ${newPlaylistUid}, ${newStreamUid}, ${joinIndex} );
        ''');
        joinIndex++;
      }

      dbFrom.dispose();
      dbTo.dispose();
    });

    return newPlaylistUid;
  }

  /// Create the special joined table which allows us to link playlists
  /// and streams.
  /// This doesn't need to be wrapped in [SqlHelpers.tryAndCatchErrors()] as
  /// the caller function already is.
  static createPlaylistStreamJoinTable(Database db) {
    db.execute('''
      CREATE TEMP TABLE ${BS.tempPlaylistStreamJoinedTableId} AS
        SELECT
          service_id,
          url,
          title,
          stream_type,
          duration,
          uploader,
          ${BS.localPlaylistsTableId}.name
            AS playlist_name,
          ${BS.streamsTableId}.uid
            AS stream_uid,
          ${BS.localPlaylistsTableId}.uid
            AS playlist_uid
        FROM ${BS.streamsTableId}
        INNER JOIN ${BS.joinTableId}
          ON ${BS.joinTableId}.stream_id = ${BS.streamsTableId}.uid
        INNER JOIN ${BS.localPlaylistsTableId}
          ON ${BS.localPlaylistsTableId}.uid = ${BS.joinTableId}.playlist_id
    ''');
  }

  /// Process playlist data (as stored in the database) into something
  /// (a [Map]) easier to work with.
  ///
  /// NewPipe stores playlists and their individual streams (eg video/audio URLS) in separate
  /// tables. Join them together here and convert them into [Map]s which we (via [DbManager]) can
  /// work with more easily.
  static Map<int, TableModel> getLocalPlaylistsMap(String dbPath) {
    AppState.debug('getLocalPlaylistsMap()');
    AppState.debug('\tdbPath: $dbPath');
    Map<int, TableModel> localPlaylists = Map<int, TableModel>.from({});
    String dbPathHash = getMd5Hash(dbPath);

    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);

      /// Create the temporary table with all the data we need joined together.
      createPlaylistStreamJoinTable(db);

      /// Select only the unique playlist names from the temporary table.
      /// Each stream stores its playlist name in a column in the temporary joined table, so they are repeated lots of times.
      /// Select in case-insensitive, alphabetical order by 'playlist_name'
      /// ONLY STREAMS ARE LOOKED AT, SO ANY EMPTY (NON STREAM-CONTAINING) PLAYLISTS WILL BE OMITTED AT THIS POING
      List<Map> localPlaylistsWithStreamsMetaData = db
          .select('''
            SELECT DISTINCT	playlist_name, playlist_uid
            FROM ${BS.tempPlaylistStreamJoinedTableId}
            ORDER BY playlist_name COLLATE NOCASE
            ;
          ''')
          .rows
          .map((value) => {'name': value[0].toString(), 'uid': value[1]})
          .toList();

      int searchPlaylistUid = DbManager.getSearchPlaylistUid(dbPath) ?? -1;
      //AppState.debug('SqlLocalPlaylists::searchPlaylistUid: $searchPlaylistUid');

      /// Create a [Map] of data for each custom playlist [localPlaylistsWithStreamsMetaData] and store them in a [localPlaylists] container
      for (Map metaData in localPlaylistsWithStreamsMetaData) {
        String columnList = LocalPlaylistColumns.values
            .asNameMap()
            .entries
            .map((e) => '${e.key} AS ${e.value.displayName}')
            .toList()
            .join(', ');

        localPlaylists[metaData['uid']] = TableModel(
          uid: metaData['uid'],
          displayName: metaData['name'],
          dbPathHash: dbPathHash,
          tableType: (metaData['uid'] == searchPlaylistUid)
              ? TableType.search
              : TableType.localPlaylist,
          resultSet: db.select('''
            SELECT ${columnList}
            FROM ${BS.tempPlaylistStreamJoinedTableId}
            WHERE playlist_uid = ${metaData['uid']}
          '''),
        );
      }

      // Add any empty playlists to [localPlaylists]

      // 1. Get all playlist UIDs and names from the localPlaylistsTableId
      ResultSet allPlaylistsResultSet = db.select('''
				SELECT uid, name
				FROM ${BS.localPlaylistsTableId}
			''');

      // Get the list of column names for the ResultSet.
      // These will be consistent across all playlists (empty or not).
      List<String> emptyResultSetColumnNames = LocalPlaylistColumns.values
          .map((e) => e.name) // 'e.name' gives the actual column key (e.g., 'url', 'title')
          .toList();

      // 2. Iterate through all playlists and add those not yet in localPlaylists
      for (final row in allPlaylistsResultSet) {
        int playlistUid = row['uid'];
        String playlistName = row['name'].toString();

        if (!localPlaylists.containsKey(playlistUid)) {
          // This playlist has no streams and hasn't been added ye, so it's an empty playlist
          localPlaylists[playlistUid] = TableModel(
            uid: playlistUid,
            displayName: playlistName,
            dbPathHash: dbPathHash,
            tableType: (row['uid'] == searchPlaylistUid)
                ? TableType.search
                : TableType.localPlaylist,
            // Use the derived column names for the empty ResultSet
            resultSet: ResultSet(emptyResultSetColumnNames, [], []),
          );
        }
      }

      // Finish up
      db.dispose();
    });

    //AppState.debug(localPlaylists.toString());

    return localPlaylists;
  }

  /// Rewrite local playlist in a different order.
  /// Although we usually use the row index rather than stream uid for other operations in Local
  /// Playlists, here it's safe to use stream uids because nothing we do is affected by duplicates.
  static void reOrderLocalPlaylist(String dbPath, int playlistUid, List<int> streamUidsInNewOrder) {
    AppState.debug('streamUidsInNewOrder: $streamUidsInNewOrder');
    int joinIndex = 0;
    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);
      // Delete old playlist as we'll recreate it with the new order.
      db.execute('''
        DELETE FROM ${BS.joinTableId}
          WHERE playlist_id = ${playlistUid};
      ''');

      // Re-enter the streams into the join table. Join index increments with each added stream.
      for (final streamUid in streamUidsInNewOrder) {
        db.execute('''
          INSERT INTO ${BS.joinTableId} 
            ( 'playlist_id', 'stream_id', 'join_index' )
          VALUES 
            ( ${playlistUid}, ${streamUid}, ${joinIndex} );
        ''');
        joinIndex++;
      }

      db.dispose();
    });
  }

  static void deduplicateLocalPlaylist(String dbPath, int playlistUid) {
    AppState.debug('deduplicateLocalPlaylist():');

    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);

      db.execute('''
      DELETE FROM ${BS.joinTableId}
      WHERE playlist_id = $playlistUid
        AND join_index NOT IN (
          SELECT MIN(join_index)
          FROM ${BS.joinTableId}
          WHERE playlist_id = $playlistUid
          GROUP BY stream_id
        );
    ''');

      db.dispose();
    });
  }

  static Map<int, Map> getLocalPlaylistDuplicateInfo(String dbPath, int playlistUid) {
    final Map<int, Map> dupeInfo = {};

    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);

      createPlaylistStreamJoinTable(db);
      // Get all stream_ids for the playlist
      final rows = db.select('''
        SELECT *
        FROM ${BS.tempPlaylistStreamJoinedTableId}
        WHERE playlist_uid = $playlistUid
      ''');

      // Count occurrences
      for (final row in rows) {
        final int streamId = row['stream_uid'] as int;
        if (dupeInfo.containsKey(streamId)) {
          dupeInfo[streamId]!['occurrences'] = (dupeInfo[streamId]!['occurrences'] ?? 0) + 1;
        } else {
          dupeInfo[streamId] = {'title': row['title'], 'occurrences': 1};
        }
      }

      db.dispose();
    });

    dupeInfo.removeWhere((int streamUid, Map info) => info['occurrences'] == 1);

    return dupeInfo;
  }

  static void importPlaylistFromJson(String dbPath, String playlistName, List<dynamic> streams) {
    AppState.debug('SqlLocalPlaylists::importPlaylistFromJson()::');

    /// Create new or get uid for existing playlist if name already exists
    int playlistUid = createNewEmptyLocalPlaylist(
      dbPath,
      name: playlistName,
      skipIfNameExists: true,
    );

    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);
      if (playlistUid == -1) {
        playlistUid = getPlaylistUidFromName(db, playlistName)!;
        AppState.debug('\tPlaylist name already exists, importing into existing playlist');
      } else {
        AppState.debug('\tPlaylist name not found, new playlist created');
      }

      int joinIndex = SqlLocalPlaylists.getNextJoinIndex(db, playlistUid);

      AppState.debug('\tplaylistUid: $playlistUid');
      AppState.debug('\tjoinIndex: $joinIndex');

      List<List<Object?>> jsonAsRows = [];
      for (dynamic stream in streams) {
        List<Object?> row = BS.jsonImportColumnSpec.entries.map((MapEntry columnSpecEntry) {
          String columnName = columnSpecEntry.key;
          Map columnRequirements = columnSpecEntry.value;
          dynamic streamColumnValue = stream[columnName];

          dynamic columnValue;
          if (streamColumnValue == null) {
            /// Missing field/column in JSON
            if (columnRequirements['required']) {
              throw (columnRequirements['error']);
            } else {
              columnValue = columnRequirements['default'] ?? 'NULL';
            }
          } else {
            /// Found field/column in JSON
            if (streamColumnValue.runtimeType != columnRequirements['type']) {
              AppState.debug('\t\tstreamColumnValue.runtimeType: ${streamColumnValue.runtimeType}');
              throw ("'$columnName' must be ${columnRequirements['type']}");
            } else {
              columnValue = streamColumnValue;
            }
          }

          return columnValue;
        }).toList();
        jsonAsRows.add(row);
      }

      List<int> newStreamUids = insertNewAndGetExistingStreams(
        playlistStreamJoinResultSet: ResultSet(
          BS.jsonImportColumnSpec.keys.toList(),
          null,
          jsonAsRows,
        ),
        db: db,
      );

      /// Loop through each stream, for each:
      for (int streamUid in newStreamUids) {
        /// - Insert into 'playlist_stream_join' (playlist_id, stream_id, join_index)
        db.execute('''
            INSERT INTO ${BS.joinTableId} (
              'playlist_id', 'stream_id', 'join_index'
            )
            VALUES ( ${playlistUid}, ${streamUid}, ${joinIndex} );
          ''');
        joinIndex++;
      }

      db.dispose();
    });
  }

  /// Get the UID for an existing playlist with a specific name
  static int? getPlaylistUidFromName(Database db, String playlistName) {
    AppState.debug('SqlLocalPlaylists::getPlaylistUidFromName()');
    AppState.debug('\tdb: $db');
    AppState.debug('\tplaylistName: $playlistName');
    int? nameMatchUid;

    ResultSet playlistUidResultSet = db.select('''
        SELECT uid FROM ${BS.localPlaylistsTableId}
        WHERE name = '${playlistName}';
      ''');
    AppState.debug('\tplaylistUidResultSet: $playlistUidResultSet');

    if (playlistUidResultSet.isNotEmpty) {
      nameMatchUid = playlistUidResultSet.first['uid'];
    }

    AppState.debug('\tnameMatchUid: $nameMatchUid');
    return nameMatchUid;
  }

  /// Get the next join index for a specific playlist. Conceptually this is a 'track number'. If no
  /// tracks exist we want 0 for the new track, otherwise we want (current highest track index + 1).
  static int getNextJoinIndex(Database db, int playlistUid) {
    int maxJoinIndex = 0;

    ResultSet maxJoinIndexResultSet = db.select('''
        SELECT MAX(join_index) FROM ${BS.joinTableId}
        WHERE playlist_id = ${playlistUid};
      ''');

    if (maxJoinIndexResultSet.isNotEmpty) {
      if (maxJoinIndexResultSet[0]['MAX(join_index)'] != null) {
        maxJoinIndex = maxJoinIndexResultSet[0]['MAX(join_index)'] + 1;
      }
    }

    return maxJoinIndex;
  }

  /// STREAMS SHOULD ONLY EXIST ONCE IN A DATABASE, UNIQUE BASED ON THEIR `service_id` AND `url`.
  /// When copying streams from one database to another, some streams will already exist in the
  /// destination database, and some will be new. For new streams we want to insert a row and get
  /// the new uids, for existing streams we want to skip the insert and just get the existing uid.
  static List<int> insertNewAndGetExistingStreams({
    required Database db,
    required ResultSet playlistStreamJoinResultSet,
  }) {
    final PreparedStatement streamSelectStatement = db.prepare('''
      SELECT uid FROM ${BS.streamsTableId} 
      WHERE url = ? AND service_id = ?
    ''');

    final PreparedStatement streamInsertStatement = db.prepare('''
      INSERT INTO ${BS.streamsTableId} ( ${SqlStringMaker.streamColumnsWithQuotesAndCommas} )
      VALUES ( ${SqlStringMaker.streamColumnsUnspecifiedParams} )
      ON CONFLICT(service_id, url) DO UPDATE SET uid = uid
      RETURNING uid;
    ''');

    List<int> newStreamUids = [];

    for (final row in playlistStreamJoinResultSet) {
      final ResultSet existingStreamResultSet = streamSelectStatement.select([
        row['url'],
        row['service_id'],
      ]);

      if (existingStreamResultSet.isNotEmpty) {
        newStreamUids.add(existingStreamResultSet.first['uid']);
      } else {
        final ResultSet uidResultSet = streamInsertStatement.select(
          StreamColumns.values.map((e) => SqlHelpers.escapeQuotes(row[e.name])).toList(),
        );

        if (uidResultSet.isNotEmpty) {
          newStreamUids.add(uidResultSet.first['uid']);
        }
      }
    }

    // Dispose both statements
    streamInsertStatement.dispose();
    streamSelectStatement.dispose();

    //AppState.debug('newStreamUids: $newStreamUids');

    return newStreamUids;
  }

  static int? getPlaylistUidFromPathAndName({required String dbPath, required String name}) {
    AppState.debug('SqlLocalPlaylists::getPlaylistUidFromPathAndName()');
    AppState.debug('\tdbPath: $dbPath');
    AppState.debug('\tname: $name');
    int? foundUid;

    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);
      foundUid = getPlaylistUidFromName(db, name);
      db.dispose();
    });
    AppState.debug('\tfoundUid: $foundUid');

    return foundUid;
  }
}
