// SPDX-License-Identifier: GPL-3.0-only

import 'package:sqlite3/sqlite3.dart';

import 'app_state.dart';
import 'constants.dart';
import 'db_manager.dart';
import 'sql_schema_logic.dart';
import 'transaction_schema_details.dart';

/// Class with static functions to perform some generic tasks on the database.
class SqlHelpers {
  /// Wrapper to be used around all sqlite calls eg `select`, `execute`.
  /// [function] is the function (containing sqlite calls) we want to try.
  /// [transactionSchemaDetails] may be provided in which case its [description] field is used to add info to any error messages.
  static void tryAndCatchErrors(function, [TransactionSchemaDetails? transactionSchemaDetails]) {
    String schemaDescription = (transactionSchemaDetails == null)
        ? ''
        : transactionSchemaDetails.description;

    try {
      function();
    } on Exception catch (e) {
      AppState.debug('${schemaDescription}EXCEPTION:\n$e\n', alwaysOnScreen: true);
    } catch (e) {
      AppState.debug('${schemaDescription}ERROR:\n$e\n', alwaysOnScreen: true);
    }
  }

  static SchemaVersion deduceSchemaVersion(String dbPath) {
    SchemaVersion schemaVersion = SchemaVersion.unknown;
    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);
      schemaVersion = detectSchemaVersion(db);
      db.dispose();
    });
    return schemaVersion;
  }

  /// Given path/s to read/write (from/to) databases, return a [TransactionSchemaDetails] with appropriate [SchemaVersion]s and a description.
  static TransactionSchemaDetails getTransactionSchemaDetails({
    SchemaVersion? toSchemaVersion,
    SchemaVersion? fromSchemaVersion,
  }) {
    String description = '';

    SqlHelpers.tryAndCatchErrors(() {
      if (toSchemaVersion != null) {
        description += 'Writing to: ${toSchemaVersion.name} (${toSchemaVersion.description})\n';
      }
      if (fromSchemaVersion != null) {
        description +=
            'Reading from: ${fromSchemaVersion.name} (${fromSchemaVersion.description})\n';
      }
    });
    if (description.isNotEmpty) {
      description = 'DATABASE SCHEMA INFO:\n${description}\n\n';
    }

    return TransactionSchemaDetails(
      fromSchemaVersion: fromSchemaVersion,
      toSchemaVersion: toSchemaVersion,
      description: description,
    );
  }

  /// Escape single quotes in a string by converting them to two single quotes.
  static String escapeQuotes(str) {
    str ??= '';
    str = str.toString();
    return str.replaceAll("'", "''");
  }

  /// Given a [Map] of ['columnName': value, ...], create an INSERT statement
  /// Where the value is a string it will be wrapped in quotes.
  static String getInsertStatement({required String tableName, required Map map}) {
    return '''
      INSERT INTO $tableName
        ( ${map.keys.join(', ')} )
      VALUES 
        ( ${map.values.map((value) => value.runtimeType == String ? "'$value'" : value).join(', ')} );
    ''';
  }

  /// Check if a database file is a NewPipe database in a recognised format.
  /// If not valid, display schema as it may be useful for improving
  /// compatibility with old dbs.
  static bool isCompatibleDatabase(dbPath) {
    bool isCompatible = false;

    tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);
      //print('--- Starting isCompatibleDatabase Check on: $dbPath ---');
      //final allTablesResult = db.select("SELECT name FROM sqlite_master WHERE type='table';");
      //final List<String> foundTables = allTablesResult.map((row) => row['name'] as String).toList();
      //print('SQLite found the following tables: $foundTables');
      // --- END OF DEBUGGING STEP ---
      isCompatible = true;
      List<String> missingTables = [];

      /// Check that the tables we need exist in the database.
      /// The query returns a resultset of length 1 if the table exists,
      /// otherwise 0.
      for (final tableName in BS.requiredTables) {
        if (db.select('''
          SELECT name FROM sqlite_master 
          WHERE type='table' AND name='${tableName}';
        ''').isEmpty) {
          missingTables.add(tableName);
          isCompatible = false;
        }
      }

      if (!isCompatible) {
        String databaseDebugInfo = missingTables.length == BS.requiredTables.length
            ? '** FILE SEEMS NOT TO BE A NEWPIPE ZIP! **\n\n'
            : '';
        databaseDebugInfo += 'Missing Tables:\n$missingTables\n\n';

        /// Get list of tables in the database we're checking
        List<String?> actualTables = getDbTableList(dbPath);
        databaseDebugInfo += 'Actual Tables:\n$actualTables\n\n';

        /// And its schema...
        databaseDebugInfo += 'Schema:\n';
        for (final tableName in actualTables) {
          ResultSet tableResultSet = db.select('''
            SELECT sql FROM sqlite_schema
            WHERE name = '${tableName}';
          ''');
          databaseDebugInfo += '${tableResultSet.toString()}\n';
        }

        databaseDebugInfo += '\n-------\n\n';

        AppState.debug(databaseDebugInfo, alwaysOnScreen: true);
      }
      db.dispose();
    });

    return isCompatible;
  }

  /// Return a list of all table names from a database.
  static List<String?> getDbTableList(String dbPath) {
    final List<String> tableNames = [];

    tryAndCatchErrors(() {
      final ResultSet resultSet = SqlHelpers.getSelectResultSet(
        dbPath,
        "SELECT name FROM sqlite_master WHERE type='table';",
      );

      for (final row in resultSet) {
        tableNames.add(row['name']);
      }
    });

    return tableNames;
  }

  /// Run a SELECT query and return the results.
  static ResultSet getSelectResultSet(String dbPath, String sqlQuery) {
    ResultSet resultSet = ResultSet([], [], []);

    tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);
      resultSet = db.select(sqlQuery);
      db.dispose();
    });

    return resultSet;
  }

  /// Select/return specific columns from a single table.
  static ResultSet getFilteredColumns(String dbPath, dynamic tableId, String columns) {
    return getSelectResultSet(dbPath, 'SELECT $columns FROM $tableId;');
  }

  /// Copy remote playlists from one database to another.
  /// General strategy:
  /// - Copy the rows we need into a temporary table, using `SELECT *` so we get column definitions
  /// - Compare the schema we're copying from, with where we're copying to
  /// - If there are some columns missing in the temp table, add them and fill with default values
  /// - If there are columns we don't need in the temp table, drop them
  /// - We should now have matching columns, do the `INSERT`, specifying column names
  static void safeCopyBetweenDbs({
    required String dbPathFrom,
    required String dbPathTo,
    required String tableName,
    required String columnToMatch,
    required List<int> valuesToMatch,
    Map<String, Map>? inconsistentColumns,
    String? uidToEnforce,
  }) {
    TransactionSchemaDetails txSchemaDetails = SqlHelpers.getTransactionSchemaDetails(
      toSchemaVersion: DbManager.getSchemaVersion(dbPathTo),
      fromSchemaVersion: DbManager.getSchemaVersion(dbPathFrom),
    );

    SqlHelpers.tryAndCatchErrors(() {
      /// We need a db in this/Dart context in order to call `execute`, `select` etc.
      /// Arbitrarily use the 'from' db (could have been 'to' instead).
      final Database db = sqlite3.open(dbPathFrom);

      /// Some vars to reuse throughout this function.
      String sql = '';
      String tempTableId = '${tableName}_temp';
      ResultSet debugResultSet = ResultSet([], [], []);

      /// Make some aliases (schema names) to keep the SQL readable.
      db.execute('''

ATTACH DATABASE '$dbPathTo' AS 'dbTo';
ATTACH DATABASE '$dbPathFrom' AS 'dbFrom';

      ''');

      /// Create temporary table to hold the rows we want to copy.
      sql =
          ('''

CREATE TABLE temp.$tempTableId AS
  SELECT * FROM dbFrom.$tableName
  WHERE $columnToMatch IN ( ${valuesToMatch.join(', ')} );

      ''');
      //AppState.debug(sql);
      db.execute(sql);

      //sql =
      //    '''
      //  SELECT * FROM temp.$tempTableId
      //''';
      ////AppState.debug(sql);
      //debugResultSet = db.select(sql);
      ////AppState.debug('SqlHelpers::safeCopyBetweenDbs 1.: ${debugResultSet.toString()}');

      /// List the columns we just created in the temp table.
      List<String> tempTableColumnList = db
          .select("PRAGMA table_info('$tempTableId');")
          .map((row) => row['name'] as String)
          .toList();
      AppState.debug('tempTableColumnList (before]: ${tempTableColumnList.join(', ')}');

      inconsistentColumns?.forEach((String columnName, Map columnDetails) {
        /// If there are columns we don't need in the temp table, remove them.
        if (tempTableColumnList.contains(columnName) &&
            txSchemaDetails.toSchemaVersion!.hasAbsentColumn(tableName, columnName)) {
          sql =
              '''

ALTER TABLE temp.$tempTableId
  DROP COLUMN $columnName;

          ''';
          //AppState.debug(sql);
          db.execute(sql);
        }

        /// If there are missing columns in the temp table, add them and fill with defaults.
        if (!tempTableColumnList.contains(columnName) &&
            txSchemaDetails.toSchemaVersion!.hasPresentColumn(tableName, columnName)) {
          sql =
              '''

ALTER TABLE temp.$tempTableId
  ADD COLUMN $columnName
    ${columnDetails['dataType']} NOT NULL DEFAULT ${columnDetails['defaultValue']};

          ''';
          //AppState.debug(sql);
          db.execute(sql);
        }
      });

      /// Update the list of columns
      tempTableColumnList = db
          .select("PRAGMA table_info('$tempTableId');")
          .map((row) => row['name'] as String)
          .toList();
      AppState.debug('tempTableColumnList (after): ${tempTableColumnList.join(', ')}');

      String tempTableColumns = tempTableColumnList.join(', ');

      if (uidToEnforce != null) {
        /// This table uses `uid` but doesn't enforce `UNIQUE`. We have to do it manually, otherwise
        /// copied playlists might overwrite existing playlists (where they share a `uid`).
        sql =
            '''

WITH idmax_cte AS (
    SELECT MAX($uidToEnforce) AS max_$uidToEnforce FROM dbTo.$tableName
)
UPDATE temp.$tempTableId
SET $uidToEnforce = $uidToEnforce + IFNULL((SELECT max_$uidToEnforce FROM idmax_cte), 0);


        ''';
        //AppState.debug(sql);
        db.execute(sql);
      }

      /// COPY ALL ROWS FROM THE TEMP TABLE TO THE `TO` TABLE
      /// two playlists from the same service with the same URL cannot coexist (so ignore if constraint is broken)
      sql =
          '''

INSERT OR IGNORE INTO dbTo.$tableName ( $tempTableColumns )
  SELECT $tempTableColumns
    FROM temp.$tempTableId;

      ''';
      //AppState.debug(sql);
      db.execute(sql);

      //sql =
      //    '''
      //  SELECT * FROM dbTo.$tableName
      //''';
      //AppState.debug(sql);
      //debugResultSet = db.select(sql);
      //AppState.debug('SqlHelpers::safeCopyBetweenDbs 2.: ${debugResultSet.toString()}');

      /// Finish off
      db.dispose();
    }, txSchemaDetails);
  }

  ///
  ///
  ///
  ///
  ///
  ///
  static void logSelect(Database db, String sql) {
    ResultSet debugResultSet = db.select(sql);
    AppState.debug(sql);
    AppState.debug(debugResultSet.toString());
  }
}
