// SPDX-License-Identifier: GPL-3.0-only

import 'package:flex_color_scheme/flex_color_scheme.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';

import 'app_state.dart';
import 'constants.dart';
import 'outer_wrapper_widget.dart';
import 'unanimated_page_route.dart';

void main() async {
  WidgetsFlutterBinding.ensureInitialized();

  /// App preferences [SharedPreferencesWithCache] require async setup
  await AppState.initPreferences();
  runApp(const BendyStrawApp());
}

/// Main app wrapper.
class BendyStrawApp extends StatefulWidget {
  const BendyStrawApp({super.key});

  @override
  State<BendyStrawApp> createState() => _BendyStrawAppState();
}

class _BendyStrawAppState extends AppState<BendyStrawApp> {
  /// We want to rebuild when 'preferences' changes, so the widgets eg
  /// checkboxes show their correct current state.
  @override
  List<String>? listenForChanges = ['preferences'];

  // This key can be used as a reference when we need to close the 'sure to quit?' dialog
  final GlobalKey<NavigatorState> _navigatorKey = GlobalKey<NavigatorState>();

  @override
  void initState() {
    /// A reference to [_navigatorKey] is stored in [AppState] (as
    /// [mainNavigatorKey]) so that other classes can access it.
    AppState.update('mainNavigatorKey', _navigatorKey);

    super.initState();
  }

  Route _onGenerateRoute(RouteSettings settings) {
    AppState.debug('\nmain::_onGenerateRoute: ${settings.name}\n');
    late Widget page;

    if (settings.name == BS.routeHome) {
      page = const OuterWrapperWidget(setupPageRoute: BS.routeInnerStartPage);
    } else if (settings.name!.startsWith(BS.routePrefixInner)) {
      final innerRoute = settings.name!.substring(BS.routePrefixInner.length);
      page = OuterWrapperWidget(setupPageRoute: innerRoute);
    } else {
      throw Exception('Unknown route: ${settings.name}');
    }

    return AppState.getPreference('disableUiAnimations')
        ? UnanimatedPageRoute<dynamic>(
            builder: (context) {
              return page;
            },
            settings: settings,
          )
        : MaterialPageRoute<dynamic>(
            builder: (context) {
              return page;
            },
            settings: settings,
          );
  }

  @override
  Widget build(BuildContext context) {
    String brightnessPref = AppState.getPreference('brightnessOverride');
    bool darkIsTrueBlack = AppState.getPreference('darkIsTrueBlack');
    bool disableUiAnimations = AppState.getPreference('disableUiAnimations');

    return MaterialApp(
      /// Theme animations eg when changing from light to dark
      themeAnimationDuration: disableUiAnimations ? Duration.zero : kThemeAnimationDuration,
      debugShowCheckedModeBanner: false,
      navigatorKey: _navigatorKey,
      onGenerateRoute: _onGenerateRoute,
      title: BS.appName,
      theme: FlexThemeData.light(scheme: FlexScheme.pinkM3),
      darkTheme: FlexThemeData.dark(scheme: FlexScheme.pinkM3, darkIsTrueBlack: darkIsTrueBlack),
      themeMode: brightnessPref == 'dark'
          ? ThemeMode.dark
          : brightnessPref == 'light'
          ? ThemeMode.light
          : ThemeMode.system,
      home: PopScope(
        canPop: false,
        onPopInvokedWithResult: (bool didPop, Object? result) async {
          AppState.debug('\n- - - - - - - - - - main::onPopInvokedWithResult:');
          AppState.debug('\tdidPop: ${didPop}');
          AppState.update('dialogToShow', {
            'title': 'Exit BendyStraw?',
            'actions': [
              ElevatedButton(
                child: Text('Cancel'),
                onPressed: () {
                  _navigatorKey.currentState?.pop(result);
                },
              ),
              ElevatedButton(
                child: Text('Exit app'),
                onPressed: () {
                  SystemChannels.platform.invokeMethod('SystemNavigator.pop');
                },
              ),
            ],
          });
        },
        child: const OuterWrapperWidget(setupPageRoute: BS.routeInnerStartPage),
      ),
    );
  }
}
