// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'app_state.dart';
import 'constants.dart';
import 'db_manager.dart';
import 'destination_list_radio_button.dart';
import 'enum.dart';
import 'location.dart';

/// A list of potential destinations (playlists) within a specific database.
class DestinationListRadioGroup extends StatefulWidget {
  /// tableId can be an int uid, or a string in the case of eg 'subscriptions'
  final Location locationTo;
  final Location locationFrom;
  final Function chosenLocationGetter;
  final Function chosenLocationSetter;
  final Iterable tables;

  const DestinationListRadioGroup({
    super.key,
    required this.locationTo,
    required this.locationFrom,
    required this.chosenLocationGetter,
    required this.chosenLocationSetter,
    required this.tables,
  });

  @override
  State<DestinationListRadioGroup> createState() => _DestinationListRadioGroupState();
}

class _DestinationListRadioGroupState extends State<DestinationListRadioGroup> {
  final ScrollController _scrollController = ScrollController();

  @override
  Widget build(BuildContext context) {
    List<Widget> radioButtons = [];

    if (widget.locationTo.tableType == TableType.remotePlaylists ||
        widget.locationTo.tableType == TableType.subscriptions) {
      /// Known tables --- can only copy items to the same table in another db
      //AppState.debug('Known tables --- can only copy items to the same table in another db');
      radioButtons = [
        DestinationListRadioButton(
          title: BS.knownTables[widget.locationTo.tableId]?['displayName'],
          location: widget.locationTo,
        )
      ];
    } else {
      if (widget.locationFrom.tableType == TableType.search && widget.tables.isEmpty) {
        /// Copying entire search table
        //AppState.debug('Copying entire search table');
        radioButtons = [
          DestinationListRadioButton(
            title: 'Create \'${DbManager.getSearchPlaylistCopyName(widget.locationFrom.dbPath)}\'',
            location: widget.locationTo,
          )
        ];
      } else {
        if (widget.tables.isEmpty) {
          /// Copying some other (so must be local playlist) entire table
          //AppState.debug('Copying some other (so must be local playlist) entire table');
          radioButtons = [
            DestinationListRadioButton(
              title:
                  'Create \'${DbManager.getLocalPlaylistName(widget.locationFrom.dbPath, widget.locationFrom.tableId)}\'',
              location: widget.locationTo,
            )
          ];
        } else {
          /// Copying selected items from one playlist to another
          //AppState.debug('Copying selected items from one playlist to another');
          //AppState.debug('${widget.tables.toString()}');
          radioButtons = widget.tables
              // Don't show the playlist we're copying from as a destination.
              .where((table) => (!(table['id'] == widget.locationFrom.tableId &&
                  widget.locationTo.dbPath == widget.locationFrom.dbPath)))
              .map<Widget>((table) => DestinationListRadioButton(
                    title: DbManager.getLocalPlaylistName(widget.locationTo.dbPath, table['id']),
                    location: Location(
                      dbPath: widget.locationTo.dbPath,
                      tableType: widget.locationTo.tableType,
                      tableId: table['id'],
                    ),
                  ))
              .toList();
        }
      }
    }

    return Theme(
      data: DbManager.getDbCurrentThemeData(widget.locationTo.dbPath),
      child: Builder(builder: (context) {
        ColorScheme colorScheme = Theme.of(context).colorScheme;
        return ClipRRect(
          borderRadius: BorderRadius.circular(BS.cornerRadius),
          child: ColoredBox(
            color: colorScheme.primaryContainer,
            child: Padding(
              padding: const EdgeInsets.all(BS.paddingSmall),
              child: Column(
                mainAxisAlignment: MainAxisAlignment.start,
                mainAxisSize: MainAxisSize.min,
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Padding(
                    padding: const EdgeInsets.only(
                      top: BS.paddingSmall,
                      bottom: BS.paddingSmall,
                      left: BS.paddingSmall,
                    ),
                    // RadioGroup heading
                    child: Text(
                      DbManager.getPrettyName(widget.locationTo.dbPath),
                      softWrap: false,
                      maxLines: 1,
                      overflow: TextOverflow.ellipsis,
                    ),
                  ),
                  Flexible(
                    child: ClipRRect(
                      borderRadius: BorderRadius.circular(BS.cornerRadius),
                      child: ColoredBox(
                        color: colorScheme.surface,
                        child: ScrollbarTheme(
                          data: Theme.of(context).scrollbarTheme.copyWith(
                                thumbColor: WidgetStatePropertyAll(colorScheme.secondary),
                                thickness: WidgetStatePropertyAll(BS.scrollbarThicknessThin),
                              ),
                          child: Scrollbar(
                            controller: _scrollController,
                            thumbVisibility: true,
                            child: SingleChildScrollView(
                                controller: _scrollController,
                                child: RadioGroup<Location>(
                                  groupValue: widget.chosenLocationGetter(),
                                  onChanged: (Location? value) {
                                    widget.chosenLocationSetter(value);
                                  },
                                  child: Flex(
                                    direction: Axis.vertical,
                                    crossAxisAlignment: CrossAxisAlignment.start,
                                    children: radioButtons,
                                  ),
                                )),
                          ),
                        ),
                      ),
                    ),
                  ),
                ],
              ),
            ),
          ),
        );
      }),
    );
  }
}
