// SPDX-License-Identifier: GPL-3.0-only

import 'dart:io';

import 'package:flutter/material.dart';
import 'package:sqlite3/sqlite3.dart';

import 'app_state.dart';
import 'bs_database.dart';
import 'databases.dart';
import 'sql_helpers.dart';
import 'sql_local_playlists.dart';
import 'sql_remote_playlists.dart';
import 'sql_schema_logic.dart';
import 'sql_search_playlist.dart';
import 'sql_subscriptions.dart';
import 'table_model.dart';

/// Handle all database management tasks.
///
/// [_databases] is a [Map] containing info for all imported databases. Each is
/// stored in that map under its key ([dbPath]), which is a filesystem path to
/// the database as extracted from its containing `zip`. Throughout the app,
/// [dbPath] is used whenever we want to specify which database we are dealing
/// with for an operation.
class DbManager {
  static final Databases _databases = Databases();

  static Databases get databases {
    return _databases;
  }

  static int get numberOfDatabases {
    return _databases.length;
  }

  /// Using [dbPath] as a key, store some info about a database in a [Map].
  /// Return `true` if a change is made, otherwise `false`.
  static bool addDbByPathKey(String dbPath, String originalZipPath) {
    bool didChange = false;

    if (!_databases.containsKey(dbPath) && SqlHelpers.isCompatibleDatabase(dbPath)) {
      BsDatabase db = BsDatabase(path: dbPath, originalZipPath: originalZipPath);
      _databases.add(key: dbPath, database: db);

      db.initSearchPlaylist();
      db.rebuildLocalAndSearchPlaylistMaps(setDbDirty: false);
      db.rebuildRemotePlaylists();
      db.rebuildChannelSubscriptions();

      /// Some of the above dirties db, but reset it as nothing important from a user perspective
      /// has changed (ie the file does not need saving).
      setDbDirtyState(dbPath, false);

      didChange = true;
    }

    return didChange;
  }

  static void removeAllDbs() {
    _databases.removeAll();
    AppState.update('currentSelectedDbPath', '');
  }

  /// Using [dbPath] as a key, remove info about a database.
  /// Return `true` if a change is made, otherwise `false`.
  static bool removeDbByPathKey(String dbPath) {
    bool didChange = false;
    if (_databases.containsKey(dbPath)) {
      _databases.remove(dbPath);
      didChange = true;
    }
    return didChange;
  }

  /// Return a [Map] containing data for each of the custom playlists.
  static SchemaVersion getSchemaVersion(String dbPath) {
    return _databases.getByKey(dbPath)!.schemaVersion;
  }

  /// Return a [Map] containing data for each of the custom playlists.
  static Map<int, TableModel> getLocalPlaylists(String dbPath) {
    return _databases.getByKey(dbPath)!.localPlaylists;
  }

  /// Number of local playlists in a database.
  static int getNumberOfLocalPlaylists(String dbPath) {
    return _databases.getByKey(dbPath)!.localPlaylists.length;
  }

  /// Total number of local playlists in all opened database.
  static int getTotalNumberOfLocalPlaylistsInAllDbs() {
    int total = 0;
    for (String key in _databases.keys) {
      total += _databases.getByKey(key)!.localPlaylists.length;
    }
    return total;
  }

  /// Return a [Map] containing data for the special search playlist. Although there is only one
  /// search playlist per db, it's presented in a map as if there might be many. This is to allow it
  /// to be processed in the same way as normal local playlists.
  static Map<int, TableModel> getSearchPlaylists(String dbPath) {
    return _databases.getByKey(dbPath)!.searchPlaylists;
  }

  /// Return the display name of a local playlist.
  static String getLocalPlaylistName(String dbPath, int playlistUid) {
    BsDatabase db = _databases.getByKey(dbPath)!;
    return db.localPlaylists[playlistUid]?.displayName ??
        db.searchPlaylists[playlistUid]?.displayName ??
        'Playlist name not found';
  }

  /// Return the display name assigned to a database.
  static String getPrettyName(String dbPath) {
    return _databases.getByKey(dbPath)!.prettyName;
  }

  /// Return the directory path a playlist was originally opened from.
  static String getOriginalDirectoryPath(String dbPath) {
    return _databases.getByKey(dbPath)!.originalDirectoryPath;
  }

  /// Return the current theme assigned to a database, based on current system
  /// brightness or an overriding prefrerence variable.
  static ThemeData getDbCurrentThemeData(String dbPath) {
    Brightness systemBrightness = MediaQuery.platformBrightnessOf(AppState.globalContext);
    String brightnessOverride = AppState.getPreference('brightnessOverride');

    /// Default to prevent crashes or flashes of wrong brigtness if this function is called where a
    /// db is momentarily null.
    ThemeData themeData =
        (brightnessOverride == 'dark' ||
            (brightnessOverride == 'system' && systemBrightness == Brightness.dark))
        ? ThemeData.dark()
        : ThemeData();

    if (_databases.containsKey(dbPath)) {
      BsDatabase db = _databases.getByKey(dbPath)!;
      themeData =
          (brightnessOverride == 'dark' ||
              (brightnessOverride == 'system' && systemBrightness == Brightness.dark))
          ? db.themeDataDark
          : db.themeData;
    }

    return themeData;
  }

  static bool getDbDirtyState(String dbPath) {
    return _databases.getByKey(dbPath)!.hasUnsavedChanges;
  }

  // Allows 'save' button to activate/deactivate
  static void setDbDirtyState(String dbPath, bool isDirty) {
    _databases.getByKey(dbPath)!.hasUnsavedChanges = isDirty;
    // `aDatabaseWasDirtied`: it doesn't matter what the value is as we just
    // want the message to be broadcast so listening widgets redraw, pulling in
    // whichever fresh data they need.
    AppState.update('aDatabaseWasDirtied', null, forceRebuild: true);
  }

  /// Wrapper just to avoid classes other than this calling [SqlHelpers]
  /// functions directly.
  ///
  /// [columns] is a string describing columns, ready to be passed into an SQL
  /// query, eg:
  /// ```
  /// "col1Name AS col1DisplayName, col2Name AS col2DisplayName"
  /// ```
  static ResultSet getFilteredColumns(String dbPath, dynamic tableId, String columns) {
    return SqlHelpers.getFilteredColumns(dbPath, tableId, columns);
  }

  ///////////////////////
  ///
  /// Local/Custom Playlists
  ///
  ///////////////////////

  static int copyLocalPlaylist(
    String dbPathFrom,
    String dbPathTo,
    int playlistUid,
    bool isMoveMode,
  ) {
    //AppState.debug(
    //    'DbManager.copyLocalPlaylist(): $dbPathFrom, $dbPathTo, $playlistUid, $isMoveMode');
    int newPlaylistUid = SqlLocalPlaylists.copyLocalPlaylist(dbPathFrom, dbPathTo, playlistUid);

    /// If this playlist is the search playlist for the db we're copying from, rename it to reflect
    /// the search terms.
    if (playlistUid == getSearchPlaylistUid(dbPathFrom)) {
      AppState.debug('Playlist being copied is the search playlist, renaming it');
      renameLocalPlaylist(dbPathTo, newPlaylistUid, getSearchPlaylistCopyName(dbPathFrom));
    }

    _databases.getByKey(dbPathTo)!.rebuildLocalAndSearchPlaylistMaps();

    if (isMoveMode) {
      deleteLocalPlaylist(dbPathFrom, playlistUid);
    }

    return newPlaylistUid;
  }

  static void createNewEmptyLocalPlaylist(String dbPath) {
    AppState.debug('createNewEmptyLocalPlaylist(): $dbPath');
    SqlLocalPlaylists.createNewEmptyLocalPlaylist(dbPath);
    _databases.getByKey(dbPath)!.rebuildLocalAndSearchPlaylistMaps();
  }

  static void deleteLocalPlaylist(String dbPath, int playlistUid) {
    AppState.debug('deleteLocalPlaylist(): $dbPath, $playlistUid');
    SqlLocalPlaylists.deleteLocalPlaylist(dbPath, playlistUid);
    _databases.getByKey(dbPath)!.rebuildLocalAndSearchPlaylistMaps();
  }

  static Map<int, Map> getLocalPlaylistDuplicateInfo(String dbPath, int playlistUid) {
    AppState.debug('getLocalPlaylistDuplicateInfo(): $dbPath, $playlistUid');
    return SqlLocalPlaylists.getLocalPlaylistDuplicateInfo(dbPath, playlistUid);
  }

  static void deduplicateLocalPlaylist(String dbPath, int playlistUid) {
    AppState.debug('deduplicateLocalPlaylist(): $dbPath, $playlistUid');
    SqlLocalPlaylists.deduplicateLocalPlaylist(dbPath, playlistUid);
    _databases.getByKey(dbPath)!.rebuildLocalAndSearchPlaylistMaps();
  }

  static void renameLocalPlaylist(String dbPath, int playlistUid, String newName) {
    SqlLocalPlaylists.renameLocalPlaylist(dbPath, playlistUid, newName);
    _databases.getByKey(dbPath)!.rebuildLocalAndSearchPlaylistMaps();
  }

  static void importPlaylistFromJson(String dbPath, String playlistName, List<dynamic> streams) {
    AppState.debug('importPlaylistFromJson(): $dbPath, $playlistName');
    SqlLocalPlaylists.importPlaylistFromJson(dbPath, playlistName, streams);

    //BsDatabase db = _databases.getByKey(dbPath)!;
    //db.rebuildLocalAndSearchPlaylistMaps(setDbDirty: false);
    //AppState.update(dbPath, DbManager.databases, forceRebuild: true);
    _databases.getByKey(dbPath)!.rebuildLocalAndSearchPlaylistMaps();
  }

  static void reverseLocalPlaylistOrder(String dbPath, int playlistUid) {
    AppState.debug('reverseLocalPlaylistOrder(): $dbPath, $playlistUid');
    SqlLocalPlaylists.reverseLocalPlaylistOrder(dbPath, playlistUid);

    _databases.getByKey(dbPath)!.rebuildLocalAndSearchPlaylistMaps();
  }

  static void shuffleLocalPlaylistOrder(String dbPath, int playlistUid) {
    AppState.debug('shuffleLocalPlaylistOrder(): $dbPath, $playlistUid');
    SqlLocalPlaylists.shuffleLocalPlaylistOrder(dbPath, playlistUid);

    _databases.getByKey(dbPath)!.rebuildLocalAndSearchPlaylistMaps();
  }

  static void copyItemsFromLocalPlaylist({
    required String dbPathFrom,
    required String dbPathTo,
    required int playlistUidFrom,
    required int playlistUidTo,
    required List<int> rowIndices,
    required bool isMoveMode,
  }) {
    AppState.debug(
      'DbManager::copyItemsFromLocalPlaylist(): $dbPathFrom, $dbPathTo, $playlistUidFrom, $playlistUidTo, $rowIndices, $isMoveMode',
    );

    SqlLocalPlaylists.copyItemsFromLocalPlaylist(
      dbPathFrom: dbPathFrom,
      dbPathTo: dbPathTo,
      playlistUidFrom: playlistUidFrom,
      playlistUidTo: playlistUidTo,
      rowIndices: rowIndices,
    );

    _databases.getByKey(dbPathTo)!.rebuildLocalAndSearchPlaylistMaps();

    if (isMoveMode) {
      deleteItemsFromLocalPlaylist(
        dbPath: dbPathFrom,
        playlistUid: playlistUidFrom,
        rowIndices: rowIndices,
      );
    }
  }

  static void deleteItemsFromLocalPlaylist({
    required String dbPath,
    required int playlistUid,
    required List<int> rowIndices,
  }) {
    AppState.debug('DbManager::deleteItemsFromLocalPlaylist(): $dbPath, $playlistUid, $rowIndices');

    SqlLocalPlaylists.deleteItemsFromLocalPlaylist(dbPath, playlistUid, rowIndices);
    _databases.getByKey(dbPath)!.rebuildLocalAndSearchPlaylistMaps();
  }

  /// Get data for a specific playlist.
  static TableModel? getSinglePlaylistData(String dbPath, int playlistUid) {
    AppState.debug('getSinglePlaylistData()');
    AppState.debug('\tplaylistUid: $playlistUid');
    TableModel? playlistData;
    Map<int, TableModel> localPlaylists = getLocalPlaylists(dbPath);
    Map<int, TableModel> searchPlaylists = getSearchPlaylists(dbPath);

    if (localPlaylists.containsKey(playlistUid)) {
      playlistData = localPlaylists[playlistUid];
    } else if (searchPlaylists.containsKey(playlistUid)) {
      playlistData = searchPlaylists[playlistUid];
    } else {
      AppState.debug('\tPlaylist uid not found in local nor search playlists');
    }

    return playlistData;
  }

  /// If 1 or more playlists with [name] exists, get the uid for the first one found.
  static int? getPlaylistUidFromPathAndName({required String dbPath, required String name}) {
    return SqlLocalPlaylists.getPlaylistUidFromPathAndName(dbPath: dbPath, name: name);
  }

  /// Get a path to which a text playlist will be exported.
  static String getExportLocalPlaylistFilepath(String dbPath, dynamic playlistUid) {
    AppState.debug('getExportLocalPlaylistFilepath()');
    AppState.debug('\tplaylistUid: $playlistUid');
    TableModel? playlistData = getSinglePlaylistData(dbPath, playlistUid);

    String playlistDisplayName = playlistData?.displayName ?? 'Playlist name not found';

    /// Create/sanitise filepath
    String fileName =
        getOriginalDirectoryPath(dbPath) +
        playlistDisplayName.replaceAll(RegExp(r'[^a-zA-Z0-9]'), '-');
    fileName += '.txt';
    AppState.debug('\tfileName: $fileName');
    return fileName;
  }

  /// Export a list of URLs from a playlist.
  ///
  /// The exported file can be used with `yt-dlp` to batch download the
  /// playlist.
  static void exportLocalPlaylistToYtDlp(String dbPath, dynamic playlistUid) {
    AppState.debug('exportLocalPlaylistToM3u(): $dbPath, $playlistUid');

    TableModel? playlistData = getSinglePlaylistData(dbPath, playlistUid);
    String playlistDisplayName = playlistData?.displayName ?? 'Playlist name not found';
    ResultSet playlistRowsResultSet = playlistData?.resultSet ?? ResultSet([], [], []);

    String fileName = getExportLocalPlaylistFilepath(dbPath, playlistUid);

    // Comments header
    String fileContents = '# For use with yt-dlp to batch download eg:\n';
    fileContents += '# `yt-dlp --batch-file "$fileName"`\n\n\n';

    // Playlist name comment
    fileContents += '# Playlist Title: $playlistDisplayName\n\n\n';

    // URLs with title as a leading comment
    for (final row in playlistRowsResultSet) {
      fileContents += '# ${row['Title']} | ${row['Channel']}\n';
      fileContents += '${row['URL']}\n\n';
    }
    AppState.debug(fileContents);

    // Write the file
    final file = File(fileName);
    file.writeAsString(fileContents);
  }

  static List<String> getLocalPlaylistAsYouTubeIdList(String dbPath, int playlistUid) {
    AppState.debug('getLocalPlaylistAsYouTubeIdList(): $dbPath, $playlistUid');

    TableModel? playlistData = getSinglePlaylistData(dbPath, playlistUid);
    ResultSet playlistRowsResultSet = playlistData?.resultSet ?? ResultSet([], [], []);

    RegExp exp = RegExp(r'v=(.*$)');
    return playlistRowsResultSet.map((row) => exp.firstMatch(row['URL'])![1] ?? '').toList();
  }

  static void sortLocalPlaylistByColumn({
    required String dbPath,
    required int playlistUid,
    required int sortColumnIndex,
    required int streamUidColumnIndex,
    required bool isNumeric,
    required bool ascending,
  }) {
    AppState.debug('streamUidColumnIndex: $streamUidColumnIndex');

    List<List<Object?>> playlistRows =
        (getSinglePlaylistData(dbPath, playlistUid)?.resultSet ?? ResultSet([], [], [])).rows;

    // As everything is passed by reference, [sort] here actually changes the order of the streams directly, but only in the cached version... so it will look right in the UI, but the sqllite database isn't affected yet.
    if (isNumeric) {
      if (ascending) {
        // Ascending (a, b)
        playlistRows.sort(
          (a, b) => int.parse(
            a[sortColumnIndex].toString(),
          ).compareTo(int.parse(b[sortColumnIndex].toString())),
        );
      } else {
        // Descending (b, a)
        playlistRows.sort(
          (b, a) => int.parse(
            a[sortColumnIndex].toString(),
          ).compareTo(int.parse(b[sortColumnIndex].toString())),
        );
      }
    } else {
      // Alphabetic column
      if (ascending) {
        playlistRows.sort(
          (a, b) => a[sortColumnIndex].toString().toLowerCase().compareTo(
            b[sortColumnIndex].toString().toLowerCase(),
          ),
        );
      } else {
        playlistRows.sort(
          (b, a) => a[sortColumnIndex].toString().toLowerCase().compareTo(
            b[sortColumnIndex].toString().toLowerCase(),
          ),
        );
      }
    }

    //AppState.debug(playlistRows);
    SqlLocalPlaylists.reOrderLocalPlaylist(
      dbPath,
      playlistUid,

      /// make a list of streamUids in the new order
      playlistRows.map((row) => row[streamUidColumnIndex]).toList().cast<int>(),
    );

    _databases.getByKey(dbPath)!.rebuildLocalAndSearchPlaylistMaps();
  }

  /// Used when a row is dragged by the user to re-order a playlist.
  static void reOrderLocalPlaylist({
    required String dbPath,
    required int playlistUid,
    required List<int> streamUidsInNewOrder,
  }) {
    SqlLocalPlaylists.reOrderLocalPlaylist(dbPath, playlistUid, streamUidsInNewOrder);
    _databases.getByKey(dbPath)!.rebuildLocalAndSearchPlaylistMaps();
  }

  ///////////////////////
  ///
  /// Remote/Bookmarked Playlists
  ///
  ///////////////////////

  static void copyItemsFromRemotePlaylists({
    required String dbPathFrom,
    required String dbPathTo,
    required List<int> remotePlaylistUids,
    required bool isMoveMode,
  }) {
    AppState.debug(
      'copyItemsFromRemotePlaylists(): $dbPathFrom, $dbPathTo, $remotePlaylistUids, $isMoveMode',
    );

    SqlRemotePlaylists.copyItemsFromRemotePlaylists(dbPathFrom, dbPathTo, remotePlaylistUids);

    _databases.getByKey(dbPathTo)!.rebuildRemotePlaylists();

    if (isMoveMode) {
      deleteItemsFromRemotePlaylists(dbPath: dbPathFrom, remotePlaylistUids: remotePlaylistUids);
    }
  }

  static void deleteItemsFromRemotePlaylists({
    required String dbPath,
    required List<int> remotePlaylistUids,
  }) {
    AppState.debug('deleteItemsFromRemotePlaylists(): $dbPath, $remotePlaylistUids');

    SqlRemotePlaylists.deleteItemsFromRemotePlaylists(dbPath, remotePlaylistUids);
    _databases.getByKey(dbPath)!.rebuildRemotePlaylists();
  }

  ///////////////////////
  ///
  /// Channel Subscriptions
  ///
  ///////////////////////

  static void copyItemsFromChannelSubscriptions({
    required String dbPathFrom,
    required String dbPathTo,
    required List<int> subscriptionUids,
    required bool isMoveMode,
  }) {
    AppState.debug(
      'copyItemsFromChannelSubscriptions(): $dbPathFrom, $dbPathTo, $subscriptionUids, $isMoveMode',
    );

    SqlChannelSubscriptions.copyItemsFromChannelSubscriptions(
      dbPathFrom,
      dbPathTo,
      subscriptionUids,
    );

    _databases.getByKey(dbPathTo)!.rebuildChannelSubscriptions();

    if (isMoveMode) {
      deleteItemsFromChannelSubscriptions(dbPath: dbPathFrom, subscriptionUids: subscriptionUids);
    }
  }

  static void deleteItemsFromChannelSubscriptions({
    required String dbPath,
    required List<int> subscriptionUids,
  }) {
    AppState.debug('deleteItemsFromChannelSubscriptions(): $dbPath, $subscriptionUids');
    SqlChannelSubscriptions.deleteItemsFromChannelSubscriptions(dbPath, subscriptionUids);
    _databases.getByKey(dbPath)!.rebuildChannelSubscriptions();
  }

  ///////////////////////
  ///
  /// Search Playlist
  ///
  ///////////////////////

  static void deleteSearchPlaylist(String dbPath, {bool deleteEntriesOnly = false}) {
    int? searchPlaylistUid = getSearchPlaylistUid(dbPath);
    if (searchPlaylistUid != null) {
      SqlLocalPlaylists.deleteLocalPlaylist(
        dbPath,
        searchPlaylistUid,
        deleteEntriesOnly: deleteEntriesOnly,
      );
    } else {
      AppState.debug('\tsearch playlist not found');
    }
  }

  static int? getSearchPlaylistUid(String dbPath) {
    return _databases.getByKey(dbPath)!.searchPlaylistUid;
  }

  static String getSearchPlaylistName(String dbPath) {
    //AppState.debug('getSearchPlaylistName()');
    //AppState.debug('\tdbPath: $dbPath');
    //AppState.debug('\t_databases.keys: ${_databases.keys.toString()}');
    return _databases.getByKey(dbPath)!.searchPlaylistName;
  }

  /// When copying a search playlist we want to rename it, just use the serch terms.
  static String getSearchPlaylistCopyName(String dbPath) {
    String searchPlaylistName = getSearchPlaylistName(dbPath);
    String name = AppState.get('${searchPlaylistName}-terms');
    return "'$name' (search results)";
  }

  static void updateSearchPlaylist(
    String dbPath,
    String searchTerms, {
    bool includeTitles = true,
    bool includeChannels = true,
  }) {
    BsDatabase db = _databases.getByKey(dbPath)!;

    /// Delete only the items, keep the (empty) playlist because we want to preserve the uid.
    deleteSearchPlaylist(dbPath, deleteEntriesOnly: true);
    db.createSearchPlaylist();

    SqlSearchPlaylist.updateSearchPlaylist(
      dbPath,
      searchTerms,
      includeTitles: includeTitles,
      includeChannels: includeChannels,
    );

    db.rebuildLocalAndSearchPlaylistMaps();
    AppState.update(dbPath, DbManager.databases, forceRebuild: true);
  }

  ///////////////////////
  ///
  /// Table paths and row selections
  ///
  ///////////////////////
  static String getDbPathHash(String dbPath) {
    return _databases.getByKey(dbPath)!.pathHash;
  }
}
