// SPDX-License-Identifier: GPL-3.0-only

import 'package:sqlite3/sqlite3.dart';

import 'constants.dart';
import 'sql_helpers.dart';

/// Low level SQL functions to operate on the list of channel subscriptions.
///
/// Only [DbManager] should call functions directly from this class. Other
/// classes should call  wrapper functions in [DbManager].

/// IMPORTANT
/// Channel Subscriptions identify rows using their uid ([subscriptionUids]). This is safe because
/// the tables cannot be sorted/re-ordered, and there will never be duplicates.
class SqlChannelSubscriptions {
  static void deleteItemsFromChannelSubscriptions(String dbPath, List<int> subscriptionUids) {
    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);
      db.execute('''
          DELETE FROM ${BS.subscriptionsTableId}
          WHERE uid IN ( ${subscriptionUids.join(", ")} )
        ''');
      db.dispose();
    });
  }

  /// Copy channel subscriptions from one database to another.
  static void copyItemsFromChannelSubscriptions(
    String dbPathFrom,
    String dbPathTo,
    List<int> subscriptionUids,
  ) {
    SqlHelpers.safeCopyBetweenDbs(
      dbPathFrom: dbPathFrom,
      dbPathTo: dbPathTo,
      tableName: BS.subscriptionsTableId,
      columnToMatch: 'uid',
      valuesToMatch: subscriptionUids,
      inconsistentColumns: {
        'notification_mode': {'dataType': 'INTEGER', 'defaultValue': 0},
      },
      uidToEnforce: 'uid',
    );
  }
}
