// SPDX-License-Identifier: GPL-3.0-only

import 'package:sqlite3/sqlite3.dart';

import 'constants.dart';
import 'db_manager.dart';
import 'sql_helpers.dart';

/// Low level SQL functions to operate on the list of remote playlists.
///
/// Only [DbManager] should call functions directly from this class. Other
/// classes should call wrapper functions in [DbManager].

/// IMPORTANT
/// Remote Playlists identify rows using their uid ([playlistUids]). This is safe because the tables
/// cannot be sorted/re-ordered, and there will never be duplicates.
class SqlRemotePlaylists {
  static void deleteItemsFromRemotePlaylists(String dbPath, List<int> playlistUids) {
    SqlHelpers.tryAndCatchErrors(() {
      final Database db = sqlite3.open(dbPath);
      db.execute('''
          DELETE FROM ${BS.remotePlaylistsTableId}
          WHERE uid IN ( ${playlistUids.join(", ")} )
        ''');
      db.dispose();
    });
  }

  /// Copy remote playlists from one database to another.
  static void copyItemsFromRemotePlaylists(
    String dbPathFrom,
    String dbPathTo,
    List<int> playlistUids,
  ) {
    SqlHelpers.safeCopyBetweenDbs(
      dbPathFrom: dbPathFrom,
      dbPathTo: dbPathTo,
      tableName: BS.remotePlaylistsTableId,
      columnToMatch: 'uid',
      valuesToMatch: playlistUids,
      inconsistentColumns: {
        'display_index': {'dataType': 'INTEGER', 'defaultValue': -1},
      },
      uidToEnforce: 'uid',
    );
  }
}
