// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';
import 'package:shared_preferences/shared_preferences.dart';

import 'app_state.dart';
import 'constants.dart';

/// Display and control a multiple-choice, radio-button-type
/// [SharedPreferences] item.
class PreferenceItemRadioGroup extends StatefulWidget {
  const PreferenceItemRadioGroup({
    super.key,
    required this.screenIsCompact,
    required this.title,
    required this.options,
    required this.id,
    this.description = '',
    this.onChangeRebuild,
  });

  final bool screenIsCompact;
  final String title;
  final String id;
  final String description;
  final Map<String, dynamic> options;
  final String? onChangeRebuild;

  @override
  State<PreferenceItemRadioGroup> createState() => _PreferenceItemRadioGroupState();
}

class _PreferenceItemRadioGroupState extends State<PreferenceItemRadioGroup> {
  @override
  Widget build(BuildContext context) {
    var defaultValue = BS.defaultPreferences[widget.id];
    var layout =
        widget.screenIsCompact ? BS.preferenceItemLayoutCompact : BS.preferenceItemLayoutStandard;
    var currentValue = AppState.getPreference(widget.id);

    return Padding(
      padding: layout['outerPadding'],
      child: RadioGroup<String>(
        groupValue: currentValue,
        onChanged: (value) {
          _setValue(value ?? defaultValue);
        },
        child: Column(crossAxisAlignment: CrossAxisAlignment.start, children: [
          Row(
            mainAxisSize: MainAxisSize.min,
            children: [
              Padding(
                padding: layout['titlePadding'],
                child: Text(
                  widget.title,
                  style: const TextStyle(fontSize: BS.fontSizeHeading, height: 1),
                ),
              ),
            ],
          ),
          if (widget.description.isNotEmpty)
            Row(
              mainAxisSize: MainAxisSize.min,
              children: [
                Expanded(
                  child: Padding(
                    padding: layout['descriptionPadding'],
                    child: Text(
                      widget.description,
                      style: const TextStyle(fontSize: BS.fontSizeSmall, height: 1),
                    ),
                  ),
                ),
              ],
            ),
          SizedBox(height: widget.screenIsCompact ? BS.paddingSmall : BS.paddingMedium),
          ...widget.options.entries.map<Widget>((entry) => InkWell(
                onTap: () {
                  _setValue(entry.value);
                },
                child: Row(children: [
                  Radio<String>(
                    value: entry.value,
                  ),
                  SizedBox(
                    width: layout['descriptionPadding'].left,
                  ),
                  Text(entry.key)
                ]),
              ))
        ]),
      ),
    );
  }

  /// Update the state of this preference in the app.
  void _setValue(value) async {
    await AppState.setPreference(widget.id, value, forceRebuild: true);
    setState(() {});
    if (widget.onChangeRebuild != null) {
      AppState.forceRebuildOn(widget.onChangeRebuild!);
    }

    //if (widget.onChangeCallback != null) {
    //  widget.onChangeCallback!();
    //}
  }
}
