// SPDX-License-Identifier: GPL-3.0-only

import 'dart:async';

import 'package:flutter/material.dart';
import 'package:url_launcher/url_launcher.dart';

import 'app_state.dart';
import 'constants.dart';
import 'db_manager.dart';
import 'location.dart';
import 'destination_chooser.dart';
import 'enum.dart';
import 'export_playlist_explainer.dart';
import 'small_icon_button.dart';
import 'tooltip_icon_button.dart';
import 'screen.dart';

/// Buttons relating to a playlist as a whole.
///
/// Buttons which act on an entire playlist, such as copy/move/export. Each
/// button calls a private method to execute its action.
class PlaylistButtonsWidget extends StatelessWidget {
  final String playlistDisplayName;
  final Location location;
  final bool disabled;
  final bool isSearchPlaylist;

  const PlaylistButtonsWidget({
    super.key,
    required this.playlistDisplayName,
    required this.location,
    this.disabled = false,
    this.isSearchPlaylist = false,
  });

  @override
  Widget build(BuildContext context) {
    bool showSmallIconButtonLabels = AppState.getPreference('showSmallIconButtonLabels');

    /// COPYING ENTIRE TABLE
    /// Subscriptions/remote playlists can not be copied/moved (this widget won't exist)
    /// Local playlists need at least 1 **other** database to copy to
    /// Search playlists can always be copied
    bool copyIsEnabled =
        (location.tableType == TableType.localPlaylist && DbManager.numberOfDatabases > 1) ||
        location.tableType == TableType.search;

    return ConstrainedBox(
      constraints: const BoxConstraints(minWidth: BS.renamableTextFieldWidth),
      child: Padding(
        padding: EdgeInsets.only(top: BS.paddingSmall, bottom: BS.paddingSmall),
        child: Wrap(
          spacing: Screen.isCompact(context) ? BS.paddingMedium : BS.paddingSmall,
          //runSpacing: Screen.isCompact(context) ? BS.paddingMedium : BS.paddingSmall,
          children: [
            if (AppState.getPreference('showPlaylistReverseButton'))
              SmallIconButton(
                iconData: Icons.swap_vert,
                hoverText: 'Reverse playlist order',
                label: showSmallIconButtonLabels ? 'Reverse' : null,
                onPressed: disabled
                    ? null
                    : () {
                        _reversePlaylistOrder();
                      },
              ),
            if (AppState.getPreference('showPlaylistShuffleButton'))
              SmallIconButton(
                iconData: Icons.shuffle,
                hoverText: 'Shuffle playlist order',
                label: showSmallIconButtonLabels ? 'Shuffle' : null,
                onPressed: disabled
                    ? null
                    : () {
                        _shufflePlaylistOrder();
                      },
              ),
            if (AppState.getPreference('showPlaylistDeduplicateButton') &&
                location.tableType != TableType.search)
              SmallIconButton(
                iconData: Icons.cleaning_services,
                hoverText: 'Remove duplicate streams',
                label: showSmallIconButtonLabels ? 'Deduplicate' : null,
                onPressed: disabled
                    ? null
                    : () {
                        _confirmThenDeduplicatePlaylist(context: context);
                      },
              ),
            if (AppState.getPreference('showInstantYoutubePlaylistButton'))
              SmallIconButton(
                iconData: Icons.web,
                hoverText: 'Open as a YouTube temporary playlist',
                label: showSmallIconButtonLabels ? 'YouTube' : null,
                onPressed: disabled
                    ? null
                    : () {
                        AppState.debounceVoidFunction(
                          callerKey: location.hashPath,
                          voidFunction: _openAsYouTubePlaylist,
                          debounceDuration: BS.openYouTubeDebounceDuration,
                        );
                      },
              ),
            SmallIconButton(
              iconData: Icons.drive_file_move_rounded,
              hoverText: 'Copy/Move playlist',
              label: showSmallIconButtonLabels ? 'Copy/Move' : null,
              onPressed: (disabled || !copyIsEnabled)
                  ? null
                  : () {
                      _confirmThenCopyPlaylist(context: context);
                    },
            ),
            SmallIconButton(
              iconData: Icons.web_stories_rounded,
              hoverText: 'Export playlist',
              label: showSmallIconButtonLabels ? 'Export' : null,
              onPressed: disabled
                  ? null
                  : () {
                      _confirmThenExportPlaylist(context: context);
                    },
            ),
            if (!isSearchPlaylist)
              SmallIconButton(
                iconData: Icons.playlist_remove_rounded,
                hoverText: 'Delete playlist',
                label: showSmallIconButtonLabels ? 'Delete' : null,
                onPressed: () {
                  _confirmThenDeletePlaylist(context: context);
                },
              ),
          ],
        ),
      ),
    );
  }

  /// Start the export process by displaying a dialog with a confirm button,
  /// which will export the playlist when pressed.
  void _confirmThenExportPlaylist({required BuildContext context}) {
    String filePath = DbManager.getExportLocalPlaylistFilepath(location.dbPath, location.tableId);
    AppState.showAppDialog(
      title: 'Export streams to text:',
      content: ExportPlaylistExplainer(filePath: filePath),
      actions: <Widget>[
        TooltipIconButton(
          text: Screen.isCompact(context) ? 'Export' : 'Export to $filePath',
          iconData: Icons.web_stories_rounded,
          layout: TooltipIconButtonLayout.iconOnRight,
          onPressedCallback: () {
            DbManager.exportLocalPlaylistToYtDlp(location.dbPath, location.tableId);
            AppState.get('mainNavigatorKey').currentState?.pop();
          },
        ),
      ],
    );
  }

  /// Start the deletion process by displaying a dialog with a confirm button,
  /// which will delete the playlist when pressed.
  void _confirmThenDeletePlaylist({required BuildContext context}) {
    AppState.showAppDialog(
      title: 'Delete playlist \'$playlistDisplayName\'?',
      actions: <Widget>[
        TooltipIconButton(
          text: Screen.isCompact(context) ? 'Delete' : 'Delete \'$playlistDisplayName\'',
          iconData: Icons.playlist_remove_rounded,
          layout: TooltipIconButtonLayout.iconOnRight,
          onPressedCallback: () {
            _deletePlaylistTableFromDb();
            AppState.get('mainNavigatorKey').currentState?.pop();
          },
        ),
      ],
    );
  }

  /// Start the deduplication process by displaying a dialog with a confirm button,
  /// which will deduplicate the playlist when pressed.
  void _confirmThenDeduplicatePlaylist({required BuildContext context}) {
    final ScrollController scrollController = ScrollController();
    Map<int, Map> dupeInfo = DbManager.getLocalPlaylistDuplicateInfo(
      location.dbPath,
      location.tableId,
    );
    ColorScheme colorScheme = Theme.of(context).colorScheme;
    TextStyle labelStyle = TextStyle(
      color: colorScheme.onSurface,
      height: 1.2,
      leadingDistribution: TextLeadingDistribution.proportional,
    );
    TextStyle infoStyle = TextStyle(
      color: colorScheme.error,
      fontSize: BS.fontSizeBaseInfo,
      fontWeight: FontWeight.bold,
      leadingDistribution: TextLeadingDistribution.proportional,
    );

    int totalDuplicates = dupeInfo.values.fold(0, (sum, info) {
      return sum + ((info['occurrences'] as int) - 1);
    });

    AppState.showAppDialog(
      title: totalDuplicates == 0
          ? 'No duplicates found in \'$playlistDisplayName\''
          : 'Remove $totalDuplicates duplicates from \'$playlistDisplayName\'?',
      content: totalDuplicates == 0
          ? null
          : Scrollbar(
              thumbVisibility: true,
              controller: scrollController,
              child: SingleChildScrollView(
                controller: scrollController,
                child: Table(
                  columnWidths: {1: FixedColumnWidth(BS.infoTableLabelColumnWidthCompact)},
                  defaultVerticalAlignment: TableCellVerticalAlignment.middle,
                  children: [
                    ...dupeInfo.entries.map((entry) {
                      int duplicates = entry.value['occurrences'] - 1;

                      return TableRow(
                        children: [
                          TableCell(
                            child: Padding(
                              padding: const EdgeInsets.only(
                                bottom: BS.paddingSmall,
                                right: BS.paddingMedium,
                              ),
                              child: Text(
                                entry.value['title'],
                                style: labelStyle,
                                textAlign: TextAlign.end,
                              ),
                            ),
                          ),
                          TableCell(
                            child: Padding(
                              padding: const EdgeInsets.only(
                                bottom: BS.paddingSmall,
                                left: BS.paddingSmall + BS.paddingSuperSmall,
                              ),
                              child: Text(
                                'x ${duplicates.toString()}',
                                //'$duplicates ${duplicates == 1 ? 'duplicate' : 'duplicates'}',
                                style: infoStyle,
                              ),
                            ),
                          ),
                        ],
                      );
                    }),
                  ],
                ),
              ),
            ),

      actions: <Widget>[
        totalDuplicates == 0
            ? TooltipIconButton(
                text: 'Dismiss',
                iconData: Icons.thumb_down,
                layout: TooltipIconButtonLayout.iconOnRight,
                onPressedCallback: () {
                  AppState.get('mainNavigatorKey').currentState?.pop();
                },
              )
            : TooltipIconButton(
                text: Screen.isCompact(context) ? 'Remove' : 'Deduplicate \'$playlistDisplayName\'',
                iconData: Icons.cleaning_services,
                layout: TooltipIconButtonLayout.iconOnRight,
                onPressedCallback: () {
                  _deduplicateLocalPlaylist();
                  AppState.get('mainNavigatorKey').currentState?.pop();
                },
              ),
      ],
    );
  }

  /// Start the copy/move process by displaying a dialog with a confirm button,
  /// which will copy or move the playlist when pressed.
  Future<void> _confirmThenCopyPlaylist({required BuildContext context}) async {
    bool isMoveMode = false;
    Location? chosenLocation;

    return showDialog<void>(
      context: context,
      builder: (BuildContext context) {
        return StatefulBuilder(
          builder: (context, setState) {
            return Center(
              child: SimpleDialog(
                titlePadding: EdgeInsets.zero,
                contentPadding: EdgeInsets.zero,
                children: [
                  DestinationChooser(
                    locationFrom: Location(
                      dbPath: location.dbPath,
                      tableId: location.tableId,
                      tableType: location.tableType,
                    ),
                    copyOperationType: CopyOperationType.fullTable,
                    isMoveMode: isMoveMode,
                    moveModeSetter: (bool? value) {
                      setState(() {
                        isMoveMode = value ?? false;
                      });
                    },
                    chosenLocation: chosenLocation,
                    chosenLocationSetter: (Location destination) {
                      setState(() {
                        chosenLocation = destination;
                      });
                    },
                    chosenLocationGetter: () => chosenLocation,
                    copyFunction: _copyPlaylist,
                  ),
                ],
              ),
            );
          },
        );
      },
    );
  }

  /// Call a function to copy the playlist, then force rebuild of relevant
  /// widgets to reflect changes.
  void _copyPlaylist({
    required Location locationFrom,
    required Location locationTo,
    bool isMoveMode = false,
  }) {
    DbManager.copyLocalPlaylist(
      locationFrom.dbPath,
      locationTo.dbPath,
      locationFrom.tableId,
      isMoveMode,
    );
    AppState.debug(
      'Copy ${location.dbPath}.${location.tableId} to ${locationTo.dbPath}.${location.tableId}',
    );
  }

  /// Call a function to delete the playlist, then force rebuild of relevant
  /// widgets to reflect that the table is now gone.
  void _deduplicateLocalPlaylist() {
    DbManager.deduplicateLocalPlaylist(location.dbPath, location.tableId);
  }

  /// Call a function to delete the playlist, then force rebuild of relevant
  /// widgets to reflect that the table is now gone.
  void _deletePlaylistTableFromDb() {
    DbManager.deleteLocalPlaylist(location.dbPath, location.tableId);
  }

  /// Call a function to reverse the order of the items in a playlist, then
  /// force rebuild of relevant widgets to reflect the changes.
  void _reversePlaylistOrder() {
    DbManager.reverseLocalPlaylistOrder(location.dbPath, location.tableId);
  }

  /// Call a function to shuffle the order of the items in a playlist, then
  /// force rebuild of relevant widgets to reflect the changes.
  void _shufflePlaylistOrder() {
    DbManager.shuffleLocalPlaylistOrder(location.dbPath, location.tableId);
  }

  Future<void> _openAsYouTubePlaylist() async {
    AppState.debug('_openAsYouTubePlaylist()');
    List<String> ids = DbManager.getLocalPlaylistAsYouTubeIdList(location.dbPath, location.tableId);
    String title = location.tableType == TableType.search
        ? DbManager.getSearchPlaylistCopyName(location.dbPath)
        : DbManager.getLocalPlaylistName(location.dbPath, location.tableId);
    Uri uri = Uri.parse(
      "https://www.youtube.com/watch_videos?video_ids=${ids.join(',')}&title=${title}",
    );
    AppState.debug(uri.toString());
    if (!await launchUrl(uri)) {
      throw Exception(uri.toString());
    }
  }
}
