// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';
import 'package:flutter/services.dart';

import 'app_state.dart';
import 'base_info_widget.dart';
import 'constants.dart';
import 'db_manager.dart';
import 'dialog_dismiss_button.dart';
import 'file_manager_widget.dart';
import 'preferences_button.dart';
import 'screen.dart';
import 'tab_manager_widget.dart';
import 'tooltip_icon_button.dart';
import 'unanimated_page_route.dart';

// Outer wrapper for the entire app.
class OuterWrapperWidget extends StatefulWidget {
  const OuterWrapperWidget({super.key, required this.setupPageRoute});

  final String setupPageRoute;

  @override
  State<OuterWrapperWidget> createState() => _OuterWrapperWidgetState();
}

class _OuterWrapperWidgetState extends AppState<OuterWrapperWidget> {
  @override
  List<String>? listenForChanges = ['dialogToShow'];

  final GlobalKey<NavigatorState> _navigatorKey = GlobalKey<NavigatorState>();

  @override
  void initState() {
    super.initState();
    AppState.debug(
      '\n  \\ - - - - - - - - _OuterWrapperWidgetState::initState: ${_navigatorKey.hashCode}\n',
    );
    AppState.update('nestedNavigatorKey', _navigatorKey);
  }

  // Defining the routes for the nested Navigator
  Route _onGenerateRoute(RouteSettings settings) {
    AppState.debug(
      '\n  \\ - - - - - - - - OuterWrapperWidget::_onGenerateRoute: ${settings.name}\n',
    );
    late Widget page;

    switch (settings.name) {
      case BS.routeInnerStartPage:
        page = BaseInfoWidget();
      case BS.routeInnerTabsPage:
        page = ListenableBuilder(
          listenable: DbManager.databases,
          builder: (context, child) {
            return TabManagerWidget(databasesNotifier: DbManager.databases);
          },
        );
    }

    return AppState.getPreference('disableUiAnimations')
        ? UnanimatedPageRoute<dynamic>(
            builder: (context) {
              return page;
            },
            settings: settings,
          )
        : MaterialPageRoute<dynamic>(
            builder: (context) {
              return page;
            },
            settings: settings,
          );
  }

  @override
  Widget build(BuildContext context) {
    AppState.debug('\n  OuterWrapperWidget::build():${_navigatorKey.hashCode}\n');
    Map dialogToShow = AppState.get('dialogToShow');
    if (dialogToShow.isNotEmpty) {
      Future.delayed(Duration.zero, () {
        if (context.mounted) {
          _showDialog(context, dialogToShow);
        }
      });
    }

    return Scaffold(
      body: SafeArea(
        child: AppStateWidget(
          child: Stack(
            children: [
              /// Inner app always appears underneath the floating buttons, and can contain the [BaseInfoWidget] or [TabManagerWidget].
              Positioned.fill(
                child: Scaffold(
                  body: PopScope(
                    onPopInvokedWithResult: (bool didPop, Object? result) async {
                      AppState.debug(
                        '\n  \\ - - - - - - - - OuterWrapperWidget::onPopInvokedWithResult:',
                      );
                      AppState.debug('\t\tdidPop: ${didPop}');
                      _navigatorKey.currentState?.popUntil((route) {
                        AppState.debug(
                          '\t\t\tOuterWrapperWidget::popUntil: route.settings.name: ${route.settings.name}',
                        );
                        bool doPop = true;
                        if (route.settings.name == BS.routeInnerTabsPage) {
                          AppState.update('dialogToShow', {
                            'title': 'Close Databases?',
                            'content': Text(
                              "Your files will be closed and you'll return to the start screen",
                            ),
                            'actions': [
                              ElevatedButton(
                                child: Text('Cancel'),
                                onPressed: () {
                                  // Close dialog
                                  Navigator.of(context).pop();
                                },
                              ),
                              ElevatedButton(
                                child: Text('Close tabs'),
                                onPressed: () {
                                  /// Close dialog
                                  Navigator.of(context).pop();

                                  /// Navigate back to tabs
                                  _navigatorKey.currentState?.pop(result);

                                  /// Ensure all databases are removed.
                                  DbManager.removeAllDbs();
                                },
                              ),
                            ],
                          }, forceRebuild: true);
                        } else if (route.settings.name == BS.routeInnerStartPage) {
                          AppState.debug('\t\t\tSTRT');
                          doPop = true;
                        }

                        return doPop;
                      });
                    },
                    child: Navigator(
                      key: _navigatorKey,
                      initialRoute: widget.setupPageRoute,
                      onGenerateRoute: _onGenerateRoute,
                    ),
                  ),
                ),
              ),

              /// Floating buttons
              Positioned.directional(
                textDirection: TextDirection.ltr,
                bottom: 0,
                start: 0,
                end: 0,
                child: FileManagerWidget(),
              ),
              Positioned.directional(
                textDirection: TextDirection.ltr,
                top: 0,
                end: 0,
                child: PreferencesButton(),
              ),
            ],
          ),
        ),
      ),
    );
  }

  /// Reusable dialog to be used by other classes as much as possible.
  void _showDialog(context, dialogToShow) {
    AppState.debug(
      '\n  \\ - - - - - - - - OuterWrapperWidget::_showDialog: ${_navigatorKey.hashCode}\n',
    );
    final Map dialogLayout = Screen.isCompact(context)
        ? BS.dialogLayoutCompact
        : BS.dialogLayoutStandard;

    final bool isError = dialogToShow['isError'] ?? false;

    showDialog(
      context: context,
      builder: (BuildContext context) {
        ColorScheme colorScheme = Theme.of(context).colorScheme;
        return AlertDialog(
          titlePadding: EdgeInsets.only(
            top: dialogLayout['topPad'],
            right: dialogLayout['rightPad'],
            bottom: dialogLayout['bottomPad'],
            left: dialogLayout['leftPad'],
          ),
          contentPadding: EdgeInsets.only(
            right: dialogLayout['innerRightPad'],
            bottom: dialogLayout['innerBottomPad'],
            left: dialogLayout['innerLeftPad'],
          ),
          actionsPadding: dialogToShow['actions'] == null && !isError
              ? EdgeInsets.zero
              : EdgeInsets.only(
                  top: dialogLayout['innerTopPad'],
                  left: dialogLayout['innerLeftPad'],
                  right: dialogLayout['innerRightPad'],
                  bottom: dialogLayout['innerBottomPad'],
                ),
          title: Row(
            mainAxisAlignment: MainAxisAlignment.spaceBetween,
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Expanded(
                child: Padding(
                  padding: EdgeInsets.only(
                    top: dialogLayout['innerTopPad'],
                    left: dialogLayout['innerLeftPad'],
                    //right: dialogLayout['innerRightPad'],
                    //bottom: dialogLayout['innerBottomPad'],
                  ),
                  child: Text(dialogToShow['title'], textAlign: TextAlign.center),
                ),
              ),
              const DialogDismissButton(),
            ],
          ),
          content:
              dialogToShow['content'] ??
              Container(
                decoration: isError
                    ? BoxDecoration(
                        color: colorScheme.onError,
                        borderRadius: BorderRadius.circular(BS.cornerRadius),
                      )
                    : null,
                child: dialogToShow['message'] == null
                    ? const Text('')
                    : SingleChildScrollView(
                        scrollDirection: Axis.vertical,
                        child: isError
                            ? Padding(
                                padding: EdgeInsets.only(
                                  top: dialogLayout['innerTopPad'],
                                  right: dialogLayout['innerRightPad'],
                                  bottom: dialogLayout['innerBottomPad'],
                                  left: dialogLayout['innerLeftPad'],
                                ),
                                child: Text(
                                  dialogToShow['message'],
                                  style: TextStyle(color: colorScheme.error),
                                ),
                              )
                            : Text(dialogToShow['message']),
                      ),
              ),
          actions: <Widget>[
            Align(
              alignment: Alignment.centerRight,
              child: Row(
                mainAxisAlignment: MainAxisAlignment.end,
                children: [
                  if (isError)
                    TooltipIconButton(
                      text: 'Copy error to clipboard',
                      iconData: Icons.copy_rounded,
                      layout: TooltipIconButtonLayout.iconOnLeft,
                      onPressedCallback: () async {
                        await Clipboard.setData(ClipboardData(text: dialogToShow['message']));
                      },
                    ),
                  if (dialogToShow['actions'] != null) ...dialogToShow['actions'],
                ],
              ),
            ),
          ],
        );
      },
    );

    /// Important to wipe [dialogToShow] otherwise the dialog gets redrawn
    /// every time build occurs for this widget (eg when screen size or
    /// orientation changes), resulting in multiple instances remaining
    /// on-screen.
    AppState.update('dialogToShow', {});
  }
}
